/* glickomania.c -- a puzzle game.

   Copyright (C) 2002 Kenneth Oksanen
   Copyright (C) 1992, 1995, 1996, 1997 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* AIX requires this to be the first thing in the file.  */
#if defined (_AIX) && !defined (__GNUC__)
 #pragma alloca
#endif

#include "includes.h"
#include "game_logic.h"


int height, width, n_tiles;


/* Current game history. */
static board_t game[(MAX_WIDTH + 1) * (MAX_HEIGHT + 1) / 2];
static int game_ix, game_ix_last;

column_t *board;


/* A precomputed array of solvable and interesting initial board
   positions.  (The program to compute these is NOT included in this
   distribution.) */

static struct {
  float difficulty;
  unsigned char width, height, n_tiles;
  unsigned int rng_kind : 1;
  unsigned int is_solved : 1;
  unsigned long seed;
} problem[] = {
  { 0.2323, 4, 4, 4, 0, 0, 103 },
  { 0.3673, 4, 4, 4, 0, 0, 29 },
  { 0.3787, 4, 4, 4, 0, 0, 62 },
  { 0.4253, 4, 4, 4, 0, 0, 6 },
  { 0.4311, 4, 4, 4, 0, 0, 20 },
  { 0.5102, 4, 4, 4, 0, 0, 86 },
  { 0.5487, 4, 4, 4, 0, 0, 78 },
  { 0.5562, 4, 4, 4, 0, 0, 69 },
  { 0.5711, 4, 4, 4, 0, 0, 38 },
  { 0.6944, 4, 4, 4, 0, 0, 50 },
  { 0.7160, 4, 4, 4, 0, 0, 14 },
  { 0.7257, 4, 4, 4, 0, 0, 105 },
  { 0.7511, 4, 4, 4, 0, 0, 61 },
  { 0.7656, 4, 4, 4, 0, 0, 108 },
  { 0.7901, 4, 4, 4, 0, 0, 89 },
  { 0.8789, 4, 4, 4, 0, 0, 102 },
  { 0.8858, 4, 4, 4, 0, 0, 94 },
  { 0.9070, 4, 4, 4, 0, 0, 40 },
  { 0.9652, 4, 4, 4, 0, 0, 109 },
  { 0.0161, 5, 5, 5, 0, 0, 310 },
  { 0.1438, 5, 5, 5, 0, 0, 102 },
  { 0.2500, 5, 5, 5, 0, 0, 80 },
  { 0.2637, 5, 5, 5, 0, 0, 35 },
  { 0.3427, 5, 5, 5, 0, 0, 58 },
  { 0.3489, 5, 5, 5, 0, 0, 77 },
  { 0.3691, 5, 5, 5, 0, 0, 57 },
  { 0.4218, 5, 5, 5, 0, 0, 134 },
  { 0.4760, 5, 5, 5, 0, 0, 48 },
  { 0.5239, 5, 5, 5, 0, 0, 82 },
  { 0.5713, 5, 5, 5, 0, 0, 85 },
  { 0.5907, 5, 5, 5, 0, 0, 50 },
  { 0.6068, 5, 5, 5, 0, 0, 125 },
  { 0.6944, 5, 5, 5, 0, 0, 63 },
  { 0.7091, 5, 5, 5, 0, 0, 39 },
  { 0.7901, 5, 5, 5, 0, 0, 7 },
  { 0.8158, 5, 5, 5, 0, 0, 101 },
  { 0.8464, 5, 5, 5, 0, 0, 64 },
  { 0.9176, 5, 5, 5, 0, 0, 103 },
  { 0.9562, 5, 5, 5, 0, 0, 95 },
  { 0.9596, 5, 5, 5, 0, 0, 115 },
  { 0.9649, 5, 5, 5, 0, 0, 459 },
  { 0.9931, 5, 5, 5, 0, 0, 33209 },
  { 0.0893, 6, 6, 7, 0, 0, 7966 },
  { 0.4630, 6, 6, 7, 0, 0, 48 },
  { 0.5325, 6, 6, 7, 0, 0, 693 },
  { 0.5533, 6, 6, 7, 0, 0, 14 },
  { 0.5598, 6, 6, 7, 0, 0, 180 },
  { 0.6193, 6, 6, 7, 0, 0, 1182 },
  { 0.6944, 6, 6, 7, 0, 0, 122 },
  { 0.7183, 6, 6, 7, 0, 0, 954 },
  { 0.7263, 6, 6, 7, 0, 0, 724 },
  { 0.7692, 6, 6, 7, 0, 0, 475 },
  { 0.8038, 6, 6, 7, 0, 0, 240 },
  { 0.8158, 6, 6, 7, 0, 0, 1196 },
  { 0.8166, 6, 6, 7, 0, 0, 229 },
  { 0.8169, 6, 6, 7, 0, 0, 757 },
  { 0.8366, 6, 6, 7, 0, 0, 775 },
  { 0.8461, 6, 6, 7, 0, 0, 974 },
  { 0.9049, 6, 6, 7, 0, 0, 706 },
  { 0.9140, 6, 6, 7, 0, 0, 38 },
  { 0.9155, 6, 6, 7, 0, 0, 960 },
  { 0.9261, 6, 6, 7, 0, 0, 859 },
  { 0.9502, 6, 6, 7, 0, 0, 1178 },
  { 0.9816, 6, 6, 7, 0, 0, 1205 },
  { 0.9675, 6, 6, 4, 0, 0, 68 },
  { 0.9517, 6, 6, 4, 0, 0, 138 },
  { 0.9538, 6, 6, 4, 0, 0, 203 },
  { 0.1498, 7, 7, 5, 0, 0, 0 },
  { 0.2399, 7, 7, 5, 0, 0, 2 },
  { 0.0546, 7, 7, 6, 0, 0, 27 },
  { 0.0793, 7, 7, 6, 0, 0, 28 },
  { 0.1255, 7, 7, 6, 0, 0, 31 },
  { 0.1259, 7, 7, 6, 0, 0, 61 },
  { 0.1465, 7, 7, 6, 0, 0, 51 },
  { 0.1600, 7, 7, 6, 0, 0, 30 },
  { 0.2304, 7, 7, 6, 0, 0, 2 },
  { 0.2761, 7, 7, 6, 0, 0, 55 },
  { 0.2776, 7, 7, 6, 0, 0, 44 },
  { 0.2954, 7, 7, 6, 0, 0, 29 },
  { 0.3031, 7, 7, 6, 0, 0, 19 },
  { 0.3265, 7, 7, 6, 0, 0, 10 },
  { 0.3673, 7, 7, 6, 0, 0, 58 },
  { 0.4100, 7, 7, 6, 0, 0, 15 },
  { 0.4444, 7, 7, 6, 0, 0, 52 },
  { 0.4521, 7, 7, 6, 0, 0, 69 },
  { 0.5074, 7, 7, 6, 0, 0, 12 },
  { 0.5797, 7, 7, 6, 0, 0, 64 },
  { 0.6087, 7, 7, 6, 0, 0, 59 },
  { 0.9350, 7, 7, 6, 0, 0, 54 },
  { 0.0010, 7, 7, 7, 0, 0, 203 },
  { 0.0071, 7, 7, 7, 0, 0, 215 },
  { 0.0285, 7, 7, 7, 0, 0, 392 },
  { 0.0784, 7, 7, 7, 0, 0, 86 },
  { 0.1128, 7, 7, 7, 0, 0, 222 },
  { 0.1259, 7, 7, 7, 0, 0, 77 },
  { 0.3710, 7, 7, 7, 0, 0, 302 },
  { 0.4148, 7, 7, 7, 0, 0, 241 },
  { 0.4918, 7, 7, 7, 0, 0, 137 },
  { 0.5524, 7, 7, 7, 0, 0, 331 },
  { 0.6470, 7, 7, 7, 0, 0, 433 },
  { 0.7322, 7, 7, 7, 0, 0, 32 },
  { 0.8220, 7, 7, 7, 0, 0, 242 },
  { 0.8264, 7, 7, 7, 0, 0, 80 },
  { 0.8690, 7, 7, 7, 0, 0, 71 },
  { -1 },
};


static unsigned int n_problems, n_unsolved_problems;
static unsigned int problem_ix, unsolved_problem_ix;


void problem_stats(char *buf, size_t n_bytes)
{
  snprintf(buf, n_bytes,
	   "  %u/%u solved.  This is #%u",
	   n_problems - n_unsolved_problems,
	   n_problems,
	   problem_ix + 1);
}


static void update_problem_stats(void)
{
  int i;

  n_problems = n_unsolved_problems = unsolved_problem_ix = 0;
  for (i = 0; problem[i].difficulty >= 0; i++) {
    if (!problem[i].is_solved) {
      n_unsolved_problems++;
      if (i <= problem_ix)
	unsolved_problem_ix++;
    }
    n_problems++;
  }
}


void save_problems(FILE *fp)
{
  int i;

  for (i = 0; problem[i].difficulty >= 0; i++)
    fprintf(fp, "1 %u %u %u %u %lu %u\n",
	    problem[i].width,
	    problem[i].height,
	    problem[i].n_tiles,
	    problem[i].rng_kind,
	    problem[i].seed,
	    problem[i].is_solved);
  fprintf(fp, "0\n");
}

int load_problems(FILE *fp)
{
  int i;
  
  for (i = 0; ; i++) {
    int ok, w, h, n, k, is, n_matched;
    unsigned long s;

    n_matched = fscanf(fp, "%d %u %u %u %u %lu %u\n",
		       &ok, &w, &h, &n, &k, &s, &is);
    if (problem[i].difficulty < 0
	&& n_matched == 1
	&& ok == 0)
      return 1;
    if (problem[i].difficulty < 0
	|| n_matched != 7
	|| !ok
	|| problem[i].width != w
	|| problem[i].height != h
	|| problem[i].n_tiles != n
	|| problem[i].rng_kind != k
	|| problem[i].seed != s)
      /* In future this function should be implemented so that almost
         arbitrary updates and changes in the problems table
         between different releases are possible. */
      return 0;
    problem[i].is_solved = is;
  }
}

void reset_problems(void)
{
  int i;

  for (i = 0; problem[i].difficulty >= 0; i++)
    problem[i].is_solved = 0;
}



static void setup_board(int nth)
{
  int x, y, kind = problem[nth].rng_kind;
  unsigned long seed = problem[nth].seed;

  width = problem[nth].width;
  height = problem[nth].height;
  n_tiles = problem[nth].n_tiles;
  game_ix = 0;

  for (x = 0; x < width; x++)
    for (y = height-1; y >= 0; y--) {
      switch (kind) {
      case 0:
	game[0][x][y] = (seed >> 8) % n_tiles;
	seed = (1103515245 * seed + 12345) & 0x7FFFFFFF;
	break;
      default:
	abort();
      }
    }
}


void first_problem(void)
{
  int i;

  /* Give the first unsolved initial position.  If all solved, then
     give the first initial position. */
  for (i = 0; problem[i].difficulty >= 0; i++)
    if (!problem[i].is_solved) {
      problem_ix = i;
      goto found;
    }
  problem_ix = 0;
 found:
  update_problem_stats();
  setup_board(problem_ix);
  board = &game[game_ix][0];
  game_ix = game_ix_last = 0;
}


int next_problem(void)
{
  if (problem[problem_ix+1].difficulty >= 0) {
    problem_ix++;
    update_problem_stats();
    setup_board(problem_ix);
    board = &game[game_ix][0];
    game_ix = game_ix_last = 0;
    return 1;
  }
  return 0;
}


int next_unsolved_problem(void)
{
  int i;

  for (i = problem_ix + 1;
       problem[i].difficulty >= 0 && problem[i].is_solved;
       i++)
    ;
  if (problem[i].difficulty >= 0) {
    problem_ix = i;
    update_problem_stats();
    setup_board(problem_ix);
    board = &game[game_ix][0];
    game_ix = game_ix_last = 0;
    return 1;
  }
  return 0;
}


int prev_problem(void)
{
  if (problem_ix > 0) {
    problem_ix--;
    update_problem_stats();
    setup_board(problem_ix);
    board = &game[game_ix][0];
    game_ix = game_ix_last = 0;
    return 1;
  }
  return 0;
}


int prev_unsolved_problem(void)
{
  int i;

  for (i = problem_ix - 1; i >= 0 && problem[i].is_solved; i--)
    ;
  if (i > 0) {
    problem_ix = i;
    update_problem_stats();
    setup_board(problem_ix);
    board = &game[game_ix][0];
    game_ix = game_ix_last = 0;
    return 1;
  }
  return 0;
}


/* Grouping and making moves. */

static unsigned char is_in_group[MAX_WIDTH][MAX_HEIGHT];
static unsigned int n_tiles_in_group;

static int rec_group(int x, int y, tile_t tile)
{
  if (x >= 0 && x < width
      && y >= 0 && y < height
      && board[x][y] == tile
      && !is_in_group[x][y]) {
    is_in_group[x][y] = 1;
    n_tiles_in_group++;
    rec_group(x-1, y, tile);
    rec_group(x+1, y, tile);
    rec_group(x, y-1, tile);
    rec_group(x, y+1, tile);
    return 1;
  }
  return 0;
}


int make_move(int x, int y)
{
  int to_x, to_y;
  tile_t tile;

  /* Some hedging in case there's a bug in the GUI... */
  if (x < 0 
      || x >= width 
      || y < 0 
      || y >= height)
    return 0;
  tile = board[x][y];
  if (tile == EMPTY)
    return 0;

  /* Clear old grouping. */
  memset(is_in_group, 0, sizeof(is_in_group));

  n_tiles_in_group = 1;
  is_in_group[x][y] = 1;

  rec_group(x-1, y, tile);
  rec_group(x+1, y, tile);
  rec_group(x, y-1, tile);
  rec_group(x, y+1, tile);
  
  if (n_tiles_in_group == 1)
    return 0;

  /* Ok, the clicked tile belongs to a group.  Now perform the
     dropping, so that in the process we copy the board at
     game[game_ix] to game[game_ix+1].  */
  for (to_x = x = 0; x < width; x++) {
    for (to_y = y = 0; y < height && game[game_ix][x][y] != EMPTY; y++)
      if (!is_in_group[x][y])
	game[game_ix+1][to_x][to_y++] = game[game_ix][x][y];
    if (to_y != 0) {
      while (to_y < width)
	game[game_ix+1][to_x][to_y++] = EMPTY;
      to_x++;
    }
  }
  while (to_x < width) {
    for (to_y = 0; to_y < height; to_y++)
      game[game_ix+1][to_x][to_y] = EMPTY;
    to_x++;
  }
  game_ix++;
  game_ix_last = game_ix;

  board = &game[game_ix][0];

  if (board[0][0] == EMPTY
      && !problem[problem_ix].is_solved) {
    /* The board is now cleared, mark it solved. */
    problem[problem_ix].is_solved = 1;
    update_problem_stats();
    if (n_unsolved_problems == 0)
      return 3;
    return 2;
  }

  return 1;
}


int unmake_move(void)
{
  if (game_ix > 0) {
    board = &game[--game_ix][0];
    return 1;
  }
  return 0;
}


int remake_move(void)
{
  if (game_ix < game_ix_last) {
    board = &game[++game_ix][0];
    return 1;
  }
  return 0;
}
