/*
  gdbm-put.c
  
  Jonathan B. Leffert <jonathan@leffert.net>
  24 April 2000
  
  $Id: gdbm-put.c,v 1.23 2000/05/15 21:37:37 jonathan Exp $
  
  Copyright (c) 2000 Jonathan B. Leffert <jonathan@leffert.net>
  All rights reserved.
*/

#include "common.h"

/* prototypes */
void parse_args(int *, char ***, char **);
void usage(char *);

/* function defs */

void parse_args(int *argc, char ***argv, char **dbName)
{
  char *options = "d:hv";
  int c;

  while ( (c = getopt(*argc, *argv, options)) != EOF )
    {
      switch (c)
	{
	case 'd':
	  /* database name */
	  *dbName = optarg;
	  break;
	case 'h':
	  /* request help */
	  usage((*argv)[0]);
	  exit(0);
	  break;
	case 'v':
	  /* version info */
	  printf("Version %f\n", VERSION);
	  exit(0);
	  break;
	case '?':
	  /* unknown option */
	  usage((*argv)[0]);
	  exit(1);
	  break;
	default:
	  /* default: do nothing */
	  break;
	}
    }
}

void usage(char *program_name)
{
  printf("Usage:\n");
  printf("\n");
  printf("%s [-hv] [-d DB] key value\n", program_name);
  printf("\n");
  printf("-d DB = Use database with specified name.\n");
  printf("        Must be supplied.\n");
  printf("-h    = This help message.\n");
  printf("-v    = Prints version information.\n");
  printf("\n");
  printf("This program puts the give key-value pair into the given\n");
  printf("database.  If the key already exists in the database, its\n");
  printf("value is overwritten.\n");
  printf("\n");
  printf("Error Codes:\n");
  printf("  0 on success\n");
  printf("  1 on non-database failure\n");
  printf("  2 on database failure\n");
}

int main(int argc, char **argv)
{
  char *name = NULL;
  GDBM_FILE db = NULL;
  datum key;
  datum value;
  int i;

  parse_args(&argc, &argv, &name);

  /*
    we expect a total of 5 things on the command line:

    o program
    o -d
    o database name
    o key
    o value

    so, argc should be 5, otherwise there is a problem
  */

  if ( (argc != 5) || (name == NULL) )
    {
      usage(argv[0]);
      exit(1);
    }
  else
    {
      /* open the database read write */
      db = open_gdbm_database_rw(name);

      if ( db != NULL )
	{
      
	  /* build the key and value objects */
	  key.dptr = argv[optind];
	  key.dsize = (strlen(key.dptr) + 1) * sizeof(char);
	  value.dptr = argv[optind+1];
	  value.dsize = (strlen(value.dptr) + 1) * sizeof(char);
	  
	  i = gdbm_store(db, key, value, GDBM_REPLACE);
	  if ( i == 0 )
	    {
	      /* success case: do nothing */
	    }
	  else if ( i < 0 )
	    {
	      /* db failure */
	      perror(argv[0]);
	      exit(2);
	    }
	  else if ( i == 1 )
	    {
	      /* key collision failure */
	      perror(argv[0]);
	      exit(2);
	    }
	  /* close the database */
	  gdbm_close(db);
	}
      else
	{
	  /* error occured */
	  perror(argv[0]);
	  exit(2);
	}
    }
  
  return 0;
}
