/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <signal.h>
#include <time.h>
#include <stdlib.h>

#include "progressbars_gui.h"
#include "vector_commands.h"
#include "common_gtk.h"
#include "modify_file_set.h"
#include "command.h"
#include "contractions.h"
#include "mainwindow.h"
#include "preferences.h"
#include "status_text.h"
#include "read_image.h"
#include "globals.h"

GtkWidget *progress_win = NULL, *info_label, *progress_bar, *progress_vbox, *stop_button;
GtkWidget *ask_win;
int bailed_out; /* this so cdrecord/mksisofs combination know we've quitted */

static GtkWidget *fifo_label;
static int sig;
static time_t start_time;
static int progress_created = FALSE;

void estimate_finnish_time_start(void) {
	if (((time_t)-1) == (start_time = time(NULL)))
		g_warning("%s::%i: time() failed", __FILE__, __LINE__);
}

void estimate_finnish_time_update(float p) {
	time_t time_now;
	double diff;
	if (((time_t)-1) == (time_now = time(NULL)))
		g_warning("%s::%i: time() failed", __FILE__, __LINE__);

	diff = difftime(time_now, start_time);

	/* When p is small, the estimate is way too optimistic, so some
	 * (quite arbitrary) correction is applied.  I'm not sure what's
	 * the cause of this error, it might just be that the task
	 * performed has been running for too short a time to produce any
	 * useful estimate. A better solution would perhaps be to no output
	 * at all for small values of p. */

	if (p < 0.02)
		p /= 2;

	if (0 != p) {
		char *buf;
		int sec, min;

		min = ((int) ((diff*(1.0 - p))/p)) / 60;
		sec = ((int) ((diff*(1.0 - p))/p)) % 60;

		if (0 != min)
			buf = g_strdup_printf(_("Done in: %d min, %d sec"), min, sec);
		else
			buf = g_strdup_printf(_("Done in %d sec"), sec);
		gtk_progress_set_format_string(GTK_PROGRESS(progress_bar), buf);
		gtk_progress_set_show_text(GTK_PROGRESS(progress_bar), TRUE);
		g_free(buf);
	}
}

void set_track_burning_now(int track) {
	char *title = g_strdup(N_("Burn in progress, please wait.\nTrack: 00 / 00 "));
	if (-1 == g_snprintf(title, strlen(title),
				_("Burn in progress, please wait.\nTrack: %02d / %02d"),
				track, no_of_tracks))
		g_warning("progressbars_gui.c::set_track_burning_now: Could not fit track data in info_label!");
	gtk_label_set(GTK_LABEL(info_label), title);
	free(title);
}

void set_track_burning_now_total(int track, int total) {
	char *title = g_strdup(N_("Burn in progress, please wait.\nTrack: 00 / 00 "));
	if (-1 == g_snprintf(title, strlen(title),
				_("Burn in progress, please wait.\nTrack: %02d / %02d"),
				track, total))
		g_warning("progressbars_gui.c::set_track_burning_now: Could not fit track data in info_label!");
	gtk_label_set(GTK_LABEL(info_label), title);
	free(title);
}

void set_track_dumping_now(int track, int total) {
	char *title = g_strdup(N_("Dump in progress, please wait.\nTrack: 00 / 00 "));
	if (-1 == g_snprintf(title, strlen(title),
				_("Dump in progress, please wait.\nTrack: %02d / %02d"),
				track, total))
		g_warning("progressbars_gui.c::set_track_dumping_now: Could not fit track data in info_label!");
	gtk_label_set(GTK_LABEL(info_label), title);

	free(title);
}

void set_mb_fifo_info(char *mb_fifo) {
	if ((NULL == mb_fifo) || (2 >= strlen(mb_fifo))) { /* shouldn't happen, but better make sure */
		g_warning("%s::%i: set_mb_fifo_info: mb_fifo is bogous!", __FILE__, __LINE__);
		return;
	}
	/* FIXME: better check here or before calling it; on errors the error message
	 *        gets appended to the mb_info */
	mb_fifo[strlen(mb_fifo) - 2] = ' '; /* don't want the trailing . */
	gtk_label_set(GTK_LABEL(fifo_label), mb_fifo);
}

void set_copy_info(char *speed, char *stats) {
	gtk_label_set(GTK_LABEL(fifo_label), speed);
	gtk_label_set(GTK_LABEL(info_label), stats);
}

static void bail_out(GtkWidget *widget, gpointer data) {

	gtk_widget_destroy(ask_win);
	ask_win = 0;

	if (0 != sig)
		gtk_signal_disconnect(GTK_OBJECT(progress_win), sig);
	sig = 0;

	if (cdr != 0)
		if (kill(-cdr, SIGINT) == -1)
			g_warning("%s::%i: cdr kill problemo!", __FILE__, __LINE__);
	if (iso != 0)
		if (kill(-iso, SIGINT) == -1)
			g_warning("%s::%i: iso kill problemo!", __FILE__, __LINE__);
	if (cd_file_child != 0) {
		if (kill(cd_file_child, SIGINT) == -1)
			g_warning("%s::%i: cd_file_child kill problemo!", __FILE__, __LINE__);
		status_pipe_ok = FALSE;
	}

	alert_user_of_error(_("Operation aborted."));

	bailed_out = TRUE;
}

static void dont_quit(GtkWidget *widget, gpointer data) {
	gtk_widget_destroy(ask_win);
	ask_win = 0;
}

static void really_wanna_quit(GtkWidget *widget, gpointer data) {
	GtkWidget *label, *yes, *no;

	if (ask_win)
		return;

	ask_win = gtk_dialog_new();
	gtk_widget_show(ask_win);

	label = gtk_label_new(_("Really sure you want to abort?"));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(ask_win)->vbox), label, TRUE,
			TRUE, 10);
	gtk_widget_show(label);

	yes = gtk_button_new_with_label(_("Yes"));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(ask_win)->action_area), yes, TRUE,
			TRUE, 10);
	gtk_widget_show(yes);
	no = gtk_button_new_with_label(_("No"));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(ask_win)->action_area), no, TRUE,
			TRUE, 10);
	gtk_widget_show(no);

	gtk_signal_connect(GTK_OBJECT(yes), "clicked",
			GTK_SIGNAL_FUNC(bail_out), (gpointer) ask_win);
	gtk_signal_connect(GTK_OBJECT(no), "clicked",
			GTK_SIGNAL_FUNC(dont_quit), (gpointer) ask_win);

}

static void create_progress_thing(char *title) {
	GtkWidget *hbox, *fill, *box;

	if (misc_prefs.own_progresswin)
		progress_win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	else {
		progress_win = gtk_frame_new(_("Status"));
		gtk_box_pack_start(GTK_BOX(status_vbox),
				progress_win, FALSE, FALSE, 0);
	}

	progress_vbox = gtk_vbox_new(0, 0);
	gtk_widget_show(progress_vbox);

	box = gtk_handle_box_new();
	gtk_widget_show(box);
	gtk_container_add(GTK_CONTAINER(box), progress_vbox);

	gtk_container_add(GTK_CONTAINER(progress_win), box);

	info_label = gtk_label_new(title);
	gtk_widget_show(info_label);
	gtk_box_pack_start(GTK_BOX(progress_vbox), info_label, TRUE, TRUE, 5);

	fifo_label = gtk_label_new("");
	gtk_widget_show(fifo_label);
	gtk_box_pack_start(GTK_BOX(progress_vbox), fifo_label, TRUE, TRUE, 5);

	progress_bar = gtk_progress_bar_new();
	gtk_widget_show(progress_bar);

	if (misc_prefs.own_progresswin) {
		gtk_box_pack_start(GTK_BOX(progress_vbox), progress_bar, TRUE, TRUE, 5);
	} else {
		gtk_box_pack_start(GTK_BOX(progress_vbox), progress_bar, TRUE, TRUE, 5);
		gtk_container_border_width(GTK_CONTAINER(progress_vbox), 5);
	}

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(progress_vbox), hbox, FALSE, FALSE, 0);

	fill = gtk_label_new("");
	gtk_widget_show(fill);
	gtk_box_pack_start(GTK_BOX(hbox), fill, TRUE, FALSE, 0);

	stop_button = gtk_button_new_with_label(_("STOP"));
	gtk_box_pack_start(GTK_BOX(hbox), stop_button, TRUE, TRUE, 10);
	gtk_widget_show(stop_button);

	fill = gtk_label_new("");
	gtk_widget_show(fill);
	gtk_box_pack_start(GTK_BOX(hbox), fill, TRUE, FALSE, 0);

	gtk_signal_connect(GTK_OBJECT(stop_button), "clicked",
			GTK_SIGNAL_FUNC(really_wanna_quit), (gpointer) progress_win);

	gtk_signal_handlers_destroy(GTK_OBJECT(progress_win));
	sig = gtk_signal_connect(GTK_OBJECT(progress_win), "delete_event",
                        GTK_SIGNAL_FUNC(really_wanna_quit), NULL);

	gtk_widget_show(progress_win);
}

GtkWidget *open_progress_win(char *title) {
	ask_win = 0;

	if (misc_prefs.own_progresswin) {
		create_progress_thing(title);
		return progress_win;
	} else {
		gtk_label_set_text(GTK_LABEL(info_label), title);
		gtk_widget_set_sensitive(progress_vbox, TRUE);
		return progress_win;
	}
}

void attach_progresswin(void) {
	if (!progress_created) {
		create_progress_thing("Idle");
		gtk_widget_set_sensitive(progress_vbox, FALSE);
		progress_created = TRUE;
	}
}

void detach_progresswin(void) {
	if (NULL != progress_win) {
		gtk_widget_destroy(progress_win);
		progress_win = NULL;
	}
	progress_created = FALSE;
}
