/*****************************************************************************
 *  FTools - Freenet Client Protocol Tools
 *
 *  Copyright (C) 2002 Juergen Buchmueller <juergen@pullmoll.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *	$Id: ftfcp.c,v 1.34 2005/06/24 11:59:27 pullmoll Exp $
 *****************************************************************************/
#include "ftfcp.h"
#include "ftlog.h"

#define	MAXBUFF	131072

typedef enum client_arg_e {
	ARG_NONE,
	ARG_BOOL,
	ARG_INT,
	ARG_STRING
}	client_arg_t;

typedef struct client_cmd_s {
	int type;			/* type of argument */
	const char *name;	/* FCP name */
	const void *arg;	/* FCP argument */
	const void *def;	/* default argument (don't send if arg == def) */
}	client_cmd_t;

void fcp_signal(int sig)
{
	signal(sig, SIG_DFL);
	if (SIGHUP == sig) {
		_exit(1);
	}
}

static int int_zero = 0;	/* default argument for ARG_INT == 0 */
static int bool_no = 0;		/* default argument for ARG_BOOL == false */
static int bool_yes = 1;	/* default argument for ARG_BOOL == true */

void set_signal(int sig, void (*my_signal_handler)(int))
{
	struct sigaction sa;
	signal(sig, my_signal_handler);
	sigaction(sig, (struct sigaction *)0, &sa);
#ifdef	SA_RESTART
	sa.sa_flags |= SA_RESTART;
#endif
#ifdef	SA_INTERRUPT
	sa.sa_flags &= ~SA_INTERRUPT;
#endif
	sigaction(sig, &sa, (struct sigaction *)0);
}

int fcp_new(fcp_t **pf, const char *hostname, int port)
{
	fcp_t *f = calloc(sizeof(fcp_t), 1);

	if (NULL == f) {
		return -1;
	}
	if (0 != sk_outgoing(&f->conn, hostname, port)) {
		free(f);
		*pf = NULL;
		return -1;
	}

	/* guess default SVK extension based on port number */
	if (8481 == port) {
		f->svk_extension = strdup("PAgM");
	} else {
		f->svk_extension = strdup("BCMA");
	}

	*pf = f;
	return 0;
}

int fcp_free(fcp_t **pf)
{
	fcp_t *f;

	if (NULL == pf) {
		return -1;
	}
	f = *pf;
	if (NULL == f) {
		return 0;
	}

	sk_shut(&f->conn);

	if (NULL != f->uri) {
		free(f->uri);
		f->uri = NULL;
	}
	if (NULL != f->public_key) {
		free(f->public_key);
		f->public_key = NULL;
	}
	if (NULL != f->private_key) {
		free(f->private_key);
		f->private_key = NULL;
	}
	if (NULL != f->chk) {
		free(f->chk);
		f->chk = NULL;
	}
	if (NULL != f->protocol) {
		free(f->protocol);
		f->node = NULL;
	}
	if (NULL != f->node) {
		free(f->node);
		f->node = NULL;
	}
	if (NULL != f->svk_extension) {
		free(f->svk_extension);
		f->svk_extension = NULL;
	}
	if (NULL != f->reason) {
		free(f->reason);
		f->reason = NULL;
	}
	if (NULL != f->meta) {
		free(f->meta);
		f->meta = NULL;
	}
	if (NULL != f->data) {
		free(f->data);
		f->data = NULL;
	}
	free(f);
	*pf = NULL;

	return 0;
}

int fcp_reset(fcp_t *f, int reset_data)
{
	if (NULL != f->uri) {
		free(f->uri);
		f->uri = NULL;
	}

	if (NULL != f->public_key) {
		free(f->public_key);
		f->public_key = NULL;
	}

	if (NULL != f->private_key) {
		free(f->private_key);
		f->private_key = NULL;
	}

	if (0 != reset_data) {
		f->metasize = 0;
		f->metaoffs = 0;
		if (NULL != f->meta) {
			free(f->meta);
			f->meta = NULL;
		}

		f->datasize = 0;
		f->dataoffs = 0;
		if (NULL != f->data) {
			free(f->data);
			f->data = NULL;
		}
	}

	if (NULL != f->protocol) {
		free(f->protocol);
		f->protocol = NULL;
	}
	if (NULL != f->node) {
		free(f->node);
		f->node = NULL;
	}
	if (NULL != f->reason) {
		free(f->reason);
		f->reason = NULL;
	}
	f->length = 0;

	f->seen_success = 0;
	f->seen_failed = 0;
	f->seen_pending = 0;
	f->seen_key_collision = 0;
	f->seen_size_error = 0;
	f->seen_end_message = 0;
	f->seen_data_found = 0;
	f->seen_uri_error = 0;
	f->seen_restarted = 0;
	f->seen_node_hello = 0;
	f->seen_node_get = 0;
	f->seen_node_put = 0;
	f->seen_data_not_found = 0;
	f->seen_route_not_found = 0;

	f->seen_data_chunk = 0;
	f->seen_data = 0;
	f->seen_data_complete = 0;

	memset(&f->sh, 0, sizeof(f->sh));

	return 0;
}

int fcp_send_cmds(pid_t pid, fcp_t *f, client_cmd_t *cmds, int count)
{
	client_cmd_t *cmd;
	const int *pint, *pdef;
	char arg[32];
	int i, rc;

	if (0 != sk_write(&f->conn, "\0\0\0\2", 4)) {
		LOG(L_ERROR,("sending fcp control failed (%s)\n",
			strerror(errno)));
		return -1;
	}
	LOG(L_DEBUG,("{%d} sent 00 00 00 02\n", pid));
	for (i = 0, cmd = cmds; i < count; i++, cmd++) {
		switch (cmd->type) {
		case ARG_NONE:
			rc = sk_printf(&f->conn, "%s\n", cmd->name);
			if (0 != rc) {
				LOG(L_ERROR,("{%d} sending command '%s' failed (%s)\n",
					pid, cmd->name, strerror(errno)));
				return -1;
			}
			LOG(L_DEBUG,("{%d} sent %s\n", pid, cmd->name));
			break;
		case ARG_BOOL:
			pint = cmd->arg;
			pdef = cmd->def;
			snprintf(arg, sizeof(arg), "%s",
				0 == *pint ? "no" : "yes");
			if (NULL != pdef && *pint == *pdef) {
				LOG(L_DEBUG,("{%d} skip %s=%s\n", pid, cmd->name, arg));
				break;
			}
			rc = sk_printf(&f->conn, "%s=%s\n", cmd->name, arg);
			if (0 != rc) {
				LOG(L_ERROR,("{%d} sending command '%s=%s' failed (%s)\n",
					pid, cmd->name, arg, strerror(errno)));
				return -1;
			}
			LOG(L_DEBUG,("{%d} sent %s=%s\n", pid, cmd->name, arg));
			break;
		case ARG_INT:
			pint = cmd->arg;
			pdef = cmd->def;
			snprintf(arg, sizeof(arg), "%x", *pint);
			if (NULL != pdef && *pint == *pdef) {
				LOG(L_DEBUG,("{%d} skip %s=%s\n", pid, cmd->name, arg));
				break;
			}
			rc = sk_printf(&f->conn, "%s=%s\n", cmd->name, arg);
			if (0 != rc) {
				LOG(L_ERROR,("{%d} sending command '%s=%s' failed (%s)\n",
					pid, cmd->name, arg, strerror(errno)));
				return -1;
			}
			LOG(L_DEBUG,("{%d} sent %s=%s\n", pid, cmd->name, arg));
			break;
		case ARG_STRING:
			rc = sk_printf(&f->conn, "%s=%s\n", cmd->name, cmd->arg);
			if (0 != rc) {
				LOG(L_ERROR,("{%d} sending command '%s=%s' failed (%s)\n",
					pid, cmd->name, cmd->arg, strerror(errno)));
				return -1;
			}
			LOG(L_DEBUG,("{%d} sent %s=%s\n", pid, cmd->name, cmd->arg));
			break;
		default:
			LOG(L_ERROR,("{%d} invalid cmd[%d]->type (%d)\n",
				pid, i, cmd->type));
			return -1;
		}
	}

	return 0;
}

int check_success(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "Success")) {
		f->seen_success = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd Success\n", pid));
		return 0;
	}
	return -1;
}

int check_failed(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "Failed")) {
		f->seen_failed = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_pending(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "Pending")) {
		f->seen_pending = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_restarted(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "Restarted")) {
		f->seen_restarted = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_key_collision(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "KeyCollision")) {
		f->seen_key_collision = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_size_error(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "SizeError")) {
		f->seen_size_error = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_uri_error(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "URIError")) {
		f->seen_uri_error = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_node_hello(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "NodeHello")) {
		f->seen_node_hello = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_node_get(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "NodeGet")) {
		f->seen_node_get = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_node_put(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "NodePut")) {
		f->seen_node_put = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_data_found(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "DataFound")) {
		f->seen_data_found = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_data_not_found(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "DataNotFound")) {
		f->seen_data_not_found = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_route_not_found(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "RouteNotFound")) {
		f->seen_route_not_found = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_data_chunk(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "DataChunk")) {
		f->seen_data_chunk = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_data(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "Data")) {
		f->seen_data = 1;
		f->seen_end_message = 0;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_segment_header(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "SegmentHeader")) {
		f->seen_segment_header = 1;
		f->seen_end_message = 0;
		memset(&f->sh0, 0, sizeof(f->sh0));
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_end_message(pid_t pid, fcp_t *f, char *var)
{
	if (0 == strcasecmp(var, "EndMessage")) {
		f->seen_end_message = 1;
		LOG(L_DEBUG,("{%d} rcvd %s\n", pid, var));
		return 0;
	}
	return -1;
}

int check_uri(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "URI")) {
		if (f->uri) {
			free(f->uri);
			f->uri = NULL;
		}
		if (NULL != val) {
			f->uri = strdup(val);
			if (strstr(val, "CHK@")) {
				if (f->chk) {
					free(f->chk);
					f->chk = NULL;
				}
				f->chk = strdup(val);
			}
		}
		LOG(L_DEBUG,("{%d} rcvd %s=%s\n", pid, var, f->uri));
		return 0;
	}
	return -1;
}

int check_public_key(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "PublicKey")) {
		if (f->public_key) {
			free(f->public_key);
			f->public_key = NULL;
		}
		f->public_key = strdup(val);
		LOG(L_DEBUG,("{%d} rcvd %s=%s\n", pid, var, f->public_key));
		return 0;
	}
	return -1;
}

int check_private_key(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "PrivateKey")) {
		if (f->private_key) {
			free(f->private_key);
			f->private_key = NULL;
		}
		f->private_key = strdup(val);
		LOG(L_DEBUG,("{%d} rcvd %s=%s\n", pid, var, f->private_key));
		return 0;
	}
	return -1;
}

int check_protocol(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "Protocol")) {
		if (f->protocol) {
			free(f->protocol);
			f->protocol = NULL;
		}
		f->protocol = strdup(val);
		LOG(L_DEBUG,("{%d} rcvd %s=%s\n", pid, var, f->protocol));
		return 0;
	}
	return -1;
}

int check_node(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "Node")) {
		if (f->node) {
			free(f->node);
			f->node = NULL;
		}
		f->node = strdup(val);
		LOG(L_DEBUG,("{%d} rcvd %s=%s\n", pid, var, f->node));
		return 0;
	}
	return -1;
}

int check_max_filesize(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "MaxFileSize")) {
		f->max_filesize = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->max_filesize));
		return 0;
	}
	return -1;
}

int check_max_hopstolive(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "MaxHopsToLive")) {
		f->max_htl = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->max_htl));
		return 0;
	}
	return -1;
}

int check_svk_extension(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "SVKExtension")) {
		if (NULL != f->svk_extension) {
			free(f->svk_extension);
			f->svk_extension = NULL;
		}
		f->svk_extension = strdup(val);
		LOG(L_DEBUG,("{%d} rcvd %s=%s\n", pid, var, f->svk_extension));
		return 0;
	}
	return -1;
}

int check_reason(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "Reason")) {
		if (f->reason) {
			free(f->reason);
			f->reason = NULL;
		}
		f->reason = strdup(val);
		LOG(L_DEBUG,("{%d} rcvd %s=%s\n", pid, var, f->reason));
		return 0;
	}
	return -1;
}

int check_timeout(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "Timeout")) {
		f->timeout = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->timeout));
		return 0;
	}
	return -1;
}

int check_data_length(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "DataLength")) {
		f->datasize = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->datasize));
		return 0;
	}
	return -1;
}

int check_meta_length(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "MetadataLength")) {
		f->metasize = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->metasize));
		return 0;
	}
	return -1;
}

int check_length(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "Length")) {
		f->length = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->length));
		return 0;
	}
	return -1;
}

int check_fec_algorithm(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "FECAlgorithm")) {
		strncpy(f->sh0.fec_algorithm, val, sizeof(f->sh0.fec_algorithm));
		LOG(L_DEBUG,("{%d} rcvd %s=%s\n", pid, var, f->sh0.fec_algorithm));
		return 0;
	}
	return -1;
}

int check_file_length(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "FileLength")) {
		f->sh0.file_length = strtoull(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%llx\n", pid, var, f->sh0.file_length));
		return 0;
	}
	return -1;
}

int check_offset(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "Offset")) {
		f->sh0.offset = strtoull(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%llx\n", pid, var, f->sh0.offset));
		return 0;
	}
	return -1;
}

int check_block_count(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "BlockCount")) {
		f->sh0.block_count = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->sh0.block_count));
		return 0;
	}
	return -1;
}

int check_block_size(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "BlockSize")) {
		f->sh0.block_size = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->sh0.block_size));
		return 0;
	}
	return -1;
}

int check_check_block_count(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "CheckBlockCount")) {
		f->sh0.check_block_count = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->sh0.check_block_count));
		return 0;
	}
	return -1;
}

int check_check_block_size(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "CheckBlockSize")) {
		f->sh0.check_block_size = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->sh0.check_block_size));
		return 0;
	}
	return -1;
}

int check_segments(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "Segments")) {
		f->sh0.segments = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->sh0.segments));
		return 0;
	}
	return -1;
}

int check_segment_num(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "SegmentNum")) {
		f->sh0.segment_num = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->sh0.segment_num));
		return 0;
	}
	return -1;
}

int check_blocks_required(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "BlocksRequired")) {
		f->sh0.blocks_required = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->sh0.blocks_required));
		return 0;
	}
	return -1;
}

int check_id(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "ID")) {
		f->id = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->id));
		return 0;
	}
	return -1;
}

int check_min_id(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "MinID")) {
		f->min_id = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->min_id));
		return 0;
	}
	return -1;
}

int check_max_id(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "MaxID")) {
		f->max_id = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->max_id));
		return 0;
	}
	return -1;
}

int check_sha1(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "SHA1")) {
		if (f->sha1)
			free(f->sha1);
		f->sha1 = strdup(val);
		LOG(L_DEBUG,("{%d} rcvd %s=%s\n", pid, var, f->sha1));
		return 0;
	}
	return -1;
}

int check_time(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "Time")) {
		f->time = strtoul(val, NULL, 16);
		LOG(L_DEBUG,("{%d} rcvd %s=%x\n", pid, var, f->time));
		return 0;
	}
	return -1;
}

int check_channel(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "Channel")) {
		if (NULL != f->channel) {
			free(f->channel);
			f->channel = NULL;
		}
		f->channel = strdup(val);
		LOG(L_DEBUG,("{%d} rcvd %s=%s\n", pid, var, f->channel));
		return 0;
	}
	return -1;
}

int check_sender(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "Sender")) {
		if (NULL != f->sender) {
			free(f->sender);
			f->sender = NULL;
		}
		f->sender = strdup(val);
		LOG(L_DEBUG,("{%d} rcvd %s=%s\n", pid, var, f->sender));
		return 0;
	}
	return -1;
}

int check_payload(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == strcasecmp(var, "Payload")) {
		if (NULL != f->payload) {
			free(f->payload);
			f->payload = NULL;
		}
		f->payload = strdup(val);
		LOG(L_DEBUG,("{%d} rcvd %s=%s\n", pid, var, f->payload));
		return 0;
	}
	return -1;
}

int check_reply(pid_t pid, fcp_t *f, char *var, char *val)
{
	if (0 == check_success(pid, f, var)) {
		return 0;
	} else if (0 == check_failed(pid, f, var)) {
		return 0;
	} else if (0 == check_pending(pid, f, var)) {
		return 0;
	} else if (0 == check_restarted(pid, f, var)) {
		return 0;
	} else if (0 == check_key_collision(pid, f, var)) {
		return 0;
	} else if (0 == check_size_error(pid, f, var)) {
		return 0;
	} else if (0 == check_uri_error(pid, f, var)) {
		return 0;
	} else if (0 == check_node_hello(pid, f, var)) {
		return 0;
	} else if (0 == check_node_get(pid, f, var)) {
		return 0;
	} else if (0 == check_node_put(pid, f, var)) {
		return 0;
	} else if (0 == check_data_found(pid, f, var)) {
		return 0;
	} else if (0 == check_data_not_found(pid, f, var)) {
		return 0;
	} else if (0 == check_route_not_found(pid, f, var)) {
		return 0;
	} else if (0 == check_data_chunk(pid, f, var)) {
		return 0;
	} else if (0 == check_data(pid, f, var)) {
		return 0;
	} else if (0 == check_end_message(pid, f, var)) {
		return 0;
	} else if (0 == check_uri(pid, f, var, val)) {
		return 0;
	} else if (0 == check_public_key(pid, f, var, val)) {
		return 0;
	} else if (0 == check_private_key(pid, f, var, val)) {
		return 0;
	} else if (0 == check_max_filesize(pid, f, var, val)) {
		return 0;
	} else if (0 == check_max_hopstolive(pid, f, var, val)) {
		return 0;
	} else if (0 == check_svk_extension(pid, f, var, val)) {
		return 0;
	} else if (0 == check_protocol(pid, f, var, val)) {
		return 0;
	} else if (0 == check_node(pid, f, var, val)) {
		return 0;
	} else if (0 == check_reason(pid, f, var, val)) {
		return 0;
	} else if (0 == check_timeout(pid, f, var, val)) {
		return 0;
	} else if (0 == check_data_length(pid, f, var, val)) {
		return 0;
	} else if (0 == check_meta_length(pid, f, var, val)) {
		return 0;
	} else if (0 == check_length(pid, f, var, val)) {
		return 0;
	} else if (0 == check_segment_header(pid, f, var)) {
		return 0;
	} else if (0 == check_fec_algorithm(pid, f, var, val)) {
		return 0;
	} else if (0 == check_file_length(pid, f, var, val)) {
		return 0;
	} else if (0 == check_offset(pid, f, var, val)) {
		return 0;
	} else if (0 == check_block_count(pid, f, var, val)) {
		return 0;
	} else if (0 == check_block_size(pid, f, var, val)) {
		return 0;
	} else if (0 == check_check_block_count(pid, f, var, val)) {
		return 0;
	} else if (0 == check_check_block_size(pid, f, var, val)) {
		return 0;
	} else if (0 == check_segments(pid, f, var, val)) {
		return 0;
	} else if (0 == check_segment_num(pid, f, var, val)) {
		return 0;
	} else if (0 == check_blocks_required(pid, f, var, val)) {
		return 0;
	} else if (0 == check_id(pid, f, var, val)) {
		return 0;
	} else if (0 == check_min_id(pid, f, var, val)) {
		return 0;
	} else if (0 == check_max_id(pid, f, var, val)) {
		return 0;
	} else if (0 == check_sha1(pid, f, var, val)) {
		return 0;
	} else if (0 == check_time(pid, f, var, val)) {
		return 0;
	} else if (0 == check_channel(pid, f, var, val)) {
		return 0;
	} else if (0 == check_sender(pid, f, var, val)) {
		return 0;
	} else if (0 == check_payload(pid, f, var, val)) {
		return 0;
	}

	return -1;
}

int fcp_hello(fcp_t *f)
{
	pid_t pid = getpid();
	char *reply = calloc(MAXBUFF, 1);
	char *var, *val, *eol;
	int rc = -1;

	client_cmd_t cmds[] = {
		{ARG_NONE,	"ClientHello",	NULL,	NULL},
		{ARG_NONE,	"EndMessage",	NULL,	NULL}
	};

	(void)cmds;

	if (NULL == reply) {
		LOG(L_ERROR,("{%d} barf: calloc(%d,%d) failure (%s)\n",
			pid, MAXBUFF, 1, strerror(errno)));
		goto bailout;
	}
	if (0 != fcp_send_cmds(pid, f, cmds, sizeof(cmds)/sizeof(cmds[0]))) {
		LOG(L_DEBUG,("{%d} sending cmds failed\n", pid));
		goto bailout;
	}

	fcp_reset(f, 1);
	while (0 == (rc = sk_gets(&f->conn, reply, MAXBUFF)) &&
		0 == f->seen_end_message) {
		eol = strchr(reply, '\r');
		if (NULL == eol)
			eol = strchr(reply, '\n');
		if (NULL != eol)
			*eol = '\0';
		var = val = reply;
		while (*val != '\0' && *val != '=')
			val++;
		if (*val == '=')
			*val++ = '\0';
		if (0 != check_reply(pid, f, var, val)) {
			LOG(L_ERROR,("{%d} rcvd NOT HANDLED %s=%s\n",
				pid, var, val));
		}
	}
	rc = (f->seen_node_hello && f->seen_end_message) ? 0 : -1;

bailout:
	if (NULL != reply) {
		free(reply);
		reply = NULL;
	}
	return rc;
}

int fcp_key(fcp_t *f)
{
	pid_t pid = getpid();
	char *reply = calloc(MAXBUFF, 1);
	char *var, *val, *eol;
	int rc = -1;

	client_cmd_t cmds[] = {
		{ARG_NONE,	"GenerateSVKPair",	NULL,	NULL},
		{ARG_NONE,	"EndMessage",		NULL,	NULL}
	};

	(void)cmds;

	if (NULL == reply) {
		LOG(L_ERROR,("{%d} barf: calloc(%d,%d) failure (%s)\n",
			pid, MAXBUFF, 1, strerror(errno)));
		goto bailout;
	}
	if (0 != fcp_send_cmds(pid, f, cmds, sizeof(cmds)/sizeof(cmds[0]))) {
		LOG(L_DEBUG,("{%d} sending cmds failed\n", pid));
		goto bailout;
	}

	fcp_reset(f, 0);
	while (0 == (rc = sk_gets(&f->conn, reply, MAXBUFF)) &&
		0 == f->seen_end_message) {
		eol = strchr(reply, '\r');
		if (NULL == eol)
			eol = strchr(reply, '\n');
		if (NULL != eol)
			*eol = '\0';
		var = val = reply;
		while (*val != '\0' && *val != '=')
			val++;
		if (*val == '=')
			*val++ = '\0';
		if (0 != check_reply(pid, f, var, val)) {
			LOG(L_ERROR,("{%d} rcvd NOT HANDLED %s=%s\n",
				pid, var, val));
		}
	}
	rc = (f->seen_success && f->seen_end_message) ? 0 : -1;

bailout:
	if (NULL != reply) {
		free(reply);
		reply = NULL;
	}
	return rc;
}

int fcp_get(fcp_t *f, const char *uri, int slimit, int htl)
{
	pid_t pid = getpid();
	char *reply = calloc(MAXBUFF, 1);
	char *var, *val, *eol;
	int rc = -1;

	client_cmd_t cmds[] = {
		{ARG_NONE,	"ClientGet",		NULL,		NULL},
		{ARG_STRING,"URI",				uri,		NULL},
		{ARG_INT,	"HopsToLive",		&htl,		&int_zero},
		{ARG_INT,	"MaxFileSize",		&slimit,	&int_zero},
		{ARG_BOOL,	"RemoveLocalKey",	&delete,	&bool_no},
		{ARG_NONE,	"EndMessage",		NULL,		NULL}
	};

	(void)cmds;

	if (NULL == reply) {
		LOG(L_ERROR,("barf: calloc(%d,%d) failure (%s)\n",
			MAXBUFF, 1, strerror(errno)));
		goto bailout;
	}
	if (0 != fcp_send_cmds(pid, f, cmds, sizeof(cmds)/sizeof(cmds[0]))) {
		LOG(L_DEBUG,("{%d} sending cmds failed\n", pid));
		goto bailout;
	}

	do {
		fcp_reset(f, 1);

		while (0 == (rc = sk_gets(&f->conn, reply, MAXBUFF))) {
			eol = strchr(reply, '\r');
			if (NULL == eol)
				eol = strchr(reply, '\n');
			if (NULL != eol)
				*eol = '\0';
			var = val = reply;
			while (*val != '\0' && *val != '=')
				val++;
			if (*val == '=')
				*val++ = '\0';
			if (0 != check_reply(pid, f, var, val)) {
				LOG(L_ERROR,("{%d} rcvd NOT HANDLED %s=%s\n",
					pid, var, val));
			}
			if (0 != f->seen_end_message) {
				if (0 != f->seen_data_found)
					break;
				if (0 != f->seen_failed)
					break;
				if (0 != f->seen_uri_error)
					break;
				if (0 != f->seen_data_not_found)
					break;
				if (0 != f->seen_route_not_found)
					break;
				if (0 != f->seen_restarted)
					break;
			}
		}

		if (0 != rc) {
			LOG(L_DEBUG,("bail: rc = %d\n", rc));
			goto bailout;
		}
		if (0 == f->seen_end_message) {
			LOG(L_DEBUG,("bail: did not see EndMessage\n"));
			rc = -1;
			goto bailout;
		}
		if (0 != f->seen_failed) {
			LOG(L_DEBUG,("bail: seen Failed '%s'\n",
				f->reason));
			rc = -1;
			goto bailout;
		}
		if (0 != f->seen_uri_error) {
			LOG(L_DEBUG,("bail: seen URIError\n"));
			rc = -1;
			goto bailout;
		}
		if (0 != f->seen_data_not_found) {
			LOG(L_DEBUG,("bail: seen DataNotFound\n"));
			rc = -1;
			goto bailout;
		}
		if (0 != f->seen_route_not_found) {
			LOG(L_DEBUG,("bail: seen RouteNotFound\n"));
			rc = -1;
			goto bailout;
		}

		if (f->metasize > 0) {
			f->meta = calloc(f->metasize + 31, 1);
			if (NULL == f->meta) {
				LOG(L_DEBUG,("bail: meta alloc(%d) failed\n", f->metasize));
				goto bailout;
			}
		}
		if (f->datasize > 0) {
			if (f->metasize > f->datasize) {
				LOG(L_DEBUG,("bail: metasize > datasize\n"));
				goto bailout;
			}
			f->datasize -= f->metasize;
			if (f->datasize > 0) {
				f->data = calloc(f->datasize + 31, 1);
				if (NULL == f->data) {
					LOG(L_DEBUG,("bail: data alloc(%d) failed\n", f->datasize));
					goto bailout;
				}
			}
		}

		f->seen_restarted = 0;
		f->seen_data_complete = 0;
		f->seen_end_message = 0;
		while (0 == f->seen_restarted && 0 == f->seen_data_complete) {
			f->seen_data_chunk = 0;
			f->seen_data = 0;
			f->length = 0;
			while (0 == (rc = sk_gets(&f->conn, reply, MAXBUFF))) {
				eol = strchr(reply, '\r');
				if (NULL == eol)
					eol = strchr(reply, '\n');
				if (NULL != eol)
					*eol = '\0';
				var = val = reply;
				while (*val != '\0' && *val != '=')
					val++;
				if (*val == '=')
					*val++ = '\0';
				if (0 != check_reply(pid, f, var, val)) {
					LOG(L_ERROR,("{%d} rcvd NOT HANDLED %s=%s\n",
						pid, var, val));
				}
				if (0 != f->seen_end_message || 0 != f->seen_data) {
					if (0 != f->seen_data_chunk)
						break;
					if (0 != f->seen_restarted)
						break;
				}
			}
			if (0 != f->seen_restarted) {
				LOG(L_DEBUG,("bail: seen Restarted\n"));
				continue;
			}
			if (0 != f->seen_failed) {
				LOG(L_DEBUG,("bail: seen Failed\n"));
				continue;
			}
			if (0 == f->seen_data_chunk) {
				LOG(L_DEBUG,("bail: did not see DataChunk\n"));
				rc = -1;
				goto bailout;
			}
			if (0 == f->length) {
				LOG(L_DEBUG,("bail: did not see Length=\n"));
				rc = -1;
				goto bailout;
			}
			if (0 == f->seen_data) {
				LOG(L_DEBUG,("bail: did not see Data\n"));
				rc = -1;
				goto bailout;
			}
			if (f->metaoffs < f->metasize) {
				size_t size = f->length;
				if (size > f->metasize - f->metaoffs) {
					size = f->metasize - f->metaoffs;
				}
				LOG(L_DEBUG,("reading max 0x%x bytes metadata\n",
					size));
				if (0 != sk_read(&f->conn, &f->meta[f->metaoffs], size)) {
					LOG(L_DEBUG,("bail: read(%d) failed\n", size));
					rc = -1;
					goto bailout;
				}
				f->metaoffs += size;
				f->length -= size;
				LOG(L_DEBUG,("{%d} rcvd [%x bytes metadata]\n",
					pid, size));
				if (0 == f->datasize && f->metaoffs >= f->metasize) {
					LOG(L_DEBUG,("{%d} data complete\n",
						pid));
					f->seen_data_complete = 1;
				}
			}
			if (f->dataoffs < f->datasize) {
				size_t size = f->length;
				if (size > f->datasize - f->dataoffs) {
					size = f->datasize - f->dataoffs;
				}
				LOG(L_DEBUG,("reading max 0x%x bytes data\n",
					size));
				if (0 != sk_read(&f->conn, &f->data[f->dataoffs], size)) {
					LOG(L_DEBUG,("bail: read(%d) failed\n", size));
					rc = -1;
					goto bailout;
				}
				f->dataoffs += size;
				f->length -= size;
				LOG(L_DEBUG,("{%d} rcvd [%x bytes data]\n",
					pid, size));
				if (f->dataoffs >= f->datasize) {
					LOG(L_DEBUG,("{%d} data complete\n",
						pid));
					f->seen_data_complete = 1;
				}
			}
		}
	} while (0 != f->seen_restarted);

	rc = 0;
bailout:
	if (NULL != reply) {
		free(reply);
		reply = NULL;
	}
	LOG(L_DEBUG,("{%d} return %d\n",
		pid, rc));
	return rc;
}

int fcp_put(fcp_t *f, const char *uri, int htl)
{
	pid_t pid = getpid();
	char *reply = calloc(MAXBUFF, 1);
	char *var, *val, *eol;
	size_t size;
	int datalength, metalength;
	int rc = -1;

	client_cmd_t cmds[] = {
		{ARG_NONE,	"ClientPut",		NULL,		NULL},
		{ARG_STRING,"URI",				uri,		NULL},
		{ARG_INT,	"HopsToLive",		&htl,		&int_zero},
		{ARG_BOOL,	"RemoveLocalKey",	&delete,	&bool_no},
		{ARG_BOOL,	"MetadataOnly",		&meta_only,	&bool_no},
		{ARG_INT,	"DataLength",		&datalength,NULL},
		{ARG_INT,	"MetaDatalength",	&metalength,&int_zero},
		{ARG_NONE,	"Data",				NULL,		NULL}
	};

	(void)cmds;

	set_signal(SIGPIPE, fcp_signal);

	if (NULL == reply) {
		LOG(L_ERROR,("{%d} barf: calloc(%d,%d) failure (%s)\n",
			pid, MAXBUFF, 1, strerror(errno)));
		goto bailout;
	}
	if (0 != sk_write(&f->conn, "\0\0\0\2", 4)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent 00 00 00 02\n", pid));

	if (0 != sk_printf(&f->conn, "ClientPut\n")) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent ClientPut\n", pid));

	if (htl > 0) {
		if (0 != sk_printf(&f->conn, "HopsToLive=%x\n", htl)) {
			goto bailout;
		}
		LOG(L_DEBUG,("{%d} sent HopsToLive=%x\n", pid, htl));
	}

	if (0 != sk_printf(&f->conn, "URI=%s\n", uri)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent URI=%s\n", pid, uri));

	size = f->datasize + f->metasize;
	if (0 != sk_printf(&f->conn, "DataLength=%x\n", size)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent DataLength=%x\n", pid, size));

	if (f->metasize > 0) {
		size = f->metasize;
		if (0 != sk_printf(&f->conn, "MetadataLength=%x\n", size)) {
			goto bailout;
		}
		LOG(L_DEBUG,("{%d} sent MetadataLength=%x\n", pid, size));
	}

	if (0 != delete) {
		if (0 != sk_printf(&f->conn, "RemoveLocalKey=%s\n", "yes")) {
			goto bailout;
		}
		LOG(L_DEBUG,("{%d} sent RemoveLocalKey=%s\n", pid, "yes"));
	}

	if (0 != meta_only) {
		if (0 != sk_printf(&f->conn, "MetadataOnly=%s\n", "yes")) {
			goto bailout;
		}
		LOG(L_DEBUG,("{%d} sent MetadataOnly=%s\n", pid, "yes"));
	}

	if (0 != sk_printf(&f->conn, "Data\n")) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent Data\n", pid));

	if (f->metasize > 0) {
		if (0 != (rc = sk_write(&f->conn, f->meta, f->metasize))) {
			goto bailout;
		}
		LOG(L_DEBUG,("{%d} sent [0x%x bytes metadata]\n", pid, f->metasize));
	}

	if (f->datasize > 0) {
		if (0 != (rc = sk_write(&f->conn, f->data, f->datasize))) {
			goto bailout;
		}
		LOG(L_DEBUG,("{%d} sent [0x%x bytes data]\n", pid, f->datasize));
	}

	fcp_reset(f, 0);
	while (0 == (rc = sk_gets(&f->conn, reply, MAXBUFF))) {
		eol = strchr(reply, '\r');
		if (NULL == eol)
			eol = strchr(reply, '\n');
		if (NULL != eol)
			*eol = '\0';
		var = val = reply;
		while (*val != '\0' && *val != '=')
			val++;
		if (*val == '=')
			*val++ = '\0';

		if (0 != check_reply(pid, f, var, val)) {
			LOG(L_ERROR,("{%d} rcvd NOT HANDLED %s=%s\n",
				pid, var, val));
		}
		if (0 != f->seen_end_message) {
			if (0 != f->seen_success)
				break;
			if (0 != f->seen_failed)
				break;
			if (0 != f->seen_uri_error)
				break;
			if (0 != f->seen_route_not_found)
				break;
			if (0 != f->seen_key_collision)
				break;
			if (0 != f->seen_size_error)
				break;
		}
	}

	rc = -1;
	if (0 == f->seen_end_message) {
		LOG(L_DEBUG,("bail: did not see EndMessage\n"));
		goto bailout;
	}
	if (0 != f->seen_failed) {
		LOG(L_DEBUG,("bail: seen Failed '%s'\n", f->reason));
		goto bailout;
	}
	if (0 != f->seen_uri_error) {
		LOG(L_DEBUG,("bail: seen URIError\n"));
		goto bailout;
	}
	if (0 != f->seen_data_not_found) {
		LOG(L_DEBUG,("bail: seen DataNotFound\n"));
		goto bailout;
	}
	if (0 != f->seen_route_not_found) {
		LOG(L_DEBUG,("bail: seen RouteNotFound\n"));
		goto bailout;
	}
	rc = 0;

bailout:
	if (NULL != reply) {
		free(reply);
		reply = NULL;
	}
	return rc;
}

int fcp_chk(fcp_t *f)
{
	pid_t pid = getpid();
	char *reply = calloc(MAXBUFF, 1);
	char *var, *val, *eol;
	size_t size;
	int datalength, metalength;
	int rc = -1;

	client_cmd_t cmds[] = {
		{ARG_NONE,	"GenerateCHK",		NULL,		NULL},
		{ARG_INT,	"DataLength",		&datalength,NULL},
		{ARG_INT,	"MetaDatalength",	&metalength,&int_zero},
		{ARG_NONE,	"Data",				NULL,		NULL}
	};

	(void)cmds;

	if (NULL == reply) {
		LOG(L_ERROR,("{%d} barf: calloc(%d,%d) failure (%s)\n",
			pid, MAXBUFF, 1, strerror(errno)));
		goto bailout;
	}
	if (0 != sk_write(&f->conn, "\0\0\0\2", 4)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent 00 00 00 02\n", pid));

	if (0 != sk_printf(&f->conn, "GenerateCHK\n")) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent GenerateCHK\n", pid));

	size = f->datasize + f->metasize;
	if (0 != sk_printf(&f->conn, "DataLength=%x\n", size)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent DataLength=%x\n",
		pid, size));

	if (f->metasize > 0) {
		size = f->metasize;
		if (0 != sk_printf(&f->conn, "MetadataLength=%x\n", size)) {
			goto bailout;
		}
		LOG(L_DEBUG,("{%d} sent MetadataLength=%x\n", pid, size));
	}
	if (0 != sk_printf(&f->conn, "Data\n")) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent Data\n", pid));

	if (f->metasize > 0) {
		if (0 != (rc = sk_write(&f->conn, f->meta, f->metasize))) {
			goto bailout;
		}
		LOG(L_DEBUG,("{%d} sent metadata (%x bytes)\n", pid, f->metasize));
	}

	if (f->datasize > 0) {
		if (0 != (rc = sk_write(&f->conn, f->data, f->datasize))) {
			goto bailout;
		}
		LOG(L_DEBUG,("{%d} sent data (%x bytes)\n", pid, f->datasize));
	}

	f->seen_success = 0;
	f->seen_restarted = 0;
	f->seen_pending = 0;
	if (NULL != f->public_key) {
		free(f->public_key);
		f->public_key = NULL;
	}
	if (NULL != f->private_key) {
		free(f->private_key);
		f->private_key = NULL;
	}
	f->seen_end_message = 0;
	while (0 == (rc = sk_gets(&f->conn, reply, MAXBUFF))) {
		eol = strchr(reply, '\r');
		if (NULL == eol)
			eol = strchr(reply, '\n');
		if (NULL != eol)
			*eol = '\0';
		var = val = reply;
		while (*val != '\0' && *val != '=')
			val++;
		if (*val == '=')
			*val++ = '\0';

		if (0 != check_reply(pid, f, var, val)) {
			LOG(L_ERROR,("{%d} rcvd NOT HANDLED %s=%s\n",
				pid, var, val));
		}
		if (0 != f->seen_end_message) {
			if (0 != f->seen_success)
				break;
		}
	}

	if (0 != rc) {
		LOG(L_DEBUG,("bail: rc = %d\n", rc));
		goto bailout;
	}

	if (0 == f->seen_end_message) {
		LOG(L_DEBUG,("bail: did not see EndMessage\n"));
		rc = -1;
		goto bailout;
	}

bailout:
	if (NULL != reply) {
		free(reply);
		reply = NULL;
	}
	return rc;
}

int fcp_inv(fcp_t *f, const char *private_ssk, char *public_ssk)
{
	pid_t pid = getpid();
	char *reply = calloc(MAXBUFF, 1);
	char *var, *val, *eol;
	int rc = -1;

	client_cmd_t cmds[] = {
		{ARG_NONE,	"InvertPrivateKey",	NULL,		NULL},
		{ARG_STRING,"Private",			private_ssk,NULL},
		{ARG_NONE,	"EndMessage",		NULL,		NULL}
	};

	(void)cmds;

	if (NULL == reply) {
		LOG(L_ERROR,("{%d} barf: calloc(%d,%d) failure (%s)\n",
			pid, MAXBUFF, 1, strerror(errno)));
		goto bailout;
	}
	if (0 != fcp_send_cmds(pid, f, cmds, sizeof(cmds)/sizeof(cmds[0]))) {
		LOG(L_DEBUG,("{%d} sending cmds failed\n", pid));
		goto bailout;
	}

	f->seen_success = 0;
	f->seen_restarted = 0;
	f->seen_pending = 0;
	if (NULL != f->public_key) {
		free(f->public_key);
		f->public_key = NULL;
	}
	if (NULL != f->private_key) {
		free(f->private_key);
		f->private_key = NULL;
	}
	f->seen_end_message = 0;
	while (0 == (rc = sk_gets(&f->conn, reply, MAXBUFF))) {
		eol = strchr(reply, '\r');
		if (NULL == eol)
			eol = strchr(reply, '\n');
		if (NULL != eol)
			*eol = '\0';
		var = val = reply;
		while (*val != '\0' && *val != '=')
			val++;
		if (*val == '=')
			*val++ = '\0';

		if (0 != check_reply(pid, f, var, val)) {
			LOG(L_ERROR,("{%d} rcvd NOT HANDLED %s=%s\n",
				pid, var, val));
		}
		if (0 != f->seen_end_message) {
			if (0 != f->seen_success)
				break;
		}
	}

	if (0 != rc) {
		LOG(L_DEBUG,("bail: rc = %d\n", rc));
		goto bailout;
	}

	if (0 == f->seen_end_message) {
		LOG(L_DEBUG,("bail: did not see EndMessage\n"));
		rc = -1;
		goto bailout;
	}
	strcpy(public_ssk, f->public_key);

bailout:
	if (NULL != reply) {
		free(reply);
		reply = NULL;
	}
	return rc;
}

int fcp_del(fcp_t *f, const char *uri)
{
	pid_t pid = getpid();
	char *reply = calloc(MAXBUFF, 1);
	char *var, *val, *eol;
	int rc = -1;

	client_cmd_t cmds[] = {
		{ARG_NONE,	"ClientGet",		NULL,		NULL},
		{ARG_STRING,"URI",				uri,		NULL},
		{ARG_BOOL,	"RemoveLocalKey",	&delete,	&bool_yes},
		{ARG_NONE,	"EndMessage",		NULL,		NULL}
	};

	if (NULL == reply) {
		LOG(L_ERROR,("{%d} barf: calloc(%d,%d) failure (%s)\n",
			pid, MAXBUFF, 1, strerror(errno)));
		goto bailout;
	}
	if (0 != fcp_send_cmds(pid, f, cmds, sizeof(cmds)/sizeof(cmds[0]))) {
		LOG(L_DEBUG,("{%d} sending cmds failed\n", pid));
		goto bailout;
	}

	fcp_reset(f, 0);
	while (0 == (rc = sk_gets(&f->conn, reply, MAXBUFF))) {
		eol = strchr(reply, '\r');
		if (NULL == eol)
			eol = strchr(reply, '\n');
		if (NULL != eol)
			*eol = '\0';
		var = val = reply;
		while (*val != '\0' && *val != '=')
			val++;
		if (*val == '=')
			*val++ = '\0';

		if (0 != check_reply(pid, f, var, val)) {
			LOG(L_ERROR,("{%d} rcvd NOT HANDLED %s=%s\n",
				pid, var, val));
		}
		if (0 != f->seen_end_message) {
			if (0 != f->seen_success)
				break;
		}
	}

	if (0 != rc) {
		LOG(L_DEBUG,("bail: rc = %d\n", rc));
		goto bailout;
	}
	if (0 == f->seen_end_message) {
		LOG(L_DEBUG,("bail: did not see EndMessage\n"));
		rc = -1;
		goto bailout;
	}

bailout:
	if (NULL != reply) {
		free(reply);
		reply = NULL;
	}
	return rc;
}

int fcp_encode_segment(fcp_t *f, size_t i, FILE *fp, u_int64_t file_length)
{
	pid_t pid = getpid();
	segment_header_t *sh = &f->sh[i];
	char *buff = malloc(MAXBUFF);
	char *reply = calloc(1,MAXBUFF);
	char *var, *val, *eol;
	size_t offs, datalen;
	size_t block_count = 0, block_size = 0;
	int rc = -1;

	printf("Encoding segment #%d/%d, offset %llx, size %llx\n",
		i, sh->segments, sh->offset, (u_int64_t)sh->block_count * sh->block_size);

	f->metasize = MAXBUFF;
	f->meta = calloc(f->metasize + 31, 1);
	if (NULL == f->meta) {
		LOG(L_DEBUG,("bail: meta alloc(%d) failed\n", f->metasize));
		goto bailout;
	}

	f->metasize = snprintf(f->meta, f->metasize,
		"SegmentHeader\n" \
		"FECAlgorithm=%s\n" \
		"FileLength=%llx\n" \
		"Offset=%llx\n" \
		"BlockCount=%x\n" \
		"BlockSize=%x\n" \
		"CheckBlockCount=%x\n" \
		"CheckBlockSize=%x\n" \
		"Segments=%x\n" \
		"SegmentNum=%x\n" \
		"BlocksRequired=%x\n" \
		"EndMessage\n",
		sh->fec_algorithm, sh->file_length, sh->offset,
		sh->block_count, sh->block_size,
		sh->check_block_count, sh->check_block_size,
		sh->segments, sh->segment_num, sh->blocks_required);

	datalen = sh->block_count * sh->block_size;
	if (0 != sk_write(&f->conn, "\0\0\0\2", 4)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent 00 00 00 02\n", pid));

	if (0 != sk_printf(&f->conn, "FECEncodeSegment\n")) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent FECEncodeSegment\n", pid));

#if 0
	if (0 != sk_printf(&f->conn, "RequestedLists=\n")) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent RequestedList=\n", pid));
#endif
	if (0 != sk_printf(&f->conn, "DataLength=%x\n", f->metasize + datalen)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent DataLength=%x\n", pid, f->metasize + datalen));

#if	0	/* This is stupid, but the metadata size is not specified */
	if (0 != sk_printf(&f->conn, "MetadataLength=%x\n", f->metasize)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent MetadataLength=%x\n", pid, f->metasize));
#endif
	if (0 != sk_printf(&f->conn, "Data\n")) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent Data\n", pid));

	if (0 != sk_write(&f->conn, f->meta, f->metasize)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent SegmentHeader (%x bytes)\n", pid, f->metasize));

	/* send down datalen bytes to the node */
	LOG(L_NORMAL,("{%d} sending data blocks (%x bytes)\n",
		pid, datalen));
	for (offs = 0; offs < datalen; /* */) {
		size_t done, size;

		size = MAXBUFF;
		if (offs >= file_length) {
			if (offs + MAXBUFF > datalen) {
				size = datalen - offs;
			}
			/* pad with zeroes */
			memset(buff, 0, size);
			done = size;
		} else {
			if (offs + MAXBUFF > file_length) {
				size = file_length - offs;
			}
			if (size != (done = fread(buff, 1, size, fp))) {
				LOG(L_ERROR,("fread(%x,%x,%x,%x) call failed (%s)\n",
					buff, 1, size, fp, strerror(errno)));
				rc = -1;
				goto bailout;
			}
		}
		if (0 != (rc = sk_write(&f->conn, buff, done))) {
			LOG(L_ERROR,("sk_write(%x,%x,%x) call failed (%s)\n",
				f, buff, done, strerror(errno)));
			goto bailout;
		}
		offs += size;
	}
	LOG(L_MINOR,("{%d} sent data (%x bytes)\n",
		pid, datalen));

	while (0 == (rc = sk_gets(&f->conn, reply, MAXBUFF))) {
		eol = strchr(reply, '\r');
		if (NULL == eol)
			eol = strchr(reply, '\n');
		if (NULL != eol)
			*eol = '\0';
		var = val = reply;
		while (*val != '\0' && *val != '=')
			val++;
		if (*val == '=')
			*val++ = '\0';

		if (0 == strcasecmp(var, "BlocksEncoded")) {
			f->seen_success = 1;
			f->seen_end_message = 0;
			LOG(L_DEBUG,("{%d} rcvd BlocksEncoded\n", pid));
		} else if (0 == strcasecmp(var, "BlockCount")) {
			block_count = strtoul(val, NULL, 16);
			LOG(L_DEBUG,("{%d} rcvd BlockCount=%x\n", pid, block_count));
		} else if (0 == strcasecmp(var, "BlockSize")) {
			block_size = strtoul(val, NULL, 16);
			LOG(L_DEBUG,("{%d} rcvd BlockSize=%x\n", pid, block_size));
		} else if (0 == strcasecmp(var, "Failed")) {
			f->seen_failed = 1;
		} else if (0 == strcasecmp(var, "Reason")) {
			if (NULL != f->reason) {
				free(f->reason);
				f->reason = NULL;
			}
			f->reason = strdup(val);
			LOG(L_DEBUG,("{%d} rcvd Reason=%s\n", pid, f->reason));
		} else if (0 == strcasecmp(var, "EndMessage")) {
			LOG(L_DEBUG,("{%d} rcvd EndMessage\n", pid));
			f->seen_end_message = 1;
		}
		if (0 != f->seen_end_message) {
			if (0 != f->seen_success)
				break;
		}
	}

	datalen = block_count * block_size;
	LOG(L_NORMAL,("receiving check blocks (%x bytes)\n", datalen));
	for (offs = 0; offs < datalen; /* */) {
		size_t size;

		if (offs + MAXBUFF > datalen) {
			size = datalen - offs;
		} else {
			size = MAXBUFF;
		}
		if (0 != (rc = sk_read(&f->conn, buff, size))) {
			LOG(L_ERROR,("sk_read(%x,%x,%x) call failed (%s)\n",
				f, buff, size, strerror(errno)));
			goto bailout;
		}
		LOG(L_MINOR,("read %x bytes @ %x\n", size, offs));
		offs += size;
	}
	LOG(L_MINOR,("read %x bytes\n", datalen));

	rc = 0;

bailout:
	if (NULL != buff) {
		free(buff);
		buff = NULL;
	}
	if (NULL != reply) {
		free(reply);
		reply = NULL;
	}
	return rc;
}

int fcp_segment_file(fcp_t **pf, const char *fcphost, int fcpport,
	 const char *algo_name, FILE *fp, u_int64_t file_length)
{
	pid_t pid = getpid();
	fcp_t *f = *pf;
	char *reply = calloc(MAXBUFF, 1);
	char *var, *val, *eol;
	size_t i;
	int rc = -1;

	if (NULL == reply) {
		LOG(L_ERROR,("{%d} barf: calloc(%d,%d) failure (%s)\n",
			pid, MAXBUFF, 1, strerror(errno)));
		goto bailout;
	}
	if (0 != sk_write(&f->conn, "\0\0\0\2", 4)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent 00 00 00 02\n", pid));

	if (0 != sk_printf(&f->conn, "FECSegmentFile\n")) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent FECSegmentFile\n", pid));

	if (0 != sk_printf(&f->conn, "AlgoName=%s\n", algo_name)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent AlgoName=%s\n", pid, algo_name));

	if (0 != sk_printf(&f->conn, "FileLength=%x\n", file_length)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent FileLength=%x\n", pid, file_length));

	if (0 != sk_printf(&f->conn, "EndMessage\n")) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent EndMessage\n", pid));

	i = 0;
	fcp_reset(f, 1);
	while (0 == (rc = sk_gets(&f->conn, reply, MAXBUFF))) {
		eol = strchr(reply, '\r');
		if (NULL == eol)
			eol = strchr(reply, '\n');
		if (NULL != eol)
			*eol = '\0';
		var = val = reply;
		while (*val != '\0' && *val != '=')
			val++;
		if (*val == '=')
			*val++ = '\0';

		if (0 != check_reply(pid, f, var, val)) {
			LOG(L_ERROR,("{%d} rcvd NOT HANDLED %s=%s\n",
				pid, var, val));
		}
		if (0 != f->seen_end_message) {
			if (0 != f->seen_segment_header) {
				if (f->sh0.segment_num >= MAX_SEGMENTS) {
					LOG(L_ERROR,("{%d} SegmentNum (%x) is out of range!!!\n",
						pid, f->sh0.segment_num));
					rc = -1;
					goto bailout;
				}
				f->sh[f->sh0.segment_num] = f->sh0;
			}
		}
	}

	/* read until EOF is okay */
	rc = 0;

	if (0 == f->seen_end_message) {
		LOG(L_DEBUG,("bail: did not see EndMessage\n"));
		rc = -1;
		goto bailout;
	}

	/* if an open file was specified */
	if (NULL != fp) {	
		/* encode the FEC segments */
		for (i = 0; i < f->sh[0].segments; i++) {
			/* reconnect */
			sk_shut(&f->conn);
			if (0 != (rc = sk_outgoing(&f->conn, fcphost, fcpport))) {
				LOG(L_ERROR,("Connection to %s:%d failed (%s)\n",
					fcphost, fcpport, strerror(errno)));
				goto bailout;
			}
			if (0 != (rc = fcp_encode_segment(f, i, fp, file_length))) {
				LOG(L_ERROR,("FEC encode segment #%d failed\n",
					i));
				goto bailout;
			}
		}
	}

bailout:
	if (NULL != reply) {
		free(reply);
		reply = NULL;
	}
	return rc;
}

int fcp_broadcast(fcp_t *f, int htl,
	const char *channel, const char *sender, const char *payload)
{
	pid_t pid = getpid();
	char *reply = calloc(MAXBUFF, 1);
	char *var, *val, *eol;
	size_t i;
	int rc = -1;

	if (NULL == reply) {
		LOG(L_ERROR,("{%d} barf: calloc(%d,%d) failure (%s)\n",
			pid, MAXBUFF, 1, strerror(errno)));
		goto bailout;
	}
	if (0 != sk_write(&f->conn, "\0\0\0\2", 4)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent 00 00 00 02\n", pid));

	if (0 != sk_printf(&f->conn, "ClientBroadcast\n")) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent ClientBroadcast\n", pid));

	if (0 != sk_printf(&f->conn, "HopsToLive=%x\n", htl)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent HopsToLive=%x\n", pid, htl));

	if (0 != sk_printf(&f->conn, "Channel=%s\n", channel)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent Channel=%s\n", pid, channel));

	if (0 != sk_printf(&f->conn, "Sender=%s\n", sender)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent Sender=%s\n", pid, sender));

	if (0 != sk_printf(&f->conn, "Payload=%s\n", payload)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent Payload=%s\n", pid, payload));

	if (0 != sk_printf(&f->conn, "EndMessage\n")) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent EndMessage\n", pid));

	i = 0;
	fcp_reset(f, 1);
	while (0 == (rc = sk_gets(&f->conn, reply, MAXBUFF))) {
		eol = strchr(reply, '\r');
		if (NULL == eol)
			eol = strchr(reply, '\n');
		if (NULL != eol)
			*eol = '\0';
		var = val = reply;
		while (*val != '\0' && *val != '=')
			val++;
		if (*val == '=')
			*val++ = '\0';

		if (0 != check_reply(pid, f, var, val)) {
			LOG(L_ERROR,("{%d} rcvd NOT HANDLED %s=%s\n",
				pid, var, val));
		}
		if (0 != f->seen_end_message) {
			break;
		}
	}

	/* read until EOF is okay */
	rc = 0;

	if (0 == f->seen_end_message) {
		LOG(L_DEBUG,("bail: did not see EndMessage\n"));
		rc = -1;
		goto bailout;
	}

bailout:
	if (NULL != reply) {
		free(reply);
		reply = NULL;
	}
	return rc;
}

int fcp_receive(fcp_t *f, int id)
{
	pid_t pid = getpid();
	char *reply = calloc(MAXBUFF, 1);
	char *var, *val, *eol;
	size_t i;
	int rc = -1;

	if (NULL == reply) {
		LOG(L_ERROR,("{%d} barf: calloc(%d,%d) failure (%s)\n",
			pid, MAXBUFF, 1, strerror(errno)));
		goto bailout;
	}
	if (0 != sk_write(&f->conn, "\0\0\0\2", 4)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent 00 00 00 02\n", pid));

	if (0 != sk_printf(&f->conn, "ClientReceive\n")) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent ClientReceive\n", pid));

	if (0 != sk_printf(&f->conn, "ID=%x\n", id)) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent ID=%x\n", pid, id));

	if (0 != sk_printf(&f->conn, "EndMessage\n")) {
		goto bailout;
	}
	LOG(L_DEBUG,("{%d} sent EndMessage\n", pid));

	i = 0;
	fcp_reset(f, 1);
	while (0 == (rc = sk_gets(&f->conn, reply, MAXBUFF))) {
		eol = strchr(reply, '\r');
		if (NULL == eol)
			eol = strchr(reply, '\n');
		if (NULL != eol)
			*eol = '\0';
		var = val = reply;
		while (*val != '\0' && *val != '=')
			val++;
		if (*val == '=')
			*val++ = '\0';

		if (0 != check_reply(pid, f, var, val)) {
			LOG(L_ERROR,("{%d} rcvd NOT HANDLED %s=%s\n",
				pid, var, val));
		}
		if (0 != f->seen_end_message) {
			break;
		}
	}

	/* read until EOF is okay */
	rc = 0;

	if (0 == f->seen_end_message) {
		LOG(L_DEBUG,("bail: did not see EndMessage\n"));
		rc = -1;
		goto bailout;
	}

bailout:
	if (NULL != reply) {
		free(reply);
		reply = NULL;
	}
	return rc;
}
