/* freezetag
 * id3 Editor for Linux
 * Requires GTK 1.2.x
 * Copyright (C), 1999, Jeremy Katz <katzj@linuxpower.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#include "freezetag.h"

void add_dir (gchar *path, GtkWidget *filelist);
void add_to_filelist (GtkWidget *clist, char *file);
void about_box (GtkWidget *widget, gpointer data);
void double_check_add_filelist (GtkWidget *clist, char *file);
void dialog_destroy (GtkWidget *widget, gpointer data);
void refresh_filelist( GtkWidget *widget, gpointer data);

static GtkItemFactoryEntry menu_items[] = {
	{ "/_File",         NULL,         NULL, 0, "<Branch>" },
	{ "/File/Quit",     "<control>Q", gtk_main_quit, 0, NULL },
	{ "/_Help",         NULL,         NULL, 0, "<LastBranch>" },
	{ "/_Help/About",   NULL,         about_box, 0, NULL },
	
};

GtkWidget *filelist;


void get_main_menu(GtkWidget *window, GtkWidget **menubar) {
	GtkItemFactory *itemfactory;
	GtkAccelGroup *accelgroup;
	gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);

	accelgroup = gtk_accel_group_new();

	itemfactory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", accelgroup);
	
	gtk_item_factory_create_items(itemfactory, nmenu_items, menu_items, NULL);
	gtk_accel_group_attach (accelgroup, GTK_OBJECT (window));

        if (menubar)
          *menubar = gtk_item_factory_get_widget (itemfactory, "<main>");
}


void about_box (GtkWidget *widget, gpointer data) {
	GtkWidget *dialog, *ok, *label,*box;

	dialog = gtk_window_new(GTK_WINDOW_DIALOG);
	box = gtk_vbox_new(FALSE, 10);

	gtk_container_add (GTK_CONTAINER (dialog), box);
	gtk_widget_show(box);
	
	ok = gtk_button_new_with_label("Ok");
	gtk_box_pack_end( GTK_BOX (box), ok, TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS(ok, GTK_CAN_DEFAULT);

	label = gtk_label_new("freezetag\nid3 Tagger for Linux\nCopyright (c) 1999\nJeremy Katz <katzj@linuxpower.org>");
	gtk_box_pack_start( GTK_BOX (box), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	gtk_signal_connect(GTK_OBJECT (ok), "clicked", GTK_SIGNAL_FUNC (dialog_destroy), GTK_OBJECT(dialog) );
	gtk_widget_show(ok);

	gtk_widget_show(dialog);

}


void add_files (GtkWidget *widget, GtkWidget *clist) {
	GtkCTreeNode *node;
	DirNode *dirnode;
	void (*handler)(gchar *);
	gchar *line[7];

	if (GTK_CLIST(tree)->selection) {
		node = GTK_CLIST(tree)->selection->data;
		dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(tree),node);
		handler = (void (*)(gchar *))gtk_object_get_user_data(GTK_OBJECT(tree));
		if(handler)
			handler(dirnode->path);
	}
	add_dir((gchar *) dirnode->path, clist);
}


void add_dir (gchar *path, GtkWidget *filelist) {
	DIR *dir;
	struct dirent *dirent;
	struct stat statbuf;
	gchar *filename;

	if(dir=opendir(path)) {
		while(dirent=readdir(dir)) {
			if(dirent->d_name[0]!='.') {
				filename=g_strconcat(path,"/",dirent->d_name,NULL);
				lstat(filename,&statbuf);
				if(S_ISDIR(statbuf.st_mode))
					add_dir(filename,filelist);
				else if (strstr(filename, ".mp3"))
/*					add_to_filelist(filelist,filename);*/
					double_check_add_filelist(filelist, filename);
				g_free(filename);
			}
		}
		closedir(dir);
	}
}


void double_check_add_filelist (GtkWidget *clist, char *file) {
	gint i;
	gchar *tmp;

	for(i = 0; i < GTK_CLIST (clist)->rows; i++) {
		gtk_clist_get_text( GTK_CLIST(clist), i, 0, &tmp);
		if (strcmp(file, tmp) == 0) {
			return;
		}
	}

	add_to_filelist( GTK_WIDGET (clist), file);
}


void add_to_filelist (GtkWidget *clist, char *file) {
	gchar *line[7] = { file,  (gchar *) " ", (gchar *) " ", (gchar *) " ", (gchar *) " ",(gchar *) " ",(gchar *) " "};
	int fd;
	gchar buf[129];
	gpointer artist, song, year, album, comment,genre;
	gint genrenum;
	extern char *genre_table[];
	extern int genre_count;

	
	if( (fd = open(file, O_RDWR) ) < 0) {
#ifdef DEBUG
		g_print("Whoops... error opening file %s\n",file);
#endif
		return;
	} 

	lseek(fd, -128, SEEK_END);
	if ( !(read(fd, buf, 128) ) ) {
		g_print("Whoops... error reading 128 bytes\n");
	}

	song = substr(buf, 3, 30);
	artist = substr(buf, 33, 30);
	album = substr(buf, 63, 30);
	year = substr(buf, 93, 4);
	comment = substr(buf, 97, 30);
	genrenum = buf[127];
	if( (genrenum < 0) || (genrenum >= genre_count) ) {
		/* Set genre to Blues if invalid specified 
		 * This is pretty much the normal way to do things 
		 */
		genrenum = 0;
	}
	genre = genre_table[genrenum];


	if ( (buf[0] != 'T') && (buf[1] != 'A') && (buf[2] != 'G') ) {
		gtk_clist_append( GTK_CLIST (clist), line );
	} else {
		gchar *row[7] = { file, song, artist, album, year, comment, genre };
		gtk_clist_append( GTK_CLIST (clist), row );
	}
	close(fd);

	g_free(song);
	g_free(artist);
	g_free(album);
	g_free(year);
	g_free(comment);

}


void rename_selected (GtkWidget *widget, gpointer data) {
	GList *list;
	guint row;
	
#ifdef DEBUG
	list = GTK_CLIST(data)->selection;
	
	while(list) {
		row = (gint) list->data;
		g_print("Row %d is selected\n", row);
		list = list->next;
	}
	g_list_free(list);
#endif

	list = GTK_CLIST(data)->selection;
	if (g_list_length(list) == 0)  {
#ifdef DEBUG
		g_print("Whoops... didn't select anything to tag\n");
#endif
		return;
	}

	rename_box( GTK_WIDGET(data) );
}


void call_tag (GtkWidget *widget, gpointer data) {
	GList *list;
	guint row;

#ifdef DEBUG
	list = GTK_CLIST(data)->selection;
	
	while(list) {
		row = (gint) list->data;
		g_print("Row %d is selected\n", row);
		list = list->next;
	}
	g_list_free(list);
#endif

	list = GTK_CLIST(data)->selection;
	if (g_list_length(list) == 0)  {
#ifdef DEBUG
		g_print("Whoops... didn't select anything to tag\n");
#endif
		return;
	}
	
	edit_box( GTK_WIDGET(data) );
}


void clear_filelist (GtkWidget *widget, gpointer data) {
	gtk_clist_clear( GTK_CLIST (data) );
}


void refresh_filelist( GtkWidget *widget, gpointer data) {
	gint rows, i;
	gchar *tmp;

	rows = (gint) GTK_CLIST (data)->rows;
	
	gtk_clist_freeze( GTK_CLIST(data) );

	for(i = 0; i < rows; i++) {
		gtk_clist_get_text( GTK_CLIST(data), 0, 0, &tmp);
		add_to_filelist( GTK_WIDGET (data), tmp);	
		gtk_clist_remove( GTK_CLIST (data), 0);
	}

	gtk_clist_thaw( GTK_CLIST(data) );
}


void sort_column(GtkCList *list, gint column) {
#ifdef DEBUG
	g_print("Selected column %d\n", column);
#endif
	gtk_clist_set_sort_column(list, column);
	gtk_clist_sort(list);
}


static gint menu_popup( GtkWidget *widget, GdkEvent *event) {
	if (event->type == GDK_BUTTON_PRESS) {
	    GdkEventButton *bevent = (GdkEventButton *) event;
	    if(bevent->button == 3) {
		gtk_menu_popup (GTK_MENU (widget), NULL, NULL, NULL, NULL, bevent->button, bevent->time );
		return TRUE;
	    }
	}
	return FALSE;
}


int main(int argc, char *argv[]) {
	GtkWidget *window, *mainbox, *menubar, *bottomscrollable, *directorylist, *topbox;
	GtkWidget *add, *quit, *retag, *clear, *rename; /*buttons*/
	GtkWidget *button_table;
	GtkWidget *leftbuttons, *rightbuttons, *buttons, *buttons2; /*button containers*/
	GtkWidget *popup, *tag_menu, *rename_menu;	/* popup menu stuff */
	extern GtkWidget *tree;
	gint *selected;

	gchar *titles[8] = { "Filename", "Song Name", "Artist", "Album", "Year", "Comment", "Genre" };

	gtk_init(&argc,&argv);
	gtk_rc_parse("~/.gnome/gtkrc");
	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect (GTK_OBJECT (window), "destroy", GTK_SIGNAL_FUNC (gtk_main_quit), "WM destroy");
	gtk_window_set_title (GTK_WINDOW(window), "freezetag");
	/* FIXME: Need to work on sizings and getting dynamic sizing to work */
	gtk_widget_set_usize (GTK_WIDGET(window), 650, 540);
	
	mainbox = gtk_vbox_new(FALSE, 2);
	gtk_container_border_width (GTK_CONTAINER (mainbox), 1);
	gtk_container_add (GTK_CONTAINER (window), mainbox);
	gtk_widget_show (mainbox);
        
	get_main_menu (window, &menubar);
	gtk_box_pack_start (GTK_BOX (mainbox), menubar, FALSE, TRUE, 0);
	gtk_widget_show (menubar);

	topbox = gtk_hbox_new(FALSE, 5);
	gtk_container_border_width (GTK_CONTAINER (topbox), 1);
	gtk_box_pack_start (GTK_BOX (mainbox), topbox, FALSE, TRUE, 0);
	gtk_widget_show(topbox);

	directorylist = create_dir_browser((gchar *) get_current_dir_name(), window);
	gtk_widget_set_usize( GTK_WIDGET (directorylist), 400, 100);
	gtk_box_pack_start (GTK_BOX (topbox), directorylist, FALSE, TRUE, 0);
	gtk_widget_show (directorylist);

	button_table = gtk_table_new (3, 2, TRUE);
	gtk_box_pack_start (GTK_BOX (topbox), button_table, FALSE, TRUE, 10);
	gtk_widget_show(button_table);

	add = gtk_button_new_with_label ("Add Directory");
	gtk_table_attach_defaults (GTK_TABLE (button_table), add, 0, 1, 0, 1);
	gtk_widget_show(add);

	clear = gtk_button_new_with_label ("Clear Filelist");
	gtk_table_attach_defaults (GTK_TABLE (button_table), clear, 0, 1, 1, 2);
	gtk_widget_show(clear);

	retag = gtk_button_new_with_label ("Retag");
	gtk_table_attach_defaults (GTK_TABLE (button_table), retag, 1, 2, 0, 1);
	gtk_widget_show(retag);

	rename = gtk_button_new_with_label ("Rename\nAccording to Tag");
	gtk_table_attach_defaults (GTK_TABLE (button_table), rename, 1, 2, 1, 2);
	gtk_widget_show(rename);

	quit = gtk_button_new_with_label ("Quit");
	gtk_table_attach_defaults (GTK_TABLE (button_table), quit, 0, 2, 2, 3);
	gtk_widget_show(quit);
	
	bottomscrollable = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (bottomscrollable), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_set_usize(bottomscrollable,1000, 400);
	gtk_box_pack_start (GTK_BOX (mainbox), bottomscrollable, FALSE, TRUE, 0);
	gtk_widget_show(bottomscrollable);

	filelist = gtk_clist_new_with_titles(7, titles);
	gtk_clist_set_selection_mode (GTK_CLIST(filelist), GTK_SELECTION_EXTENDED);
	gtk_clist_set_shadow_type (GTK_CLIST(filelist), GTK_SHADOW_OUT);
	gtk_clist_set_column_width (GTK_CLIST(filelist), 0, 125);
	gtk_clist_set_column_width (GTK_CLIST(filelist), 1, 100);
	gtk_clist_set_column_width (GTK_CLIST(filelist), 2, 100);
	gtk_clist_set_column_width (GTK_CLIST(filelist), 3, 100);
	gtk_clist_set_column_width (GTK_CLIST(filelist), 5, 80);
	gtk_scrolled_window_add_with_viewport ( GTK_SCROLLED_WINDOW (bottomscrollable) , filelist);
	gtk_widget_show(filelist);

	popup = gtk_menu_new();
	tag_menu = gtk_menu_item_new_with_label("Edit Tags");
	rename_menu = gtk_menu_item_new_with_label("Rename According to Tag");
	gtk_menu_append( GTK_MENU(popup), tag_menu);
	gtk_menu_append( GTK_MENU(popup), rename_menu);
	gtk_signal_connect( GTK_OBJECT (tag_menu), "activate", GTK_SIGNAL_FUNC (call_tag), GTK_OBJECT (filelist) );
	gtk_signal_connect( GTK_OBJECT (rename_menu), "activate", GTK_SIGNAL_FUNC (rename_selected), GTK_OBJECT (filelist) );
	gtk_widget_show(tag_menu);
	gtk_widget_show(rename_menu);


	/* Time for the signal handlers for all the miscellany buttons I'm putting in here */
	gtk_signal_connect(GTK_OBJECT (quit), "clicked", GTK_SIGNAL_FUNC (gtk_main_quit), "WM destroy");
	gtk_signal_connect(GTK_OBJECT (add), "clicked", GTK_SIGNAL_FUNC (add_files), filelist);
	gtk_signal_connect( GTK_OBJECT (clear), "clicked", GTK_SIGNAL_FUNC (clear_filelist), GTK_OBJECT (filelist) );
	gtk_signal_connect( GTK_OBJECT (retag), "clicked", GTK_SIGNAL_FUNC (call_tag), GTK_OBJECT (filelist) );
	gtk_signal_connect( GTK_OBJECT (rename), "clicked", GTK_SIGNAL_FUNC (rename_selected), GTK_OBJECT (filelist) );


	/* The signal handlers for the clist of files */
	gtk_signal_connect( GTK_OBJECT (filelist), "click_column", GTK_SIGNAL_FUNC(sort_column), GTK_OBJECT(filelist) );
	gtk_signal_connect_object ( GTK_OBJECT (filelist), "event", GTK_SIGNAL_FUNC(menu_popup), GTK_OBJECT (popup) );

	gtk_widget_show(window);

	gtk_main();

	return 0;
}
