/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-mime-viewer.c,v 1.4 2003/11/24 20:04:19 hoa Exp $
 */

#include "etpan-mime-viewer.h"

#include "etpan-app-subapp.h"
#include "etpan-errors.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <unistd.h>
#include <sys/wait.h>
#include "etpan-tools.h"
#include <string.h>
#include <stdlib.h>
#include "etpan-app.h"

static int viewer_process_pipe(struct etpan_app * app,
    char * viewer, char * filename)
{
  int r;
  int res;
  int status;
  int fd;
  struct stat stat_info;
  size_t written;
  char * mapping;
  FILE * p;

  fd = open(filename, O_RDONLY);
  if (fd == -1) {
    res = ERROR_FILE;
    goto err;
  }

  r = fstat(fd, &stat_info);
  if (r < 0) {
    res = ERROR_FILE;
    goto close;
  }

  mapping = mmap(NULL, stat_info.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
  if (mapping == MAP_FAILED) {
    res = ERROR_FILE;
    goto close;
  }

  etpan_app_display_done(app);
  
  p = popen(viewer, "w");
  if (p == NULL) {
    etpan_app_display_init(app);
    etpan_app_set_colors(app);
    res = ERROR_COMMAND;
    goto unmap;
  }
  
  written = fwrite(mapping, 1, stat_info.st_size, p);
  if (written != stat_info.st_size) {
    pclose(p);
    etpan_app_display_init(app);
    etpan_app_set_colors(app);
    res = ERROR_FILE;
    goto unmap;
  }
  
  status = pclose(p);
  etpan_app_display_init(app);
  etpan_app_set_colors(app);
  
  munmap(mapping, stat_info.st_size);
  close(fd);
  
  if (WEXITSTATUS(status) != 0) {
    res = ERROR_COMMAND;
    goto err;
  }
  
  return NO_ERROR;
  
 unmap:
  munmap(mapping, stat_info.st_size);
 close:
  close(fd);
 err:
  return res;
}


static int get_viewer_command(char * command, size_t size,
    char * cmd, char * filename)
{
  MMAPString * mmapstr;
  char * s;
  int res;
  char quoted_filename[PATH_MAX];

  etpan_quote_filename(quoted_filename,
      sizeof(quoted_filename), filename);

  mmapstr = mmap_string_new("");
  if (mmapstr == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  s = cmd;
  while (* s != '\0') {
    switch (*s) {
    case '%':
      s ++;
      switch (*s) {
      case '\0':
	break;
      case '%':
	if (mmap_string_append_c(mmapstr, '%') == NULL) {
          res = ERROR_MEMORY;
          goto free;
        }
	s++;
	break;
      case 'f':
	if (mmap_string_append(mmapstr, quoted_filename) == NULL) {
          res = ERROR_MEMORY;
          goto free;
        }
	s++;
	break;
      }
      break;

    default:
      if (mmap_string_append_c(mmapstr, * s) == NULL) {
        res = ERROR_MEMORY;
        goto free;
      }
      s++;
      break;
    }
  }
  
  fflush(stdout);
  
  strncpy(command, mmapstr->str, size);
  command[size - 1] = '\0';

  mmap_string_free(mmapstr);

  return NO_ERROR;
  
 free:
  mmap_string_free(mmapstr);
 err:
  return res;
}

static int viewer_process_command(struct etpan_app * app,
    char * viewer, char * filename)
{
  char command[PATH_MAX];
  int status;
  int res;
  int r;
  
  r = get_viewer_command(command, sizeof(command), viewer, filename);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  etpan_app_display_done(app);
  status = system(command);
  etpan_app_display_init(app);
  etpan_app_set_colors(app);
  if (WEXITSTATUS(status) != 0) {
    res = ERROR_COMMAND;
    goto err;
  }
  
  return NO_ERROR;
  
 err:
  return res;
}

int etpan_viewer_process(struct etpan_app * app,
    char * filename,
    struct etpan_mime_info * mime_info)
{
  int r;
  int res;

  switch (mime_info->viewer_type) {
  case COMMAND_TYPE_PIPE:
    r = viewer_process_pipe(app, mime_info->viewer, filename);
    if (r != NO_ERROR) {
      res = r;
      goto err;
    }

    return NO_ERROR;

  case COMMAND_TYPE_PARAM:
    r = viewer_process_command(app, mime_info->viewer, filename);
    if (r != NO_ERROR) {
      res = r;
      goto err;
    }
    
    return NO_ERROR;
  }
  
  return ERROR_INVAL;
  
 err:
  return res;
}
