/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-mime-common.c,v 1.22 2004/12/27 15:27:38 hoa Exp $
 */

#include "etpan-mime-common.h"

#include "etpan-mime-common-types.h"
#include <libetpan/charconv.h>
#include <stdlib.h>
#include <unistd.h>
#include <ncurses.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "etpan-errors.h"
#include "etpan-subapp-thread.h"
#include "etpan-part-viewer-app.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-subapp.h"
#include "etpan-tools.h"
#include "etpan-msg-renderer.h"
#include "etpan-mime-params.h"
#include "etpan-imf-helper.h"
#include "etpan-msg-reply.h"
#include "etpan-msg-new.h"
#include "etpan-mime-viewer.h"
#include "etpan-msg-supersedes.h"
#include "etpan-filename-input.h"
#include "etpan-mime-tools.h"
#if 0
#include "etpan-security.h"
#endif

#include <libetpan/mailprivacy.h>

void etpan_mime_common_set_fd(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    fd_set * fds,
    int * maxfd)
{
  etpan_subapp_thread_set_fd(app, fds, maxfd);
}

struct partviewer_matcher {
  mailmessage * msg;
  struct mailmime * mime;
  int part_type;
};

static int get_part_type(int app_op_type);

static int
partviewer_match_mime(struct etpan_subapp * partviewer_app, void * data)
{
  struct partviewer_matcher * state;
  
  state = data;
  
  return (etpan_part_viewer_app_get_msg(partviewer_app) ==
      state->msg)
    && (etpan_part_viewer_app_get_mime(partviewer_app) ==
        state->mime)
    && (etpan_part_viewer_app_get_part_type(partviewer_app) ==
        state->part_type);
}

static int open_viewer(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    struct mailfolder * folder,
    mailmessage * msg, struct mailmime * mime,
    int part_type,
    char * filename, FILE * f, carray * attr)
{
  struct etpan_subapp * partviewer_app;
  int r;
  struct partviewer_matcher match_data;
  
  match_data.msg = msg;
  match_data.mime = mime;
  match_data.part_type = part_type;
  
  partviewer_app = etpan_app_find_subapp(app->app, "part-viewer",
    1, partviewer_match_mime, &match_data);
  if (partviewer_app != NULL) {
    etpan_app_switch_subapp(partviewer_app, 0);
    return NO_ERROR;
  }
  
  /* reuse a previous interface */
  partviewer_app = etpan_app_find_subapp(app->app, "part-viewer",
      0, NULL, NULL);
  if (partviewer_app == NULL) {
    /* create new */
    partviewer_app = etpan_part_viewer_app_new(app->app);
    if (partviewer_app == NULL)
      return ERROR_MEMORY;
  }

  etpan_subapp_set_parent(partviewer_app, app);
  
  r = etpan_part_viewer_app_set_part(partviewer_app, folder, msg,
      mime, part_type, filename, f, attr);
  if (r != NO_ERROR)
    return r;
  
  etpan_app_switch_subapp(partviewer_app, 0);
  
  return NO_ERROR;
}


static void thread_part_source_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct etpan_message_fetch_result * fetch_result;
  FILE * f;
  char filename[PATH_MAX];
  int r;
  struct etpan_mime_common_app_state * state;
  int part_type;
  size_t write_len;
  struct mailmime_single_fields single_fields;
  int encoding;
  size_t cur_token;
  char * result;
  size_t result_len;
  mode_t old_mask;
  
  if (op->err != NO_ERROR) {
    mailmessage * msg;

    msg = op->data.mailaccess.msg;
    
    if (msg->msg_uid != NULL)
      ETPAN_APP_LOG((app->app,
                        "view part - error while retrieving part of message %i, %s",
                        msg->msg_index, msg->msg_uid));
    else
      ETPAN_APP_LOG((app->app,
                        "view part - error while retrieving part of message %i",
                        msg->msg_index));
    return;
  }
  
  fetch_result = op->result;
  
  encoding = MAILMIME_MECHANISM_8BIT;
  switch (op->cmd) {
  case ETPAN_THREAD_MESSAGE_FETCH_SECTION:
    if (op->data.mailaccess.mime != NULL) {
      mailmime_single_fields_init(&single_fields,
          op->data.mailaccess.mime->mm_mime_fields,
          op->data.mailaccess.mime->mm_content_type);
      
      if (single_fields.fld_encoding != NULL)
        encoding = single_fields.fld_encoding->enc_type;
    }
    break;
  }
  
  cur_token = 0;
  r = mailmime_part_parse(fetch_result->content, fetch_result->len,
      &cur_token, encoding, &result, &result_len);
  if (r != MAILIMF_NO_ERROR) {
    goto exit;
  }
  
  if (app_op_type == THREAD_ID_MIME_FETCH_FOR_SAVE) {
    char * save_filename;
    
    /* in this case, data is the name of the file to write */
    
    save_filename = data;
    
    /* write result to a file */
    old_mask = umask(0077);
    f = fopen(save_filename, "w");
    umask(old_mask);
    if (f == NULL) {
      mmap_string_unref(result);
      goto exit;
    }
    write_len = fwrite(result, 1, result_len, f);
    if (write_len != result_len) {
      mmap_string_unref(result);
      fclose(f);
      unlink(save_filename);
      goto exit;
    }
    fclose(f);
    mmap_string_unref(result);
    
    goto exit;
  }
  
  /* write result to a file */
  f = etpan_get_mime_tmp_file(filename, sizeof(filename));
  if (f == NULL) {
    mmap_string_unref(result);
    goto exit;
  }
  
  write_len = fwrite(result, 1, result_len, f);
  if (write_len != result_len) {
    mmap_string_unref(result);
    goto unlink;
  }
  
  mmap_string_unref(result);
  
  if (app_op_type == THREAD_ID_MIME_FETCH_FOR_VIEWER) {
    struct etpan_mime_info * mime_info;
    char filename_ext[PATH_MAX];
    
    fclose(f);
    
    /* external mime part viewer */
    mime_info = etpan_get_mime_info(app->app->config.mime_config,
        op->data.mailaccess.mime);
    
    if (mime_info->ext != NULL) {
      snprintf(filename_ext, sizeof(filename_ext), "%s%s",
          filename, mime_info->ext);
      r = rename(filename, filename_ext);
    }
    else {
      snprintf(filename_ext, sizeof(filename_ext), "%s",
          filename);
      r = 0;
    }
    
    if (r == 0) {
      r = etpan_viewer_process(app->app,
          filename_ext, mime_info);
      if (r != NO_ERROR) {
        ETPAN_APP_LOG((app->app, "mime viewer - failed"));
      }
      unlink(filename_ext);
    }
    else {
      unlink(filename);
    }
  }
  else {
    part_type = get_part_type(app_op_type);
    
    /* give the message to the part viewer */
    
    /* in this case, data is the mime_common_state */
    state = data;
    
    r = open_viewer(app, state,
        op->data.mailaccess.folder,
        op->data.mailaccess.msg,
        op->data.mailaccess.mime, part_type,
        filename, f, NULL);
    if (r != NO_ERROR) {
      goto unlink;
    }
  }
  
  /* no error */
  goto exit;
  
 unlink:
  fclose(f);
  unlink(filename);
 exit:
  
  /* free string returned by message */
  etpan_subapp_thread_op_add(app, THREAD_ID_MIME_FETCH_RESULT_FREE,
    ETPAN_THREAD_MESSAGE_FETCH_RESULT_FREE,
    op->data.mailaccess.storage, op->data.mailaccess.folder,
    op->data.mailaccess.msg, NULL,
    fetch_result->content,
    NULL, NULL, NULL);
  free(fetch_result);

  op->result = NULL;
}

static void thread_view_part_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct etpan_message_render_result * render_result;
  struct etpan_mime_common_app_state * state;
  int part_type;
  int r;
  
  state = data;
  
  if (op->err != NO_ERROR) {
    mailmessage * msg;

    msg = op->data.mailaccess.msg;
    
    if (msg->msg_uid != NULL)
      ETPAN_APP_LOG((app->app,
                        "view part - error while retrieving part of message %i, %s",
                        msg->msg_index, msg->msg_uid));
    else
      ETPAN_APP_LOG((app->app,
                        "view part - error while retrieving part of message %i",
                        msg->msg_index));
    return;
  }
  
  render_result = op->result;

  part_type = get_part_type(app_op_type);

  /* give the message to the part viewer */
  
  r = open_viewer(app, state,
      op->data.mailaccess.folder, op->data.mailaccess.msg,
      op->data.mailaccess.mime, part_type,
    render_result->filename, render_result->f, render_result->attr);
  
  if (r != NO_ERROR) {
    fclose(render_result->f);
    unlink(render_result->filename);
    if (render_result->attr != NULL)
      etpan_text_prop_array_free(render_result->attr);
  }
  
  free(render_result->filename);
  free(render_result);
  
  op->result = NULL;

}

static void
thread_view_handle_cancel(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  etpan_queue_unref_msg_mime(app, op->data.mailaccess.msg);
}

static void thread_view_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  switch (app_op_type) {
  case THREAD_ID_MIME_RENDER_SECTION:
    thread_view_part_callback(app, op, app_op_type, data);
    break;
    
  case THREAD_ID_MIME_FETCH_SECTION:
  case THREAD_ID_MIME_FETCH_HEADER:
  case THREAD_ID_MIME_FETCH_MIME:
  case THREAD_ID_MIME_FETCH_FOR_VIEWER:
    thread_part_source_callback(app, op, app_op_type, data);
    break;
  }
  
  etpan_queue_unref_msg_mime(app, op->data.mailaccess.msg);
}

static void
thread_save_mime_handle_cancel(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  char * filename;
  
  /* cleanup filename */
  filename = data;
  free(filename);
  
  etpan_queue_unref_msg_mime(app, op->data.mailaccess.msg);
}

static void
thread_save_mime_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  thread_part_source_callback(app, op, app_op_type, data);
  etpan_queue_unref_msg_mime(app, op->data.mailaccess.msg);
}



void etpan_mime_common_handle_fd(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    fd_set * fds)
{
  etpan_subapp_thread_handle_fd(app, fds);
}

static int forward_search(struct etpan_mime_common_app_state * state,
    unsigned int first);
static int backward_search(struct etpan_mime_common_app_state * state,
    unsigned int last);
static int view_mime(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    int app_op_type);

static int save_mime(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state);

static int reply_message(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    int reply_type);

static int bounce_message(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    int do_redirect);

static int expand_node(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    struct mailmime * mime);

void etpan_mime_common_handle_key(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    int key)
{
  struct mailmime * mime;
  
  switch (key) {
  case '0':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
  case 'G':
    break;
  default:
    state->chosen = 0;
    break;
  }

  switch (key) {
  case '0':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
    state->chosen *= 10;
    state->chosen += key - '0';
    break;
    
  case 'j':
  case KEY_DOWN:
    if (state->selected_window_forward_count > 1)
      state->selected = state->selected_window_forward[1];
    break;

  case 'k':
  case KEY_UP:
    if (state->selected_window_backward_count > 1)
      state->selected = state->selected_window_backward[1];
    break;

  case KEY_NPAGE:
    if (state->selected_window_forward_count > 0)
      state->selected = state->selected_window_forward[state->selected_window_forward_count - 1];
    break;

  case KEY_PPAGE:
    if (state->selected_window_backward_count > 0)
      state->selected = state->selected_window_backward[state->selected_window_backward_count - 1];
    break;

  case KEY_HOME:
    state->selected = forward_search(state, 0);
    break;
  case KEY_END:
    state->selected = backward_search(state, state->mime_tab->len - 1);
    break;

  case '+':
    mime = etpan_mime_common_get_selected_mime(app, state);
    if (mime != NULL)
      etpan_mime_change_opened(state->params, mime);
    break;

  case '*':
    expand_node(app, state, state->root);
    break;
    
  case '\n':
  case KEY_RIGHT:
    {
      int do_viewer;
      
      do_viewer = 0;
      
      mime = etpan_mime_common_get_selected_mime(app, state);
      if (mime != NULL) {
        struct etpan_mime_info * mime_info;
        
        mime_info = etpan_get_mime_info(app->app->config.mime_config, mime);
        
        if (mime_info != NULL)
          if (mime_info->viewer != NULL)
            do_viewer = 1;
      }
      
      if (do_viewer)
        view_mime(app, state, THREAD_ID_MIME_FETCH_FOR_VIEWER);
      else
        view_mime(app, state, THREAD_ID_MIME_RENDER_SECTION);
    }
    break;
    
  case 's':
    /* save mime part to file */
    save_mime(app, state);
    break;
    
  case 'h':
    view_mime(app, state, THREAD_ID_MIME_FETCH_HEADER);
    break;

  case KEY_CTRL('H'):
    view_mime(app, state, THREAD_ID_MIME_FETCH_MIME);
    break;

  case 't':
    view_mime(app, state, THREAD_ID_MIME_FETCH_SECTION);
    break;
    
  case 'm':
    etpan_compose_new(app, state->folder);
    break;
    
  case 'a':
    reply_message(app, state, ETPAN_THREAD_REPLY);
    break;
    
  case 'r':
    reply_message(app, state, ETPAN_THREAD_REPLY_FOLLOWUP_TO);
    break;
    
  case 'g':
    reply_message(app, state, ETPAN_THREAD_REPLY_ALL);
    break;
    
  case 'f':
    reply_message(app, state, ETPAN_THREAD_REPLY_FORWARD);
    break;
    
  case KEY_CTRL('F'):
    reply_message(app, state, ETPAN_THREAD_REPLY_FORWARD_AS_ATTACHMENT);
    break;

  case 'b':
    /* bounce message */
    bounce_message(app, state, 0);
    break;

  case 'B':
    /* redirect message */
    bounce_message(app, state, 1);
    break;
  }
}


static void
etpan_mime_list_display_precalc_free(struct etpan_mime_common_app_state *
    state);
static int
etpan_mime_list_display_precalc(struct etpan_mime_common_app_state * state);


void etpan_mime_common_set_color(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state)
{
  etpan_app_set_color(app->app, "main",
      &state->main_attr, A_NORMAL);
  etpan_app_set_color(app->app, "selection",
      &state->selection_attr, A_REVERSE);
  etpan_app_set_color(app->app, "status",
      &state->status_attr, A_REVERSE);
}

int etpan_mime_common_init(struct etpan_mime_common_app_state * state)
{
  /* root MIME */
  state->folder = NULL;
  state->msg = NULL;
  state->root = NULL;
  
  /* colors */
  state->main_attr = A_NORMAL;
  state->selection_attr = A_REVERSE;
  state->status_attr = A_REVERSE;

  /* cursor */
  state->chosen = 0;
  state->first = 0;
  state->selected = 0;
  
  /* mime info */
  state->params = etpan_mime_params_new();
  if (state->params == NULL)
    goto err;
  
  /* display optimization */
  state->mime_tab = carray_new(128);
  if (state->mime_tab == NULL)
    goto free_params;
  
  state->prefix_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (state->prefix_hash == NULL)
    goto free_msg_tree_tab;
  
  state->window_size = 0;
  state->selected_window_backward = NULL;
  state->selected_window_backward_count = 0;
  state->selected_window_forward = NULL;
  state->selected_window_forward_count = 0;

  return NO_ERROR;
  
 free_msg_tree_tab:
  carray_free(state->mime_tab);
 free_params:
  etpan_mime_params_free(state->params);
 err:
  return ERROR_MEMORY;
}

void etpan_mime_common_flush(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state)
{
  if (state->msg != NULL) {
    etpan_queue_unref_msg_mime(app, state->msg);
  }
  
  etpan_mime_list_display_precalc_free(state);
  etpan_mime_params_clear(state->params);
  state->folder = NULL;
  state->msg = NULL;
  state->root = NULL;
}

void etpan_mime_common_done(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state)
{
  if (state->selected_window_backward != NULL)
    free(state->selected_window_backward);
  if (state->selected_window_forward != NULL)
    free(state->selected_window_forward);
  
  chash_free(state->prefix_hash);
  carray_free(state->mime_tab);
  etpan_mime_params_free(state->params);
}

int etpan_mime_common_set_msg(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    struct mailfolder * folder,
    mailmessage * msg)
{
  int r;
  int res;
  
  etpan_mime_common_flush(app, state);

  r = etpan_queue_ref_msg_mime(app, msg);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "view message - not enough memory"));
    res = r;
    goto err;
  }
  
  state->root = msg->msg_mime;
  state->msg = msg;
  state->folder = folder;
  state->first = 0;
  state->selected = 0;
  
  etpan_mime_params_clear(state->params);
  
  r = etpan_mime_params_add_recursive(state->params, state->root);
  if (r != NO_ERROR) {
    res = r;
    goto unref;
  }
  
  etpan_mime_list_display_precalc_free(state);
  r = etpan_mime_list_display_precalc(state);
  if (r != NO_ERROR) {
    res = r;
    goto clear_params;
  }
  
  return NO_ERROR;

 clear_params:
  etpan_mime_params_clear(state->params);
 unref:
  etpan_queue_unref_msg_mime(app, msg);
  state->root = NULL;
  state->msg = NULL;
  state->folder = NULL;
 err:
  return res;
}

mailmessage * etpan_mime_common_get_msg(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state)
{
  return state->msg;
}

struct mailfolder * etpan_mime_common_get_folder(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state)
{
  return state->folder;
}


void etpan_mime_common_leave(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    struct etpan_subapp * new_app)
{
#if 0
  etpan_subapp_thread_cancel_all(app);
#endif
 
#if 0
  while (1) {
    struct etpan_subapp * subapp;
    
    subapp = etpan_app_find_child_subapp(app, 1);
    if (subapp != NULL)
      etpan_app_leave_subapp(subapp, new_app);
    else
      break;
  }
#endif

  etpan_mime_common_flush(app, state);
}


/* ***************************************** */
/* implementation */

static void snprint_mime_info(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    char * output, char * buffer, char * fill, int width,
    struct mailmime * mime)
{
  chashdatum key;
  chashdatum value;
  char * prefix;
  char * thread_prefix;
  int has_child;
  int r;
  char * content_str;
  char * decoded_content_str;
  char * dsp_content_str;
  char * decoded_filename;
  char * dsp_filename;
  char * decoded_description;
  char * dsp_description;
  struct mailmime_single_fields single_fields;
  char * filename;
  char * description;
#if 0
  char * security_driver;
  char * security_encryption;
  char * security_name;
#endif
  char * privacy_driver;
  char * privacy_encryption;
  char * privacy_name;
  
  mailmime_single_fields_init(&single_fields, mime->mm_mime_fields,
      mime->mm_content_type);
  
  key.data = &mime;
  key.len = sizeof(mime);
  r = chash_get(state->prefix_hash, &key, &value);
  if (r < 0)
    prefix = "";
  else
    prefix = value.data;

  switch (mime->mm_type) {
  case MAILMIME_SINGLE:
    has_child = FALSE;
    break;

  case MAILMIME_MULTIPLE:
    has_child = (!clist_isempty(mime->mm_data.mm_multipart.mm_mp_list));
    break;

  case MAILMIME_MESSAGE:
    has_child = (mime->mm_data.mm_message.mm_msg_mime != NULL);
    break;

  default:
    has_child = FALSE;
    break;
  }

  if (has_child) {
    if (etpan_mime_is_opened(state->params, mime))
      thread_prefix = "(-)";
    else
      thread_prefix = "(+)";
  }
  else
    thread_prefix = "   ";

  content_str = etpan_get_content_type_str(mime->mm_content_type);
  if (content_str == NULL)
    return;
  
  decoded_content_str = NULL;
#if 0
  r = charconv(app->app->config.global_config->display_charset,
      app->app->config.global_config->message_charset,
      content_str, strlen(content_str),
      &decoded_content_str);
  if (r != MAIL_CHARCONV_NO_ERROR)
    dsp_content_str = content_str;
  else
    dsp_content_str = decoded_content_str;
#endif
  dsp_content_str = content_str;

  filename = single_fields.fld_disposition_filename;
  if (filename == NULL)
    filename = single_fields.fld_content_name;
  
  dsp_filename = NULL;
#if 0
  decoded_filename = NULL;
#endif
  decoded_filename = NULL;
  if (filename != NULL) {
#if 0
    char * new_decoded_filename;
    
    r = charconv(app->app->config.global_config->display_charset,
        app->app->config.global_config->message_charset,
        filename, strlen(filename),
        &decoded_filename);
    if (r != MAIL_CHARCONV_NO_ERROR)
      dsp_filename = filename;
    else
      dsp_filename = decoded_filename;
#endif
    
    dsp_filename = filename;
    decoded_filename = etpan_decode_mime_header(app->app, filename);
    if (decoded_filename != NULL) {
#if 0
      free(decoded_filename);
      decoded_filename = new_decoded_filename;
#endif
      dsp_filename = decoded_filename;
    }
  }
  
  description = single_fields.fld_description;
  
  dsp_description = NULL;
  decoded_description = NULL;
  if (description != NULL) {
#if 0
    r = charconv(app->app->config.global_config->display_charset,
        app->app->config.global_config->message_charset,
        description, strlen(description),
        &decoded_description);
    if (r != MAIL_CHARCONV_NO_ERROR)
      dsp_description = description;
    else
      dsp_description = decoded_description;
#endif
    dsp_description = description;
    decoded_description = etpan_decode_mime_header(app->app, description);
    if (decoded_description != NULL) {
      dsp_description = decoded_description;
    }
  }
  
  snprintf(buffer, width, "%s %s %s", thread_prefix, prefix, dsp_content_str);
  if (filename != NULL)
    snprintf(output, width, "%s %s", buffer, dsp_filename);
  else
    snprintf(output, width, "%s", buffer);
  
  if (description != NULL)
    snprintf(buffer, width, "%s (%s)", output, dsp_description);
  else
    snprintf(buffer, width, "%s", output);
  
  etpan_mime_get_privacy(state->params,
      mime, &privacy_driver, &privacy_encryption);
  
  if ((privacy_driver != NULL) && (privacy_encryption != NULL)) {
#if 0
    security_name = etpan_security_get_encryption_name(app->app,
        security_driver, security_encryption);
#endif
    privacy_name = mailprivacy_get_encryption_name(app->app->privacy,
        privacy_driver, privacy_encryption);
  }
  else {
#if 0
    security_name = NULL;
#endif
    privacy_name = NULL;
  }
  
  if (privacy_name != NULL)
#if 0
    snprintf(output, width, "%s (%s)", buffer, security_name);
#endif
    snprintf(output, width, "%s (%s)", buffer, privacy_name);
  else
    snprintf(output, width, "%s", buffer);
  
  snprintf(buffer, width, "%s%s", output, fill);
  snprintf(output, width, "%s", buffer);
  
#if 0
  if (filename != NULL) {
    if (description != NULL) {
      snprintf(output, width, "%s %s%s %s (%s)%s",
          thread_prefix, prefix, dsp_content_str,
          dsp_filename, dsp_description, fill);
    }
    else {
      snprintf(output, width, "%s %s%s %s%s",
          thread_prefix, prefix, dsp_content_str,
          dsp_filename, fill);
    }
  }
  else {
    if (description != NULL) {
      snprintf(output, width, "%s %s%s (%s)%s",
          thread_prefix, prefix, dsp_content_str, dsp_description, fill);
    }
    else {
      snprintf(output, width, "%s %s%s%s",
          thread_prefix, prefix, dsp_content_str, fill);
    }
  }
#endif

  if (decoded_description != NULL)
    free(decoded_description);
  if (decoded_filename != NULL)
    free(decoded_filename);
  if (decoded_content_str != NULL)
    free(decoded_content_str);
  if (content_str != NULL)
    free(content_str);
}

/* this function will display the list of messages */

static void draw_line(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    WINDOW * w, char * output, char * buffer, char * fill, char * help_str)
{
  unsigned int i;
  unsigned int count;
  unsigned int percent;
  unsigned int list_lines;
  
  list_lines = app->display_height - 1;
  
  /* list */
  
  wattron(w, state->main_attr);
  count = 0;
  for(i = state->first ; count < list_lines ; i++) {
    struct mailmime * mime;
    
    if (i >= state->mime_tab->len)
      break;

    mime = carray_get(state->mime_tab, i);

    if (etpan_mime_is_visible(state->params, mime)) {
      
      snprint_mime_info(app, state, output, buffer,
          fill, app->display_width + 1, mime);
      
      if (i == state->selected) {
        wattroff(w, state->main_attr);
        wattron(w, state->selection_attr);
      }
      mvwaddstr(w, count, 0, output);
      if (i == state->selected) {
        wattroff(w, state->selection_attr);
        wattron(w, state->main_attr);
      }
      count ++;
    }
  }
  
  while (count < list_lines) {
    mvwaddstr(w, count, 0, fill);
    count ++;
  }
  attroff(state->main_attr);

  /* status bar */

  wattron(w, state->status_attr);
  
  if (state->mime_tab->len == 0)
    percent = 0;
  else if (state->mime_tab->len == 1)
	percent = 100;
  else
    percent = state->selected * 100 / (state->mime_tab->len-1);
  
  if (help_str != NULL)
    snprintf(output, app->display_width + 1,
        " %3i %% | %s%s", percent, help_str, fill);
  else
    snprintf(output, app->display_width + 1,
        " %3i %%%s", percent, fill);
  
  mvwprintw(w, app->display_height - 1, 0, "%s", output);
  wattroff(w, state->status_attr);
}


/*
  This function will prepare and display the message list
*/

static int update_view(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state)
{
  int count;
  unsigned int i;
  int list_lines;
  
  list_lines = app->display_height - 1;
  
  if (state->selected >= carray_count(state->mime_tab))
    state->selected = carray_count(state->mime_tab) - 1;

  if (state->window_size != app->display_height) {
    unsigned int * buf;

    buf = realloc(state->selected_window_backward,
        list_lines * sizeof(unsigned int));
    if (buf == NULL)
      return ERROR_MEMORY;
    state->selected_window_backward = buf;
    
    buf = realloc(state->selected_window_forward,
        list_lines * sizeof(unsigned int));
    if (buf == NULL)
      return ERROR_MEMORY;
    state->selected_window_forward = buf;
    
    state->window_size = list_lines;
  }
  
  count = 0;
  for(i = state->selected ; count < list_lines ; i++) {
    struct mailmime * mime;
    
    if (i >= state->mime_tab->len)
      break;

    mime = carray_get(state->mime_tab, i);

    if (etpan_mime_is_visible(state->params, mime)) {
      state->selected_window_forward[count] = i;
      count ++;
    }
  }
  state->selected_window_forward_count = count;
  
  count = 0;
  i = state->selected;
  while (count < list_lines) {
    struct mailmime * mime;
    
    mime = carray_get(state->mime_tab, i);

    if (etpan_mime_is_visible(state->params, mime)) {
      state->selected_window_backward[count] = i;
      count ++;
    }

    if (i == 0)
      break;

    i --;
  }
  state->selected_window_backward_count = count;

  if (state->first <
      state->selected_window_backward[state->selected_window_backward_count - 1])
    state->first =
      state->selected_window_backward[state->selected_window_backward_count - 1];

  if (state->selected < state->first)
    state->first = state->selected;

  return NO_ERROR;
}


int etpan_mime_common_display(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    WINDOW * w, char * help_str)
{
  char * output;
  char * fill;
  char * buffer;
  
  output = app->app->output;
  buffer = app->app->buffer;
  fill = app->app->fill;
  
  update_view(app, state);
  
  draw_line(app, state, w, output, buffer, fill, help_str);
  
  return NO_ERROR;
}

static int
etpan_mime_list_display_precalc_sub(struct etpan_mime_common_app_state * state,
    MMAPString * prefix, struct mailmime * mime,
    int level, int has_next)
{
  int r;
  int has_child;
  clistiter * cur;
  
  chashdatum key;
  chashdatum value;

  r = carray_add(state->mime_tab, mime, NULL);
  if (r < 0)
    return ERROR_MEMORY;

  key.data = &mime;
  key.len = sizeof(mime);

  value.data = prefix->str;
  value.len = prefix->len + 1;

  r = chash_set(state->prefix_hash, &key, &value, NULL);
  if (r < 0)
    return ERROR_MEMORY;

  switch (mime->mm_type) {
  case MAILMIME_SINGLE:
    has_child = FALSE;
    break;

  case MAILMIME_MULTIPLE:
    has_child = (!clist_isempty(mime->mm_data.mm_multipart.mm_mp_list));
    break;

  case MAILMIME_MESSAGE:
    has_child = (mime->mm_data.mm_message.mm_msg_mime != NULL);
    break;

  default:
    has_child = FALSE;
    break;
  }

  if (has_child) {
    char old_prefix[2];
    struct mailmime * child;
    int sub_has_next;

    if (level >= 1) {
      memcpy(old_prefix, prefix->str + prefix->len - 2, 2);
      if (has_next)
        memcpy(prefix->str + prefix->len - 2, "| ", 2);
      else
        memcpy(prefix->str + prefix->len - 2, "  ", 2);
    }

    switch (mime->mm_type) {
    case MAILMIME_SINGLE:
      break;
      
    case MAILMIME_MULTIPLE:
      for(cur = clist_begin(mime->mm_data.mm_multipart.mm_mp_list) ;
          cur != NULL ; cur = clist_next(cur)) {
        
        child = cur->data;
        
        if (cur->next != NULL) {
          if (level >= 0) {
            if (mmap_string_append(prefix, "+-") == NULL)
              return ERROR_MEMORY;
          }
          sub_has_next = TRUE;
        }
        else {
          if (level >= 0) {
            if (mmap_string_append(prefix, "\\-") == NULL)
              return ERROR_MEMORY;
          }
          sub_has_next = FALSE;
        }
        
        r = etpan_mime_list_display_precalc_sub(state,
            prefix, child, level + 1, sub_has_next);
        if (r != NO_ERROR)
          return r;
        
        if (mmap_string_truncate(prefix, prefix->len - 2) == NULL) {
          return ERROR_MEMORY;
        }
      }
      break;
      
    case MAILMIME_MESSAGE:
      child = mime->mm_data.mm_message.mm_msg_mime;

      if (mmap_string_append(prefix, "\\-") == NULL)
        return ERROR_MEMORY;
      sub_has_next = FALSE;

      r = etpan_mime_list_display_precalc_sub(state,
          prefix, child, level + 1, has_next);
      if (r != NO_ERROR)
        return r;

      if (mmap_string_truncate(prefix, prefix->len - 2) == NULL) {
        return ERROR_MEMORY;
      }

      break;
    }

    if (level >= 1) {
      memcpy(prefix->str + prefix->len - 2, old_prefix, 2);
    }
  }

  return NO_ERROR;
}

static int
etpan_mime_list_display_precalc(struct etpan_mime_common_app_state * state)
{
  MMAPString * prefix;

  prefix = mmap_string_new("");
  if (prefix == NULL)
    return ERROR_MEMORY;

  etpan_mime_list_display_precalc_sub(state, prefix,
      state->root, 0, FALSE);

  mmap_string_free(prefix);

  return NO_ERROR;
}

static void
etpan_mime_list_display_precalc_free(struct etpan_mime_common_app_state *
    state)
{
  chash_clear(state->prefix_hash);

  carray_set_size(state->mime_tab, 0);
}

static int forward_search(struct etpan_mime_common_app_state * state,
                          unsigned int first)
{
  unsigned int i;

  for(i = first ; i < carray_count(state->mime_tab) ; i++) {
    struct mailmime * mime;

    mime = carray_get(state->mime_tab, i);
    
    if (etpan_mime_is_visible(state->params, mime)) {
      return i;
    }
  }

  return first;
}

static int backward_search(struct etpan_mime_common_app_state * state,
    unsigned int last)
{
  unsigned int i;

  i = last;
  do {
    struct mailmime * mime;

    mime = carray_get(state->mime_tab, i);
    
    if (etpan_mime_is_visible(state->params, mime))
      return i;
    

    if (i == 0)
      break;

    i --;

  } while (1);

  return last;
}


int etpan_subapp_thread_has_match_op(struct etpan_subapp * app,
    int app_op_type,
    struct mailstorage * storage, struct mailfolder * folder,
    mailmessage * msg, struct mailmime * mime);


static int get_part_type(int app_op_type)
{
  switch (app_op_type) {
  case THREAD_ID_MIME_FETCH_FOR_VIEWER:
    return ETPAN_PART_VIEWER_TEXT;
  case THREAD_ID_MIME_FETCH_SECTION:
    return ETPAN_PART_VIEWER_TEXT;
  case THREAD_ID_MIME_FETCH_MIME:
    return ETPAN_PART_VIEWER_MIME;
  case THREAD_ID_MIME_FETCH_HEADER:
    return ETPAN_PART_VIEWER_HEADER;
  case THREAD_ID_MIME_RENDER_SECTION:
    return ETPAN_PART_VIEWER_RENDER;
  default:
    return ETPAN_PART_VIEWER_NONE;
  }
}

static int get_cmd(int app_op_type)
{
  switch (app_op_type) {
  case THREAD_ID_MIME_FETCH_FOR_VIEWER:
    return ETPAN_THREAD_MESSAGE_FETCH_SECTION;
  case THREAD_ID_MIME_FETCH_SECTION:
    return ETPAN_THREAD_MESSAGE_FETCH_SECTION;
  case THREAD_ID_MIME_FETCH_MIME:
    return ETPAN_THREAD_MESSAGE_FETCH_SECTION_MIME;
  case THREAD_ID_MIME_FETCH_HEADER:
    return ETPAN_THREAD_MESSAGE_FETCH_SECTION_HEADER;
  case THREAD_ID_MIME_RENDER_SECTION:
    return ETPAN_THREAD_MESSAGE_RENDER_MIME;
  default:
    return ETPAN_PART_VIEWER_NONE;
  }
}

static int view_mime(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    int app_op_type)
{
  struct mailmime * mime;
  int part_type;
  struct partviewer_matcher match_data;
  struct etpan_subapp * partviewer_app;
  int cmd;
  int r;
  
  mime = etpan_mime_common_get_selected_mime(app, state);
  if (mime == NULL)
    return ERROR_INVAL;
  
  if (etpan_subapp_thread_has_match_op(app, app_op_type,
          NULL, NULL, state->msg, mime))
    return ERROR_BUSY;
  
  part_type = get_part_type(app_op_type);
  
  match_data.part_type = part_type;
  match_data.mime = mime;
  match_data.msg = state->msg;
  
  partviewer_app = etpan_app_find_subapp(app->app, "part-viewer",
      1, partviewer_match_mime, &match_data);
  if (partviewer_app != NULL) {
    /* found a corresponding app */
    etpan_app_switch_subapp(partviewer_app, 0);
    return NO_ERROR;
  }
  
  if (state->msg->msg_uid != NULL)
    ETPAN_APP_LOG((app->app, "retrieving part of message %i, %s",
                      state->msg->msg_index, state->msg->msg_uid));
  else
    ETPAN_APP_LOG((app->app, "retrieving part of message %i",
                      state->msg->msg_index));
  
  cmd = get_cmd(app_op_type);

  r = etpan_queue_ref_msg_mime(app, state->msg);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "mime message - not enough memory"));
    return r;
  }
  
  r = etpan_subapp_thread_msg_op_add(app, app_op_type, cmd,
      state->msg, mime,
      NULL,
      thread_view_callback, state, thread_view_handle_cancel);
  if (r != NO_ERROR) {
    etpan_queue_unref_msg_mime(app, state->msg);
    ETPAN_APP_LOG((app->app, "mime viewer - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}

struct mailmime *
etpan_mime_common_get_selected_mime(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state)
{
  struct mailmime * mime;
  
  if (carray_count(state->mime_tab) == 0)
    return NULL;
  
  mime = carray_get(state->mime_tab, state->selected);
  
  return mime;
}

static int expand_node(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    struct mailmime * mime)
{
  clistiter * cur;

  etpan_mime_set_opened(state->params, mime, TRUE);
  
  switch(mime->mm_type) {
  case MAILMIME_SINGLE:
    break;
    
  case MAILMIME_MULTIPLE:
    for(cur = clist_begin(mime->mm_data.mm_multipart.mm_mp_list) ;
        cur != NULL ; cur = clist_next(cur)) {
      expand_node(app, state, clist_content(cur));
    }
    
    break;
    
  case MAILMIME_MESSAGE:
    
    if (mime->mm_data.mm_message.mm_msg_mime != NULL) {
      expand_node(app, state, mime->mm_data.mm_message.mm_msg_mime);
    }

    break;
  }

  return NO_ERROR;
}

static int reply_message(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    int reply_type)
{
  struct mailmime * mime;
  
  mime = etpan_mime_common_get_selected_mime(app, state);
 
  return etpan_reply_message(app, reply_type,
      state->msg, mime, NULL, NULL, state->folder);
}

static int bounce_message(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state,
    int do_redirect)
{
  struct mailmime * mime;
  
  mime = etpan_mime_common_get_selected_mime(app, state);
  
  return etpan_bounce_message(app, state->msg, mime, do_redirect);
}

static void save_upcall(struct etpan_subapp * input_app, int valid,
    void * data);

struct save_mime_upcall_data {
  struct etpan_subapp * app;
  struct etpan_mime_common_app_state * state;
};

static int save_mime(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state)
{
  struct etpan_subapp * input_app;
  int r;
  struct mailmime * mime;
  char * mime_filename;
  struct mailmime_single_fields single_fields;
  struct save_mime_upcall_data * save_mime_info;
  char filename[PATH_MAX];
  char current_dir[PATH_MAX];
  char * default_filename;
  
  state = app->data;

  mime = etpan_mime_common_get_selected_mime(app, state);
  if (mime == NULL) {
    ETPAN_APP_LOG((app->app, "save attachment - no mime part"));
    return ERROR_INVAL;
  }
  
  mailmime_single_fields_init(&single_fields, mime->mm_mime_fields,
      mime->mm_content_type);
  mime_filename = single_fields.fld_disposition_filename;
  if (mime_filename == NULL)
    mime_filename = single_fields.fld_content_name;
  
  input_app = etpan_app_find_subapp(app->app, "filename-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_filename_input_new(app->app);
    if (input_app == NULL) {
      ETPAN_APP_LOG((app->app, "save attachment - not enough memory"));
      return ERROR_MEMORY;
    }
  }
  
  default_filename = NULL;
  
  if (getcwd(current_dir, sizeof(current_dir)) != NULL) {
    size_t len;
    
    len = strlen(current_dir);
    if (current_dir[len - 1] != '/') {
      if (len + 1 < sizeof(current_dir) - 1) {
        current_dir[len] = '/';
        current_dir[len + 1] = '\0';
      }
    }
    
    if (mime_filename != NULL) {
      snprintf(filename, sizeof(filename), "%s%s",
          current_dir, mime_filename);
      default_filename = filename;
    }
    else {
      default_filename = current_dir;
    }
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  save_mime_info = malloc(sizeof(* save_mime_info));
  if (save_mime_info == NULL) {
    ETPAN_APP_LOG((app->app, "save attachment - not enough memory"));
    return ERROR_MEMORY;
  }
  
  save_mime_info->app = app;
  save_mime_info->state = state;
  
  r = etpan_filename_input_set(input_app,
      "filename: ", 256, default_filename,
      save_upcall, save_mime_info);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "save attachment - not enough memory"));
    return ERROR_MEMORY;
  }
  
  etpan_app_switch_subapp(input_app, 0);
  
  return NO_ERROR;
}



static void save_upcall(struct etpan_subapp * input_app, int valid,
    void * data)
{
  char * val;
  int r;
  struct save_mime_upcall_data * save_mime_info;
  struct etpan_subapp * app;
  struct etpan_mime_common_app_state * state;
  char * filename;
  struct mailmime * mime;
  
  save_mime_info = data;
  app = save_mime_info->app;
  state = save_mime_info->state;
  free(save_mime_info);
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    ETPAN_APP_LOG((app->app, "save attachment - cancel"));
    goto err;
  }
  
  mime = etpan_mime_common_get_selected_mime(app, state);
  if (mime == NULL) {
    ETPAN_APP_LOG((app->app, "save attachment - invalid"));
    goto err;
  }
  
  val = etpan_filename_input_get_value(input_app);
  if (val == NULL) {
    ETPAN_APP_LOG((app->app, "save attachment - BUG detected"));
    goto err;
  }

  filename = strdup(val);
  if (filename == NULL) {
    ETPAN_APP_LOG((app->app, "save attachment - not enough memory"));
    goto err;
  }

  r = etpan_queue_ref_msg_mime(app, state->msg);
  if (r != NO_ERROR) {
    free(filename);
    ETPAN_APP_LOG((app->app, "save attachment - not enough memory"));
    goto err;
  }
  
  r = etpan_subapp_thread_folder_op_add(app, THREAD_ID_MIME_FETCH_FOR_SAVE,
      ETPAN_THREAD_MESSAGE_FETCH_SECTION,
      state->folder,
      state->msg, mime,
      NULL,
      thread_save_mime_callback, filename,
      thread_save_mime_handle_cancel);
  if (r != NO_ERROR) {
    etpan_queue_unref_msg_mime(app, state->msg);
    free(filename);
    ETPAN_APP_LOG((app->app, "save attachment - not enough memory"));
    goto err;
  }
  
  etpan_app_quit_subapp(input_app);
  
  return;
  
 err:
  etpan_app_quit_subapp(input_app);
  return;
}

void etpan_mime_common_display_update(struct etpan_subapp * app,
    struct etpan_mime_common_app_state * state)
{
  int r;
  
  etpan_mime_list_display_precalc_free(state);
  r = etpan_mime_list_display_precalc(state);
  r = etpan_mime_params_add_recursive(state->params, state->root);
}
