#!/usr/bin/perl -w
#
# generate a vfolder tree from a local mbox directory
# for etpan-ng's config.
# script TODO: extend to work with mh, maildir... and IMAP.
# etpan TODO: integrate a vtree feature with subscribe/unsubscribe interface :)
#
# usage: etpan-make-vtree.pl [-p] [-v] [-C] <etpan-vfolder> <local-path>
# -p : activate polling on created storage (can be slow if many folders)
# -v : be verbose
# -C : disable caching for created folders*
#
# The script tries hard to be as idempotent as possible and so should be
# usable several times with various parameters without damaging
# the configuration.
#
# Gal Roualland <gael.roualland@dial.oleane.com>
# $Id: etpan-make-vtree.pl,v 1.1 2004/01/04 23:40:29 g_roualland Exp $

use strict;
use Cwd 'abs_path';
use DirHandle;
use Getopt::Std;

my $HOME = $ENV{'HOME'};
my $config = "$HOME/.libetpan/config";

# parse an etpan-like config file
sub read_conf($) {
    my $conf = shift;
    my @data;
    my $cur;

    open(CONFIG, "< $config/$conf") || die "can't open $config/$conf for reading: $!";
    while (<CONFIG>) {
	chomp;
	next unless /^\s*(\S+)\s*=\s*(.*)\s*$/;

	my $key = lc($1);
	my $value = $2;
	
	if ($key eq 'id') {
	    push @data, $cur if defined $cur;
	    undef $cur;
	}
	$cur->{$key} = $value;
    }
    close CONFIG;
    push @data, $cur if defined $cur;
    return \@data;
}

# save an etpan-like config file
sub save_conf ($$) {
    my $conf = shift;
    my $data = shift;
    my $id = 1;
    my %usedid;
    
    open(CONF, "> $config/$conf.new.$$") || die "can't create new config $config: $!";

    foreach my $item (@{$data}) {
	$usedid{$item->{'id'}} = 1 if defined $item->{'id'};
    }
    foreach my $item (sort { (!defined($a->{'name'}) ? -1 :
			      (!defined($b->{'name'}) ? 1 :
			       ($a->{'name'} cmp $b->{'name'}))) } (@{$data})) {
	if (!defined($item->{'id'})) {
	    while (defined($usedid{$id})) {
		$id++;
	    }
	    $item->{'id'} = $id;
	    $usedid{$id} = 1;
	}
	print CONF "id = $item->{id}\n";
	foreach my $key (keys %{$item}) {
	    next if $key eq 'id';
	    print CONF "$key = $item->{$key}\n";
	}
	print CONF "\n";
    }
    close CONF;
    rename("$config/$conf", "$config/$conf.old") || print STDERR "can't rename old $config: $!\n";
    rename("$config/$conf.new.$$", "$config/$conf") || print STDERR "can't rename new $config: $!\n";
}

# browse a directory and returns list of files in it
sub browse {
    my $dir = shift;
    my @data;
    
    my $d = new DirHandle $dir;
    if (!defined ($d)) {
	print STDERR "can't open $dir: $!\n";
	return undef;
    }
    while (defined($_ = $d->read)) {
	next if /^\./;
	my $f = "$dir/$_";
	if (-f "$f") {
	    push @data, "$f";
	} elsif (-d "$f") {
	    my $subdata = browse($f);
	    push @data, @{$subdata} if defined $subdata;
	}
    }
    undef $d;
    return \@data;
}

sub usage {
    die "usage: $0 [-p] [-v] [-C] folder local-path";
}

my %opts;
getopts('pvC', \%opts) || usage();

usage unless scalar @ARGV == 2;

my $vfolder = $ARGV[0];
my $source = abs_path($ARGV[1]);
my %stats;

die "bad local path" unless (defined($source) && ($source ne ''));

# load the relevant parts of the current etpan config
print "* loading configuration...\n" if defined $opts{'v'};
my $vfolders = read_conf("vfolder");
my $storages = read_conf("storage");

# browse folders and remove hierarchy under our branch point
# storage referenced under it are saved for later expunge if useless
print "* checking vfolders...\n" if defined $opts{'v'};
my %storuse;
my @vfold;
foreach my $folder (@{$vfolders}) {
    $storuse{$folder->{'storage'}}++
	if defined $folder->{'storage'};
    if (defined($folder->{'name'}) &&
	$folder->{'name'} =~ /^$vfolder(\/.*)?$/) {
	$storuse{$folder->{'storage'}}--
	    if defined $folder->{'storage'};
	$stats{'folder.del'}++;
	print "removing folder $folder->{'name'} (under $vfolder)\n" if defined $opts{'v'};
	next;
    }
    print "keeping folder " . (defined($folder->{'name'}) ? $folder->{'name'}  : "noname (id=$folder->{'id'})") .
	"\n" if defined $opts{'v'};
    push @vfold, $folder;
}
$vfolders = \@vfold;

# expunge unused mbox storages referenced from deleted vfolders
print "* checking storages...\n" if defined $opts{'v'};
my @stor;
foreach my $storage (@{$storages}) {
    if (!defined($storage->{'type'}) ||
	$storage->{'type'} !~ /^mbox$/i ||
	!defined($storage->{'name'}) ||
	!defined($storuse{$storage->{'name'}}) ||
	$storuse{$storage->{'name'}} > 0) {
	push @stor, $storage;
	print "keeping storage " . (defined($storage->{'name'}) ? $storage->{'name'}  : "noname (id=$storage->{'id'})") .
	    "\n" if defined $opts{'v'};
	next;
    }
    $stats{'storage.del'}++;
    print "removing storage $storage->{'name'} (no longer used)\n" if defined $opts{'v'};
}
$storages = \@stor;

# browse local-path and add new mailbox/storage
print "* checking directory $source for mbox files...\n" if defined $opts{'v'};
my $files = browse($source);

my %nfolders;
foreach my $file (@{$files}) {
    # check to see if that's a mailbox file
    if (!open(M, "< $file")) {
	print STDERR "can't open $file: $!. SKIPPING\n";
	next;
    }
    my $l = <M>;
    close (M);
    if (defined($l) && $l !~ /^From /) {
	print "skipping $file (not a valid mbox)\n" if defined $opts{'v'};
	next;
    }

    # convert file name to etpan-folder name
    my $folder = $file;

    $folder =~ s,^$source/,$vfolder/,g;
    $folder =~ s,\.sbd/,/,g; # netscape subdirectories

    if (defined($nfolders{$folder})) {
	print "skipping $file (folder $folder already used)\n" if defined $opts{'v'};
	next;
    }
    print "selecting file $file as folder $folder\n" if defined $opts{'v'};
    $nfolders{$folder} = $file;
}

print "* linking configuration...\n" if defined $opts{'v'};
# create empty folders for hierarchy if needed
foreach my $folder (keys %nfolders) {
    while ($folder =~ s,(/[^/]+)$,,) {
	unless (exists($nfolders{$folder})) {
	    print "creating non selectable vfolder $folder\n" if defined $opts{'v'};
	    $nfolders{$folder} = undef;
	}
    }
}

# browse existent storage to build hash of name and locations
my %name2storage;
my %path2storage;
foreach my $storage (@{$storages}) {
    next unless defined $storage->{'name'};
    $name2storage{$storage->{'name'}} = 1;
    if (defined($storage->{'type'}) && $storage->{'type'} =~ /^mbox$/i
	&& defined($storage->{'location'})) {
	my $p = abs_path($storage->{'location'});
	$path2storage{$p} = $storage->{'name'} if defined $p;
    }
}

# add vfolders for new folders and create new storage when needed
foreach my $folder (keys %nfolders) {
    if (!defined($nfolders{$folder})) { # simple directory
	push @{$vfolders}, { 'name' => $folder, location => '' };
	print "adding non selectable vfolder $folder\n" if defined $opts{'v'};
	$stats{'folder.add'}++;
	next;
    }
    if (!defined($path2storage{$nfolders{$folder}})) { # need to create a storage
	my $name = $nfolders{$folder};
	$name =~ s,[^\w\d],_,g;
	$name = "mbox$name";
	while (defined($name2storage{$name})) {
	    $name .= "_";
	}
	push @{$storages}, { 'name' => $name, 'type' => "mbox",
			     'location' => $nfolders{$folder},
			     'cached' => defined($opts{'C'}) ? 0 : 1 };
	$name2storage{$name} = 1;
	$path2storage{$nfolders{$folder}} = $name;
	$stats{'storage.add'}++;
	print "adding storage $name for file $nfolders{$folder}\n" if defined $opts{'v'};
    } else {
	print "using existing storage $path2storage{$nfolders{$folder}} for file $nfolders{$folder}\n" if defined $opts{'v'};  
    }
    push @{$vfolders}, { 'name' => $folder, 'storage' => $path2storage{$nfolders{$folder}},
			 'poll' => defined($opts{'p'}) ? 1 : 0 };
    $stats{'folder.add'}++;
    print "adding vfolder $folder for file $nfolders{$folder}\n" if defined $opts{'v'};
}

print "* saving new configuration...\n" if defined $opts{'v'};
save_conf("vfolder", $vfolders);
save_conf("storage", $storages);

print "" . (defined($stats{'folder.add'}) ? $stats{'folder.add'} : 0) . " folders added, " .
    (defined($stats{'folder.del'}) ? $stats{'folder.del'} : 0) . " folders deleted, " .
    (defined($stats{'storage.add'}) ? $stats{'storage.add'} : 0) . " storages added, " .
    (defined($stats{'storage.del'}) ? $stats{'storage.del'} : 0) . " storages deleted.\n";

exit 0;




