# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to show the output of the svn proplist command process.
"""

import types

from qt import *

from PropListForm import PropListForm

class SvnPropListDialog(PropListForm):
    """
    Module implementing a dialog to show the output of the svn proplist command process.
    """
    def __init__(self, vcs, parent = None):
        """
        Constructor
        
        @param vcs reference to the vcs object
        @param parent parent widget (QWidget)
        """
        PropListForm.__init__(self, parent)
        
        self.setWFlags(self.getWFlags() | Qt.WDestructiveClose)
        self.process = QProcess()
        self.vcs = vcs
        
        self.propsList.setSorting(0)
        
        self.connect(self.process, SIGNAL('readyReadStdout()'),
            self.handleReadStdout)
        self.connect(self.process, SIGNAL('readyReadStderr()'),
            self.handleReadStderr)
        self.connect(self.process, SIGNAL('processExited()'),
            self.handleProcessExited)
            
        self.rx_path = QRegExp("Properties on '([^']+)':")
        self.rx_prop = QRegExp("  (.*) : (.*)")
        self.lastPath = None
        self.lastProp = None
        self.propBuffer = QString('')
            
    def closeEvent(self, e):
        """
        Private slot implementing a close event handler.
        
        @param e close event (QCloseEvent)
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        e.accept()
        
    def start(self, fn, recursive = 0):
        """
        Public slot to start the svn status command.
        
        @param fn filename(s) (string or list of string)
        @param recursive flag indicating a recursive list is requested
        """
        self.process.kill()
        
        self.process.clearArguments()
        self.process.addArgument('svn')
        self.process.addArgument('proplist')
        self.vcs.addArguments(self.process, self.vcs.options['global'])
        self.process.addArgument('--verbose')
        if recursive:
            self.process.addArgument('--recursive')
        if type(fn) is types.ListType:
            dname, fnames = self.vcs.splitPathList(fn)
            self.vcs.addArguments(self.process, fnames)
        else:
            dname, fname = self.vcs.splitPath(fn)
            self.process.addArgument(fname)
        self.process.setWorkingDirectory(QDir(dname))
        
        self.process.start()
        
    def finish(self):
        """
        Private slot called when the process finished or the user pressed the button.
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        self.cancelButton.setText(self.trUtf8('OK'))
        self.cancelButton.setDefault(1)
        
        self.process = None
        if self.lastProp:
            itm = QListViewItem(self.propsList,
                self.lastPath, self.lastProp, self.propBuffer)
        
    def buttonPressed(self):
        """
        Private slot connected to the button clicked signal.
        """
        if self.process is None:
            self.close()
        else:
            self.finish()
            
    def handleProcessExited(self):
        """
        Private slot to handle the processExited signal.
        
        Just call finish().
        """
        self.finish()
        
        if self.lastPath is None:
            itm = QListViewItem(self.propsList, '', 'None', '')
        
    def handleReadStdout(self):
        """
        Private slot to handle the readyReadStdout signal.
        
        It reads the output of the process, formats it and inserts it into
        the contents pane.
        """
        while self.process.canReadLineStdout():
            s = self.process.readLineStdout()
            if self.rx_path.exactMatch(s):
                if self.lastProp:
                    itm = QListViewItem(self.propsList,
                        self.lastPath, self.lastProp, self.propBuffer)
                self.lastPath = self.rx_path.cap(1)
                self.lastProp = None
                self.propBuffer = QString('')
            elif self.rx_prop.exactMatch(s):
                if self.lastProp:
                    itm = QListViewItem(self.propsList,
                        self.lastPath, self.lastProp, self.propBuffer)
                self.lastProp = self.rx_prop.cap(1)
                self.propBuffer = self.rx_prop.cap(2)
            else:
                self.propBuffer.append(' ')
                self.propBuffer.append(s)
                
    def handleReadStderr(self):
        """
        Private slot to handle the readyReadStderr signal.
        
        It reads the error output of the process and inserts it into the
        error pane.
        """
        while self.process.canReadLineStderr():
            s = self.process.readLineStderr()
            self.errors.append(s)
