# -*- coding: utf-8 -*-

# Copyright (c) 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the VCS project helper for Mercurial.
"""

from qt import *

from VCS.ProjectHelper import VcsProjectHelper
from UI.E3Action import E3Action

class HgProjectHelper(VcsProjectHelper):
    """
    Class implementing the VCS project helper for Mercurial.
    """
    def __init__(self, vcsObject, projectObject, parent=None, name=None):
        """
        Constructor
        
        @param vcsObject reference to the vcs object
        @param projectObject reference to the project object
        @param parent parent widget (QWidget)
        @param name name of this object (string or QString)
        """
        VcsProjectHelper.__init__(self, vcsObject, projectObject, parent, name)
        
    def initActions(self):
        """
        Private method to generate the action objects.
        """
        self.vcsNewAct = E3Action(self.trUtf8('New from repository'),
                self.trUtf8('&New from repository...'),0,0,self,'mercurial_new')
        self.vcsNewAct.setStatusTip(self.trUtf8(
            'Create a new project from the VCS repository'
        ))
        self.vcsNewAct.setWhatsThis(self.trUtf8(
            """<b>New from repository</b>"""
            """<p>This creates a new local project from the VCS repository.</p>"""
        ))
        self.vcsNewAct.connectIt(SIGNAL('activated()'),self.vcsCheckout)
        self.actions.append(self.vcsNewAct)
        
        self.vcsUpdateAct = E3Action(self.trUtf8('Update from repository'),
                self.trUtf8('&Update from repository'),0,0,self,
                'mercurial_update')
        self.vcsUpdateAct.setStatusTip(self.trUtf8(
            'Update the local project from the VCS repository'
        ))
        self.vcsUpdateAct.setWhatsThis(self.trUtf8(
            """<b>Update from repository</b>"""
            """<p>This updates the local project from the VCS repository.</p>"""
        ))
        self.vcsUpdateAct.connectIt(SIGNAL('activated()'),self.vcsUpdate)
        self.actions.append(self.vcsUpdateAct)
        
        self.vcsMergeAct = E3Action(self.trUtf8('Update (merge) from repository'),
                self.trUtf8('Update (mer&ge) from repository'),
                0, 0, self, 'mercurial_merge')
        self.vcsMergeAct.setStatusTip(self.trUtf8(
            'Update the local project from the VCS repository with merging'
        ))
        self.vcsMergeAct.setWhatsThis(self.trUtf8(
            """<b>Update (merge) from repository</b>"""
            """<p>This updates the local project from the VCS repository and"""
            """ performs a merge operation.</p>"""
        ))
        self.vcsMergeAct.connectIt(SIGNAL('activated()'),self.hgMerge)
        self.actions.append(self.vcsMergeAct)
        
        self.vcsCommitAct = E3Action(self.trUtf8('Commit changes to repository'),
                self.trUtf8('&Commit changes to repository...'),0,0,self,
                'mercurial_commit')
        self.vcsCommitAct.setStatusTip(self.trUtf8(
            'Commit changes to the local project to the VCS repository'
        ))
        self.vcsCommitAct.setWhatsThis(self.trUtf8(
            """<b>Commit changes to repository</b>"""
            """<p>This commits changes to the local project to the VCS repository.</p>"""
        ))
        self.vcsCommitAct.connectIt(SIGNAL('activated()'),self.vcsCommit)
        self.actions.append(self.vcsCommitAct)
        
        self.vcsAddAct = E3Action(self.trUtf8('Add to repository'),
                self.trUtf8('&Add to repository...'),0,0,self,'mercurial_add')
        self.vcsAddAct.setStatusTip(self.trUtf8(
            'Add the local project to the VCS repository'
        ))
        self.vcsAddAct.setWhatsThis(self.trUtf8(
            """<b>Add to repository</b>"""
            """<p>This adds (imports) the local project to the VCS repository.</p>"""
        ))
        self.vcsAddAct.connectIt(SIGNAL('activated()'),self.vcsImport)
        self.actions.append(self.vcsAddAct)
        
        self.vcsRemoveAct = E3Action(self.trUtf8(
                'Remove from repository (and disk)'),
                self.trUtf8('&Remove from repository (and disk)'),
                0,0,self,'mercurial_remove')
        self.vcsRemoveAct.setStatusTip(self.trUtf8(
            'Remove the local project from the VCS repository (and  disk)'
        ))
        self.vcsRemoveAct.setWhatsThis(self.trUtf8(
            """<b>Remove from repository</b>"""
            """<p>This removes the local project from the VCS repository (and disk).</p>"""
        ))
        self.vcsRemoveAct.connectIt(SIGNAL('activated()'),self.vcsRemove)
        self.actions.append(self.vcsRemoveAct)
        
        self.vcsLogAct = E3Action(self.trUtf8(
                'Show log'),
                self.trUtf8('Show &log'),
                0, 0, self, 'mercurial_log')
        self.vcsLogAct.setStatusTip(self.trUtf8(
            'Show the log of the local project'
        ))
        self.vcsLogAct.setWhatsThis(self.trUtf8(
            """<b>Show log</b>"""
            """<p>This shows the log of the local project.</p>"""
        ))
        self.vcsLogAct.connectIt(SIGNAL('activated()'),self.vcsLog)
        self.actions.append(self.vcsLogAct)
        
        self.vcsDiffAct = E3Action(self.trUtf8('Show difference'),
                self.trUtf8('Show &difference'),
                0, 0, self, 'mercurial_diff')
        self.vcsDiffAct.setStatusTip(self.trUtf8(
            'Show the difference of the local project to the repository'
        ))
        self.vcsDiffAct.setWhatsThis(self.trUtf8(
            """<b>Show difference</b>"""
            """<p>This shows the difference of the local project to the repository.</p>"""
        ))
        self.vcsDiffAct.connectIt(SIGNAL('activated()'),self.vcsDiff)
        self.actions.append(self.vcsDiffAct)
        
        self.hgExtDiffAct = E3Action(self.trUtf8('Show difference (extended)'),
                self.trUtf8('Show difference (extended)'),
                0, 0, self, 'mercurial_extendeddiff')
        self.hgExtDiffAct.setStatusTip(self.trUtf8(
            'Show the difference of revisions of the project to the repository'
        ))
        self.hgExtDiffAct.setWhatsThis(self.trUtf8(
            """<b>Show difference (extended)</b>"""
            """<p>This shows the difference of selectable revisions of the project.</p>"""
        ))
        self.hgExtDiffAct.connectIt(SIGNAL('activated()'),self.hgExtendedDiff)
        self.actions.append(self.hgExtDiffAct)
        
        self.vcsStatusAct = E3Action(self.trUtf8('Show status'),
                self.trUtf8('Show &status'),
                0, 0, self, 'mercurial_status')
        self.vcsStatusAct.setStatusTip(self.trUtf8(
            'Show the status of the local project'
        ))
        self.vcsStatusAct.setWhatsThis(self.trUtf8(
            """<b>Show status</b>"""
            """<p>This shows the status of the local project.</p>"""
        ))
        self.vcsStatusAct.connectIt(SIGNAL('activated()'),self.vcsStatus)
        self.actions.append(self.vcsStatusAct)
        
        self.vcsTagAct = E3Action(self.trUtf8('Tag in repository'), 
                self.trUtf8('&Tag in repository...'),
                0, 0, self, 'mercurial_tag')
        self.vcsTagAct.setStatusTip(self.trUtf8(
            'Tag the local project in the repository'
        ))
        self.vcsTagAct.setWhatsThis(self.trUtf8(
            """<b>Tag in repository</b>"""
            """<p>This tags the local project in the repository.</p>"""
        ))
        self.vcsTagAct.connectIt(SIGNAL('activated()'),self.vcsTag)
        self.actions.append(self.vcsTagAct)
        
        self.vcsExportAct = E3Action(self.trUtf8('Export from repository'), 
                self.trUtf8('&Export from repository...'),
                0, 0, self, 'mercurial_export')
        self.vcsExportAct.setStatusTip(self.trUtf8(
            'Export a project from the repository'
        ))
        self.vcsExportAct.setWhatsThis(self.trUtf8(
            """<b>Export from repository</b>"""
            """<p>This exports a project from the repository.</p>"""
        ))
        self.vcsExportAct.connectIt(SIGNAL('activated()'),self.vcsExport)
        self.actions.append(self.vcsExportAct)
        
##        self.vcsPropsAct = E3Action(self.trUtf8('Command options'),
##                self.trUtf8('Command &options...'),0,0,self,
##                'mercurial_options')
##        self.vcsPropsAct.setStatusTip(self.trUtf8('Show the VCS command options'))
##        self.vcsPropsAct.setWhatsThis(self.trUtf8(
##            """<b>Command options...</b>"""
##            """<p>This shows a dialog to edit the VCS command options.</p>"""
##        ))
##        self.vcsPropsAct.connectIt(SIGNAL('activated()'),self.vcsCommandOptions)
##        self.actions.append(self.vcsPropsAct)
##        
        self.vcsRevertAct = E3Action(self.trUtf8('Revert changes'),
                self.trUtf8('Re&vert changes'),
                0, 0, self, 'mercurial_revert')
        self.vcsRevertAct.setStatusTip(self.trUtf8(
            'Revert all changes made to the local project'
        ))
        self.vcsRevertAct.setWhatsThis(self.trUtf8(
            """<b>Revert changes</b>"""
            """<p>This reverts all changes made to the local project.</p>"""
        ))
        self.vcsRevertAct.connectIt(SIGNAL('activated()'),self.vcsRevert)
        self.actions.append(self.vcsRevertAct)
        
        self.vcsSwitchAct = E3Action(self.trUtf8('Switch'),
                self.trUtf8('S&witch...'),
                0, 0, self, 'mercurial_switch')
        self.vcsSwitchAct.setStatusTip(self.trUtf8(
            'Switch the local copy to another tag/revision'
        ))
        self.vcsSwitchAct.setWhatsThis(self.trUtf8(
            """<b>Switch</b>"""
            """<p>This switches the local copy to another tag/revision.</p>"""
        ))
        self.vcsSwitchAct.connectIt(SIGNAL('activated()'),self.vcsSwitch)
        self.actions.append(self.vcsSwitchAct)
        
        self.vcsCleanupAct = E3Action(self.trUtf8('Cleanup'),
                self.trUtf8('Cleanu&p'),
                0, 0, self, 'mercurial_cleanup')
        self.vcsCleanupAct.setStatusTip(self.trUtf8(
            'Cleanup the local project'
        ))
        self.vcsCleanupAct.setWhatsThis(self.trUtf8(
            """<b>Cleanup</b>"""
            """<p>This performs a cleanup of the local project.</p>"""
        ))
        self.vcsCleanupAct.connectIt(SIGNAL('activated()'),self.vcsCleanup)
        self.actions.append(self.vcsCleanupAct)
        
        self.vcsCommandAct = E3Action(self.trUtf8('Execute command'),
                self.trUtf8('E&xecute command...'),
                0, 0, self, 'mercurial_command')
        self.vcsCommandAct.setStatusTip(self.trUtf8(
            'Execute an arbitrary VCS command'
        ))
        self.vcsCommandAct.setWhatsThis(self.trUtf8(
            """<b>Execute command</b>"""
            """<p>This opens a dialog to enter an arbitrary VCS command.</p>"""
        ))
        self.vcsCommandAct.connectIt(SIGNAL('activated()'),self.vcsCommand)
        self.actions.append(self.vcsCommandAct)
        
        self.hgTagListAct = E3Action(self.trUtf8('List tags'), 
                self.trUtf8('List tags...'),
                0, 0, self, 'mercurial_list_tags')
        self.hgTagListAct.setStatusTip(self.trUtf8(
            'List tags of the project'
        ))
        self.hgTagListAct.setWhatsThis(self.trUtf8(
            """<b>List tags</b>"""
            """<p>This lists the tags of the project.</p>"""
        ))
        self.hgTagListAct.connectIt(SIGNAL('activated()'),self.hgTagList)
        self.actions.append(self.hgTagListAct)
        
        self.hgPullAct = E3Action(self.trUtf8('Pull'), 
                self.trUtf8('Pull...'),
                0, 0, self, 'mercurial_pull')
        self.hgPullAct.setStatusTip(self.trUtf8(
            'Pull changesets'
        ))
        self.hgPullAct.setWhatsThis(self.trUtf8(
            """<b>Pull</b>"""
            """<p>Pull changesets not present in the repository.</p>"""
        ))
        self.hgPullAct.connectIt(SIGNAL('activated()'),self.hgPull)
        self.actions.append(self.hgPullAct)
        
        self.hgPushAct = E3Action(self.trUtf8('Push'), 
                self.trUtf8('Push...'),
                0, 0, self, 'mercurial_push')
        self.hgPushAct.setStatusTip(self.trUtf8(
            'Push changesets'
        ))
        self.hgPushAct.setWhatsThis(self.trUtf8(
            """<b>Push</b>"""
            """<p>Push changesets, which are new in the repository.</p>"""
        ))
        self.hgPushAct.connectIt(SIGNAL('activated()'),self.hgPush)
        self.actions.append(self.hgPushAct)
        
        self.hgIncomingAct = E3Action(self.trUtf8('Incoming'), 
                self.trUtf8('Incoming...'),
                0, 0, self, 'mercurial_incoming')
        self.hgIncomingAct.setStatusTip(self.trUtf8(
            'Check incoming changesets'
        ))
        self.hgIncomingAct.setWhatsThis(self.trUtf8(
            """<b>Incoming</b>"""
            """<p>Check changesets incoming to the repository.</p>"""
        ))
        self.hgIncomingAct.connectIt(SIGNAL('activated()'),self.hgIncoming)
        self.actions.append(self.hgIncomingAct)
        
        self.hgOutgoingAct = E3Action(self.trUtf8('Outgoing'), 
                self.trUtf8('Outgoing...'),
                0, 0, self, 'mercurial_outgoing')
        self.hgOutgoingAct.setStatusTip(self.trUtf8(
            'Check outgoing changesets'
        ))
        self.hgOutgoingAct.setWhatsThis(self.trUtf8(
            """<b>Outgoing</b>"""
            """<p>Check changesets outgoing from the repository.</p>"""
        ))
        self.hgOutgoingAct.connectIt(SIGNAL('activated()'),self.hgOutgoing)
        self.actions.append(self.hgOutgoingAct)
        
        self.hgManifestAct = E3Action(self.trUtf8('Show manifest'),
                self.trUtf8('Show &manifest'),
                0, 0, self, 'mercurial_manifest')
        self.hgManifestAct.setStatusTip(self.trUtf8(
            'Show the manifest of the local project'
        ))
        self.hgManifestAct.setWhatsThis(self.trUtf8(
            """<b>Show manifest</b>"""
            """<p>This shows the manifest of the local project.</p>"""
        ))
        self.hgManifestAct.connectIt(SIGNAL('activated()'),self.hgManifest)
        self.actions.append(self.hgManifestAct)
        
        self.hgUndoAct = E3Action(self.trUtf8('Undo last commit/pull'),
                self.trUtf8('&Undo last commit/pull'),
                0, 0, self, 'mercurial_undo')
        self.hgUndoAct.setStatusTip(self.trUtf8(
            'Undo the last commit or pull transaction'
        ))
        self.hgUndoAct.setWhatsThis(self.trUtf8(
            """<b>Undo last commit/pull</b>"""
            """<p>This undoes the last commit or pull transaction.</p>"""
        ))
        self.hgUndoAct.connectIt(SIGNAL('activated()'),self.hgUndo)
        self.actions.append(self.hgUndoAct)
    
    def initMenu(self):
        """
        Private method to generate the VCS menu.
        """
        try:
            menu = self.project.vcsMenu
        except:
            return
        menu.clear()

        lbl = QLabel(self.vcs.vcsName(), menu)
        lbl.setFrameStyle( QFrame.Panel | QFrame.Sunken )
        lbl.setAlignment(Qt.AlignHCenter)
        font = lbl.font()
        font.setBold(1)
        lbl.setFont(font)
        menu.insertItem(lbl)
        
        self.vcsUpdateAct.addTo(menu)
        self.vcsMergeAct.addTo(menu)
        self.vcsCommitAct.addTo(menu)
        menu.insertSeparator()
        self.hgPullAct.addTo(menu)
        self.hgPushAct.addTo(menu)
        self.hgIncomingAct.addTo(menu)
        self.hgOutgoingAct.addTo(menu)
        menu.insertSeparator()
        self.hgUndoAct.addTo(menu)
        menu.insertSeparator()
        self.vcsNewAct.addTo(menu)
        self.vcsExportAct.addTo(menu)
        menu.insertSeparator()
        self.vcsAddAct.addTo(menu)
        self.vcsRemoveAct.addTo(menu)
        menu.insertSeparator()
        self.vcsTagAct.addTo(menu)
        self.hgTagListAct.addTo(menu)
        menu.insertSeparator()
        self.vcsLogAct.addTo(menu)
        self.vcsStatusAct.addTo(menu)
        self.vcsDiffAct.addTo(menu)
        self.hgExtDiffAct.addTo(menu)
        self.hgManifestAct.addTo(menu)
        menu.insertSeparator()
        self.vcsRevertAct.addTo(menu)
        menu.insertSeparator()
        self.vcsSwitchAct.addTo(menu)
        menu.insertSeparator()
        self.vcsCleanupAct.addTo(menu)
        menu.insertSeparator()
        self.vcsCommandAct.addTo(menu)
##        menu.insertSeparator()
##        self.vcsPropsAct.addTo(menu)
    
    def hgTagList(self):
        """
        Private slot used to list the tags of the project.
        """
        self.vcs.hgListTags(self.project.ppath)
        
    def hgExtendedDiff(self):
        """
        Private slot used to perform a hg diff with the selection of revisions.
        """
        self.vcs.hgExtendedDiff(self.project.ppath)
        
    def hgMerge(self):
        """
        Private slot used to update the local project from the repository with merging.
        """
        self.vcs.hgMerge(self.project.ppath)
        
    def hgPull(self):
        """
        Private slot used to pull changesets into the repository.
        """
        self.vcs.hgPull(self.project.ppath)
        
    def hgPush(self):
        """
        Private slot used to push changesets from the repository.
        """
        self.vcs.hgPush(self.project.ppath)
        
    def hgIncoming(self):
        """
        Private slot used to check changesets incoming to the repository.
        """
        self.vcs.hgIncoming(self.project.ppath)
        
    def hgOutgoing(self):
        """
        Private slot used to check changesets outgoing from the repository.
        """
        self.vcs.hgOutgoing(self.project.ppath)

    def hgManifest(self):
        """
        Private slot used to show the manifest of the repository.
        """
        self.vcs.hgManifest(self.project.ppath)

    def hgUndo(self):
        """
        Private slot used to undo the last commit or pull transaction.
        """
        self.vcs.hgUndo(self.project.ppath)
