# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog starting a process and showing its output.
"""

import os

from qt import *

from KdeQt import KQMessageBox

from mercurial.ui import ui

from HgForm import HgForm
import Preferences

class HgDialog(HgForm, ui):
    """
    Module implementing a dialog as the Mercurial UI.
    """
    def __init__(self, text, command = '', parent = None, debug = 0):
        """
        Constructor
        
        @param text text to be shown by the label (string or QString)
        @param command command executed in the dialog (string)
        @param parent parent widget (QWidget)
        @param debug flag indicating debug operation (boolean)
        """
        HgForm.__init__(self, parent, None, 0)
        ui.__init__(self, verbose = 1, debug = debug, quiet = 0, interactive = 1)
        
        self.outputGroup.setTitle(text)
        self.intercept = 0
        self.inputText = ''
        
        command = unicode(command)
        if command:
            self.resultbox.append(command)
            self.resultbox.append('')
        
        self.show()
        qApp.processEvents()
        
    def finish(self):
        """
        Public slot called when the function is finished.
        """
        self.button.setEnabled(1)
        self.button.setDefault(1)
        self.button.setFocus()
        
        self.inputGroup.setEnabled(0)
        
        if Preferences.getVCS("AutoClose") and \
           not self.errors.length() > 0:
            self.accept()
        
    def passwordMode(self, isOn):
        """
        Private slot to handle the password checkbox toggled.
        
        @param isOn flag indicating the status of the check box (boolean)
        """
        if isOn:
            self.input.setEchoMode(QLineEdit.Password)
        else:
            self.input.setEchoMode(QLineEdit.Normal)
        
    def sendInput(self):
        """
        Private slot to send the input to mercurial.
        """
        self.inputText = unicode(self.input.text())
        
        self.resultbox.moveCursor(QTextEdit.MoveEnd, 0)
        if self.passwordCheckBox.isChecked():
            self.resultbox.insert(os.linesep)
        else:
            self.resultbox.insert(self.inputText + os.linesep)
        
        self.passwordCheckBox.setChecked(0)
        self.input.clear()
        
    def returnPressed(self):
        """
        Private slot to handle the press of the return key in the input field.
        """
        self.intercept = 1
        self.sendInput()
        
    def keyPressEvent(self, evt):
        """
        Protected slot to handle a key press event.
        
        @param evt the key press event (QKeyEvent)
        """
        if self.intercept:
            self.intercept = 0
            evt.accept()
            return
        HgForm.keyPressEvent(self, evt)

    ############################################################################
    # Overridden methods from mercurial.ui
    ############################################################################
    
    def write(self, *args):
        """
        Public method to write something to the output of the dialog.
        
        @param *args texts to be written
        """
        for a in args:
            self.resultbox.moveCursor(QTextEdit.MoveEnd, 0)
            self.resultbox.insert(unicode(a))

    def write_err(self, *args):
        """
        Public method to write something to the errors output of the dialog.
        
        @param *args error texts to be written
        """
        for a in args:
            self.errors.append(unicode(a))

    def prompt(self, msg, pat, default = "y"):
        """
        Public method to prompt the user for some input.
        
        @param msg prompt message to be shown (string)
        @param pat pattern of acceptable input (string)
        @param default default answer if we are in noninteractive mode (string)
        @return the entered text
        """
        if not self.interactive:
            return default
        
        while 1:
            self.write(msg, " ")
            while not self.inputText:
                qApp.processEvents()
            r = self.inputText
            self.inputText = ''
            if re.match(pat, r):
                return r
            else:
                self.write("unrecognized response\n")
    
    def status(self, *msg):
        """
        Public method to output a status message.
        
        @param status messages to show (strings)
        """
        if not self.quiet:
            self.write(*msg)
    
    def warn(self, *msg):
        """
        Public method to output a warning message.
        
        @param warning messages to show (strings)
        """
        self.write_err(*msg)
    
    def note(self, *msg):
        """
        Public method to output a note.
        
        @param notes to show (strings)
        """
        if self.verbose:
            self.write(*msg)
    
    def debug(self, *msg):
        """
        Public method to output a debug message.
        
        @param debug messages to show (strings)
        """
        if self.debugflag:
            self.write(*msg)
    
    def edit(self, text):
        """
        Public method to enter some text.
        
        @exception RuntimeError not implemented
        """
        raise RuntimeError, 'Not implemented'
    
class HgDummyDialog(ui):
    """
    Module implementing a dummy dialog as the Mercurial UI.
    """
    def __init__(self):
        """
        Constructor
        """
        ui.__init__(self, verbose = 0, debug = 0, quiet = 0, interactive = 1)

    ############################################################################
    # Overridden methods from mercurial.ui
    ############################################################################
    
    def write(self, *args):
        """
        Public method to write something to the output of the dialog.
        
        @param *args texts to be written
        """
        return

    def write_err(self, *args):
        """
        Public method to write something to the errors output of the dialog.
        
        @param *args error texts to be written
        """
        return

    def prompt(self, msg, pat, default = "y"):
        """
        Public method to prompt the user for some input.
        
        @param msg prompt message to be shown (string)
        @param pat pattern of acceptable input (string)
        @param default default answer if we are in noninteractive mode (string)
        @return the entered text
        """
        raise RuntimeError, 'Not implemented'
    
    def status(self, *msg):
        """
        Public method to output a status message.
        
        @param status messages to show (strings)
        """
        return
    
    def warn(self, *msg):
        """
        Public method to output a warning message.
        
        @param warning messages to show (strings)
        """
        return
    
    def note(self, *msg):
        """
        Public method to output a note.
        
        @param notes to show (strings)
        """
        return
    
    def debug(self, *msg):
        """
        Public method to output a debug message.
        
        @param debug messages to show (strings)
        """
        return
    
    def edit(self, text):
        """
        Public method to enter some text.
        
        @exception RuntimeError not implemented
        """
        raise RuntimeError, 'Not implemented'
    
