# -*- coding: utf-8 -*-

# Copyright (c) 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the VCS project browser helper for CVS.
"""

import os

from qt import *

from VCS.ProjectBrowserHelper import VcsProjectBrowserHelper
from UI.Browser import BrowserFile
from Project.ProjectBrowser import ProjectBrowserSimpleDir, ProjectBrowserDirectory

import Preferences

class CvsProjectBrowserHelper(VcsProjectBrowserHelper):
    """
    Class implementing the VCS project browser helper for CVS.
    """
    def __init__(self, vcsObject, browserObject, projectObject, isTranslationsBrowser,
        parent=None, name=None):
        """
        Constructor
        
        @param vcsObject reference to the vcs object
        @param browserObject reference to the project browser object
        @param projectObject reference to the project object
        @param isTranslationsBrowser flag indicating, the helper is requested for the
            translations browser (this needs some special treatment)
        @param parent parent widget (QWidget)
        @param name name of this object (string or QString)
        """
        VcsProjectBrowserHelper.__init__(self, vcsObject, browserObject, projectObject,
            isTranslationsBrowser, parent, name)
    
    def handleShowPopupMenu(self, menu, standardItems):
        """
        Slot called before the context menu is shown. 
        
        It enables/disables the VCS menu entries depending on the overall 
        VCS status and the file status.
        
        @param menu reference to the menu to be shown
        @param standardItems array of standard items that need activation/deactivation
            depending on the overall VCS status
        """
        if unicode(self.browser.currentItem().text(1)) == self.vcs.vcsName():
            for itm in self.vcsMenuItems:
                menu.setItemEnabled(itm, 1)
            for itm in self.vcsAddMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in standardItems:
                menu.setItemEnabled(itm, 0)
            if os.path.isdir(unicode(self.browser.currentItem().fileName())):
                menu.setItemEnabled(self.cvsMenuEdit, 0)
                menu.setItemEnabled(self.cvsMenuUnedit, 0)
        else:
            for itm in self.vcsMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in self.vcsAddMenuItems:
                menu.setItemEnabled(itm, 1)
            if 1 in self.browser.specialMenuEntries and \
               not os.path.isdir(unicode(self.browser.currentItem().fileName())):
                menu.setItemEnabled(self.vcsMenuAddTree, 0)
            for itm in standardItems:
                menu.setItemEnabled(itm, 1)
    
    def handleShowPopupMenuMulti(self, menu, standardItems):
        """
        Slot called before the context menu (multiple selections) is shown. 
        
        It enables/disables the VCS menu entries depending on the overall 
        VCS status and the files status.
        
        @param menu reference to the menu to be shown
        @param standardItems array of standard items that need activation/deactivation
            depending on the overall VCS status
        """
        vcsName = self.vcs.vcsName()
        items = self.browser.getSelectedItems()
        vcsItems = 0
        # determine number of selected items under VCS control
        for itm in items:
            if unicode(itm.text(1)) == vcsName:
                vcsItems += 1
        
        if vcsItems > 0:
            if vcsItems != len(items):
                for itm in self.vcsMultiMenuItems:
                    menu.setItemEnabled(itm, 0)
            else:
                for itm in self.vcsMultiMenuItems:
                    menu.setItemEnabled(itm, 1)
            for itm in self.vcsAddMultiMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in standardItems:
                menu.setItemEnabled(itm, 0)
            if self.__itemsHaveDirectories(items):
                menu.setItemEnabled(self.cvsMultiMenuEdit, 0)
                menu.setItemEnabled(self.cvsMultiMenuUnedit, 0)
        else:
            for itm in self.vcsMultiMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in self.vcsAddMultiMenuItems:
                menu.setItemEnabled(itm, 1)
            if 1 in self.browser.specialMenuEntries and \
               self.__itemsHaveFiles(items):
                menu.setItemEnabled(self.vcsMultiMenuAddTree, 0)
            for itm in standardItems:
                menu.setItemEnabled(itm, 1)
    
    def handleShowPopupMenuDir(self, menu, standardItems):
        """
        Slot called before the context menu is shown. 
        
        It enables/disables the VCS menu entries depending on the overall 
        VCS status and the directory status.
        
        @param menu reference to the menu to be shown
        @param standardItems array of standard items that need activation/deactivation
            depending on the overall VCS status
        """
        if unicode(self.browser.currentItem().text(1)) == self.vcs.vcsName():
            for itm in self.vcsDirMenuItems:
                menu.setItemEnabled(itm, 1)
            for itm in self.vcsAddDirMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in standardItems:
                menu.setItemEnabled(itm, 0)
        else:
            for itm in self.vcsDirMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in self.vcsAddDirMenuItems:
                menu.setItemEnabled(itm, 1)
            for itm in standardItems:
                menu.setItemEnabled(itm, 1)
    
    def handleShowPopupMenuDirMulti(self, menu, standardItems):
        """
        Slot called before the context menu is shown. 
        
        It enables/disables the VCS menu entries depending on the overall 
        VCS status and the directory status.
        
        @param menu reference to the menu to be shown
        @param standardItems array of standard items that need activation/deactivation
            depending on the overall VCS status
        """
        vcsName = self.vcs.vcsName()
        items = self.browser.getSelectedItems()
        vcsItems = 0
        # determine number of selected items under VCS control
        for itm in items:
            if unicode(itm.text(1)) == vcsName:
                vcsItems += 1
        
        if vcsItems > 0:
            if vcsItems != len(items):
                for itm in self.vcsDirMultiMenuItems:
                    menu.setItemEnabled(itm, 0)
            else:
                for itm in self.vcsDirMultiMenuItems:
                    menu.setItemEnabled(itm, 1)
            for itm in self.vcsAddDirMultiMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in standardItems:
                menu.setItemEnabled(itm, 0)
        else:
            for itm in self.vcsDirMultiMenuItems:
                menu.setItemEnabled(itm, 0)
            for itm in self.vcsAddDirMultiMenuItems:
                menu.setItemEnabled(itm, 1)
            for itm in standardItems:
                menu.setItemEnabled(itm, 1)

    ############################################################################
    # Private menu generation methods below
    ############################################################################

    def addVCSMenu(self, menu):
        """
        Public method used to add the VCS menu to all project browsers.
        
        @param menu reference to the menu to be amended
        """
        self.vcsMenuItems = []
        self.vcsAddMenuItems = []
        
        lbl = QLabel(self.vcs.vcsName(), menu)
        lbl.setFrameStyle( QFrame.Panel | QFrame.Sunken )
        lbl.setAlignment(Qt.AlignHCenter)
        font = lbl.font()
        font.setBold(1)
        lbl.setFont(font)
        menu.insertItem(lbl)
        
        itm = menu.insertItem(self.trUtf8('Update from repository'), self.handleVCSUpdate)
        self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Commit changes to repository...'), self.handleVCSCommit)
        self.vcsMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Add to repository'), self.handleVCSAdd)
        self.vcsAddMenuItems.append(itm)
        if 1 in self.browser.specialMenuEntries:
            itm = menu.insertItem(self.trUtf8('Add to repository (binary)'), self.handleCVSAddBinary)
            self.vcsAddMenuItems.append(itm)
            self.vcsMenuAddTree = menu.insertItem(self.trUtf8('Add tree to repository'), self.handleVCSAddTree)
            self.vcsAddMenuItems.append(self.vcsMenuAddTree)
        itm = menu.insertItem(self.trUtf8('Remove from repository (and disk)'), self.handleVCSRemove)
        self.vcsMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Edit'), self.handleCVSEdit)
        self.vcsMenuItems.append(itm)
        self.cvsMenuEdit = itm
        itm = menu.insertItem(self.trUtf8('Unedit'), self.handleCVSUnedit)
        self.vcsMenuItems.append(itm)
        self.cvsMenuUnedit = itm
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Show log'), self.handleVCSLog)
        self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show status'), self.handleVCSStatus)
        self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show difference'), self.handleVCSDiff)
        self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show history'), self.handleCVSHistory)
        self.vcsMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Revert changes'), self.handleVCSRevert)
        self.vcsMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Merge changes'), self.handleVCSMerge)
        self.vcsMenuItems.append(itm)
        menu.insertSeparator()
        menu.insertItem(self.trUtf8('Select all local file entries'),
                        self.browser.handleSelectLocalEntries)
        menu.insertItem(self.trUtf8('Select all versioned file entries'),
                        self.browser.handleSelectVCSEntries)
        menu.insertItem(self.trUtf8('Select all local directory entries'),
                        self.browser.handleSelectLocalDirEntries)
        menu.insertItem(self.trUtf8('Select all versioned directory entries'),
                        self.browser.handleSelectVCSDirEntries)
        
    def addVCSMenuMulti(self, menu):
        """
        Public method used to add the VCS menu for multi selection to all project browsers.
        
        @param menu reference to the menu to be amended
        """
        self.vcsMultiMenuItems = []
        self.vcsAddMultiMenuItems = []
        
        lbl = QLabel(self.vcs.vcsName(), menu)
        lbl.setFrameStyle( QFrame.Panel | QFrame.Sunken )
        lbl.setAlignment(Qt.AlignHCenter)
        font = lbl.font()
        font.setBold(1)
        lbl.setFont(font)
        menu.insertItem(lbl)
        
        itm = menu.insertItem(self.trUtf8('Update from repository'), self.handleVCSUpdate)
        self.vcsMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Commit changes to repository...'), self.handleVCSCommit)
        self.vcsMultiMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Add to repository'), self.handleVCSAdd)
        self.vcsAddMultiMenuItems.append(itm)
        if 1 in self.browser.specialMenuEntries:
            itm = menu.insertItem(self.trUtf8('Add to repository (binary)'), self.handleCVSAddBinary)
            self.vcsAddMultiMenuItems.append(itm)
            self.vcsMultiMenuAddTree = menu.insertItem(self.trUtf8('Add tree to repository'), self.handleVCSAddTree)
            self.vcsAddMultiMenuItems.append(self.vcsMultiMenuAddTree)
        itm = menu.insertItem(self.trUtf8('Remove from repository (and disk)'), self.handleVCSRemove)
        self.vcsMultiMenuItems.append(itm)
        self.vcsRemoveMultiMenuItem = itm
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Edit'), self.handleCVSEdit)
        self.vcsMultiMenuItems.append(itm)
        self.cvsMultiMenuEdit = itm
        itm = menu.insertItem(self.trUtf8('Unedit'), self.handleCVSUnedit)
        self.vcsMultiMenuItems.append(itm)
        self.cvsMultiMenuUnedit = itm
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Show status'), self.handleVCSStatus)
        self.vcsMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show difference'), self.handleVCSDiff)
        self.vcsMultiMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Revert changes'), self.handleVCSRevert)
        self.vcsMultiMenuItems.append(itm)
        menu.insertSeparator()
        menu.insertItem(self.trUtf8('Select all local file entries'),
                        self.browser.handleSelectLocalEntries)
        menu.insertItem(self.trUtf8('Select all versioned file entries'),
                        self.browser.handleSelectVCSEntries)
        menu.insertItem(self.trUtf8('Select all local directory entries'),
                        self.browser.handleSelectLocalDirEntries)
        menu.insertItem(self.trUtf8('Select all versioned directory entries'),
                        self.browser.handleSelectVCSDirEntries)
        
    def addVCSMenuBack(self, menu):
        """
        Public method used to add the VCS menu to all project browsers.
        
        @param menu reference to the menu to be amended
        """
        lbl = QLabel(self.vcs.vcsName(), menu)
        lbl.setFrameStyle( QFrame.Panel | QFrame.Sunken )
        lbl.setAlignment(Qt.AlignHCenter)
        font = lbl.font()
        font.setBold(1)
        lbl.setFont(font)
        menu.insertItem(lbl)
        
        menu.insertItem(self.trUtf8('Select all local file entries'),
                        self.browser.handleSelectLocalEntries)
        menu.insertItem(self.trUtf8('Select all versioned file entries'),
                        self.browser.handleSelectVCSEntries)
        menu.insertItem(self.trUtf8('Select all local directory entries'),
                        self.browser.handleSelectLocalDirEntries)
        menu.insertItem(self.trUtf8('Select all versioned directory entries'),
                        self.browser.handleSelectVCSDirEntries)
        
    def addVCSMenuDir(self, menu):
        """
        Public method used to add the VCS menu to all project browsers.
        
        @param menu reference to the menu to be amended
        """
        if menu is None:
            return
        
        self.vcsDirMenuItems = []
        self.vcsAddDirMenuItems = []
        
        lbl = QLabel(self.vcs.vcsName(), menu)
        lbl.setFrameStyle( QFrame.Panel | QFrame.Sunken )
        lbl.setAlignment(Qt.AlignHCenter)
        font = lbl.font()
        font.setBold(1)
        lbl.setFont(font)
        menu.insertItem(lbl)
        
        itm = menu.insertItem(self.trUtf8('Update from repository'), self.handleVCSUpdate)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Commit changes to repository...'), self.handleVCSCommit)
        self.vcsDirMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Add to repository'), self.handleVCSAdd)
        self.vcsAddDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Remove from repository (and disk)'), self.handleVCSRemove)
        self.vcsDirMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Show log'), self.handleVCSLog)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show status'), self.handleVCSStatus)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show difference'), self.handleVCSDiff)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show history'), self.handleCVSHistory)
        self.vcsDirMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Revert changes'), self.handleVCSRevert)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Merge changes'), self.handleVCSMerge)
        self.vcsDirMenuItems.append(itm)
        menu.insertSeparator()
        menu.insertItem(self.trUtf8('Select all local file entries'),
                        self.browser.handleSelectLocalEntries)
        menu.insertItem(self.trUtf8('Select all versioned file entries'),
                        self.browser.handleSelectVCSEntries)
        menu.insertItem(self.trUtf8('Select all local directory entries'),
                        self.browser.handleSelectLocalDirEntries)
        menu.insertItem(self.trUtf8('Select all versioned directory entries'),
                        self.browser.handleSelectVCSDirEntries)
        
    def addVCSMenuDirMulti(self, menu):
        """
        Public method used to add the VCS menu to all project browsers.
        
        @param menu reference to the menu to be amended
        """
        if menu is None:
            return
        
        self.vcsDirMultiMenuItems = []
        self.vcsAddDirMultiMenuItems = []
        
        lbl = QLabel(self.vcs.vcsName(), menu)
        lbl.setFrameStyle( QFrame.Panel | QFrame.Sunken )
        lbl.setAlignment(Qt.AlignHCenter)
        font = lbl.font()
        font.setBold(1)
        lbl.setFont(font)
        menu.insertItem(lbl)
        
        itm = menu.insertItem(self.trUtf8('Update from repository'), self.handleVCSUpdate)
        self.vcsDirMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Commit changes to repository...'), self.handleVCSCommit)
        self.vcsDirMultiMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Add to repository'), self.handleVCSAdd)
        self.vcsAddDirMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Remove from repository (and disk)'), self.handleVCSRemove)
        self.vcsDirMultiMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Show log'), self.handleVCSLog)
        self.vcsDirMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show status'), self.handleVCSStatus)
        self.vcsDirMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show difference'), self.handleVCSDiff)
        self.vcsDirMultiMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Show history'), self.handleCVSHistory)
        self.vcsDirMultiMenuItems.append(itm)
        menu.insertSeparator()
        itm = menu.insertItem(self.trUtf8('Revert changes'), self.handleVCSRevert)
        self.vcsDirMenuItems.append(itm)
        itm = menu.insertItem(self.trUtf8('Merge changes'), self.handleVCSMerge)
        self.vcsDirMenuItems.append(itm)
        menu.insertSeparator()
        menu.insertItem(self.trUtf8('Select all local file entries'),
                        self.browser.handleSelectLocalEntries)
        menu.insertItem(self.trUtf8('Select all versioned file entries'),
                        self.browser.handleSelectVCSEntries)
        menu.insertItem(self.trUtf8('Select all local directory entries'),
                        self.browser.handleSelectLocalDirEntries)
        menu.insertItem(self.trUtf8('Select all versioned directory entries'),
                        self.browser.handleSelectVCSDirEntries)
        
    ############################################################################
    # Menu handling methods below
    ############################################################################
    
    def handleCVSAddBinary(self):
        """
        Private slot called by the context menu.
        
        It is used to add the selected
        file/directory in binary mode to the VCS repository.
        """
        items = self.browser.getSelectedItems()
        names = [unicode(itm.fileName()) for itm in items]
        if len(names) == 1:
            self.vcs.vcsAddBinary(names[0], os.path.isdir(names[0]))
        else:
            self.vcs.vcsAddBinary(names)
        for itm, fn in zip(items, names):
            self.browser.updateVCSStatus(itm, fn)
        
    def handleCVSHistory(self):
        """
        Private slot called by the context menu to show the history of a file.
        """
        itm = self.browser.currentItem()
        fn = unicode(itm.fileName())
        self.vcs.vcsHistory(fn)
        
    def handleCVSEdit(self):
        """
        Private slot called by the context menu to edit a file (CVS).
        """
        names = [unicode(itm.fileName()) for itm in self.browser.getSelectedItems()]
        if self.isTranslationsBrowser:
            names += ['%s.qm' % os.path.splitext(name)[0] for name in names]
        self.vcs.cvsEdit(names)
        
    def handleCVSUnedit(self):
        """
        Private slot called by the context menu to unedit a file (CVS).
        """
        names = [unicode(itm.fileName()) for itm in self.browser.getSelectedItems()]
        if self.isTranslationsBrowser:
            names += ['%s.qm' % os.path.splitext(name)[0] for name in names]
        self.vcs.cvsUnedit(names)

    ############################################################################
    # Some private utility methods below
    ############################################################################
    
    def __itemsHaveFiles(self, items):
        """
        Private method to check, if items contain file type items.
        
        @param items items to check (list of QListViewItems)
        @return flag indicating items contain file type items (boolean)
        """
        for itm in items:
            if isinstance(itm, BrowserFile):
                return 1
        return 0

    def __itemsHaveDirectories(self, items):
        """
        Private method to check, if items contain directory type items.
        
        @param items items to check (list of QListViewItems)
        @return flag indicating items contain directory type items (boolean)
        """
        for itm in items:
            if isinstance(itm, ProjectBrowserDirectory):
                return 1
        return 0
