# -*- coding: utf-8 -*-

# Copyright (c) 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to show the output of the Mercurial manifest command.
"""

import types
import os

from qt import *

import mercurial.commands as commands
import mercurial.hg as hg
from mercurial.ui import ui

from ManifestForm import ManifestForm
import Preferences

class HgManifestDialog(ManifestForm, ui):
    """
    Module implementing a dialog to show the output of the Mercurial manifest command.
    """
    def __init__(self, vcs, parent = None):
        """
        Constructor
        
        @param vcs -- reference to the vcs object
        @param parent -- parent widget (QWidget)
        """
        ManifestForm.__init__(self, parent)
        ui.__init__(self, verbose = 0, debug = 0, quiet = 0, interactive = 1)
        
        self.setWFlags(self.getWFlags() | Qt.WDestructiveClose)
        self.vcs = vcs
        
        self.manifestList.setSorting(2)
        
    def start(self, ppath):
        """
        Public slot to start the hg manifest command.
        
        @param ppath directory name of the project (string)
        """
        try:
            self.setActiveWindow()
            self.raiseW()
        except:
            pass
        
        cwd = os.getcwd()
        os.chdir(ppath)
        try:
            repo = hg.repository(self, ppath)
            n = repo.manifest.tip()
            m = repo.manifest.read(n)
            mf = repo.manifest.readflags(n)
            files = m.keys()
            files.sort()
        
            for f in files:
                itm = QListViewItem(self.manifestList, 
                                    hg.hex(m[f]), mf[f] and "755" or "644", f)
        except:
            pass
        os.chdir(cwd)
    
    def finish(self):
        """
        Private slot called when the process finished or the user pressed the button.
        """
        self.okButton.setDefault(1)
        self.okButton.setFocus()
        
        self.inputGroup.setEnabled(0)
        self.refreshButton.setEnabled(1)

    ############################################################################
    # Overridden methods from mercurial.ui
    ############################################################################
    
    def write(self, *args):
        """
        Public method to write something to the output of the dialog.
        
        @param *args texts to be written
        """
        for a in args:
            self.errors.moveCursor(QTextEdit.MoveEnd, 0)
            self.errors.insert(unicode(a))

    def write_err(self, *args):
        """
        Public method to write something to the errors output of the dialog.
        
        @param *args error texts to be written
        """
        for a in args:
            self.errors.append(unicode(a))

    def prompt(self, msg, pat, default = "y"):
        """
        Public method to prompt the user for some input.
        
        @param msg prompt message to be shown (string)
        @param pat pattern of acceptable input (string)
        @param default default answer if we are in noninteractive mode (string)
        @return the entered text
        """
        if not self.interactive:
            return default
        
        while 1:
            self.write(msg, " ")
            while not self.inputText:
                qApp.processEvents()
            r = self.inputText
            self.inputText = ''
            if re.match(pat, r):
                return r
            else:
                self.write("unrecognized response\n")
    
    def status(self, *msg):
        """
        Public method to output a status message.
        
        @param status messages to show (strings)
        """
        if not self.quiet:
            self.write(*msg)
    
    def warn(self, *msg):
        """
        Public method to output a warning message.
        
        @param warning messages to show (strings)
        """
        self.write_err(*msg)
    
    def note(self, *msg):
        """
        Public method to output a note.
        
        @param notes to show (strings)
        """
        if self.verbose:
            self.write(*msg)
    
    def debug(self, *msg):
        """
        Public method to output a debug message.
        
        @param debug messages to show (strings)
        """
        if self.debugflag:
            self.write(*msg)
    
    def edit(self, text):
        """
        Public method to enter some text.
        
        @exception RuntimeError not implemented
        """
        raise RuntimeError, 'Not implemented'
