# -*- coding: utf-8 -*-

# Copyright (c) 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to show the output of the hg incoming command.
"""

import os

from qt import *

import mercurial.commands as commands
from mercurial.hg import repository, RepoError
from mercurial.ui import ui

from LogForm import LogForm

import Utilities

class HgIncomingDialog(LogForm, ui):
    """
    Module implementing a dialog to show the output of the hg incoming command.
    """
    def __init__(self, vcs, parent = None):
        """
        Constructor
        
        @param vcs reference to the vcs object
        @param parent parent widget (QWidget)
        """
        LogForm.__init__(self, parent)
        ui.__init__(self, verbose = 1, debug = 0, quiet = 0, interactive = 1)
        
        self.saveButton.hide()
        
        self.contentsGroup.setTitle(self.trUtf8('Incoming:'))
        QWhatsThis.add(self.contents,self.tr("<b>Mercurial Incoming</b>\n"
            "<p>This shows the output of the hg incoming command.</p>"
        ))
        QWhatsThis.add(self.errors,self.tr("<b>Mercurial incoming errors</b>\n"
            "<p>This shows possible error messages of the hg incoming command.</p>"
        ))
        
        self.setWFlags(self.getWFlags() | Qt.WDestructiveClose)
        self.vcs = vcs
        
    def start(self, ppath, source):
        """
        Public slot to start the cvs incoming command.
        
        @param ppath directory name of the project (string)
        @param source directory name or URL of the source repository (string)
        """
        self.changesetsfound = 0
        self.firstoutput = 1

        try:
            self.setActiveWindow()
            self.raiseW()
        except:
            pass
        
        cwd = os.getcwd()
        os.chdir(ppath)
        try:
            repo = repository(self, ppath)
            if not source:
                source = self.expandpath("default")
                if source == "default":
                    source = ""
            self.setCaption(self.trUtf8('Mercurial Incoming %1').arg(source))
            commands.incoming(self, repo, source)
        except RepoError, inst:
            self.warn("abort: ", inst, "\n")
            self.changesetsfound = 1
        except:
            pass
        os.chdir(cwd)
        
        # at exit
        self.closeButton.setFocus()
        
        self.contents.setCursorPosition(0, 0)
        self.contents.ensureCursorVisible()
        
        if not self.changesetsfound:
            self.contents.append(\
                self.trUtf8("<br>No incoming changesets available from '%1'")\
                .arg(source))

    ############################################################################
    # Overridden methods from mercurial.ui
    ############################################################################
    
    def write(self, *args):
        """
        Public method to write something to the output of the dialog.
        
        @param *args texts to be written
        """
        for a in args:
            for s in a.splitlines():
                if self.firstoutput:
                    self.contents.clear()
                    self.firstoutput = 0
                if s.startswith("changeset:"):
                    self.changesetsfound = 1
                    self.contents.append('<hr>')
                    dstr = QString('<b>%1</b>').arg(s)
                elif s.startswith("user:"):
                    dstr = QString('<i>%1</i>').arg(s)
                elif s.startswith("date:"):
                    dstr = QString('<i>%1</i>').arg(s)
                else:
                    if s:
                        dstr = QString(Utilities.html_encode(unicode(s)))
                    else:
                        dstr = QString('<br>')
                self.contents.append(dstr)

    def write_err(self, *args):
        """
        Public method to write something to the errors output of the dialog.
        
        @param *args error texts to be written
        """
        for a in args:
            self.errors.append(unicode(a))

    def prompt(self, msg, pat, default = "y"):
        """
        Public method to prompt the user for some input.
        
        @param msg prompt message to be shown (string)
        @param pat pattern of acceptable input (string)
        @param default default answer if we are in noninteractive mode (string)
        @return the entered text
        """
        raise RuntimeError, 'Not implemented'
    
    def status(self, *msg):
        """
        Public method to output a status message.
        
        @param status messages to show (strings)
        """
        if not self.quiet:
            self.write(*msg)
    
    def warn(self, *msg):
        """
        Public method to output a warning message.
        
        @param warning messages to show (strings)
        """
        self.write_err(*msg)
    
    def note(self, *msg):
        """
        Public method to output a note.
        
        @param notes to show (strings)
        """
        if self.verbose:
            self.write(*msg)
    
    def debug(self, *msg):
        """
        Public method to output a debug message.
        
        @param debug messages to show (strings)
        """
        if self.debugflag:
            self.write(*msg)
    
    def edit(self, text):
        """
        Public method to enter some text.
        
        @exception RuntimeError not implemented
        """
        raise RuntimeError, 'Not implemented'
