# -*- coding: utf-8 -*-

# Copyright (c) 2004 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing functions dealing with keyboard shortcuts.

<b>Note</b>: These functions are imported into the Preferences
package and should be called like Preferences.readShortcuts(...).
"""

from qt import *

from KdeQt import KQMessageBox

from Preferences import Prefs, syncPreferences

from XML.XMLUtilities import make_parser
from XML.XMLErrorHandler import XMLErrorHandler, XMLFatalParseError
from XML.ShortcutsHandler import ShortcutsHandler
from XML.ShortcutsWriter import ShortcutsWriter
from XML.XMLEntityResolver import XMLEntityResolver

def readShortcuts(ui, prefClass = Prefs):
    """
    Module function to read the keyboard shortcuts for the defined QActions.
    
    @param ui reference to the main widget (UI.UserInterface)
    @param prefClass preferences class used as the storage area
    """
    for act in ui.getProject().getActions():
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Project/%1/Accel").arg(act.name()))
        if ok:
            act.setAccel(QKeySequence(accel))
        else:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/Project/%1").arg(act.text()))
            if ok:
                act.setAccel(QKeySequence(accel))
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Project/%1/AltAccel").arg(act.name()))
        if ok:
            act.setAlternateAccel(QKeySequence(accel))
            
    for act in ui.getActions():
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/General/%1/Accel").arg(act.name()))
        if ok:
            act.setAccel(QKeySequence(accel))
        else:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/General/%1").arg(act.text()))
            if ok:
                act.setAccel(QKeySequence(accel))
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/General/%1/AltAccel").arg(act.name()))
        if ok:
            act.setAlternateAccel(QKeySequence(accel))
            
    for act in ui.getDebugger().getActions():
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Debug/%1/Accel").arg(act.name()))
        if ok:
            act.setAccel(QKeySequence(accel))
        else:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/Debug/%1").arg(act.text()))
            if ok:
                act.setAccel(QKeySequence(accel))
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Debug/%1/AltAccel").arg(act.name()))
        if ok:
            act.setAlternateAccel(QKeySequence(accel))
            
    for act in ui.getWizards().getActions():
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Wizards/%1/Accel").arg(act.name()))
        if ok:
            act.setAccel(QKeySequence(accel))
        else:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/Wizards/%1").arg(act.text()))
            if ok:
                act.setAccel(QKeySequence(accel))
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Wizards/%1/AltAccel").arg(act.name()))
        if ok:
            act.setAlternateAccel(QKeySequence(accel))
            
    for act in ui.getViewManager().getActions('edit'):
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Edit/%1/Accel").arg(act.name()))
        if ok:
            act.setAccel(QKeySequence(accel))
        else:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/Edit/%1").arg(act.text()))
            if ok:
                act.setAccel(QKeySequence(accel))
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Edit/%1/AltAccel").arg(act.name()))
        if ok:
            act.setAlternateAccel(QKeySequence(accel))
            
    for act in ui.getViewManager().getActions('file'):
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/File/%1/Accel").arg(act.name()))
        if ok:
            act.setAccel(QKeySequence(accel))
        else:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/File/%1").arg(act.text()))
            if ok:
                act.setAccel(QKeySequence(accel))
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/File/%1/AltAccel").arg(act.name()))
        if ok:
            act.setAlternateAccel(QKeySequence(accel))
            
    for act in ui.getViewManager().getActions('search'):
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Search/%1/Accel").arg(act.name()))
        if ok:
            act.setAccel(QKeySequence(accel))
        else:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/Search/%1").arg(act.text()))
            if ok:
                act.setAccel(QKeySequence(accel))
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Search/%1/AltAccel").arg(act.name()))
        if ok:
            act.setAlternateAccel(QKeySequence(accel))
            
    for act in ui.getViewManager().getActions('view'):
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/View/%1/Accel").arg(act.name()))
        if ok:
            act.setAccel(QKeySequence(accel))
        else:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/View/%1").arg(act.text()))
            if ok:
                act.setAccel(QKeySequence(accel))
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/View/%1/AltAccel").arg(act.name()))
        if ok:
            act.setAlternateAccel(QKeySequence(accel))
            
    for act in ui.getViewManager().getActions('macro'):
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Macro/%1/Accel").arg(act.name()))
        if ok:
            act.setAccel(QKeySequence(accel))
        else:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/Macro/%1").arg(act.text()))
            if ok:
                act.setAccel(QKeySequence(accel))
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Macro/%1/AltAccel").arg(act.name()))
        if ok:
            act.setAlternateAccel(QKeySequence(accel))
            
    for act in ui.getViewManager().getActions('bookmark'):
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Bookmarks/%1/Accel").arg(act.name()))
        if ok:
            act.setAccel(QKeySequence(accel))
        else:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/Bookmarks/%1").arg(act.text()))
            if ok:
                act.setAccel(QKeySequence(accel))
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Bookmarks/%1/AltAccel").arg(act.name()))
        if ok:
            act.setAlternateAccel(QKeySequence(accel))
            
    actions = ui.getViewManager().getActions('window')
    if actions:
        for act in actions:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/Window/%1/Accel").arg(act.name()))
            if ok:
                act.setAccel(QKeySequence(accel))
            else:
                accel, ok = prefClass.settings.readEntry(\
                    QString("/eric3/Shortcuts/Window/%1").arg(act.text()))
                if ok:
                    act.setAccel(QKeySequence(accel))
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/Window/%1/AltAccel").arg(act.name()))
            if ok:
                act.setAlternateAccel(QKeySequence(accel))
                
    ref = ui.getRefactoring()
    if ref is not None:
        actions = ref.getActions()
        for act in actions:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/Refactoring/%1/Accel").arg(act.name()))
            if ok:
                act.setAccel(QKeySequence(accel))
            else:
                accel, ok = prefClass.settings.readEntry(\
                    QString("/eric3/Shortcuts/Refactoring/%1").arg(act.text()))
                if ok:
                    act.setAccel(QKeySequence(accel))
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/Refactoring/%1/AltAccel").arg(act.name()))
            if ok:
                act.setAlternateAccel(QKeySequence(accel))
    
def saveShortcuts(prefClass = Prefs):
    """
    Module function to write the keyboard shortcuts for the defined QActions.
    
    @param prefClass preferences class used as the storage area
    """
    ui = qApp.mainWidget()
    
    # step 1: clear all previously saved shortcuts
    subkeyList = prefClass.settings.subkeyList("/eric3/Shortcuts")
    for subkey in subkeyList:
        entryList = prefClass.settings.entryList(QString("/eric3/Shortcuts/%1").arg(subkey))
        for entry in entryList:
            prefClass.settings.removeEntry(\
                QString("/eric3/Shortcuts/%1/%2/Accel").arg(subkey).arg(entry))
            prefClass.settings.removeEntry(\
                QString("/eric3/Shortcuts/%1/%2/AltAccel").arg(subkey).arg(entry))
                
    # step 2: save the various shortcuts
    for act in ui.getProject().getActions():
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Project/%1/Accel").arg(act.name()), 
            QString(act.accel()))
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Project/%1/AltAccel").arg(act.name()), 
            QString(act.alternateAccel()))
        
    for act in ui.getActions():
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/General/%1/Accel").arg(act.name()), 
            QString(act.accel()))
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/General/%1/AltAccel").arg(act.name()), 
            QString(act.alternateAccel()))
        
    for act in ui.getDebugger().getActions():
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Debug/%1/Accel").arg(act.name()), 
            QString(act.accel()))
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Debug/%1/AltAccel").arg(act.name()), 
            QString(act.alternateAccel()))
        
    for act in ui.getWizards().getActions():
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Wizards/%1/Accel").arg(act.name()), 
            QString(act.accel()))
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Wizards/%1/AltAccel").arg(act.name()), 
            QString(act.alternateAccel()))
        
    for act in ui.getViewManager().getActions('edit'):
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Edit/%1/Accel").arg(act.name()), 
            QString(act.accel()))
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Edit/%1/AltAccel").arg(act.name()), 
            QString(act.alternateAccel()))
        
    for act in ui.getViewManager().getActions('file'):
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/File/%1/Accel").arg(act.name()), 
            QString(act.accel()))
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/File/%1/AltAccel").arg(act.name()), 
            QString(act.alternateAccel()))
        
    for act in ui.getViewManager().getActions('search'):
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Search/%1/Accel").arg(act.name()), 
            QString(act.accel()))
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Search/%1/AltAccel").arg(act.name()), 
            QString(act.alternateAccel()))
        
    for act in ui.getViewManager().getActions('view'):
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/View/%1/Accel").arg(act.name()), 
            QString(act.accel()))
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/View/%1/AltAccel").arg(act.name()), 
            QString(act.alternateAccel()))
        
    for act in ui.getViewManager().getActions('macro'):
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Macro/%1/Accel").arg(act.name()), 
            QString(act.accel()))
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Macro/%1/AltAccel").arg(act.name()), 
            QString(act.alternateAccel()))
        
    for act in ui.getViewManager().getActions('bookmark'):
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Bookmarks/%1/Accel").arg(act.name()), 
            QString(act.accel()))
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Bookmarks/%1/AltAccel").arg(act.name()), 
            QString(act.alternateAccel()))
        
    actions = ui.getViewManager().getActions('window')
    if actions:
        for act in actions:
            prefClass.settings.writeEntry(\
                QString("/eric3/Shortcuts/Window/%1/Accel").arg(act.name()), 
                QString(act.accel()))
            prefClass.settings.writeEntry(\
                QString("/eric3/Shortcuts/Window/%1/AltAccel").arg(act.name()), 
                QString(act.alternateAccel()))
                
    ref = ui.getRefactoring()
    if ref is not None:
        actions = ref.getActions()
        for act in actions:
            prefClass.settings.writeEntry(\
                QString("/eric3/Shortcuts/Refactoring/%1/Accel").arg(act.name()), 
                QString(act.accel()))
            prefClass.settings.writeEntry(\
                QString("/eric3/Shortcuts/Refactoring/%1/AltAccel").arg(act.name()), 
                QString(act.alternateAccel()))

def exportShortcuts(fn):
    """
    Module function to export the keyboard shortcuts for the defined QActions.
    
    @param fn filename of the export file (string)
    @return flag indicating success
    """
    try:
        if fn.lower().endswith("e3kz"):
            try:
                import gzip
            except ImportError:
                KQMessageBox.critical(None,
                    qApp.translate("Shortcuts", "Export Keyboard Shortcuts"),
                    qApp.translate("Shortcuts", """Compressed keyboard shortcut files not supported. The compression library is missing."""),
                    qApp.translate("Shortcuts", "&Abort"),
                    QString.null,
                    QString.null,
                    0, -1)
                return 0
            f = gzip.open(fn, "wb")
        else:
            f = open(fn, "wb")
        
        ShortcutsWriter(f).writeXML()
        
        f.close()
        return 1
    except IOError:
        return 0

def importShortcuts(fn):
    """
    Module function to import the keyboard shortcuts for the defined E3Actions.
    
    @param fn filename of the import file (string)
    @return flag indicating success
    """
    fn = unicode(fn)
    try:
        if fn.lower().endswith("e3kz"):
            try:
                import gzip
            except ImportError:
                KQMessageBox.critical(None,
                    qApp.translate("Shortcuts", "Import Keyboard Shortcuts"),
                    qApp.translate("Shortcuts", """Compressed keyboard shortcut files not supported. The compression library is missing."""),
                    qApp.translate("Shortcuts", "&Abort"),
                    QString.null,
                    QString.null,
                    0, -1)
                return 0
            f = gzip.open(fn, "rb")
        else:
            f = open(fn, "rb")
        try:
            line = f.readline()
            dtdLine = f.readline()
        finally:
            f.close()
    except IOError:
        KQMessageBox.critical(None,
            qApp.translate("Shortcuts", "Import Keyboard Shortcuts"),
            qApp.translate("Shortcuts", 
                "<p>The keyboard shortcuts could not be read from file <b>%1</b>.</p>")
                .arg(fn),
            qApp.translate("Shortcuts", "&Abort"))
        return 0
    
    parser = make_parser(dtdLine.startswith("<!DOCTYPE"))
    handler = ShortcutsHandler()
    er = XMLEntityResolver()
    eh = XMLErrorHandler()
    
    parser.setContentHandler(handler)
    parser.setEntityResolver(er)
    parser.setErrorHandler(eh)
    
    try:
        if fn.lower().endswith("e3kz"):
            try:
                import gzip
            except ImportError:
                KQMessageBox.critical(None,
                    qApp.translate("Shortcuts", "Import Keyboard Shortcuts"),
                    qApp.translate("Shortcuts", """Compressed keyboard shortcut files not supported. The compression library is missing."""),
                    qApp.translate("Shortcuts", "&Abort"),
                    QString.null,
                    QString.null,
                    0, -1)
                return 0
            f = gzip.open(fn, "rb")
        else:
            f = open(fn, "rb")
        try:
            parser.parse(f)
        finally:
            f.close()
    except IOError:
        KQMessageBox.critical(None,
            qApp.translate("Shortcuts", "Import Keyboard Shortcuts"),
            qApp.translate("Shortcuts", 
                "<p>The keyboard shortcuts could not be read from file <b>%1</b>.</p>")
                .arg(fn),
            qApp.translate("Shortcuts", "&Abort"))
        return 0
        
    except XMLFatalParseError:
        KQMessageBox.critical(None,
            qApp.translate("Shortcuts", "Import Keyboard Shortcuts"),
            qApp.translate("Shortcuts", 
                "<p>The keyboard shortcuts file <b>%1</b> has invalid contents.</p>")
                .arg(fn),
            qApp.translate("Shortcuts", "&Abort"))
        eh.showParseMessages()
        return 0
        
    eh.showParseMessages()
    
    shortcuts = handler.getShortcuts()
    
    ui = qApp.mainWidget()
    
    if handler.getVersion() == "3.5":
        setActions_35(shortcuts, ui)
    else:
        setActions_36(shortcuts, ui)
    
    saveShortcuts()
    syncPreferences()
    
    return 1

def setActions_36(shortcuts, ui):
    """
    Function to set actions based on new format shortcuts file.
    
    @param shortcuts dictionary containing the accelerator information 
        read from a XML file
    @param ui reference to the main window
    """
    if shortcuts.has_key("Project"):
        actions = ui.getProject().getActions()
        sdict = shortcuts["Project"]
        for act in actions:
            try:
                accel,altAccel = sdict[str(act.name())]
                act.setAccel(QKeySequence(accel))
                act.setAlternateAccel(QKeySequence(altAccel))
            except KeyError:
                pass
    
    if shortcuts.has_key("General"):
        actions = ui.getActions()
        sdict = shortcuts["General"]
        for act in actions:
            try:
                accel,altAccel = sdict[str(act.name())]
                act.setAccel(QKeySequence(accel))
                act.setAlternateAccel(QKeySequence(altAccel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Debug"):
        actions = ui.getDebugger().getActions()
        sdict = shortcuts["Debug"]
        for act in actions:
            try:
                accel,altAccel = sdict[str(act.name())]
                act.setAccel(QKeySequence(accel))
                act.setAlternateAccel(QKeySequence(altAccel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Wizards"):
        actions = ui.getWizards().getActions()
        sdict = shortcuts["Wizards"]
        for act in actions:
            try:
                accel,altAccel = sdict[str(act.name())]
                act.setAccel(QKeySequence(accel))
                act.setAlternateAccel(QKeySequence(altAccel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Edit"):
        actions = ui.getViewManager().getActions('edit')
        sdict = shortcuts["Edit"]
        for act in actions:
            try:
                accel,altAccel = sdict[str(act.name())]
                act.setAccel(QKeySequence(accel))
                act.setAlternateAccel(QKeySequence(altAccel))
            except KeyError:
                pass
    
    if shortcuts.has_key("File"):
        actions = ui.getViewManager().getActions('file')
        sdict = shortcuts["File"]
        for act in actions:
            try:
                accel,altAccel = sdict[str(act.name())]
                act.setAccel(QKeySequence(accel))
                act.setAlternateAccel(QKeySequence(altAccel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Search"):
        actions = ui.getViewManager().getActions('search')
        sdict = shortcuts["Search"]
        for act in actions:
            try:
                accel,altAccel = sdict[str(act.name())]
                act.setAccel(QKeySequence(accel))
                act.setAlternateAccel(QKeySequence(altAccel))
            except KeyError:
                pass
    
    if shortcuts.has_key("View"):
        actions = ui.getViewManager().getActions('view')
        sdict = shortcuts["View"]
        for act in actions:
            try:
                accel,altAccel = sdict[str(act.name())]
                act.setAccel(QKeySequence(accel))
                act.setAlternateAccel(QKeySequence(altAccel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Macro"):
        actions = ui.getViewManager().getActions('macro')
        sdict = shortcuts["Macro"]
        for act in actions:
            try:
                accel,altAccel = sdict[str(act.name())]
                act.setAccel(QKeySequence(accel))
                act.setAlternateAccel(QKeySequence(altAccel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Bookmarks"):
        actions = ui.getViewManager().getActions('bookmark')
        sdict = shortcuts["Bookmarks"]
        for act in actions:
            try:
                accel,altAccel = sdict[str(act.name())]
                act.setAccel(QKeySequence(accel))
                act.setAlternateAccel(QKeySequence(altAccel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Window"):
        actions = ui.getViewManager().getActions('window')
        if actions:
            sdict = shortcuts["Window"]
            for act in actions:
                try:
                    accel,altAccel = sdict[str(act.name())]
                    act.setAccel(QKeySequence(accel))
                    act.setAlternateAccel(QKeySequence(altAccel))
                except KeyError:
                    pass
    
    if shortcuts.has_key("Refactoring"):
        ref = ui.getRefactoring()
        if ref is not None:
            actions = ref.getActions()
            sdict = shortcuts["Refactoring"]
            for act in actions:
                try:
                    accel,altAccel = sdict[str(act.name())]
                    act.setAccel(QKeySequence(accel))
                    act.setAlternateAccel(QKeySequence(altAccel))
                except KeyError:
                    pass
    
def setActions_35(shortcuts, ui):
    """
    Function to set actions based on old format shortcuts file.
    
    @param shortcuts dictionary containing the accelerator information 
        read from a XML file
    @param ui reference to the main window
    """
    if shortcuts.has_key("Project"):
        actions = ui.getProject().getActions()
        sdict = shortcuts["Project"]
        for act in actions:
            try:
                accel = sdict[str(act.text())][0]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("General"):
        actions = ui.getActions()
        sdict = shortcuts["General"]
        for act in actions:
            try:
                accel = sdict[str(act.text())][0]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Debug"):
        actions = ui.getDebugger().getActions()
        sdict = shortcuts["Debug"]
        for act in actions:
            try:
                accel = sdict[str(act.text())][0]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Wizards"):
        actions = ui.getWizards().getActions()
        sdict = shortcuts["Wizards"]
        for act in actions:
            try:
                accel = sdict[str(act.text())][0]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Edit"):
        actions = ui.getViewManager().getActions('edit')
        sdict = shortcuts["Edit"]
        for act in actions:
            try:
                accel = sdict[str(act.text())][0]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("File"):
        actions = ui.getViewManager().getActions('file')
        sdict = shortcuts["File"]
        for act in actions:
            try:
                accel = sdict[str(act.text())][0]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Search"):
        actions = ui.getViewManager().getActions('search')
        sdict = shortcuts["Search"]
        for act in actions:
            try:
                accel = sdict[str(act.text())][0]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("View"):
        actions = ui.getViewManager().getActions('view')
        sdict = shortcuts["View"]
        for act in actions:
            try:
                accel = sdict[str(act.text())][0]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Macro"):
        actions = ui.getViewManager().getActions('macro')
        sdict = shortcuts["Macro"]
        for act in actions:
            try:
                accel = sdict[str(act.text())][0]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Bookmarks"):
        actions = ui.getViewManager().getActions('bookmark')
        sdict = shortcuts["Bookmarks"]
        for act in actions:
            try:
                accel = sdict[str(act.text())][0]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Window"):
        actions = ui.getViewManager().getActions('window')
        if actions:
            sdict = shortcuts["Window"]
            for act in actions:
                try:
                    accel = sdict[str(act.text())][0]
                    act.setAccel(QKeySequence(accel))
                except KeyError:
                    pass
    
    if shortcuts.has_key("Refactoring"):
        ref = ui.getRefactoring()
        if ref is not None:
            actions = ref.getActions()
            sdict = shortcuts["Refactoring"]
            for act in actions:
                try:
                    accel = sdict[str(act.text())][0]
                    act.setAccel(QKeySequence(accel))
                except KeyError:
                    pass
