/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: TargetDB_Write.c,v 2.0 1995/02/13 20:48:42 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "File.h"
#include "TargetDB.h"

static char TargetDBSpecialChars[] =
{
  ' ', '\t', '\n', '\'', '\"', '\\',
  '\0'
};

static int NTargetDBSpecialChars =
sizeof(TargetDBSpecialChars) /
sizeof(TargetDBSpecialChars[0]) - 1;


static int TargetSource_Write();
static int TargetSourceList_Write();
static int TargetDB_Tree_Write();
static int TargetDB_WriteUncompressed();


static int 
TargetSource_Write(fd, sourcep, nodename, sourcepathset)
     FILE *fd;
     TARGETSOURCE *sourcep;
     char *nodename;
     STRINGSET *sourcepathset;
{
  TARGETSOURCE *secondarysourcep;

  char potentialparentpath[MAXPATHLEN + 1];
  char *last_slash, *leaf_node;
  int sourceindex;
  Boolean childofsource;

  if (sourcep != NULL) {
    if (TARGETSOURCE_Path(sourcep) == NULL) {
      FatalError(E_BADTARGETSOURCEPATH,
		 "No path fournd in target source.\n");
    } else {
      sourceindex = -1;
      childofsource = FALSE;
      (void) strcpy(potentialparentpath, TARGETSOURCE_Path(sourcep));
      last_slash = strrchr(potentialparentpath, '/');
      leaf_node = (last_slash == NULL) ? potentialparentpath : last_slash + 1;
      if (String_Comparator(leaf_node, nodename) == 0) {
	if (last_slash == NULL) {
	  (void) strcpy(potentialparentpath, "/");
	} else {		/* potentialparentpath ends at last_slash */
	  *last_slash = '\0';
	}
	sourceindex = StringSet_Index(sourcepathset, potentialparentpath);
      }
      if (sourceindex >= 0) {
	childofsource = TRUE;
      } else {
	sourceindex = StringSet_Index(sourcepathset,
				      TARGETSOURCE_Path(sourcep));
      }

      if (sourceindex < 0) {
	fprintf(fd, " -1 ");
	String_Write(fd, TARGETSOURCE_Path(sourcep),
		     '\\', ' ',
		     TargetDBSpecialChars, NTargetDBSpecialChars);
      } else {
	fprintf(fd, " %d", sourceindex);
	if (childofsource) {
	  fprintf(fd, " 1");
	} else {		/* !childofsource */
	  fprintf(fd, " 0");
	}
      }
    }
    if (PROGRAM_ErrorNo == E_NULL) {
      fprintf(fd, " %d", TARGETSOURCE_UpdateSpec(sourcep));
      fprintf(fd, " %d", TARGETSOURCE_CollectionId(sourcep));
      secondarysourcep = TARGETSOURCE_SecondarySource(sourcep);
      if (secondarysourcep == NULL) {
	fprintf(fd, " 0\n");
      } else {
	fprintf(fd, " 1\n");
	(void) TargetSource_Write(fd, secondarysourcep,
				  nodename, sourcepathset);
      }
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? 0 : -1;
}


static int 
TargetSourceList_Write(fd, sourcelist, nodename, sourcepathset)
     FILE *fd;
     TARGETSOURCELIST *sourcelist;
     char *nodename;
     STRINGSET *sourcepathset;
{
  register int i;

  if (sourcelist == NULL) {
    fprintf(fd, " 0\n");	/* zero sources */
  } else {
    fprintf(fd, " %d\n", TARGETSOURCELIST_NSources(sourcelist));
    if (TARGETSOURCELIST_NSources(sourcelist) > 0) {
      if (TARGETSOURCELIST_SourceArray(sourcelist) == NULL) {
	FatalError(E_BADTARGETSOURCELIST,
		   "No source found in source list as expected.\n");
      } else {
	for (i = 0;
	     (i < TARGETSOURCELIST_NSources(sourcelist))
	     && (PROGRAM_ErrorNo == E_NULL);
	     i++) {
	  TargetSource_Write(fd,
			     TARGETSOURCELIST_Source(sourcelist, i),
			     nodename,
			     sourcepathset);
	}
      }
    }
  }

  return (PROGRAM_ErrorNo == E_NULL) ? 0 : -1;
}


static int 
TargetDB_Tree_Write(fd, tdb, sourcepathset)
     FILE *fd;
     TARGETDB *tdb;
     STRINGSET *sourcepathset;
{
  register TARGETDB **tp;
  register int i;

  STRINGSET *childsourcepathset;

  if (tdb != NULL) {
    if (TARGETDB_Name(tdb) == NULL) {
      FatalError(E_BADTARGETNAME, "Unnamed entry in target database.\n");
    }
    if (PROGRAM_ErrorNo == E_NULL) {
      String_Write(fd, TARGETDB_Name(tdb),
		   '\\', ' ',
		   TargetDBSpecialChars, NTargetDBSpecialChars);
    }
    if (PROGRAM_ErrorNo == E_NULL) {
      fprintf(fd, " %d", TARGETDB_UpdateSpec(tdb));
      if (TARGETDB_UpdateSpec(tdb)
	  & (U_OWNER | U_GROUP | U_MODE | U_SETUID | U_SETGID)) {
	if (TARGETDB_FileStatus(tdb) == NULL) {
	  FatalError(E_BADTARGETFILESTATUS,
		     "Unexpected NULL target file status info while writing target database.\n");
	}
	if (PROGRAM_ErrorNo == E_NULL) {
	  if (TARGETDB_UpdateSpec(tdb) & (U_OWNER | U_SETUID))
	    fprintf(fd, " %d", FILESTAT_Uid(TARGETDB_FileStatus(tdb)));
	  if (TARGETDB_UpdateSpec(tdb) & (U_GROUP | U_SETGID))
	    fprintf(fd, " %d", FILESTAT_Gid(TARGETDB_FileStatus(tdb)));
	  if (TARGETDB_UpdateSpec(tdb) & U_MODE)
	    fprintf(fd, " %o", FILESTAT_Mode(TARGETDB_FileStatus(tdb)));
	}
      }
    }
    if (PROGRAM_ErrorNo == E_NULL) {
      (void) TargetSourceList_Write(fd,
				    TARGETDB_SourceList(tdb),
				    TARGETDB_Name(tdb),
				    sourcepathset);
    }
    if (PROGRAM_ErrorNo == E_NULL) {
      fprintf(fd, "%d\n", TARGETDB_ChildCount(tdb));
      if (TARGETDB_ChildCount(tdb) > 0) {
	childsourcepathset =
	  TargetSourceList_PathSet(TARGETDB_SourceList(tdb));
	for (i = 0, tp = TARGETDB_Children(tdb);
	     (i < TARGETDB_ChildCount(tdb)) && (*tp != NULL)
	     && (PROGRAM_ErrorNo == E_NULL);
	     i++, tp++) {
	  TargetDB_Tree_Write(fd, *tp, childsourcepathset);
	}
	if (childsourcepathset != NULL)
	  StringArray_Free(childsourcepathset);
      }
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? 0 : -1;
}


static int 
TargetDB_WriteUncompressed(fd, TargetDB)
     FILE *fd;
     TARGETDB *TargetDB;
{
  STRINGSET *sourcepathset;

  if (PROGRAM_ErrorNo == E_NULL) {
    sourcepathset = TargetSourceList_PathSet(TARGETDB_SourceList(TargetDB));
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    fprintf(fd, "%d\n", StringArray_Size(sourcepathset));
    if (!StringSet_Empty(sourcepathset)) {
      StringArray_Write(fd, sourcepathset,
			'\\', '\n',
			TargetDBSpecialChars, NTargetDBSpecialChars);
      putc('\n', fd);
    }
    if (TargetDB_Tree_Write(fd, TargetDB, sourcepathset) < 0) {
      FatalError(E_TARGETDBWRITEFAILED,
		 "Attempt to write target database failed.\n");
    }
  }
  if (sourcepathset != NULL)
    StringArray_Free(sourcepathset);

  return (PROGRAM_ErrorNo == E_NULL) ? 0 : -1;
}


int 
TargetDB_Write(fp, TargetDB, compressflag)
     FILE *fp;
     TARGETDB *TargetDB;
     Boolean compressflag;
{
  int pipefd[2];
  FILE *pipeinfp;
  int pid;
  int fd_targetdb;
  long pos_targetdb;
#ifdef HAVE_UNION_WAIT
  union wait ExecStatus;
#else
  int ExecStatus;
#endif

  fprintf(fp, "%s\n", TARGETDBVERSIONSTRING);
  fflush(fp);

  if (TargetDB == NULL) {
    FatalError(E_NULLTARGETDB, "Attempt to write NULL target database.\n");
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    if (!compressflag) {
      TargetDB_WriteUncompressed(fp, TargetDB);
    } else {
      fd_targetdb = fileno(fp);
      pos_targetdb = ftell(fp);
      (void) lseek(fd_targetdb, (off_t) pos_targetdb, SEEK_SET);
      /* open a pipe to pass the stuff to compress */
      if (pipe(pipefd) < 0) {
	FatalError(E_PIPEFAILED,
		   "Could not open pipe to write target database : %s\n",
		   strerror(errno));
      }
      /* fork and write in parent, compress in child */
      if (PROGRAM_ErrorNo == E_NULL) {
	pid = vfork();
	if (pid < 0) {
	  FatalError(E_VFORKFAILED,
		     "Could not vfork to compress target database: %s\n",
		     strerror(errno));
	} else if (pid > 0) {
	  close(pipefd[0]);
	  pipeinfp = efdopen(pipefd[1], "w");
	  if (PROGRAM_ErrorNo == E_NULL) {
	    TargetDB_WriteUncompressed(pipeinfp, TargetDB);
	  }
	  fclose(pipeinfp);
	  close(pipefd[1]);
	} else {		/* pid == 0, child */
	  STRINGARRAY *ex = compress_command();

	  if (dup2(pipefd[0], STDIN_FILENO) < 0) {
	    FatalError(E_DUPFAILED, "dup2 failed in mapping stdin to pipe: %s\n",
		       strerror(errno));
	  }
	  if (dup2(fd_targetdb, STDOUT_FILENO) < 0) {
	    FatalError(E_DUPFAILED, "dup2 failed in mapping stdout to pipe: %s\n",
		       strerror(errno));
	  }
	  close(pipefd[1]);
	  execv(STRINGARRAY_String(ex,0), STRINGARRAY_Values(ex));
	  FatalError(E_EXECFAILED, "Unable to exec %s to compress targetdb: %s\n",
		     STRINGARRAY_String(ex,0), strerror(errno));
	  _exit(-1);
	}
	wait(&ExecStatus);
	if (WEXITSTATUS(ExecStatus) != 0) {
	  FatalError(E_COMPRESSFAILED,
		     "Attempt to compress target database failed.\n");
	}
      }
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? 0 : -1;
}
/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/018/src/lib/TargetDB/RCS/TargetDB_Write.c,v $ */
