/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: TargetDB_Update.c,v 1.13 1994/12/09 21:31:39 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "File.h"
#include "TargetDB.h"



static Boolean TargetDB_CleanDir();


/*
 * $$$NOTE$$$
 *      Make sure no noop or delete node has soures with commands.
 */
Boolean 
TargetDB_Noop(targetpath, nodep, flags)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
{
  Boolean UpdateStatus;

  UpdateStatus = TRUE;
  if (TARGETDB_Status(nodep) & S_OUTOFDATE) {
    if ((PROGRAM_ErrorNo == E_NULL)
	&& (TARGETDB_UpdateSpec(nodep) & U_TARGET)) {
      Filtered_Message(flags & TDB_SHOWACTIONS, "NOOP %s\n", targetpath);
    }
    TARGETDB_Status(nodep) |= S_UPTODATE;
  }
  return (PROGRAM_ErrorNo == E_NULL) ? UpdateStatus : FALSE;
}



Boolean 
TargetDB_Delete(targetpath, nodep, flags)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
{
  Boolean UpdateStatus;

  UpdateStatus = TRUE;
  if (TARGETDB_Status(nodep) & S_OUTOFDATE) {
    Filtered_Message(flags & TDB_SHOWACTIONS, "DELETE %s\n", targetpath);
    if (File_RemovePath(targetpath, NULL) >= 0)
      UpdateStatus = TRUE;
    TARGETDB_Status(nodep) |= S_UPTODATE;
  }
  return (PROGRAM_ErrorNo == E_NULL) ? UpdateStatus : FALSE;
}


Boolean 
TargetDB_MaintainDir(targetpath, nodep, flags)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
{
  Boolean UpdateStatus;

  UpdateStatus = TRUE;

  if (TARGETDB_Status(nodep) & S_OUTOFDATE) {
    UpdateStatus = TargetDB_Mkdir(targetpath, nodep, flags);
  }
  if ((PROGRAM_ErrorNo == E_NULL) && UpdateStatus) {
    TARGETDB_Status(nodep) |= S_UPTODATE;
  }
  return (PROGRAM_ErrorNo == E_NULL) ? UpdateStatus : FALSE;
}


Boolean 
TargetDB_Mkdir(targetpath, nodep, flags)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
{
  FILESTAT targetstatus;

  Boolean UpdateStatus;

  UpdateStatus = TRUE;

  if (TARGETDB_Status(nodep) & S_OUTOFDATE) {
    File_GetStatus(targetpath, &targetstatus, FALSE /* followlinks */ );
    if (PROGRAM_ErrorNo == E_NULL) {
      if (FILESTAT_Type(&targetstatus) & F_DIR) {
	if (!TargetDB_CleanDir(targetpath, nodep, flags))
	  UpdateStatus = FALSE;
      } else {
	if (FILESTAT_Type(&targetstatus) != F_NUL) {
	  Filtered_Message(flags & TDB_SHOWACTIONS,
			   "REMOVE %s\n", targetpath);
	  File_RemovePath(targetpath, &targetstatus);
	}
	Filtered_Message(flags & TDB_SHOWACTIONS,
			 "MKDIR %s\n", targetpath);
	if (File_CreateDirectory(targetpath, NULL, FSTAT_NULL) < 0)
	  UpdateStatus = FALSE;
      }
    }
  }
  if ((PROGRAM_ErrorNo == E_NULL) && UpdateStatus) {
    TARGETDB_Status(nodep) |= S_UPTODATE;
  }
  return (PROGRAM_ErrorNo == E_NULL) ? UpdateStatus : FALSE;
}



static Boolean 
TargetDB_CleanDir(targetpath, nodep, flags)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
{
  register char **cp;

  STRINGARRAY *filelistarray;
  char **filelist;
  int childindex;
  char childpath[MAXPATHLEN + 1];

  Boolean UpdateStatus;

  UpdateStatus = TRUE;
  File_ListDirectory(targetpath, &filelistarray);
  if (PROGRAM_ErrorNo == E_NULL) {
    if (filelistarray == NULL) {
      filelist = NULL;
    } else {
      filelist = STRINGARRAY_Values(filelistarray);
    }
  }
  if ((PROGRAM_ErrorNo == E_NULL) && (filelist != NULL)) {
    for (cp = filelist;
	 (PROGRAM_ErrorNo == E_NULL) && (*cp != NULL);
	 cp++) {
      childindex = TargetDB_LocateIndexToChildNode(nodep,
						   *cp,
						   TDB_LAX);
      if ((PROGRAM_ErrorNo == E_NULL) && (childindex < 0)) {	/* superfluous 
								 * child */
	if ((String_Comparator(targetpath, "/") == 0)
	    || (String_Comparator(targetpath, ".") == 0))
	  (void) strcpy(childpath, *cp);
	else
	  (void) sprintf(childpath, "%s/%s", targetpath, *cp);
	Filtered_Message(flags & TDB_SHOWACTIONS,
			 "REMOVE %s\n", childpath);
	if (File_RemovePath(childpath, NULL) < 0)
	  UpdateStatus = FALSE;
      }
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? UpdateStatus : FALSE;
}



Boolean 
TargetDB_HardLink(targetpath, nodep, flags, parentp, pathprefix, collectionpath, sourcep, depth)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
     TARGETDBENTRY *parentp;
     char *pathprefix;
     char *collectionpath;
     TARGETSOURCE *sourcep;
     unsigned depth;
{
  register TARGETSOURCE *secondaryp;

  Boolean UpdateStatus;

  char hardlinkpath[MAXPATHLEN + 1];
  TARGETDBENTRY *hardlinksourcenodep;
  FILESTAT targetstatus;
  unsigned linkmethod;
  Boolean FoundHardLinkSourceNode;

  UpdateStatus = TRUE;

  if (TARGETDB_UpdateSpec(nodep) & U_MAPCOPY)
    linkmethod = U_MAPCOPY;
  else if (TARGETDB_UpdateSpec(nodep) & U_MAPLINK)
    linkmethod = U_MAPLINK;
  else if (TARGETSOURCE_UpdateSpec(sourcep) & U_MAPCOPY)
    linkmethod = U_MAPCOPY;
  else if (TARGETSOURCE_UpdateSpec(sourcep) & U_MAPLINK)
    linkmethod = U_MAPLINK;

  if (linkmethod == U_MAPLINK) {
    UpdateStatus = TRUE;
    if (PROGRAM_ErrorNo == E_NULL) {
      sourcep = HardLink_SourceToUse(TARGETDB_SourceList(nodep));
      UpdateStatus = TargetDB_LinkFile(targetpath,
				       nodep,
				       flags,
				       collectionpath,
				       sourcep,
				       depth);
    }
  } else {			/* linkmethod == U_MAPCOPY */
    UpdateStatus = TRUE;
    sourcep = HardLink_SourceToUse(TARGETDB_SourceList(nodep));

    FoundHardLinkSourceNode = FALSE;
    secondaryp = TARGETSOURCE_SecondarySource(sourcep);
    while ((PROGRAM_ErrorNo == E_NULL)
	   && !FoundHardLinkSourceNode && (secondaryp != NULL)
	   && (String_Comparator(TARGETDB_Name(nodep),
				 TARGETSOURCE_Path(secondaryp)) != 0)) {
      if (TARGETSOURCE_Status(secondaryp) & S_HARDLINKSOURCE) {
	hardlinksourcenodep =
	  TargetDB_LocateChildNode(parentp,
				   TARGETSOURCE_Path(secondaryp),
				   TDB_LOCATE);
	if ((PROGRAM_ErrorNo == E_NULL)
	    && (TARGETDB_Status(hardlinksourcenodep) & S_UPTODATE)) {
	  FoundHardLinkSourceNode = TRUE;
	} else {
	  FatalError(E_BADHARDLINKSOURCE,
		     "%s not updated before making hard link from %s\n",
		     TARGETDB_Name(nodep),
		     TARGETSOURCE_Path(secondaryp));
	}
      }
      if (!FoundHardLinkSourceNode) {
	secondaryp = TARGETSOURCE_SecondarySource(secondaryp);
      }
    }

    if (PROGRAM_ErrorNo == E_NULL) {
      if (!FoundHardLinkSourceNode) {
	/* we have to map from this node */
	UpdateStatus = TargetDB_CopyFile(targetpath,
					 nodep,
					 flags,
					 collectionpath,
					 sourcep);
      } else {
	if ((String_Comparator(pathprefix, "/") == 0)
	    || (String_Comparator(pathprefix, ".") == 0)) {
	  if (String_Comparator(TARGETSOURCE_Path(secondaryp), "/") == 0)
	    (void) strcpy(hardlinkpath, ".");
	  else
	    (void) strcpy(hardlinkpath, TARGETSOURCE_Path(secondaryp));
	} else
	  (void) sprintf(hardlinkpath, "%s/%s",
			 pathprefix, TARGETSOURCE_Path(secondaryp));

	if (!(TARGETDB_Status(nodep) & S_UPTODATE)) {
	  File_GetStatus(targetpath, &targetstatus, FALSE /* followlinks */ );
	  if ((PROGRAM_ErrorNo == E_NULL)
	      && (FILESTAT_Type(&targetstatus) != F_NUL)) {
	    Filtered_Message(flags & TDB_SHOWACTIONS,
			     "REMOVE %s\n", targetpath);
	    File_RemovePath(targetpath, &targetstatus);
	  }
	  if (PROGRAM_ErrorNo == E_NULL) {
	    Filtered_Message(flags & TDB_SHOWACTIONS,
			     "HARD LINK %s %s\n", hardlinkpath, targetpath);
	    File_CreateLink(hardlinkpath, targetpath);
	  }
	}
	if ((PROGRAM_ErrorNo == E_NULL) && UpdateStatus) {
	  TARGETDB_Status(nodep) |= S_UPTODATE;
	}
      }
    }
  }


  return (PROGRAM_ErrorNo == E_NULL) ? UpdateStatus : FALSE;
}





Boolean 
TargetDB_CopyFile(targetpath, nodep, flags, collectionpath, sourcep)
     char *targetpath;
     TARGETDB *nodep;
     unsigned flags;
     char *collectionpath;
     TARGETSOURCE *sourcep;
{
  char sourcepath[MAXPATHLEN + 1];
  FILESTAT targetstatus, sourcestatus;
  FILESTAT copystatus;
  unsigned copyflags;
  char linksrc[MAXPATHLEN + 1];

  Boolean UpdateStatus;

  UpdateStatus = TRUE;
  /* compute sourcepath */
  (void) sprintf(sourcepath, "%s/%s",
		 collectionpath, TARGETSOURCE_Path(sourcep));

  if (TARGETDB_Status(nodep) & S_OUTOFDATE) {
    File_GetStatus(targetpath, &targetstatus, FALSE /* followlinks */ );

    if (PROGRAM_ErrorNo == E_NULL) {
      /* 
       * $$$OPTIMIZE$$$
       * carry over source info in sourcep with S_STATTHISRUN
       */
      File_GetStatus(sourcepath, &sourcestatus, FALSE /* followlinks */ );
    }
    if (PROGRAM_ErrorNo == E_NULL) {
      if (FILESTAT_Type(&sourcestatus) & F_LNK) {
	/* copy the symlink info from sourcepath */
	File_ReadSymLink(sourcepath, linksrc, MAXPATHLEN + 1);
	if (PROGRAM_ErrorNo == E_NULL) {
	  Filtered_Message(flags & TDB_SHOWACTIONS,
			   "COPY LINK %s %s\n",
			   linksrc, targetpath);
	  if (File_CreateSymLink(linksrc, targetpath) < 0)
	    UpdateStatus = FALSE;
	}
      } else {			/* (FILESTAT_Type(&sourcestatus) & F_REG) */
	/* copy the file over */
	extern Boolean Depot_KeepAttrib; /* defined in Depot_RunMode.c */

	if ((FILESTAT_Type(&targetstatus) != F_NUL)
	    && (FILESTAT_Type(&targetstatus) != F_REG)) {
	  Filtered_Message(flags & TDB_SHOWACTIONS,
			   "REMOVE %s\n", targetpath);
	  File_RemovePath(targetpath, &targetstatus);
	}
	
	if (TARGETDB_Filter(nodep)) {
	  Filtered_Message(flags & TDB_SHOWACTIONS,
			   "Filter %s %s\n",
			   sourcepath, targetpath);
	} else {
	  Filtered_Message(flags & TDB_SHOWACTIONS,
			   "COPY %s %s\n",
			   sourcepath, targetpath);
	}

	FILESTAT_Type(&copystatus) = F_NUL;

	if (Depot_KeepAttrib) {
	  copyflags = FSTAT_OWNER|FSTAT_MODE|FSTAT_GROUP;
	  if (File_GetStatus(sourcepath, &copystatus, FALSE /* followlinks */ ) < 0) {
	    return FALSE;
	  }
	  /* note that the actual attribute can still be overrwritten by the target.* options
	   * listed below */
	}

	if ((FILESTAT_Type(&copystatus) == F_NUL) || !Depot_KeepAttrib) {
	  copyflags = FSTAT_NULL;

	  /* $$$KLUDGE$$$ - 0777 below */
	  copyflags |= FSTAT_MODE;
	  FILESTAT_Mode(&copystatus) = (0777
					& FILESTAT_Mode(&sourcestatus));
	}

	if ((TARGETDB_UpdateSpec(nodep)
	     & (U_OWNER | U_GROUP | U_MODE | U_SETUID | U_SETGID))
	     && (TARGETDB_FileStatus(nodep) == NULL)) {
	  FatalError(E_BADTARGETFILESTATUS,
		     "Unexpected NULL target file status info while updating using target database node.\n");
	}
	if (PROGRAM_ErrorNo == E_NULL) {
	  if (TARGETDB_UpdateSpec(nodep) & (U_OWNER | U_SETUID)) {
	    FILESTAT_Uid(&copystatus) =
	      FILESTAT_Uid(TARGETDB_FileStatus(nodep));
	    copyflags |= FSTAT_OWNER;
	    if (TARGETDB_UpdateSpec(nodep) & U_SETUID) {
	      copyflags |= FSTAT_SETUID;
	    }
	  }
	  if (TARGETDB_UpdateSpec(nodep) & (U_GROUP | U_SETGID)) {
	    FILESTAT_Gid(&copystatus) =
	      FILESTAT_Gid(TARGETDB_FileStatus(nodep));
	    copyflags |= FSTAT_GROUP;
	    if (TARGETDB_UpdateSpec(nodep) & U_SETGID) {
	      copyflags |= FSTAT_SETGID;
	    }
	  }
	  if (TARGETDB_UpdateSpec(nodep) & U_MODE) {
	    FILESTAT_Mode(&copystatus) =
	      FILESTAT_Mode(&sourcestatus)
	      & FILESTAT_Mode(TARGETDB_FileStatus(nodep));
	  }
	  if (flags & TDB_USEMODTIMES) {
	    copyflags |= FSTAT_MTIME;
	    FILESTAT_MTime(&copystatus) = FILESTAT_MTime(&sourcestatus);
	  }
	  if (TARGETDB_Filter(nodep)) {
	    if (File_Filter(sourcepath, targetpath, &copystatus, copyflags,
			    TARGETDB_Filter(nodep)) < 0) {
	      UpdateStatus = FALSE;
	    }
	  } else {
	    if (File_Copy(sourcepath, targetpath,
			  &copystatus, copyflags) < 0) {
	      UpdateStatus = FALSE;
	    }
	  }
	}
      }
    }
  }
  if ((PROGRAM_ErrorNo == E_NULL) && UpdateStatus) {
    TARGETDB_Status(nodep) |= S_UPTODATE;
  }
  return (PROGRAM_ErrorNo == E_NULL) ? UpdateStatus : FALSE;
}



Boolean 
TargetDB_LinkFile(targetpath, nodep, flags, collectionpath, sourcep, depth)
     char *targetpath;
     TARGETDB *nodep;
     unsigned flags;
     char *collectionpath;
     TARGETSOURCE *sourcep;
     unsigned depth;
{
  register unsigned int i;
  register char *cp;

  char dotdotprefix[MAXPATHLEN + 1];
  char sourcepath[MAXPATHLEN + 1];
  Boolean UpdateStatus = TRUE;

  /* compute sourcepath with .. prefixes, if necessary */
  if (collectionpath[0] == '/') {
    (void) sprintf(sourcepath, "%s/%s",
		   collectionpath, TARGETSOURCE_Path(sourcep));
  } else {
    i = 1;
    cp = dotdotprefix;
    while (i < depth) {
      *cp++ = '.';
      *cp++ = '.';
      *cp = '/';
      i++, cp++;
    }
    *cp = '\0';
    (void) sprintf(sourcepath, "%s%s/%s",
		   dotdotprefix, collectionpath,
		   TARGETSOURCE_Path(sourcep));
  }

  if (TARGETDB_Status(nodep) & S_OUTOFDATE) {
    if (PROGRAM_ErrorNo == E_NULL) {
      Filtered_Message(flags & TDB_SHOWACTIONS,
		       "LINK %s %s\n", sourcepath, targetpath);
      if (File_CreateSymLink(sourcepath, targetpath) < 0)
	UpdateStatus = FALSE;
    }
  }
  if ((PROGRAM_ErrorNo == E_NULL) && UpdateStatus) {
    TARGETDB_Status(nodep) |= S_UPTODATE;
  }
  return (PROGRAM_ErrorNo == E_NULL) ? UpdateStatus : FALSE;
}

/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/017/src/lib/TargetDB/RCS/TargetDB_Update.c,v $ */
