/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: TargetDB_Apply.c,v 1.8 1995/05/11 17:44:24 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "Hint.h"
#include "File.h"
#include "FileSystemImage.h"
#include "DepotConf.h"
#include "TargetDB.h"
#include "Collection.h"
#include "CollectionList.h"

static TARGETSOURCE *TargetDB_VirginDirectorySource();
static Boolean TargetDB_ApplyNoop();
static Boolean TargetDB_ApplyDelete();
static Boolean TargetDB_ApplyMaintainDir();
static Boolean TargetDB_ApplyMapCopy();
static Boolean TargetDB_ApplyMapLink();
static Boolean TargetDB_ApplyMapDefault();

static Boolean TargetDB_NodeApply();
static Boolean TargetDB_TreeApply();


static TARGETSOURCE *
TargetDB_VirginDirectorySource(nodep)
     TARGETDB *nodep;
{
  register unsigned int i, j;
  register TARGETDBENTRY *childp;
  register TARGETSOURCE *sp;

  TARGETSOURCELIST *sourcelistp;
  int coll_id;
  unsigned int ci;
  Boolean VirginDirectoryNode;

  TARGETSOURCE *VirginSourcep;

  sourcelistp = TARGETDB_SourceList(nodep);
  coll_id = -1;			/* unknown, initially */

  VirginDirectoryNode = TRUE;	/* until proved otherwise */
  VirginSourcep =
    TARGETSOURCELIST_Source(sourcelistp,
			    TARGETSOURCELIST_NSources(sourcelistp) - 1);

  if (TARGETDB_UpdateSpec(nodep) & U_NONVIRGIN) {
    VirginDirectoryNode = FALSE;
  }
  if (VirginDirectoryNode == TRUE) {
    i = 0;
    while ((PROGRAM_ErrorNo == E_NULL)
	   && (VirginDirectoryNode == TRUE)
	   && (i < TARGETSOURCELIST_NSources(sourcelistp))) {
      if (TARGETSOURCE_UpdateSpec(TARGETSOURCELIST_Source(sourcelistp, i))
	  & (U_MAPCOPY | U_NONVIRGIN)) {
	VirginDirectoryNode = FALSE;
      }
      i++;
    }
  }
  if ((PROGRAM_ErrorNo == E_NULL)
      && (VirginDirectoryNode == TRUE)
      && (TARGETSOURCELIST_NSources(sourcelistp) > 1)
      && (TARGETDB_ChildCount(nodep) > 0)) {
    /* 
     * check sources of children to see if contributions from multiple
     * collections exist
     */
    coll_id = -1;		/* superfluous re-initialization */
    i = 0;
    while ((PROGRAM_ErrorNo == E_NULL)
	   && (VirginDirectoryNode == TRUE)
	   && (i < TARGETDB_ChildCount(nodep))) {
      childp = *(TARGETDB_Children(nodep) + i);
      j = 0;
      while ((PROGRAM_ErrorNo == E_NULL)
	     && (VirginDirectoryNode == TRUE)
	  && (j < TARGETSOURCELIST_NSources(TARGETDB_SourceList(childp)))) {
	sp = TARGETSOURCELIST_Source(TARGETDB_SourceList(childp), j);
	if (TARGETSOURCE_CollectionId(sp) != coll_id) {
	  if (coll_id == -1) {
	    coll_id = TARGETSOURCE_CollectionId(sp);
	  } else {		/* multiple collections in children */
	    VirginDirectoryNode = FALSE;
	  }
	}
	j++;
      }
      i++;
    }

    if ((PROGRAM_ErrorNo == E_NULL)
	&& (VirginDirectoryNode == TRUE) && (coll_id >= 0)) {
      /* 
       * Children exist and have contributions from one collection. We
       * locate the source corresponding to that collection;
       * if more than one such source exists,
       *   the node has been deflowered;
       * else
       *   move it to the end of sourcelist if flags is on.
       */
      ci = TARGETSOURCELIST_NSources(sourcelistp);
      /* initialize to non-value */
      i = 0;
      while ((PROGRAM_ErrorNo == E_NULL)
	     && (VirginDirectoryNode == TRUE)
	     && (i < TARGETSOURCELIST_NSources(sourcelistp))) {
	sp = TARGETSOURCELIST_Source(sourcelistp, i);
	if (TARGETSOURCE_CollectionId(sp) == coll_id) {
	  if (ci == TARGETSOURCELIST_NSources(sourcelistp)) {
	    ci = i;
	  } else {		/* multiple contributions from coll_id */
	    VirginDirectoryNode = FALSE;
	  }
	}
	i++;
      }

      if ((PROGRAM_ErrorNo == E_NULL)
	  && (VirginDirectoryNode == TRUE)) {
	VirginSourcep = TARGETSOURCELIST_Source(sourcelistp, ci);
      }
    }
  }
  if (VirginDirectoryNode != TRUE)
    VirginSourcep = NULL;

  return (PROGRAM_ErrorNo == E_NULL) ? VirginSourcep : NULL;
}



TARGETSOURCE *
SourceList_SourceToUse(sourcelistp)
     TARGETSOURCELIST *sourcelistp;
{
  register TARGETSOURCE *sp;

  sp = TARGETSOURCELIST_Source(sourcelistp,
			       TARGETSOURCELIST_NSources(sourcelistp) - 1);
  if (sp != NULL) {		/* call FatalError() in the other case */
    while (TARGETSOURCE_SecondarySource(sp) != NULL)
      sp = TARGETSOURCE_SecondarySource(sp);
  }
  return (PROGRAM_ErrorNo == E_NULL) ? sp : NULL;
}



static Boolean 
TargetDB_ApplyNoop(targetpath, nodep, flags)
     char *targetpath;
     TARGETDB *nodep;
     unsigned flags;
{
  Boolean ApplyStatus;

  if (flags & TDB_CHECK) {
    ApplyStatus = TargetDB_CheckNoop(targetpath, nodep, flags);
  } else {
    ApplyStatus = TargetDB_Noop(targetpath, nodep, flags);
  }

  return (PROGRAM_ErrorNo == E_NULL) ? ApplyStatus : FALSE;
}



static Boolean 
TargetDB_ApplyDelete(targetpath, nodep, flags)
     char *targetpath;
     TARGETDB *nodep;
     unsigned flags;
{
  Boolean ApplyStatus;

  if (flags & TDB_CHECK) {
    ApplyStatus = TargetDB_CheckDelete(targetpath, nodep, flags);
  } else {
    ApplyStatus = TargetDB_Delete(targetpath, nodep, flags);
  }

  return (PROGRAM_ErrorNo == E_NULL) ? ApplyStatus : FALSE;
}



static Boolean 
TargetDB_ApplyMaintainDir(targetpath, nodep, flags, applychildren)
     char *targetpath;
     TARGETDB *nodep;
     unsigned flags;
     Boolean *applychildren;
{
  Boolean ApplyStatus;

  if (flags & TDB_CHECK) {
    ApplyStatus = TargetDB_CheckMaintainDir(targetpath, nodep, flags);
  } else {
    ApplyStatus = TargetDB_MaintainDir(targetpath, nodep, flags);
  }

  *applychildren = TRUE;

  return (PROGRAM_ErrorNo == E_NULL) ? ApplyStatus : FALSE;
}



static Boolean 
TargetDB_ApplyMapCopy(targetpath, nodep, flags,
		      nodeparentp, pathprefix, collectionlist,
		      applychildren)
     char *targetpath;
     TARGETDB *nodep;
     unsigned flags;
     TARGETDB *nodeparentp;
     char *pathprefix;
     COLLECTIONLIST *collectionlist;
     Boolean *applychildren;
{
  char *collectionpath;
  TARGETSOURCELIST *sourcelistp;
  TARGETSOURCE *sourcep;

  Boolean ApplyStatus;

  *applychildren = FALSE;

  sourcelistp = TARGETDB_SourceList(nodep);
  sourcep = SourceList_SourceToUse(sourcelistp);
  collectionpath =
    COLLECTION_Path(COLLECTIONLIST_Collection(collectionlist,
				       TARGETSOURCE_CollectionId(sourcep)));
  if (TARGETSOURCE_UpdateSpec(sourcep) & U_COMMAND) {
    ApplyStatus = FALSE;
    *applychildren = FALSE;
    /* do nothing */
  } else if (TARGETSOURCE_UpdateSpec(sourcep) & U_HARDLINK) {
    if (flags & TDB_CHECK) {
      ApplyStatus = TargetDB_CheckHardLink(targetpath, nodep, flags,
					   nodeparentp, pathprefix,
					   collectionpath, sourcep,
					0 /* depth not used for copies */ );
    } else {
      ApplyStatus = TargetDB_HardLink(targetpath, nodep, flags,
				      nodeparentp, pathprefix,
				      collectionpath, sourcep,
				      0 /* depth not used for copies */ );
    }
    *applychildren = FALSE;
  } else if (TARGETSOURCE_UpdateSpec(sourcep) & U_MKDIR) {
    if (flags & TDB_CHECK) {
      ApplyStatus = TargetDB_CheckMkdir(targetpath, nodep, flags, sourcep);
    } else {
      ApplyStatus = TargetDB_Mkdir(targetpath, nodep, flags);
    }
    *applychildren = TRUE;
  } else if (TARGETSOURCE_UpdateSpec(sourcep) & U_MAP) {
    if (flags & TDB_CHECK) {
      ApplyStatus = TargetDB_CheckCopyFile(targetpath, nodep, flags,
					   collectionpath, sourcep);
    } else {
      ApplyStatus = TargetDB_CopyFile(targetpath, nodep, flags,
				      collectionpath, sourcep);
    }
  } else {
    FatalError(E_BADSOURCEUPDTSPEC,
	       "Invalid update specification %x in source for %s.\n",
	       TARGETSOURCE_UpdateSpec(sourcep), targetpath);
  }

  return (PROGRAM_ErrorNo == E_NULL) ? ApplyStatus : FALSE;
}



static Boolean 
TargetDB_ApplyMapLink(targetpath, nodep, flags,
		      nodeparentp, pathprefix, collectionlist,
		      depth, applychildren)
     char *targetpath;
     TARGETDB *nodep;
     unsigned flags;
     TARGETDB *nodeparentp;
     char *pathprefix;
     COLLECTIONLIST *collectionlist;
     unsigned depth;
     Boolean *applychildren;
{
  char *collectionpath;
  TARGETSOURCELIST *sourcelistp;
  TARGETSOURCE *sourcep;
  char *virgincollectionpath;
  TARGETSOURCE *virginsourcep;

  Boolean ApplyStatus;

  *applychildren = FALSE;

  sourcelistp = TARGETDB_SourceList(nodep);
  sourcep = SourceList_SourceToUse(sourcelistp);
  collectionpath =
    COLLECTION_Path(COLLECTIONLIST_Collection(collectionlist,
				       TARGETSOURCE_CollectionId(sourcep)));
  if (TARGETSOURCE_UpdateSpec(sourcep) & U_COMMAND) {
    ApplyStatus = FALSE;
    *applychildren = FALSE;
    /* do nothing */
  } else if (TARGETSOURCE_UpdateSpec(sourcep) & U_HARDLINK) {
    if (flags & TDB_CHECK) {
      ApplyStatus = TargetDB_CheckHardLink(targetpath, nodep, flags,
					   nodeparentp, pathprefix,
					   collectionpath, sourcep, depth);
    } else {
      ApplyStatus = TargetDB_HardLink(targetpath, nodep, flags,
				      nodeparentp, pathprefix,
				      collectionpath, sourcep, depth);
    }
    *applychildren = FALSE;
  } else if (TARGETSOURCE_UpdateSpec(sourcep) & U_MKDIR) {
    /* 
     * if (!(TARGETDB_UpdateSpec(nodep) & U_NONVIRGIN) &&
     * (TARGETSOURCELIST_NSources(TARGETDB_SourceList(nodep)) == 1) &&
     * (sourcep == TARGETSOURCELIST_Source(TARGETDB_SourceList(nodep), 0)) && 
     * !(TARGETSOURCE_UpdateSpec(sourcep) & U_NONVIRGIN)) */
    virginsourcep = TargetDB_VirginDirectorySource(nodep);
    if (PROGRAM_ErrorNo == E_NULL) {
      if (virginsourcep != NULL) {	/* virgin link possible */
	virgincollectionpath =
	  COLLECTION_Path
	  (COLLECTIONLIST_Collection(collectionlist,
				 TARGETSOURCE_CollectionId(virginsourcep)));
	if (flags & TDB_CHECK) {
	  ApplyStatus = TargetDB_CheckLinkFile(targetpath, nodep, flags,
					       virgincollectionpath,
					       virginsourcep, depth);
	} else {
	  ApplyStatus = TargetDB_LinkFile(targetpath, nodep, flags,
					  virgincollectionpath,
					  virginsourcep, depth);
	}
      } else {
	if (flags & TDB_CHECK) {
	  ApplyStatus = TargetDB_CheckMkdir(targetpath, nodep,
					    flags, sourcep);
	} else {
	  ApplyStatus = TargetDB_Mkdir(targetpath, nodep, flags);
	}
	*applychildren = TRUE;
      }
    }
  } else if (TARGETSOURCE_UpdateSpec(sourcep) & U_MAP) {
    if (flags & TDB_CHECK) {
      ApplyStatus = TargetDB_CheckLinkFile(targetpath, nodep, flags,
					   collectionpath, sourcep, depth);
    } else {
      ApplyStatus = TargetDB_LinkFile(targetpath, nodep, flags,
				      collectionpath, sourcep, depth);
    }
  } else {
    FatalError(E_BADSOURCEUPDTSPEC,
	       "Invalid update specification %x in source for %s.\n",
	       TARGETSOURCE_UpdateSpec(sourcep), targetpath);
  }

  return (PROGRAM_ErrorNo == E_NULL) ? ApplyStatus : FALSE;
}



static Boolean 
TargetDB_ApplyMapDefault(targetpath, nodep, flags, nodeparentp, pathprefix, collectionlist, depth, applychildren)
     char *targetpath;
     TARGETDB *nodep;
     unsigned flags;
     TARGETDB *nodeparentp;
     char *pathprefix;
     COLLECTIONLIST *collectionlist;
     unsigned depth;
     Boolean *applychildren;
{
  char *collectionpath;
  TARGETSOURCELIST *sourcelistp;
  TARGETSOURCE *sourcep;
  char *virgincollectionpath;
  TARGETSOURCE *virginsourcep;

  Boolean ApplyStatus;


  *applychildren = FALSE;

  sourcelistp = TARGETDB_SourceList(nodep);
  sourcep = SourceList_SourceToUse(sourcelistp);
  collectionpath =
    COLLECTION_Path(COLLECTIONLIST_Collection(collectionlist,
				       TARGETSOURCE_CollectionId(sourcep)));
  if (TARGETSOURCE_UpdateSpec(sourcep) & U_COMMAND) {
    ApplyStatus = FALSE;
    *applychildren = FALSE;
    /* do nothing */
  } else if (TARGETSOURCE_UpdateSpec(sourcep) & U_HARDLINK) {
    if (flags & TDB_CHECK) {
      ApplyStatus = TargetDB_CheckHardLink(targetpath, nodep, flags,
					   nodeparentp, pathprefix,
					   collectionpath, sourcep, depth);
    } else {
      ApplyStatus = TargetDB_HardLink(targetpath, nodep, flags,
				      nodeparentp, pathprefix,
				      collectionpath, sourcep, depth);
    }
    *applychildren = FALSE;
  } else if (TARGETSOURCE_UpdateSpec(sourcep) & U_MKDIR) {
    /* 
     * if (!(TARGETDB_UpdateSpec(nodep) & U_NONVIRGIN) &&
     * (TARGETSOURCELIST_NSources(TARGETDB_SourceList(nodep)) == 1) &&
     * (sourcep == TARGETSOURCELIST_Source(TARGETDB_SourceList(nodep), 0)) && 
     * !(TARGETSOURCE_UpdateSpec(sourcep) & (U_MAPCOPY|U_NONVIRGIN))) */
    virginsourcep = TargetDB_VirginDirectorySource(nodep);
    if (PROGRAM_ErrorNo == E_NULL) {
      if (virginsourcep != NULL) {	/* virgin link possible */
	virgincollectionpath =
	  COLLECTION_Path
	  (COLLECTIONLIST_Collection(collectionlist,
				 TARGETSOURCE_CollectionId(virginsourcep)));
	if (flags & TDB_CHECK) {
	  ApplyStatus = TargetDB_CheckLinkFile(targetpath, nodep, flags,
					       virgincollectionpath,
					       virginsourcep, depth);
	} else {
	  ApplyStatus = TargetDB_LinkFile(targetpath, nodep, flags,
					  virgincollectionpath,
					  virginsourcep, depth);
	}
      } else {
	if (flags & TDB_CHECK) {
	  ApplyStatus = TargetDB_CheckMkdir(targetpath, nodep,
					    flags, sourcep);
	} else {
	  ApplyStatus = TargetDB_Mkdir(targetpath, nodep, flags);
	}
	*applychildren = TRUE;
      }
    }
  } else if (TARGETSOURCE_UpdateSpec(sourcep) & U_MAP) {
    if (TARGETSOURCE_UpdateSpec(sourcep) & U_MAPCOPY) {
      if (flags & TDB_CHECK) {
	ApplyStatus = TargetDB_CheckCopyFile(targetpath, nodep, flags,
					     collectionpath, sourcep);
      } else {
	ApplyStatus = TargetDB_CopyFile(targetpath, nodep, flags,
					collectionpath, sourcep);
      }
    } else {
      if (flags & TDB_CHECK) {
	ApplyStatus = TargetDB_CheckLinkFile(targetpath, nodep, flags,
					     collectionpath, sourcep,
					     depth);
      } else {
	ApplyStatus = TargetDB_LinkFile(targetpath, nodep, flags,
					collectionpath, sourcep, depth);
      }
    }
  } else {
    FatalError(E_BADSOURCEUPDTSPEC,
	       "Invalid update specification %x in source for %s.\n",
	       TARGETSOURCE_UpdateSpec(sourcep), targetpath);
  }

  return (PROGRAM_ErrorNo == E_NULL) ? ApplyStatus : FALSE;
}



static Boolean 
TargetDB_NodeApply(nodep, flags, nodeparentp,
		   collectionlist, pathprefix,
		   depth, applychildren)
     TARGETDB *nodep;
     unsigned flags;
     TARGETDB *nodeparentp;
     COLLECTIONLIST *collectionlist;
     char *pathprefix;
     unsigned depth;
     Boolean *applychildren;
{
  char targetpath[MAXPATHLEN + 1];

  Boolean ApplyStatus;

  if ((String_Comparator(pathprefix, "/") == 0)
      || (String_Comparator(pathprefix, ".") == 0)) {
    if (String_Comparator(TARGETDB_Name(nodep), "/") == 0)
      (void) strcpy(targetpath, ".");
    else
      (void) strcpy(targetpath, TARGETDB_Name(nodep));
  } else
    (void) sprintf(targetpath, "%s/%s", pathprefix, TARGETDB_Name(nodep));

  if (String_Comparator(TARGETDB_Name(nodep), "..") == 0) {
    FatalError(E_BADTARGETPATH,
	       "Forbidden attempt to update .. in target %s\n", targetpath);
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    *applychildren = FALSE;

    if (TARGETDB_UpdateSpec(nodep) & U_SPECIAL) {
      if ((TARGETDB_SourceList(nodep) != NULL)
	  && (TARGETSOURCELIST_NSources(TARGETDB_SourceList(nodep)) != 0)) {
	Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
			 "Attempt to update special file at %s.\n",
			 targetpath);

      } else {
	Filtered_Message((flags & TDB_SHOWACTIONS),
			 "SPECIAL %s\n", targetpath);
	ApplyStatus = TRUE;
      }
      *applychildren = FALSE;
    } else if (TARGETDB_UpdateSpec(nodep) & U_NOOP) {
      ApplyStatus = TargetDB_ApplyNoop(targetpath, nodep, flags);
      *applychildren = FALSE;
    } else if (TARGETDB_UpdateSpec(nodep) & U_DELETE) {
      ApplyStatus = TargetDB_ApplyDelete(targetpath, nodep, flags);
      *applychildren = FALSE;
    } else if (TARGETDB_UpdateSpec(nodep) & (U_NOOPPATH | U_SPECIALPATH)) {
      ApplyStatus = TargetDB_ApplyMaintainDir(targetpath, nodep, flags,
					      applychildren);
    } else if (TARGETDB_UpdateSpec(nodep) & U_MAPCOPY) {
      ApplyStatus = TargetDB_ApplyMapCopy(targetpath, nodep, flags,
					  nodeparentp, pathprefix,
					  collectionlist, applychildren);
    } else if (TARGETDB_UpdateSpec(nodep) & U_MAPLINK) {
      ApplyStatus = TargetDB_ApplyMapLink(targetpath, nodep, flags,
					  nodeparentp, pathprefix,
					  collectionlist, depth,
					  applychildren);
    } else {
      ApplyStatus = TargetDB_ApplyMapDefault(targetpath, nodep, flags,
					     nodeparentp, pathprefix,
					     collectionlist, depth,
					     applychildren);
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? ApplyStatus : FALSE;
}



static Boolean 
TargetDB_TreeApply(targetdbp, flags, targetdbparentp,
		   collectionlist, pathprefix, depth)
     TARGETDB *targetdbp;
     unsigned flags;
     TARGETDB *targetdbparentp;
     COLLECTIONLIST *collectionlist;
     char *pathprefix;
     unsigned depth;
{
  register int i;
  register TARGETDB **childp;

  char newpathprefix[MAXPATHLEN + 1];
  unsigned newdepth;
  Boolean applychildren;

  Boolean ApplyStatus, ApplyChildrenStatus;

  if (flags & TDB_CHECK) {
    ApplyStatus = TargetDB_NodeApply(targetdbp, flags, targetdbparentp,
				     collectionlist, pathprefix,
				     depth, &applychildren);
    if (applychildren && !(TARGETDB_Status(targetdbp) & S_TARGETISDIR)) {
      flags |= TDB_CREATE;
    }
  } else if (TARGETDB_Status(targetdbp) & S_OUTOFDATE) {
    ApplyStatus = TargetDB_NodeApply(targetdbp, flags, targetdbparentp,
				     collectionlist, pathprefix,
				     depth, &applychildren);
  } else {
    ApplyStatus = TRUE;
    if (TARGETDB_Status(targetdbp) & S_OUTOFDATECHILDREN)
      applychildren = TRUE;
    else
      applychildren = FALSE;
  }

  if ((String_Comparator(pathprefix, "/") == 0)
      || (String_Comparator(pathprefix, ".") == 0)) {
    (void) strcpy(newpathprefix, TARGETDB_Name(targetdbp));
  } else {
    (void) sprintf(newpathprefix, "%s/%s",
		   pathprefix, TARGETDB_Name(targetdbp));
  }
  newdepth = depth + 1;

  ApplyChildrenStatus = TRUE;
  if (applychildren) {
    for (i = 0, childp = TARGETDB_Children(targetdbp);
	 (PROGRAM_ErrorNo == E_NULL) && (i < TARGETDB_ChildCount(targetdbp));
	 i++, childp++) {
      ApplyChildrenStatus &= TargetDB_TreeApply(*childp, flags, targetdbp,
						collectionlist,
						newpathprefix, newdepth);
    }
  }
  if ((flags & TDB_CHECK) && !ApplyChildrenStatus) {
    TARGETDB_Status(targetdbp) |= S_OUTOFDATECHILDREN;
  }
  ApplyStatus &= ApplyChildrenStatus;

  return (PROGRAM_ErrorNo == E_NULL) ? ApplyStatus : FALSE;
}



void 
TargetDB_Apply(targetdbp, flags, collectionlist)
     TARGETDB *targetdbp;
     unsigned flags;
     COLLECTIONLIST *collectionlist;
{
  (void) TargetDB_TreeApply(targetdbp, flags, NULL /* parent node */ ,
		     collectionlist, "/" /* pathprefix */ , 0 /* depth */ );

  return;
}
/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/020/src/lib/TargetDB/RCS/TargetDB_Apply.c,v $ */
