/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: Depot_RunMode.c,v 1.16 1994/12/09 21:46:50 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */


#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "Lock.h"
#include "File.h"
#include "Hint.h"
#include "FileSystemImage.h"
#include "Preference.h"
#include "DepotConf.h"
#include "TargetDB.h"
#include "Collection.h"
#include "CollectionList.h"
#include "DepotDB.h"
#include "HintService.h"
#include "Command.h"

#include "config.h"
#include "Depot.h"
#include "DepotUtil.h"

#ifdef LOG_DEPOTDB
static char log_path[MAXPATHLEN];
#endif

/* It is assumed that the two functions contained within 
 *           Depot_RunBuildMode and Depot_RunDeltaMode
 * will be exclusively run from a single thread of execution. That is
 * in any given program, the program will only call Depot_RunBuildMode or 
 * Depot_RunDeltaMode.
 */

/* configuration info */
static char *DEPOT_SpecialFiles[] =  { "depot", NULL };

static STRINGSET DEPOT_SpecialFileSet =
{
  1,	/* size */
  DEPOT_SpecialFiles	/* values */
};

/* (really) global variables */
Boolean Depot_KeepAttrib;

/* module variables */
static DEPOTDB Depot_depotdb;
static DEPOTDB *Depot_depotdbp;

static STRINGSET *cnameset = NULL;
static STRINGSET *tempstringset = NULL;
static TARGETSOURCELIST *sourcelistp;

static char filenamebuffer[MAXPATHLEN + 1];
static FILE *fpread, *fpwrite;
static HINTDB Depot_HintCache;
static int id;
static char *collectionpath;
static char *hintservicepath;
static unsigned tdb_checkflags, tdb_applyflags;
static STRINGSET *AllTargetCommandsSet = NULL;
static char DepotDBFileName[MAXPATHLEN + 1];
static char Depot_VersionDelimiter;
static STRINGSET *Depot_SearchPathList = NULL;
static Boolean Depot_UseModTimes;
static Boolean Depot_CompressTDB;
static STRINGSET *Depot_IgnoreSet = NULL;
static STRINGSET *Depot_SpecialFileSet = NULL;
static STRINGSET *Depot_CommandLabelsToRun = NULL;
static COMMAND **Depot_CommandList;

#ifdef LOG_DEPOTDB
static void
log_depotdb(tdb, name) 
  TARGETDB *tdb;
  char *name;
{
  FILE *fp2;

  if (tdb == NULL) {
    fprintf(stderr,"WARNING: Target_DB is null so not writing %s\n", name);
    return;
  }
  fp2 = efopen(name,"w");
  TargetDB_Write(fp2, tdb, 0);
  fclose(fp2);
}
#endif

static void
Initialize(updatemode) 
  unsigned updatemode;
{
  (void) sprintf(DepotDBFileName,
		 "%s/%s", DEPOTSPECIALDIRECTORY, DEPOTDBFILE);
  Depot_depotdbp = &Depot_depotdb;
  (void)memset(Depot_depotdbp, 0, sizeof(DEPOTDB));
  
  /* read in new preference options */
  if (PROGRAM_ErrorNo == E_NULL) {
    (void) sprintf(filenamebuffer,
		   "%s/%s", DEPOTSPECIALDIRECTORY, DEPOTPREFERENCESFILE);
    fpread = efopen(filenamebuffer, "r");
    if (PROGRAM_ErrorNo == E_NULL) {
      DEPOTDB_PreferenceDB(Depot_depotdbp) = PreferenceDB_Read(fpread);
      (void) fclose(fpread);
    }
  }
  /* read in new global configuration requests, if any */
  if (PROGRAM_ErrorNo == E_NULL) {
    DEPOTDB_DepotConfDB(Depot_depotdbp) = NULL;
    (void) sprintf(filenamebuffer,
		   "%s/%s", DEPOTSPECIALDIRECTORY, DEPOTCONFIGFILE);
    if ((fpread = fopen(filenamebuffer, "r")) == NULL) {
      if (errno != ENOENT) {
	FatalError(E_FOPENFAILED, "Unable to open global depot configuration file: %s\n",
		   filenamebuffer);
      }
    } else {
      DEPOTDB_DepotConfDB(Depot_depotdbp) = GlobalDepotConfDB_Read(fpread);
      (void) fclose(fpread);
    }
  }
  /* compute version delimiter */
  if (PROGRAM_ErrorNo == E_NULL) {
    char *versiondelimiterstring;
    
    versiondelimiterstring =
      Preference_GetString(DEPOTDB_PreferenceDB(Depot_depotdbp),
			   NULL,
			   "versiondelimiter",
			   NULL /* default */ );
    if (PROGRAM_ErrorNo == E_NULL) {
      if (versiondelimiterstring == NULL) {
	Depot_VersionDelimiter = DEPOT_DEFAULTDEPOTVERSIONDELIMITER;
      } else {
	Depot_VersionDelimiter = versiondelimiterstring[0];
	free(versiondelimiterstring);
      }
    }
  }
  
  /* get list of searchpaths from preferences */
  if (PROGRAM_ErrorNo == E_NULL) {
    Depot_SearchPathList =
      Preference_GetSearchPathList(DEPOTDB_PreferenceDB(Depot_depotdbp));
  }

  /* check if mod times are to be used */
  if (PROGRAM_ErrorNo == E_NULL) {
    if (updatemode & M_USEMODTIMES) {
      Depot_UseModTimes = TRUE;
    } else if (updatemode & M_NOMODTIMES) {
      Depot_UseModTimes = FALSE;
    } else {
      Depot_UseModTimes =
	Preference_GetBoolean(DEPOTDB_PreferenceDB(Depot_depotdbp),
			      NULL,
			      "usemodtimes",
			      TRUE /* default */ );
    }
  }

  /* check to see if the target database is to be compressed */
  if (PROGRAM_ErrorNo == E_NULL) {
    if (updatemode & M_COMPRESSTDB) {
      Depot_CompressTDB = TRUE;
    } else if (updatemode & M_NOCOMPRESSTDB) {
      Depot_CompressTDB = FALSE;
    } else {
      Depot_CompressTDB = 
	Preference_GetBoolean(DEPOTDB_PreferenceDB(Depot_depotdbp),
			      NULL,
			      "compresstargetdb",
			      TRUE /* default */ );
    }
  }

  /* check to see if file attributes are to be preserved */
  if (PROGRAM_ErrorNo == E_NULL) {
/* NOTE: This really doesn't belong here. It belongs in TargetDB_Update. 
 *       It is placed here because I didn't want to modify a bunch of 
 *       functions in order to pass the preference database down to that 
 *       routine. */
    if (updatemode & M_KEEPATTRIB) {
      Depot_KeepAttrib = TRUE;
    } else if (updatemode & M_DITCHATTRIB) {
      Depot_KeepAttrib = FALSE;
    } else {
      Depot_KeepAttrib =
	Preference_GetBoolean(DEPOTDB_PreferenceDB(Depot_depotdbp),
			      NULL,
			      "keepcopiedfileattributes",
			      TRUE /* default */ );
    }
  }

  /* get hints stored in all searchpaths */
  Depot_HintCache = NULL;
  if ((PROGRAM_ErrorNo == E_NULL)
      && !StringSet_Empty(Depot_SearchPathList)) {
    register char **sp;

    for (sp = STRINGSET_Values(Depot_SearchPathList);
	 (PROGRAM_ErrorNo == E_NULL) && (*sp != NULL);
	 sp++) {
#ifdef TESTRUN
      (void) sprintf(filenamebuffer,
		     "%s%s/hint.depot", DEPOT_TEMPTESTPATHPREFIX, *sp);
#else /* TESTRUN */
      (void) sprintf(filenamebuffer,
		     "%s/%s", *sp, DEPOTHINTFILE);
#endif /* TESTRUN */
      Depot_HintCache = HintService_GetRelatedHints("", filenamebuffer);
      Depot_HintCache = HintDB_SetPathPrefix(Depot_HintCache, *sp);
    }
  }
}


static DEPOTCONFDB *
Load_DepotConfDB(name, path) 
  char *name, *path;
{
  DEPOTCONFDB *coll_dcdb, *glob_dcdb;

  /* read in depotconfdb of collection */
  coll_dcdb = Depot_GetCollectionDepotConfDB(name, path);

  /* append related specs from global depotconf */
  if (PROGRAM_ErrorNo != E_NULL) 
    return NULL;

  if (DEPOTDB_DepotConfDB(Depot_depotdbp) != NULL) {
    glob_dcdb = GlobalDepotConf_ExtractCollectionDepotConf
      (DEPOTDB_DepotConfDB(Depot_depotdbp), name);
    if (PROGRAM_ErrorNo != E_NULL)
      return NULL;
    return(CollectionDepotConf_AppendGlobalDepotConf(coll_dcdb, glob_dcdb, name));
  } else {
    return(coll_dcdb);
  }
}


/* 
 * Depot_RunBuildMode is the initial building of the depot.db file and
 *      offers the greatest amount of consistency but is slower
 */
void 
Depot_RunBuildMode(ImagePath, UpdateMode, PointofNoReturn)
     char *ImagePath;
     unsigned UpdateMode;
     Boolean *PointofNoReturn;

{
  register char **sp, **namep;
  register COLLECTION *colp;
  register int i;
  register COMMAND **commandp;

  Initialize(UpdateMode);

#ifdef LOG_TIME
  timer_start(stderr);
#endif
  /* 
   * if all collections to be updated,
   * add them to cnameset and collection list
   */
  if (PROGRAM_ErrorNo == E_NULL) {
    Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL, "Loading collections...");
    /* get list of collections specified in preference database */
    cnameset =
      Preference_GetCollectionNameList(DEPOTDB_PreferenceDB(Depot_depotdbp));

    /* add names of collections found in searchpaths to this list */
    if (!StringSet_Empty(Depot_SearchPathList)) {
      for (sp = STRINGSET_Values(Depot_SearchPathList);
	   (PROGRAM_ErrorNo == E_NULL) && (*sp != NULL);
	   sp++) {
	tempstringset =
	  Depot_GetCollectionNamesInDirectory(*sp,
					      Depot_VersionDelimiter,
					      Depot_HintCache);
	if (PROGRAM_ErrorNo == E_NULL) {
	  cnameset = StringSet_Merge(cnameset, tempstringset);
	}
	StringArray_Free(tempstringset);
      }
    }
    /* add these collection names to the collection list */
    if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(cnameset)) {
      for (namep = STRINGSET_Values(cnameset);
	   (PROGRAM_ErrorNo == E_NULL) && (*namep != NULL);
	   namep++) {
	DEPOTDB_CollectionList(Depot_depotdbp) =
	  CollectionList_AddCollectionName
	  (DEPOTDB_CollectionList(Depot_depotdbp),
	   *namep, C_POSSIBLYNONEXISTENT);
      }
    }
    /* 
     * last usage of Depot_SearchPathList and cnameset 
     * so let\'s free it before we forget 
     */
    StringArray_Free(Depot_SearchPathList);
    StringArray_Free(cnameset);
  }


  /* mark collections to be ignored */
  if (PROGRAM_ErrorNo == E_NULL) {
    Depot_IgnoreSet =
      Preference_GetStringSet(DEPOTDB_PreferenceDB(Depot_depotdbp),
			      NULL, "ignore", NULL /* default */, 0 /* don't allocate space */ );
  }
  if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(Depot_IgnoreSet)) {
    for (sp = STRINGSET_Values(Depot_IgnoreSet); *sp != NULL; sp++) {
      (void) CollectionList_IgnoreCollection
	(DEPOTDB_CollectionList(Depot_depotdbp), *sp);
    }
  }
  if (PROGRAM_ErrorNo == E_NULL)
    Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL, "done.\n");
#ifdef LOG_TIME
  timer_status("Collection load");
#endif
  if (DEPOTDB_CollectionList(Depot_depotdbp) == NULL) {
    Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL, "No collections found.\n");
    return;
  }
  /* get collection paths by checking preference options and the hint cache */
  for (i = 0;
       (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
       i++) {
    colp = COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i);
    if ((colp != NULL) && !(COLLECTION_Status(colp) & C_IGNORE)) {
      collectionpath =
	Depot_GetCollectionPath(COLLECTION_Name(colp),
				DEPOTDB_PreferenceDB(Depot_depotdbp),
				Depot_VersionDelimiter,
				Depot_HintCache);

      if (collectionpath == NULL) {
	if (!(COLLECTION_Status(colp) & C_POSSIBLYNONEXISTENT)) {
	  FatalError(E_NOPATHTOCOLLECTION,
		     "Could not locate path to collection named %s.\n",
		     COLLECTION_Name(colp));
	} else {
	  COLLECTION_Path(colp) = NULL;
	  COLLECTION_Status(colp) |= C_IGNORE;
	}
      } else {
	if (PROGRAM_ErrorNo == E_NULL)
	  COLLECTION_Path(colp) = collectionpath; 
	/* note that collectionpath is allocated by Depot_GetCollectionPath */
      }
    }
  }

  /* get hints for all collection paths */
  for (i = 0;
       (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
       i++) {
    HINTDB hdb;
      
    colp = COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i);
    if ((colp != NULL) && !(COLLECTION_Status(colp) & C_IGNORE)) {

      hdb = NULL;

      /* check if the hint cache has hint info */
      if (Depot_HintCache != NULL)
	hdb = HintDB_GetRelatedHints(Depot_HintCache, COLLECTION_Path(colp));

      /* if not, seek info from collection-specific hint service */
      if ((hdb == NULL) && (PROGRAM_ErrorNo == E_NULL)) {
	hintservicepath
	  = Preference_GetString(DEPOTDB_PreferenceDB(Depot_depotdbp),
				 COLLECTION_Name(colp),
				 "hintfile",
				 NULL /* default */ );

	if ((hintservicepath != NULL) && (PROGRAM_ErrorNo == E_NULL))
	  hdb = HintService_GetRelatedHints(hintservicepath,
					    COLLECTION_Path(colp));
      }
      if ((hdb != NULL) && (PROGRAM_ErrorNo == E_NULL)) {
	hdb = HintDB_UnsetPathPrefix(hdb, COLLECTION_Path(colp));
      }
      COLLECTION_HintDB(colp) = (PROGRAM_ErrorNo == E_NULL) ? hdb : NULL;
    }
  }

  /* prune collections to be ignored */
  for (i = 0;
       (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
       i++) {
    colp = COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i);
    if ((colp != NULL) && (COLLECTION_Status(colp) & C_IGNORE)) {
      COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i)
	= NULL;
      Collection_Free(colp);
      colp = NULL;
    }
  }

  /* get installmethod for all collections */
  for (i = 0;
       (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
       i++) {
    colp = COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i);
    if (colp != NULL) {
      if (String_Comparator
	  (PreferenceDB_CollectionInstallMethod
	   (COLLECTION_Name(colp), DEPOTDB_PreferenceDB(Depot_depotdbp)),
	   "copy") == 0) {
	COLLECTION_MapMethod(colp) = U_MAPCOPY;
      } else {
	COLLECTION_MapMethod(colp) = U_MAPLINK;
      }
    }
  }

  /* 
   * for all collections,
   *    read file system image;
   *       if no hints, extract hints from file system image;
   *    get depotconfdb;
   *    append related specs from global depotconf;
   *    set status of collection to indicate depotconf change
   *    get list of commands for collection;
   *    generate targetdb;
   *    remove depot.conf and depot.image from the targetdb
   *    write out targetdb;
   *    merge targetdb into global depotconfdb's target database;
   *    free targetdb;
   */
  DEPOTDB_TargetDB(Depot_depotdbp) = NULL;
  for (i = 0;
       (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
       i++) {
    colp = COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i);

    if (colp != NULL) {
      Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
		       "Processing %s (%d)...",
		       COLLECTION_Name(colp),
		       COLLECTION_Id(colp));

      if (COLLECTION_HintDB(colp) != NULL) {
	Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
			 "got hints...");
      }
      /* read file system image */
      if (ImagePath == NULL) {
	(void) sprintf(filenamebuffer,
		       "%s/%s", COLLECTION_Path(colp), DEPOTIMAGEFILE);
      } else {
	(void) sprintf(filenamebuffer,
		       "%s/%s.image", ImagePath, COLLECTION_Name(colp));
      }
      COLLECTION_Image(colp) = FileSystemImage_Read(filenamebuffer);
      if ((COLLECTION_Image(colp) != NULL) && (PROGRAM_ErrorNo == E_NULL)) {
	Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
			 "got fs image...");
      }
      /* if no hints, extract hints from file system image */
      if ((PROGRAM_ErrorNo == E_NULL)
	  && (COLLECTION_HintDB(colp) == NULL)
	  && (COLLECTION_Image(colp) != NULL)) {
	if ((COLLECTION_HintDB(colp) = 
	     Hint_CollectAllHints(COLLECTION_Image(colp))) != NULL) {
	  Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
			   "got hints...");
	}
      }
      COLLECTION_Status(colp) |= (C_NEW | C_PATHCHANGE);

      /* get depotconfdb */
      {
	COLLECTION_DepotConfDB(colp) = Load_DepotConfDB(COLLECTION_Name(colp),
						       COLLECTION_Path(colp));
	/* set status to indicate depotconf change */
	COLLECTION_Status(colp) |= C_DEPOTCONFCHANGE;
      }

      /* get list of commands for collection */
      if (PROGRAM_ErrorNo == E_NULL) {
	COLLECTION_CommandList(colp) =
	  DepotConfDB_GetCommandList(COLLECTION_DepotConfDB(colp));
      }
      /* generate targetdb */
      if (PROGRAM_ErrorNo == E_NULL) {
	COLLECTION_TargetDB(colp) =
	  Collection_TargetDBGenerate(colp, NULL);
      }
      /* remove DEPOTCONFIGFILE and DEPOTIMAGEFILE from the targetdb */
      if ((PROGRAM_ErrorNo == E_NULL) && (COLLECTION_TargetDB(colp) != NULL)) {
	COLLECTION_TargetDB(colp) =
	  TargetDB_DeleteTargetPath(COLLECTION_TargetDB(colp),
				    DEPOTCONFIGFILE,
				    TDB_LAX);
	COLLECTION_TargetDB(colp) =
	  TargetDB_DeleteTargetPath(COLLECTION_TargetDB(colp),
				    DEPOTIMAGEFILE,
				    TDB_LAX);
      }
      if (PROGRAM_ErrorNo == E_NULL)
	Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL, "done.\n");

      /* merge targetdb into global depotconfdb's target database */
      if ((PROGRAM_ErrorNo == E_NULL) && (COLLECTION_TargetDB(colp) != NULL)) {
	DEPOTDB_TargetDB(Depot_depotdbp) =
	  TargetDB_Merge(DEPOTDB_TargetDB(Depot_depotdbp),
			 COLLECTION_TargetDB(colp),
			 TRUE /* newpathtosources */ );
      }

      if (COLLECTION_Image(colp) != NULL) {
	FileSystemImage_Free(COLLECTION_Image(colp));
	COLLECTION_Image(colp) = NULL;
      }
      if (COLLECTION_TargetDB(colp) != NULL) {
	TargetDB_Free(COLLECTION_TargetDB(colp));
	COLLECTION_TargetDB(colp) = NULL;
      }
    }
  }
#ifdef LOG_TIME
  timer_status("Collection processing");
#endif

  /* sort sources by override information */
  if (PROGRAM_ErrorNo == E_NULL) {
    Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL, "Checking for conflicts\n");

    TargetDB_SortByOverrides(DEPOTDB_TargetDB(Depot_depotdbp),
				    DEPOTDB_PreferenceDB(Depot_depotdbp),
				    DEPOTDB_CollectionList(Depot_depotdbp));
  }
  /* check for conflicts */
  if (PROGRAM_ErrorNo == E_NULL) {
    TargetDB_CheckSourceConsistency(DEPOTDB_TargetDB(Depot_depotdbp),
				       DEPOTDB_PreferenceDB(Depot_depotdbp),
				     DEPOTDB_CollectionList(Depot_depotdbp),
					   TRUE /* findallconflicts */ );
  }

  /* apply target specific mappings */
  if (PROGRAM_ErrorNo == E_NULL) {

    DEPOTDB_TargetDB(Depot_depotdbp) =
      TargetDB_SetTargetMappings(DEPOTDB_TargetDB(Depot_depotdbp),
				 DEPOTDB_PreferenceDB(Depot_depotdbp));
  }
  /* apply target specific status info */
  if (PROGRAM_ErrorNo == E_NULL) {
    DEPOTDB_TargetDB(Depot_depotdbp) =
      TargetDB_SetTargetFileStatus(DEPOTDB_TargetDB(Depot_depotdbp),
				   DEPOTDB_PreferenceDB(Depot_depotdbp));
  }
  /* check for violations of special files */
  if (PROGRAM_ErrorNo == E_NULL) {
    Depot_SpecialFileSet =
      Preference_GetStringSet(DEPOTDB_PreferenceDB(Depot_depotdbp),
			      NULL,
			      "specialfile",
			      NULL /* default */,
			      0 /* don't allocate space */);
    if (PROGRAM_ErrorNo == E_NULL) {
      Depot_SpecialFileSet =
	StringSet_Merge(Depot_SpecialFileSet, &DEPOT_SpecialFileSet);
    }
    if ((PROGRAM_ErrorNo == E_NULL)
	&& !StringSet_Empty(Depot_SpecialFileSet)) {
      DEPOTDB_TargetDB(Depot_depotdbp) =
	TargetDB_ProtectSpecialTargets(DEPOTDB_TargetDB(Depot_depotdbp),
				    STRINGSET_Values(Depot_SpecialFileSet));
    }
  }
#ifdef LOG_TIME
  timer_status("Consistency checking");
#endif

  /* check targetdb tree for need to update */
  if (PROGRAM_ErrorNo == E_NULL) {
    Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL, 
		     "Updating target directory\n");

    tdb_checkflags = TDB_CHECK | TDB_BUILD; 
    if (UpdateMode & M_SHOWACTIONSONLY) {
      tdb_checkflags |= TDB_SHOWACTIONS;
    }
    if (Depot_UseModTimes) {
      tdb_checkflags |= TDB_USEMODTIMES;
    }
    TargetDB_Apply(DEPOTDB_TargetDB(Depot_depotdbp),
		   tdb_checkflags,
		   DEPOTDB_CollectionList(Depot_depotdbp));
  }
  /* 
   * get a list of target specific command labels specified in depot.pref;
   * check which of these commands have applicable nodes in the target database;
   * for all collections which contribute to the target database
   *   add commands requested by the collection;
   * check all commands in the final list of labels;
   */

  if (PROGRAM_ErrorNo == E_NULL) {
    Depot_CommandLabelsToRun = NULL;
    /* get a list of target command labels specified in depot.pref */
    AllTargetCommandsSet =
      Preference_GetTargetCommandList(DEPOTDB_PreferenceDB(Depot_depotdbp));
    if ((PROGRAM_ErrorNo == E_NULL)
	&& !StringSet_Empty(AllTargetCommandsSet)) {
      /* check which of these commands have targets in the target database */
      for (sp = STRINGSET_Values(AllTargetCommandsSet);
	   (PROGRAM_ErrorNo == E_NULL) && (*sp != NULL);
	   sp++) {
	if (TargetDB_CommandTargetExists(*sp,
					 DEPOTDB_TargetDB(Depot_depotdbp),
				    DEPOTDB_PreferenceDB(Depot_depotdbp))) {
	  Depot_CommandLabelsToRun =
	    StringSet_AddString(Depot_CommandLabelsToRun, *sp);
	}
      }
    }
    if ((PROGRAM_ErrorNo == E_NULL)
	&& (DEPOTDB_TargetDB(Depot_depotdbp) != NULL)
	&& (TARGETDB_SourceList(DEPOTDB_TargetDB(Depot_depotdbp)) != NULL)) {
      id = -1;
      sourcelistp = TARGETDB_SourceList(DEPOTDB_TargetDB(Depot_depotdbp));
      for (i = 0;
	   (PROGRAM_ErrorNo == E_NULL)
	   && (i < TARGETSOURCELIST_NSources(sourcelistp));
	   i++) {
	if (TARGETSOURCE_CollectionId(TARGETSOURCELIST_Source(sourcelistp,
							      i))
	    != id) {		/* new collectionid in source list */
	  id =
	    TARGETSOURCE_CollectionId(TARGETSOURCELIST_Source(sourcelistp, i));
	  colp =
	    COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp),
				      id);
	  if (colp != NULL) {
	    /* add commands requested by the collection */
	    Depot_CommandLabelsToRun =
	      StringSet_Merge(Depot_CommandLabelsToRun,
			      COLLECTION_CommandList(colp));
	  }
	}
      }
    }
    Depot_CommandList = NULL;
    if ((PROGRAM_ErrorNo == E_NULL)
	&& !StringSet_Empty(Depot_CommandLabelsToRun)) {
      /* check all commands in the final list of labels */
      Depot_CommandList =
	(COMMAND **) emalloc((STRINGSET_Size(Depot_CommandLabelsToRun) + 1)
			     * sizeof(COMMAND *));
      if (PROGRAM_ErrorNo == E_NULL) {
	for (i = 0, commandp = Depot_CommandList;
	     (PROGRAM_ErrorNo == E_NULL)
	     && (i < STRINGSET_Size(Depot_CommandLabelsToRun));
	     i++, commandp++) {
	  *commandp = Command_Build(STRINGSET_String(Depot_CommandLabelsToRun, i),
				    DEPOTDB_TargetDB(Depot_depotdbp),
				    DEPOTDB_PreferenceDB(Depot_depotdbp),
				    DEPOTDB_CollectionList(Depot_depotdbp));
	}
	*(Depot_CommandList + STRINGSET_Size(Depot_CommandLabelsToRun))
	  = NULL;
      }
    }
    if ((PROGRAM_ErrorNo == E_NULL) && (Depot_CommandList != NULL)) {
      if (UpdateMode & M_SHOWACTIONSONLY) {
	for (commandp = Depot_CommandList;
	     (PROGRAM_ErrorNo == E_NULL) && (*commandp != NULL);
	     commandp++) {
	  Message("EXEC %s as:", COMMAND_Label(*commandp));
	  for (sp = STRINGARRAY_Values(COMMAND_Args(*commandp));
	       (PROGRAM_ErrorNo == E_NULL) && (*sp != NULL);
	       sp++) {
	    Message(" %s", *sp);
	  }
	  Message("\n");
	}
      }
    }
  }
  /* 
   * if targetdir needs to be updated,
   *    save depot database with suffix .NEW
   *    and mark point of no return.
   *    update targetdir, if necessary;
   *    run any commands, if necessary;
   */
#ifdef LOG_TIME
  timer_status("TargetDB Updating");
#endif

#ifdef LOG_DEPOTDB
  log_depotdb(DEPOTDB_TargetDB(Depot_depotdbp), "/tmp/db.write");
#endif

  if ((PROGRAM_ErrorNo == E_NULL) && !(UpdateMode & M_SHOWACTIONSONLY)) {
    /* save depot database with suffix .NEW and mark point of no return */
    (void) sprintf(filenamebuffer, "%s.NEW", DepotDBFileName);
    *PointofNoReturn = TRUE;
    Filtered_Message(UpdateMode & M_VERBOSE,
		     "Saving depot database to %s ...", filenamebuffer);

    fpwrite = efopen(filenamebuffer, "w");
    if (PROGRAM_ErrorNo == E_NULL) {
      (void) DepotDB_Write(fpwrite, Depot_depotdbp, Depot_CompressTDB);
      (void) fclose(fpwrite);
      Filtered_Message(UpdateMode & M_VERBOSE, " done.\n");
    }
#ifdef LOG_TIME
    timer_status("depot.db write");
#endif

    /* update targetdir, if necessary */
    if (PROGRAM_ErrorNo == E_NULL) {
      /* 
       * the nonsetting of TDB_SHOWACTIONS on PROGRAM_Verbose looks
       * counter-intuitive, but essentially, we turn it off because
       * with PROGRAM_Verbose set, FileOps routines will show all actions.
       */
      tdb_applyflags = TDB_BUILD;
      if (Depot_UseModTimes) {
	tdb_applyflags |= TDB_USEMODTIMES;
      }
      TargetDB_Apply(DEPOTDB_TargetDB(Depot_depotdbp),
		     tdb_applyflags,
		     DEPOTDB_CollectionList(Depot_depotdbp));
    }
    /* run any commands, if necessary */
    if ((PROGRAM_ErrorNo == E_NULL) && (Depot_CommandList != NULL)) {
      for (commandp = Depot_CommandList;
	   (PROGRAM_ErrorNo == E_NULL) && (*commandp != NULL);
	   commandp++) {
	if (PROGRAM_Verbose) {
	  Message("EXEC %s as\n", COMMAND_Label(*commandp));
	  Message("\t");
	  for (sp = STRINGSET_Values(COMMAND_Args(*commandp));
	       (PROGRAM_ErrorNo == E_NULL) && (*sp != NULL);
	       sp++) {
	    Message(" %s", *sp);
	  }
	  Message("\n");
	}
	Command_Run(*commandp);
      }
    }
    /* move saved new depot database to right location */
    if (PROGRAM_ErrorNo == E_NULL) {
      (void) sprintf(filenamebuffer, "%s.NEW", DepotDBFileName);
      Filtered_Message(UpdateMode & M_VERBOSE,
		       "Moving depot database %s to %s ...",
		       filenamebuffer, DepotDBFileName);
      (void) File_Move(filenamebuffer, DepotDBFileName);
      Filtered_Message(UpdateMode & M_VERBOSE, " done.\n");
    }
  }
#ifdef LOG_TIME
  timer_stop("Done");
#endif
  
  return;
}


/*
 * Depot_RunDeltaMode offers the greatest amount of speed but if the environment
 *     gets corrupt then it cannot handle things properly.
 */
void
Depot_RunDeltaMode(ImagePath, UpdateMode, PointofNoReturn)
     char *ImagePath;
     unsigned UpdateMode;
     Boolean *PointofNoReturn;
{
  register char **sp, **namep;
  register COLLECTION *colp, *oldcolp;
  register int i;
  register COMMAND **commandp;
  
  DEPOTDBHANDLE *dbh;
  HINTDB hintdb_old;
  STRINGSET *ChangedTargetCommandSet;
  Boolean Depot_AllCollectionsChanged;
  Boolean Depot_DepotConfChanged;
  Boolean Depot_ChangeThisRun;
  STRINGSET *Depot_ChangedCollections;
  STRINGSET *Depot_ChangedCommandLabels;
  PREFERENCE ***prefdiffp;

  Initialize(UpdateMode);

#ifdef LOG_TIME
  timer_start(stderr);
#endif

  if (PROGRAM_ErrorNo == E_NULL) {
    Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL, "Reading depot database...");
    dbh = DepotDB_Open(DepotDBFileName);
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    Depot_depotdbp = DepotDB_ReadPreliminaryInfo(Depot_depotdbp, dbh);
  } else {
    FatalError(E_NULLDEPOTDB, 
	       "Depot must complete successfully before the -Q switch can be used\n");
    return;
  }

  Filtered_Message(((PROGRAM_ErrorNo == E_NULL) && (PROGRAM_Verbose == PROGRAM_TRACEVAL)),
		   "done.\n");
#ifdef LOG_TIME
  timer_status("reading depot.db");
#endif

  Depot_ChangeThisRun = FALSE;
  Depot_AllCollectionsChanged = FALSE;
  Depot_ChangedCollections = NULL;
  Depot_ChangedCommandLabels = NULL;

  /* get preference differences */
  if (PROGRAM_ErrorNo == E_NULL) {
    Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL, "Checking for changes...");
    prefdiffp = Preference_Diff(DEPOTDB_PreferenceDB(Depot_depotdbp),
				DEPOTDB_OldPreferenceDB(Depot_depotdbp));
  }
  if ((PROGRAM_ErrorNo == E_NULL) && (prefdiffp != NULL)) {
    Depot_ChangeThisRun = TRUE;
    Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL, "preferences...");
  }
  /* check if global customizations have changed */
  if (PROGRAM_ErrorNo == E_NULL)
    Depot_DepotConfChanged
      = !DepotConfDB_Comparator(DEPOTDB_DepotConfDB(Depot_depotdbp),
				DEPOTDB_OldDepotConfDB(Depot_depotdbp));
  if ((PROGRAM_ErrorNo == E_NULL) && Depot_DepotConfChanged) {
    Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL, "depot.conf...");
    Depot_ChangeThisRun = TRUE;
  }

  if (PROGRAM_ErrorNo == E_NULL) {
    if (prefdiffp == NULL) {
      Depot_ChangedCollections = NULL;
    } else {			/* some preferences have changed */
      /* get list of collections whose preferences have changed */
      Depot_ChangedCollections
	= PreferenceDiff_GetChangedCollections(prefdiffp);
      if (StringSet_Member(Depot_ChangedCollections, PREF_WILDITEM)) {
	Depot_AllCollectionsChanged = TRUE;
      }
      /* get list of command labels whose preferences have changed */
      Depot_ChangedCommandLabels
	= PreferenceDiff_GetChangedCommands(prefdiffp);
    }
  }

  /* clone old collectionlist */
  if (PROGRAM_ErrorNo == E_NULL) {
    Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL, "done\n");
    DEPOTDB_CollectionList(Depot_depotdbp)
      = CollectionList(DEPOTDB_OldCollectionList(Depot_depotdbp));
  }

  /* mark all collections in the cloned list as potentially non-existent */
  if (PROGRAM_ErrorNo == E_NULL) {
    for (i = 0;
	 (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
	 i++) {
      colp =
	COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i);
      if (colp != NULL) {
	COLLECTION_Status(colp) |= C_POSSIBLYNONEXISTENT;
      }
    }
  }
#ifdef LOG_TIME
  timer_status("checking changes and marking collections");
#endif
  /* 
   * if all collections to be updated,
   * add them to cnameset and collection list
   */
  if (PROGRAM_ErrorNo == E_NULL) {
    /* get list of collections specified in preference database */
    cnameset =
      Preference_GetCollectionNameList(DEPOTDB_PreferenceDB(Depot_depotdbp));
    /* add names of collections found in searchpaths to this list */
    if (!StringSet_Empty(Depot_SearchPathList)) {
      for (sp = STRINGSET_Values(Depot_SearchPathList);
	   (PROGRAM_ErrorNo == E_NULL) && (*sp != NULL);
	   sp++) {
	tempstringset =
	  Depot_GetCollectionNamesInDirectory(*sp,
					      Depot_VersionDelimiter,
					      Depot_HintCache);
	if (PROGRAM_ErrorNo == E_NULL) {
	  cnameset = StringSet_Merge(cnameset, tempstringset);
	}
	if (tempstringset != NULL)
	  StringArray_Free(tempstringset);
      }
    }

    /* 
     * add these collection names to the collection list
     */
    if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(cnameset)) {
      for (namep = STRINGSET_Values(cnameset);
	   (PROGRAM_ErrorNo == E_NULL) && (*namep != NULL);
	   namep++) {
	DEPOTDB_CollectionList(Depot_depotdbp) =
	  CollectionList_AddCollectionName
	  (DEPOTDB_CollectionList(Depot_depotdbp),
	   *namep, C_POSSIBLYNONEXISTENT);
      }
      for (namep = STRINGSET_Values(cnameset);
	   (PROGRAM_ErrorNo == E_NULL) && (*namep != NULL);
	   namep++) {
	(void) CollectionList_SetCollectionStatus
	  (DEPOTDB_CollectionList(Depot_depotdbp), *namep, C_UPDATE);
      }
    }
  }
#ifdef LOG_TIME
  timer_status("adding collections");
#endif

  /* mark collections with changed preferences */
  if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(Depot_ChangedCollections)) {
    if (Depot_AllCollectionsChanged) {
      for (i = 0;
	   (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
	   i++) {
	colp =
	  COLLECTIONLIST_Collection
	  (DEPOTDB_CollectionList(Depot_depotdbp), i);
	if (colp != NULL) {
	  COLLECTION_Status(colp) |= C_PREFERENCECHANGE;
	}
      }
    } else {
      for (i = 0;
	   (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
	   i++) {
	colp =
	  COLLECTIONLIST_Collection
	  (DEPOTDB_CollectionList(Depot_depotdbp), i);
	if ((colp != NULL)
	    && StringSet_Member(Depot_ChangedCollections,
				COLLECTION_Name(colp))) {
	  COLLECTION_Status(colp) |= C_PREFERENCECHANGE;
	}
      }
    }
  }
  /* 
   *  only collections in cnameset are to be used
   *   mark others for deletion;
   */
  if (PROGRAM_ErrorNo == E_NULL) {
    for (i = 0;
	 (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
	 i++) {
      colp =
	COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i);
      if (colp != NULL) {
	if (!StringSet_Member(cnameset, COLLECTION_Name(colp))) {
	  COLLECTION_Status(colp) |= C_DELETE;
	}
      }
    }
  }
  /* mark collections to be ignored */
  if (PROGRAM_ErrorNo == E_NULL) {
    Depot_IgnoreSet =
      Preference_GetStringSet(DEPOTDB_PreferenceDB(Depot_depotdbp),
			      NULL, "ignore", NULL /* default */,
			      0 /* don't allocate space */);
  }
  if ((PROGRAM_ErrorNo == E_NULL) && !StringSet_Empty(Depot_IgnoreSet)) {
    for (sp = STRINGSET_Values(Depot_IgnoreSet); *sp != NULL; sp++) {
      (void) CollectionList_IgnoreCollection
	(DEPOTDB_CollectionList(Depot_depotdbp), *sp);
    }
  }
  /* 
   * get collection paths by checking preference options and the hint cache
   * if no path found for a potentially non-existent collection
   *   mark it for deletion
   */
#ifdef LOG_TIME
  timer_status("marking collections");
#endif
  for (i = 0;
       (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
       i++) {
    colp = COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i);
    if ((colp != NULL) && !(COLLECTION_Status(colp) & (C_IGNORE | C_DELETE))) {
      collectionpath =
	Depot_GetCollectionPath(COLLECTION_Name(colp),
				DEPOTDB_PreferenceDB(Depot_depotdbp),
				Depot_VersionDelimiter,
				Depot_HintCache);

      if (collectionpath == NULL) {
	if (!(COLLECTION_Status(colp) & C_POSSIBLYNONEXISTENT)) {
	  FatalError(E_NOPATHTOCOLLECTION,
		     "Could not locate path to collection named %s.\n",
		     COLLECTION_Name(colp));
	} else {
	  COLLECTION_Path(colp) = NULL;
	  COLLECTION_Status(colp) |= C_DELETE;
	}
      } else {
	if (PROGRAM_ErrorNo == E_NULL)
	  COLLECTION_Path(colp) = collectionpath;
	/* note collectionpath is allocated memory! */
      }
    }
  }
#ifdef LOG_TIME
  timer_status("locating collections");
#endif
  /* get new hints for all collection paths */
  for (i = 0;
       (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
       i++) {
    HINTDB hdb;

    colp = COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i);
    if ((colp != NULL) && !(COLLECTION_Status(colp) & (C_IGNORE | C_DELETE))) {
      hdb = NULL;

      /* check if the hint cache has hint info */
      if (Depot_HintCache != NULL)
	hdb = HintDB_GetRelatedHints(Depot_HintCache, COLLECTION_Path(colp));

      /* if not, seek info from collection-specific hint service */
      if ((hdb == NULL) && (PROGRAM_ErrorNo == E_NULL)) {
	hintservicepath
	  = Preference_GetString(DEPOTDB_PreferenceDB(Depot_depotdbp),
				 COLLECTION_Name(colp),
				 "hintfile",
				 NULL /* default */ );
	if ((hintservicepath != NULL) && (PROGRAM_ErrorNo == E_NULL))
	  hdb = HintService_GetRelatedHints(hintservicepath,
					    COLLECTION_Path(colp));
      }
      if ((hdb != NULL) && (PROGRAM_ErrorNo == E_NULL)) {
	hdb = HintDB_UnsetPathPrefix(hdb, COLLECTION_Path(colp));
      }
      COLLECTION_HintDB(colp) = (PROGRAM_ErrorNo == E_NULL) ? hdb : NULL;
    }
  }
#ifdef LOG_TIME
  timer_status("getting new hints");
#endif

  /* prune previously non-existent collections to be ignored or deleted */
  if (PROGRAM_ErrorNo == E_NULL) {
    for (i = 0;
	 (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
	 i++) {
      colp = COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i);
      oldcolp
	= COLLECTIONLIST_Collection(DEPOTDB_OldCollectionList(Depot_depotdbp), i);
      if ((colp != NULL) || (oldcolp != NULL)) {
	if (colp == NULL) {	/* (oldcolp != NULL) */
	  FatalError(E_BADDEPOTCOLLECTIONLIST,
		     "Collection unexpectedly removed from list.\n");
	} else {		/* colp != NULL */
	  if (oldcolp == NULL) {
	    if (COLLECTION_Status(colp) & (C_IGNORE | C_DELETE)) {
	      COLLECTIONLIST_Collection
		(DEPOTDB_CollectionList(Depot_depotdbp), i) = NULL;
	      Collection_Free(colp);
	      colp = NULL;
	    }
	  }
	}
      }
    }
  }
  /* get old&new installmethod for all collections not to be ignored or *
   * deleted */
  if (PROGRAM_ErrorNo == E_NULL) {
    for (i = 0;
	 (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
	 i++) {
      colp = COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i);
      oldcolp = COLLECTIONLIST_Collection(DEPOTDB_OldCollectionList(Depot_depotdbp), i);
      if ((colp != NULL) && !(COLLECTION_Status(colp) & (C_IGNORE | C_DELETE))) {
	if (String_Comparator
	    (PreferenceDB_CollectionInstallMethod
	     (COLLECTION_Name(colp), DEPOTDB_PreferenceDB(Depot_depotdbp)),
	     "copy") == 0) {
	  COLLECTION_MapMethod(colp) = U_MAPCOPY;
	} else {
	  COLLECTION_MapMethod(colp) = U_MAPLINK;
	}
	if (oldcolp != NULL) {
	  if (String_Comparator
	      (PreferenceDB_CollectionInstallMethod
	       (COLLECTION_Name(oldcolp), DEPOTDB_OldPreferenceDB(Depot_depotdbp)),
	       "copy") == 0) {
	    COLLECTION_MapMethod(oldcolp) = U_MAPCOPY;
	  } else {
	    COLLECTION_MapMethod(oldcolp) = U_MAPLINK;
	  }
	}
      }
    }
  }
#ifdef LOG_TIME
  timer_status("loading info");
#endif
  /* 
   *   for all collections which are not to be ignored
   *     if collection is to be deleted
   *       if collection existed previously, we have a change this run;
   *     else
   *       if new collection or path change or hint change
   *         read file system image;
   *         if no hints, extract hints from file system image;
   *
   *       if new collection
   *         mark as new and path as changed;
   *       else
   *         mark if path has changed;
   *         mark if hint has changed;
   *
   *       mark if mapmethod has changed;
   * 
   *       if not new collection and no hint or path change
   *         and global depotconf has not changed
   *         clone depotconfdb from old depotconfdb;
   *       else
   *         get depotconfdb of collection;
   *         append related specs from global depotconf;
   *         set status of collection appropriately if depotconf has changed;
   *       get list of commands for collection;
   */

  /* 
   *   if Depot_ChangeThisRun
   *     read in old target database as global target database;
   *     for all collections
   *       if collection is to be ignored or deleted or has changed
   *         obsolete collection in global target database;
   *       if collection is not to be deleted or ignored and has changed
   *         generate targetdb;
   *         remove depot.conf and depot.image from the targetdb;
   *         merge targetdb into global target database;
   *         free targetdb;
   *       prune obsolete contributions of collection in global target database;
   */

  if (PROGRAM_ErrorNo == E_NULL) {
    for (i = 0;
	 (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
	 i++) {
      colp =
	COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i);
      oldcolp =
	COLLECTIONLIST_Collection(DEPOTDB_OldCollectionList(Depot_depotdbp), i);

      if ((colp != NULL) && !(COLLECTION_Status(colp) & C_IGNORE)) {
	Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
			 "Checking %s (%d)...", COLLECTION_Name(colp), COLLECTION_Id(colp));
	if (COLLECTION_Status(colp) & C_DELETE) {
	  if ((oldcolp != NULL)
	      && !(COLLECTION_Status(oldcolp) & (C_IGNORE | C_DELETE))) {
	    Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
			     "collection will be updated\n");
	    Depot_ChangeThisRun = TRUE;
	  }
	} else {
	  if ((oldcolp == NULL)
	      || (String_Comparator(COLLECTION_Path(colp),
				    COLLECTION_Path(oldcolp)) != 0)
	      || (COLLECTION_HintDB(colp) == NULL)
	      || (HintDB_Comparator(COLLECTION_HintDB(colp),
				    COLLECTION_HintDB(oldcolp)) != 0)) {
	    /* new  collection or change or hint change */

	    /* read file system image */
	    if (ImagePath == NULL) {
	      (void) sprintf(filenamebuffer,
			     "%s/%s", COLLECTION_Path(colp), DEPOTIMAGEFILE);
	    } else {
	      (void) sprintf(filenamebuffer, "%s/%s.image",
			     ImagePath, COLLECTION_Name(colp));
	    }
	    COLLECTION_Image(colp) =
	      FileSystemImage_Read(filenamebuffer);
	    if ((COLLECTION_Image(colp) != NULL) 
		&& (PROGRAM_ErrorNo == E_NULL))  {
	      Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
			       "loaded fs image...");
	    }
	    /* if no hints, extract hints from file system image */
	    if ((PROGRAM_ErrorNo == E_NULL)
		&& (COLLECTION_HintDB(colp) == NULL)
		&& (COLLECTION_Image(colp) != NULL)) {
	      COLLECTION_HintDB(colp) =
		Hint_CollectAllHints(COLLECTION_Image(colp));
	      Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
			       "loaded hints...");
	    }
	  }
	  /* should the following block be thrown away for delete mode? */
	  if (oldcolp == NULL) {	/* new collection */
	    COLLECTION_Status(colp) |= (C_NEW | C_PATHCHANGE);
	  } else {
	    if (String_Comparator(COLLECTION_Path(colp),
				  COLLECTION_Path(oldcolp)) != 0)
	      COLLECTION_Status(colp) |= C_PATHCHANGE;
	    else if ((COLLECTION_HintDB(colp) == NULL)
		     || (HintDB_Comparator
			 (COLLECTION_HintDB(colp),
			  COLLECTION_HintDB(oldcolp)) != TRUE)) {
	      COLLECTION_Status(colp) |= C_MODTIMECHANGE;
	    }
	  }

	  /* mark if mapmethod has changed */
	  if ((oldcolp == NULL)
	      || (COLLECTION_MapMethod(colp) != COLLECTION_MapMethod(oldcolp))) {
	    COLLECTION_Status(colp) |= C_INSTALLMETHODCHANGE;
	  }
	  /* get depotconfdb */
	  if (!(COLLECTION_Status(colp)
		& (C_NEW | C_PATHCHANGE | C_MODTIMECHANGE))
	      && !Depot_DepotConfChanged) {
	    /* no change in either the  collection or global  depot.conf */
	    /* clone depotconfdb from old depotconfdb */
	    COLLECTION_DepotConfDB(colp) =
	      DepotConfDB(COLLECTION_DepotConfDB(oldcolp));
	  } else {
	    /* read in depotconfdb of collection */
	    COLLECTION_DepotConfDB(colp) = 
	      Load_DepotConfDB(COLLECTION_Name(colp), COLLECTION_Path(colp));

	    /* set status appropriately if depotconf has changed */
	    if ((oldcolp == NULL)
		|| !DepotConfDB_Comparator(COLLECTION_DepotConfDB(colp),
					 COLLECTION_DepotConfDB(oldcolp))) {
	      COLLECTION_Status(colp) |= C_DEPOTCONFCHANGE;
	    }
	  }

	  /* get list of commands for collection */
	  if (PROGRAM_ErrorNo == E_NULL) {
	    if (COLLECTION_Status(colp) & C_DEPOTCONFCHANGE) {
	      COLLECTION_CommandList(colp) =
		DepotConfDB_GetCommandList(COLLECTION_DepotConfDB(colp));
	    } else {
	      if (COLLECTION_CommandList(oldcolp) == NULL)
		COLLECTION_CommandList(colp) = NULL;
	      else
		COLLECTION_CommandList(colp)
		  = StringArray
		  (COLLECTION_CommandList(oldcolp),
		   StringArray_Size(COLLECTION_CommandList(oldcolp)));
	    }
	  }
	  if ((PROGRAM_ErrorNo == E_NULL)
	      && (COLLECTION_Status(colp)
	  & (C_NEW | C_PATHCHANGE | C_MODTIMECHANGE | C_DEPOTCONFCHANGE))) {
	    Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
			     "collection has changed...");
	    Depot_ChangeThisRun = TRUE;
	  }
	  Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
			   "done.\n");
	}
      }
    }
  }
  /* 
   *   if Depot_ChangeThisRun
   *     read in old target database as global target database;
   *     for all collections
   *       if collection is to be ignored or deleted or has changed
   *          obsolete collection in global target database;
   *       if collection is not to be ignored or deleted and has changed
   *          generate targetdb;
   *          remove depot.conf and depot.image from the targetdb;
   *          merge targetdb into global target database;
   *          free targetdb;
   *      prune obsolete contributions of collection in global target database;
   */
#ifdef LOG_TIME
  timer_status("processing collections");
#endif

  if ((PROGRAM_ErrorNo == E_NULL) && Depot_ChangeThisRun) {
    if (PROGRAM_ErrorNo == E_NULL) {
      Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
		       "Environment has changed. Updating..\n");
    }
    /* read in old target database as global target database */
    DEPOTDB_TargetDB(Depot_depotdbp) = DepotDB_TargetDB_Read(dbh);

#ifdef LOG_DEPOTDB
    log_depotdb(DEPOTDB_TargetDB(Depot_depotdbp), "/tmp/db.read-Q");
#endif

#ifdef LOG_TIME
    timer_status("targetdb_read");
#endif
    (void) DepotDB_Close(dbh);

    for (i = 0;
	 (PROGRAM_ErrorNo == E_NULL)
       && (i < COLLECTIONLIST_Size(DEPOTDB_CollectionList(Depot_depotdbp)));
	 i++) {
#ifdef LOG_DEPOTDB
      sprintf(log_path,"/tmp/db.D-%d", i); 
      log_depotdb(DEPOTDB_TargetDB(Depot_depotdbp), log_path);
#endif
      colp =
	COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp), i);
      oldcolp =
	COLLECTIONLIST_Collection(DEPOTDB_OldCollectionList(Depot_depotdbp), i);

      if ((colp != NULL)
	  && ((COLLECTION_Status(colp) & (C_IGNORE | C_DELETE))
	      || (COLLECTION_Status(colp)
		  & (C_NEW | C_PATHCHANGE | C_INSTALLMETHODCHANGE
		     | C_MODTIMECHANGE | C_DEPOTCONFCHANGE)))) {
	/* collection is to be ignored, deleted, or has changed */

	/* obsolete collection in global target database */
#ifdef LOG_DEPOTDB
	sprintf(log_path,"/tmp/db.C-%d",i);
  log_depotdb(DEPOTDB_TargetDB(Depot_depotdbp), log_path);
#endif

	DEPOTDB_TargetDB(Depot_depotdbp) =
	  TargetDB_ObsoleteCollection(DEPOTDB_TargetDB(Depot_depotdbp),
				      COLLECTION_Id(colp));
#ifdef LOG_DEPOTDB
	sprintf(log_path, "/tmp/db.B-%d", i);
  log_depotdb(DEPOTDB_TargetDB(Depot_depotdbp), log_path);
#endif

      }
      if ((PROGRAM_ErrorNo == E_NULL)
	  && (colp != NULL)
	  && !(COLLECTION_Status(colp) & (C_IGNORE | C_DELETE))
	  && (COLLECTION_Status(colp)
	  & (C_NEW | C_PATHCHANGE | C_MODTIMECHANGE | C_DEPOTCONFCHANGE))) {
	/* collection is not to be ignored, deleted, and has changed internally */
	Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
			 "Processing %s...",
			 COLLECTION_Name(colp));
	/* generate targetdb */
	if (PROGRAM_ErrorNo == E_NULL) {
	  if ((oldcolp == NULL)
	      || (COLLECTION_Status(colp) & (C_NEW | C_PATHCHANGE))) {
	    hintdb_old = NULL;
	  } else {
	    hintdb_old = COLLECTION_HintDB(oldcolp);
	  }
	  COLLECTION_TargetDB(colp) =
	    Collection_TargetDBGenerate(colp, hintdb_old);
	}
	/* remove DEPOTCONFIGFILE and DEPOTIMAGEFILE from the targetdb */
	if ((PROGRAM_ErrorNo == E_NULL) && (COLLECTION_TargetDB(colp)!= NULL)) {
	  COLLECTION_TargetDB(colp) =
	    TargetDB_DeleteTargetPath(COLLECTION_TargetDB(colp),
				      DEPOTCONFIGFILE,
				      TDB_LAX);
	  COLLECTION_TargetDB(colp) =
	    TargetDB_DeleteTargetPath(COLLECTION_TargetDB(colp),
				      DEPOTIMAGEFILE,
				      TDB_LAX);
	}
	/* merge targetdb into global target database */
	if ((PROGRAM_ErrorNo == E_NULL) && (COLLECTION_TargetDB(colp)!= NULL)) {
	  DEPOTDB_TargetDB(Depot_depotdbp) =
	    TargetDB_Merge(DEPOTDB_TargetDB(Depot_depotdbp),
			   COLLECTION_TargetDB(colp), 
			   COLLECTION_Status(colp) & (C_NEW | C_PATHCHANGE));
	
	  Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL, "done.\n");
	}
	if ((PROGRAM_ErrorNo == E_NULL)
	    && (COLLECTION_TargetDB(colp) != NULL)) {
	  TargetDB_Free(COLLECTION_TargetDB(colp));
	  COLLECTION_TargetDB(colp) = NULL;
	}
      } else if ((PROGRAM_ErrorNo == E_NULL)
		 && (colp != NULL)
		 && !(COLLECTION_Status(colp) & (C_IGNORE | C_DELETE))
		 && (COLLECTION_Status(colp) & C_INSTALLMETHODCHANGE)) {
        /* collection  is not to be ignored, or deleted, and installmethod has changed */

	DEPOTDB_TargetDB(Depot_depotdbp) =
	  TargetDB_UnObsoleteCollection(DEPOTDB_TargetDB(Depot_depotdbp),
					COLLECTION_Id(colp),
					COLLECTION_MapMethod(colp));
      }
      /* 
       * prune obsolete contributions of collection
       * in global target database
       */
#ifdef LOG_DEPOTDB
      sprintf(log_path, "/tmp/db.A-%d",i);
      log_depotdb(DEPOTDB_TargetDB(Depot_depotdbp), log_path);
#endif

      if ((PROGRAM_ErrorNo == E_NULL)
	  && (colp != NULL)
	  && ((COLLECTION_Status(colp) & (C_IGNORE | C_DELETE))
	      || (COLLECTION_Status(colp)
	 & (C_NEW | C_PATHCHANGE | C_MODTIMECHANGE | C_DEPOTCONFCHANGE)))) {
        /* collection is to be ignored, deleted, or has changed */
	DEPOTDB_TargetDB(Depot_depotdbp) =
	  TargetDB_PruneCollection(DEPOTDB_TargetDB(Depot_depotdbp),
				   COLLECTION_Id(colp));
      }
#ifdef LOG_DEPOTDB
      sprintf(log_path, "/tmp/db.a-%d",i);
      log_depotdb(DEPOTDB_TargetDB(Depot_depotdbp), log_path);
#endif

      if ((PROGRAM_ErrorNo == E_NULL)
	  && (colp != NULL) && (COLLECTION_Image(colp) != NULL)) {
	FileSystemImage_Free(COLLECTION_Image(colp));
	COLLECTION_Image(colp) = NULL;
      }
    }
  }
#ifdef LOG_TIME
  timer_status("processing changed collections");
#endif /* LOG_TIME */
  

  Filtered_Message(((PROGRAM_ErrorNo == E_NULL) 
		    && !Depot_ChangeThisRun && PROGRAM_Verbose), "No updates necessary\n");

  if ((PROGRAM_ErrorNo == E_NULL) && Depot_ChangeThisRun) {
#ifdef LOG_DEPOTDB
    log_depotdb(DEPOTDB_TargetDB(Depot_depotdbp), "/tmp/db.T1-Q");
#endif

    /* prune sourceless entries */
    if (PROGRAM_ErrorNo == E_NULL) {
      DEPOTDB_TargetDB(Depot_depotdbp) =
	TargetDB_PruneTargets(DEPOTDB_TargetDB(Depot_depotdbp));
    }
#ifdef LOG_DEPOTDB
    log_depotdb(DEPOTDB_TargetDB(Depot_depotdbp), "/tmp/db.T2-Q");
#endif
#ifdef LOG_TIME
    timer_status("pruning targets");
#endif

    /* sort sources by override information */
    if (PROGRAM_ErrorNo == E_NULL) {
      TargetDB_SortByOverrides(DEPOTDB_TargetDB(Depot_depotdbp),
				      DEPOTDB_PreferenceDB(Depot_depotdbp),
				     DEPOTDB_CollectionList(Depot_depotdbp));
    }
#ifdef LOG_TIME
    timer_status("Sorting by overrides");
#endif
#ifdef LOG_DEPOTDB
    log_depotdb(DEPOTDB_TargetDB(Depot_depotdbp), "/tmp/db.T3-Q");
#endif
    /* check for conflicts */
    if (PROGRAM_ErrorNo == E_NULL) {
      TargetDB_CheckSourceConsistency(DEPOTDB_TargetDB(Depot_depotdbp),
				      DEPOTDB_PreferenceDB(Depot_depotdbp),
				      DEPOTDB_CollectionList(Depot_depotdbp),
				      TRUE /* findallconflicts */ );
    }

#ifdef LOG_TIME
    timer_status("checking consistency");
#endif
#ifdef LOG_DEPOTDB
    log_depotdb(DEPOTDB_TargetDB(Depot_depotdbp), "/tmp/db.T4-Q");
#endif

    /* apply target specific mappings */
    if (PROGRAM_ErrorNo == E_NULL) {
      DEPOTDB_TargetDB(Depot_depotdbp) =
	TargetDB_SetTargetMappings(DEPOTDB_TargetDB(Depot_depotdbp),
				   DEPOTDB_PreferenceDB(Depot_depotdbp));
    }
    /* apply target specific status info */
    if (PROGRAM_ErrorNo == E_NULL) {
      DEPOTDB_TargetDB(Depot_depotdbp) =
	TargetDB_SetTargetFileStatus(DEPOTDB_TargetDB(Depot_depotdbp),
				     DEPOTDB_PreferenceDB(Depot_depotdbp));
    }

    /* check for violations of special files */
    if (PROGRAM_ErrorNo == E_NULL) {
      Depot_SpecialFileSet =
	Preference_GetStringSet(DEPOTDB_PreferenceDB(Depot_depotdbp),
				NULL,
				"specialfile",
				NULL /* default */,
				0 /* don't allocate space */);
    }
    if (PROGRAM_ErrorNo == E_NULL) {
      Depot_SpecialFileSet =
	StringSet_Merge(Depot_SpecialFileSet, &DEPOT_SpecialFileSet);
    }
    if ((PROGRAM_ErrorNo == E_NULL)
	&& !StringSet_Empty(Depot_SpecialFileSet)) {
      DEPOTDB_TargetDB(Depot_depotdbp) =
	TargetDB_ProtectSpecialTargets(DEPOTDB_TargetDB(Depot_depotdbp),
				    STRINGSET_Values(Depot_SpecialFileSet));
    }

#ifdef LOG_DEPOTDB
    log_depotdb(DEPOTDB_TargetDB(Depot_depotdbp), "/tmp/db.T5-Q");
#endif 

#ifdef LOG_TIME
    timer_status("setting target info");
#endif

    /* check targetdb tree for need to update */
    if (PROGRAM_ErrorNo == E_NULL) {
    Filtered_Message(PROGRAM_Verbose == PROGRAM_TRACEVAL,
		     "Updating targetdir.\n");
      tdb_checkflags = TDB_CHECK | TDB_TRUSTTDB ;
      if (UpdateMode & M_SHOWACTIONSONLY) {
	tdb_checkflags |= TDB_SHOWACTIONS;
      }
      if (Depot_UseModTimes) {
	tdb_checkflags |= TDB_USEMODTIMES;
      }

      TargetDB_Apply(DEPOTDB_TargetDB(Depot_depotdbp),
		     tdb_checkflags,
		     DEPOTDB_CollectionList(Depot_depotdbp));
    }
#ifdef LOG_TIME
    timer_status("targetdb update");
#endif

    /* 
     * get a list of target specific command labels specified in depot.pref;
     * check which command labels have changed command preferences;
     * check which of these have applicable nodes in the target database;
     * check which command labels have modified nodes in the target database;
     *
     * for all collections which contribute to the target database
     *   if the collection has changed
     *     add commands requested by the collection;
     *   else
     *     add commands whose preferences have changed;
     * check all commands in the final list of labels;
     */

    /* get a list of target specific command labels specified in depot.pref */
    if (PROGRAM_ErrorNo == E_NULL) {
      Depot_CommandLabelsToRun = NULL;
      ChangedTargetCommandSet = NULL;

      AllTargetCommandsSet =
	Preference_GetTargetCommandList(DEPOTDB_PreferenceDB(Depot_depotdbp));
    }
    /* check which command labels have changed command preferences */
    if ((PROGRAM_ErrorNo == E_NULL)
	&& !StringSet_Empty(AllTargetCommandsSet)) {
      for (sp = STRINGSET_Values(AllTargetCommandsSet);
	   (PROGRAM_ErrorNo == E_NULL) && (*sp != NULL);
	   sp++) {
	if ((PROGRAM_ErrorNo == E_NULL)
	    && StringSet_Member(Depot_ChangedCommandLabels, *sp)) {
	  ChangedTargetCommandSet =
	    StringSet_AddString(ChangedTargetCommandSet, *sp);
	}
      }
    }
    /* check which of these have applicable nodes in the target database */
    if ((PROGRAM_ErrorNo == E_NULL)
	&& !StringSet_Empty(ChangedTargetCommandSet)) {
      for (sp = STRINGSET_Values(ChangedTargetCommandSet);
	   (PROGRAM_ErrorNo == E_NULL) && (*sp != NULL);
	   sp++) {
	if (TargetDB_CommandTargetExists(*sp,
					 DEPOTDB_TargetDB(Depot_depotdbp),
				    DEPOTDB_PreferenceDB(Depot_depotdbp))) {
	  Depot_CommandLabelsToRun =
	    StringSet_AddString(Depot_CommandLabelsToRun, *sp);
	}
      }
    }
    /* check which command labels have modified nodes in the target database */
    if ((PROGRAM_ErrorNo == E_NULL)
	&& !StringSet_Empty(AllTargetCommandsSet)) {
      for (sp = STRINGSET_Values(AllTargetCommandsSet);
	   (PROGRAM_ErrorNo == E_NULL) && (*sp != NULL);
	   sp++) {
	if (TargetDB_CommandTargetModified(*sp,
					   DEPOTDB_TargetDB(Depot_depotdbp),
				    DEPOTDB_PreferenceDB(Depot_depotdbp))) {
	  Depot_CommandLabelsToRun =
	    StringSet_AddString(Depot_CommandLabelsToRun, *sp);
	}
      }
    }
    if ((PROGRAM_ErrorNo == E_NULL)
	&& (DEPOTDB_TargetDB(Depot_depotdbp) != NULL)
	&& (TARGETDB_SourceList(DEPOTDB_TargetDB(Depot_depotdbp)) != NULL)) {
      id = -1;
      sourcelistp = TARGETDB_SourceList(DEPOTDB_TargetDB(Depot_depotdbp));
      for (i = 0;
	   (PROGRAM_ErrorNo == E_NULL)
	   && (i < TARGETSOURCELIST_NSources(sourcelistp));
	   i++) {
	if (TARGETSOURCE_CollectionId(TARGETSOURCELIST_Source(sourcelistp, i))
	    != id) {		/* new collectionid in source list */
	  id =
	    TARGETSOURCE_CollectionId(TARGETSOURCELIST_Source(sourcelistp, i));
	  colp =
	    COLLECTIONLIST_Collection(DEPOTDB_CollectionList(Depot_depotdbp),
				      id);
	  if ((colp != NULL)
	      && !(COLLECTION_Status(colp) & (C_IGNORE | C_DELETE))) {
	    if (COLLECTION_Status(colp)
		& (C_NEW | C_PATHCHANGE
		   | C_MODTIMECHANGE | C_DEPOTCONFCHANGE)) {
	      /* collection has changed */
	      /* add commands requested by the collection */
	      Depot_CommandLabelsToRun =
		StringSet_Merge(Depot_CommandLabelsToRun,
				COLLECTION_CommandList(colp));
	    } else {		/* no change in the collection */
	      /* add commands whose preferences have changed */
	      if (!StringSet_Empty(COLLECTION_CommandList(colp))) {
		for (sp = STRINGSET_Values(COLLECTION_CommandList(colp));
		     (PROGRAM_ErrorNo == E_NULL) && (*sp != NULL);
		     sp++) {
		  if (StringSet_Member(Depot_ChangedCommandLabels, *sp)) {
		    Depot_CommandLabelsToRun =
		      StringSet_AddString(Depot_CommandLabelsToRun, *sp);
		  }
		}
	      }
	    }
	  }
	}
      }
    }
    Depot_CommandList = NULL;
    if ((PROGRAM_ErrorNo == E_NULL)
	&& !StringSet_Empty(Depot_CommandLabelsToRun)) {
      /* check all commands in the final list of labels */
      Depot_CommandList =
	(COMMAND **) emalloc((STRINGSET_Size(Depot_CommandLabelsToRun) + 1)
			     * sizeof(COMMAND *));
      if (PROGRAM_ErrorNo == E_NULL) {
	for (i = 0, commandp = Depot_CommandList;
	     (PROGRAM_ErrorNo == E_NULL)
	     && (i < STRINGSET_Size(Depot_CommandLabelsToRun));
	     i++, commandp++) {
	  *commandp = Command_Build(STRINGSET_String(Depot_CommandLabelsToRun, i),
				    DEPOTDB_TargetDB(Depot_depotdbp),
				    DEPOTDB_PreferenceDB(Depot_depotdbp),
				    DEPOTDB_CollectionList(Depot_depotdbp));
	}
	*(Depot_CommandList + STRINGSET_Size(Depot_CommandLabelsToRun))
	  = NULL;
      }
    }
    if ((PROGRAM_ErrorNo == E_NULL) && (Depot_CommandList != NULL)) {
      if (UpdateMode & M_SHOWACTIONSONLY) {
	for (commandp = Depot_CommandList;
	     (PROGRAM_ErrorNo == E_NULL) && (*commandp != NULL);
	     commandp++) {
	  Message("EXEC %s as\n", COMMAND_Label(*commandp));
	  Message("\t");
	  for (sp = STRINGSET_Values(COMMAND_Args(*commandp));
	       (PROGRAM_ErrorNo == E_NULL) && (*sp != NULL);
	       sp++) {
	    Message(" %s", *sp);
	  }
	  Message("\n");
	}
      }
    }
    /* 
     * if targetdir needs to be updated
     *  save depot database with suffix .NEW and mark point of no return;
     *  update targetdir, if necessary;
     *  run any commands, if necessary;
     */
#ifdef LOG_DEPOTDB
    log_depotdb(DEPOTDB_TargetDB(Depot_depotdbp), "/tmp/db.write-Q");
#endif

    if ((PROGRAM_ErrorNo == E_NULL) && !(UpdateMode & M_SHOWACTIONSONLY)) {
      /* save depot database with suffix .NEW and mark point of no return */
      (void) sprintf(filenamebuffer, "%s.NEW", DepotDBFileName);
      *PointofNoReturn = TRUE;

#ifdef LOG_TIME
      timer_status("before depot.db write");
#endif

      Filtered_Message(UpdateMode & M_VERBOSE,
		       "Saving depot database to %s ...", filenamebuffer);
      fpwrite = efopen(filenamebuffer, "w");


      if (PROGRAM_ErrorNo == E_NULL) {
	(void) DepotDB_Write(fpwrite, Depot_depotdbp, Depot_CompressTDB);
	(void) fclose(fpwrite);
	Filtered_Message(UpdateMode & M_VERBOSE, " done.\n");
      }
#ifdef LOG_TIME
      timer_status("depot.db write");
#endif

      /* update targetdir, if necessary */
      if (PROGRAM_ErrorNo == E_NULL) {
	/* 
	 * the nonsetting of TDB_SHOWACTIONS on PROGRAM_Verbose looks
	 * counter-intuitive, but essentially, we turn it off because
	 * with PROGRAM_Verbose set, FileOps routines will show all actions.
	 */
	tdb_applyflags = TDB_TRUSTTDB;
	if (Depot_UseModTimes) {
	  tdb_applyflags |= TDB_USEMODTIMES;
	}
	TargetDB_Apply(DEPOTDB_TargetDB(Depot_depotdbp),
		       tdb_applyflags,
		       DEPOTDB_CollectionList(Depot_depotdbp));
      }
      /* run any commands, if necessary */
      if ((PROGRAM_ErrorNo == E_NULL) && (Depot_CommandList != NULL)) {
	for (commandp = Depot_CommandList;
	     (PROGRAM_ErrorNo == E_NULL) && (*commandp != NULL);
	     commandp++) {
	  if (PROGRAM_Verbose) {
	    Message("EXEC %s as\n", COMMAND_Label(*commandp));
	    Message("\t");
	    for (sp = STRINGSET_Values(COMMAND_Args(*commandp));
		 (PROGRAM_ErrorNo == E_NULL) && (*sp != NULL);
		 sp++) {
	      Message(" %s", *sp);
	    }
	    Message("\n");
	  }
	  Command_Run(*commandp);
	}
      }
      /* save old depot database and move saved new one to right location */
      if (PROGRAM_ErrorNo == E_NULL) {
	(void) sprintf(filenamebuffer, "%s.OLD", DepotDBFileName);
	Filtered_Message(UpdateMode & M_VERBOSE,
			 "Backing up old depot database %s to %s ...",
			 DepotDBFileName, filenamebuffer);
	(void) File_Move(DepotDBFileName, filenamebuffer);
	Filtered_Message(UpdateMode & M_VERBOSE, " done.\n");
      }
      if (PROGRAM_ErrorNo == E_NULL) {
	(void) sprintf(filenamebuffer, "%s.NEW", DepotDBFileName);
	Filtered_Message(UpdateMode & M_VERBOSE,
			 "Moving depot database %s to %s ...",
			 filenamebuffer, DepotDBFileName);
	(void) File_Move(filenamebuffer, DepotDBFileName);
	Filtered_Message(UpdateMode & M_VERBOSE, " done.\n");
      }
    }
  }
#ifdef LOG_TIME
  timer_stop("done");
#endif

  return;
}

/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/017/src/lib/Depot/RCS/Depot_RunMode.c,v $ */
