/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: Preference_Write.c,v 1.6 1994/09/30 20:49:00 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "Preference.h"

static int Preference_Write();

static char PreferenceSpecialChars[] =
{
  PREF_QUOTECHAR,
  PREF_STRINGARRAYDELIM,
  PREF_SENTENCEDELIM,
  ' ', '\t', '\n', '\'', '\"', '\\',
  '\0'
};

static int NPreferenceSpecialChars =
sizeof(PreferenceSpecialChars) / sizeof(PreferenceSpecialChars[0]) - 1;


int 
PreferenceDB_Write(fp, prefdb)
     FILE *fp;
     PREFERENCEDB *prefdb;
{
  register unsigned int i;
  register PREFERENCE ***arrayp;
  register PREFERENCE **pp;

  for (i = 0, arrayp = *prefdb;
       (i < Preference_HashTableSize) && (PROGRAM_ErrorNo == E_NULL);
       i++, arrayp++) {
    if (*arrayp != NULL)
      /* not an empty hash bucket */
    {
      for (pp = *arrayp;
	   (*pp != NULL) && (PROGRAM_ErrorNo == E_NULL);
	   pp++) {
	if (Preference_Write(fp, *pp) < 0) {
	  FatalError(E_PREFWRITEFAILED,
		     "Error while writing preference entry\n");
	}
      }
    }
  }

  return (PROGRAM_ErrorNo == E_NULL) ? 0 : -1;
}


static int 
Preference_Write(fp, prefp)
     FILE *fp;
     PREFERENCE *prefp;
{
  char *pname;
  char *item;
  unsigned short ptype;

  if (prefp == NULL) {
    FatalError(E_BADPREFERENCE, "Attempt to write NULL preference\n");
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    pname = Preference_OptionName(PREFERENCE_OptionID(prefp));
    item = PREFERENCE_Operand(prefp);
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    ptype = PREFERENCE_Type(prefp);
    if (ptype & PREF_INTERNAL) { /* don't write out pref_internal entries */
      return 0;
    }

    String_Write(fp, pname, PREF_QUOTECHAR, PREF_STRINGARRAYDELIM,
		 PreferenceSpecialChars, NPreferenceSpecialChars);
    fputc('\t', fp);

    if (ptype & PREF_REGULAR) {
      String_Write(fp, item, PREF_QUOTECHAR, PREF_STRINGARRAYDELIM,
		   PreferenceSpecialChars, NPreferenceSpecialChars);
      fputc('\t', fp);
    }
    switch (ptype & ~(PREF_INTERNAL | PREF_REGULAR | PREF_GLOBAL | PREF_COMMAND | PREF_FILTER)) {
    case PREF_INTEGER:
      fprintf(fp, "%d", PREFERENCE_IVal(prefp));
      break;
    case PREF_UNSIGNED:
      fprintf(fp, "%d", PREFERENCE_UVal(prefp));
      break;
    case PREF_BOOLEAN:
      if (PREFERENCE_BoolVal(prefp) == FALSE)
	fputs("false", fp);
      else if (PREFERENCE_BoolVal(prefp) == TRUE)
	fputs("true", fp);
      else {
	FatalError(E_BADPREFVALUE,
		   "Unknown value %d for boolean preference \n",
		   PREFERENCE_BoolVal(prefp));
      }
      break;
    case PREF_STRING:
      if (PREFERENCE_StrVal(prefp) == NULL) {
	FatalError(E_BADPREFVALUE,
		   "NULL value for string preference \n");
      } else {
	String_Write(fp, PREFERENCE_StrVal(prefp),
		     PREF_QUOTECHAR, PREF_STRINGARRAYDELIM,
		     PreferenceSpecialChars, NPreferenceSpecialChars);
      }
      break;
    case PREF_STRINGARRAY:
      if (PREFERENCE_StrArrVal(prefp) == NULL) {
	FatalError(E_BADPREFVALUE,
		   "NULL value for string array preference\n");
      } else {
	StringArray_Write(fp, PREFERENCE_StrArrVal(prefp),
			  PREF_QUOTECHAR, PREF_STRINGARRAYDELIM,
			  PreferenceSpecialChars,
			  NPreferenceSpecialChars);
      }
      break;
    case PREF_STRINGSET:
      if (PREFERENCE_StrSetVal(prefp) == NULL) {
	FatalError(E_BADPREFVALUE,
		   "NULL value for string set preference\n");
      } else {
	StringArray_Write(fp, PREFERENCE_StrArrVal(prefp),
			  PREF_QUOTECHAR, PREF_STRINGARRAYDELIM,
			  PreferenceSpecialChars,
			  NPreferenceSpecialChars);
      }
      break;
    case PREF_SENTENCE:
      if (PREFERENCE_SentenceVal(prefp) == NULL) {
	FatalError(E_BADPREFVALUE,
		   "NULL value for sentence preference\n");
      } else {
	StringArray_Write(fp, PREFERENCE_StrArrVal(prefp),
			  PREF_QUOTECHAR, PREF_SENTENCEDELIM,
			  PreferenceSpecialChars,
			  NPreferenceSpecialChars);
      }
      break;
    default:
      FatalError(E_BADPREFERENCE,
		 "Attempt to write unknown preference type\n");
      break;
    }
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    fputc('\n', fp);
  }
  return (PROGRAM_ErrorNo == E_NULL) ? 0 : -1;
}

/* $Source: /afs/andrew/system/src/local/depot2/013/src/lib/Preference/RCS/Preference_Write.c,v $ */
