/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[]="$Id: Preference_Lookup.c,v 1.11 1994/09/30 20:48:26 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "Preference.h"


static PREFERENCE *Preference_Create();
static PREFERENCE ***PreferenceDB_HashBucket();


/*
 * PREFERENCE *Preference_Lookup(prefdbp, item, prefname, Create)
 *	PREFERENCEDB *prefdbp;
 *	char *item, *name;
 *	Boolean Create;
 */
PREFERENCE *Preference_Lookup(prefdbp, item, name, Create)
     PREFERENCEDB *prefdbp;
     char *item, *name;
     Boolean Create;
{
  register int i;
  register PREFERENCE **fp, **tp, **pp;
  Boolean FoundPreference, FoundPreferenceLocation;
  int Comparison;
  int ArraySize;

  PREFERENCE *prefentry;
  PREFERENCE ***hashp;
  int PrefIndex;

  /* sanity checks */
  if (prefdbp == NULL)
    {
      FatalError(E_BADPREFERENCEDB,
		 "Attempt to search NULL preference database\n");
    }
  if (name == NULL)
    {
      FatalError(E_BADPREFERENCE, "Attempt to lookup NULL preference\n");
    }
  else if (Preference_Type(name) & PREF_UNKNOWN)
    {
      FatalError(E_BADPREFERENCE,
		 "Attempt to lookup unknown preference type %s\n", name);
    }

  /*
   * if database hasn't been initialized and Create is set,
   * set up hash table etc.
   */
  if ((PROGRAM_ErrorNo == E_NULL)
      && (*prefdbp == NULL) && Create)
    *prefdbp = PreferenceDB_InitializeSetup();

  prefentry = NULL; PrefIndex = 0;
  FoundPreference = FALSE; FoundPreferenceLocation = FALSE;
  /* lookup preference */
  if ((PROGRAM_ErrorNo == E_NULL) && (*prefdbp != NULL))
    {
      /* get hash entry */
      hashp = PreferenceDB_HashBucket(*prefdbp, item, name);
      if (hashp == NULL) {
	FatalError(E_BADPREFERENCEDB,
		   "Unable to locate hash entry for item %s, preference %s\n",
		   item, name);
      }
      /* search from hash entry for index of preference being looked up */
      if ((PROGRAM_ErrorNo == E_NULL)
	  && (*hashp != NULL))
	{
	  pp = *hashp;
	  while ( !FoundPreference && !FoundPreferenceLocation
		 && (*pp != NULL))
	    {
	      Comparison = Preference_Comparator(*pp, item, name);
	      if (Comparison > 0)
		FoundPreferenceLocation = TRUE;
	      else if (Comparison == 0)
		FoundPreference = TRUE;
	      else
		PrefIndex++, pp++;
	    }
	}
    }

  /*
   * if (!FoundPreference && Create)
   *   create item and insert into database;
   */
  if ((PROGRAM_ErrorNo == E_NULL)
      && (!FoundPreference && Create))
    {
      if (*hashp == NULL)
	{
	  *hashp = (PREFERENCE **)emalloc(sizeof(PREFERENCE *));
	  if (PROGRAM_ErrorNo == E_NULL) **hashp = NULL;
	}
      ArraySize = PreferenceArray_Size(*hashp) + 1;
      if (!FoundPreference && !FoundPreferenceLocation)
	{ PrefIndex = ArraySize - 1; }
      ArraySize++;
      *hashp = (PREFERENCE **)erealloc((void *)*hashp,
				       ArraySize*sizeof(PREFERENCE *));
      if (PROGRAM_ErrorNo == E_NULL)
	{
	  i = ArraySize - 1; fp = *hashp + i - 1; tp = *hashp + i;
	  while (i > PrefIndex)
	    { *tp-- = *fp-- ; i--; }
	  pp = *hashp + PrefIndex;
	  *pp = Preference_Create(item, name);
	  FoundPreference = TRUE;
	}
    }

  if ((PROGRAM_ErrorNo == E_NULL) && FoundPreference)
    prefentry = *(*hashp + PrefIndex);

  return (PROGRAM_ErrorNo == E_NULL) ? prefentry : NULL;
}


PREFERENCEDB PreferenceDB_InitializeSetup()
{
  PREFERENCEDB prefdb;

  prefdb = (PREFERENCEDB)ecalloc(Preference_HashTableSize,
				 sizeof(PREFERENCE **));

  return prefdb;
}


static PREFERENCE ***PreferenceDB_HashBucket(prefdb, item, name)
     PREFERENCEDB prefdb;
     char *item, *name;
{
  register PREFERENCE ***hashp;
  register char *cp;
  register unsigned hashval;

  hashval = 0;
  if (item != NULL)
    {
      for (cp = item; *cp != '\0'; cp++)
	{ hashval += (hashval << 6) + *cp - 'a' ; }
    }
  if (name != NULL)
    {
      for (cp = name; *cp != '\0'; cp++)
	{ hashval += (hashval << 6) + *cp - 'a' ; }
    }
  hashval = hashval % Preference_HashTableSize;
  hashp = prefdb + hashval;

  return hashp;
}


static PREFERENCE *Preference_Create(item, name)
     char *item, *name;
{
  PREFERENCE *prefp;

  prefp = (PREFERENCE *)emalloc(sizeof(PREFERENCE));
  if (PROGRAM_ErrorNo == E_NULL)
    {
      PREFERENCE_OptionID(prefp) = Preference_OptionID(name);
      if (PREFERENCE_OptionID(prefp) < 0)
	{
	  FatalError(E_UNKNOWNPREFERENCE,
		     "Unknown preference option %s\n", name);
	}
      else
	{
	  PREFERENCE_Operand(prefp) =
	    (item == NULL) ? NULL : String(item, strlen(item));
	  PREFERENCE_Type(prefp) = Preference_Type(name);
	  switch (PREFERENCE_Type(prefp) &
		  ~(PREF_INTERNAL|PREF_REGULAR|PREF_GLOBAL|PREF_COMMAND|PREF_FILTER))
	    {
	    case PREF_INTEGER:
	      PREFERENCE_IVal(prefp)
 		= (int) Preference_DefaultValue(PREF_INTEGER);
	      break;
	    case PREF_UNSIGNED:
	      PREFERENCE_UVal(prefp)
		= (unsigned) Preference_DefaultValue(PREF_UNSIGNED);
	      break;
	    case PREF_BOOLEAN:
	      PREFERENCE_BoolVal(prefp)
		= (Boolean) Preference_DefaultValue(PREF_BOOLEAN);
	      break;
	    case PREF_STRING:
	      PREFERENCE_StrVal(prefp)
		= (char *) Preference_DefaultValue(PREF_STRING);
	      break;
	    case PREF_STRINGARRAY:
	      PREFERENCE_StrArrVal(prefp)
		= (STRINGARRAY *) Preference_DefaultValue(PREF_STRINGARRAY);
	      break;
	    case PREF_STRINGSET:
	      PREFERENCE_StrSetVal(prefp)
		= (STRINGSET *) Preference_DefaultValue(PREF_STRINGSET);
	      break;
	    case PREF_SENTENCE:
	      PREFERENCE_SentenceVal(prefp)
		= (SENTENCE *)  Preference_DefaultValue(PREF_SENTENCE);
	      break;
	    default:
	      FatalError(E_UNKNOWNPREFERENCE,
			 "Unknown preference type %s\n", name);
	      break;
	    }
	}
    }

  return (PROGRAM_ErrorNo == E_NULL) ? prefp : NULL;
}

/* $Source: /afs/andrew/system/src/local/depot2/013/src/lib/Preference/RCS/Preference_Lookup.c,v $ */
