/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: Depot_CollectionUtil.c,v 1.13 1994/10/12 20:56:06 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */


#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "Lock.h"
#include "File.h"
#include "Hint.h"
#include "FileSystemImage.h"
#include "Preference.h"
#include "DepotConf.h"
#include "TargetDB.h"
#include "Collection.h"
#include "CollectionList.h"
#include "DepotDB.h"
#include "HintService.h"
#include "Command.h"

#include "config.h"
#include "DepotUtil.h"


static char *DEPOT_DEFAULTSEARCHPATH[] = { "depot",0 };

static STRINGARRAY DEPOT_DEFAULTSEARCHPATHLIST =
{
  1,	/* size */
  DEPOT_DEFAULTSEARCHPATH	/* values */
};

static char * DEPOTCONF_ROOTMAPPINGVALUE[] =
{
  "/", "/", NULL
};

static STRINGARRAY DEPOTCONF_ROOTMAPPINGARRAY =
{
  2, DEPOTCONF_ROOTMAPPINGVALUE
};

static DEPOTCONF DEPOTCONF_ROOTMAPPING =
{
  "*", DC_MAP, &DEPOTCONF_ROOTMAPPINGARRAY
};

static DEPOTCONF *DEPOT_DEFAULTDEPOTCONFLIST[] =
{
  &DEPOTCONF_ROOTMAPPING,
  NULL
};

static DEPOTCONFDB DEPOT_DEFAULTDEPOTCONFDB = DEPOT_DEFAULTDEPOTCONFLIST;


static char *Depot_LocateCollectionPathInHints();


STRINGSET *
Depot_GetCollectionNamesInDirectory(path, versiondelimiter, hintdb)
     char *path;
     char versiondelimiter;
     HINTDB hintdb;
{
  register char **ssp;
  STRINGSET *filenamelist = NULL;
  STRINGSET *collectionnamelist = NULL;

  if (File_ListDirectory(path, &filenamelist) < 0) {
    return NULL;
  }

  if (StringSet_Empty(filenamelist)) {
    return NULL;
  }
  
  for (ssp = STRINGSET_Values(filenamelist); *ssp != NULL; ssp++) {
    char *cp;
    char colname[MAXPATHLEN];
    char colpath[MAXPATHLEN];
    FILESTAT filestatusbuffer;

    (void)strcpy(colname, *ssp);

    if ((cp = strrchr(colname, versiondelimiter)) != NULL) {
      *cp = '\0';
    }

    if (!StringSet_Member(collectionnamelist, colname)) {
      (void) strcpy(colpath, path);
      (void) strcat(colpath, "/");
      (void) strcat(colpath, *ssp);
      if ((hintdb == NULL) || (HintDB_GetHint(hintdb, colpath) == NULL)) {
	if (PROGRAM_ErrorNo != E_NULL)
	  return NULL;
	
	if (File_GetStatus(colpath, &filestatusbuffer, TRUE /* followlinks */ ) < 0) {
	  return NULL;
	}

	if (FILESTAT_Type(&filestatusbuffer) & F_DIR) {
	  collectionnamelist = StringSet_AddString(collectionnamelist,
						   colname);
	}
      } else {
	collectionnamelist = StringSet_AddString(collectionnamelist,
						   colname);
      }
    }
  }
  if (filenamelist != NULL) {
    StringArray_Free(filenamelist);
  }
  return (PROGRAM_ErrorNo == E_NULL) ? collectionnamelist : NULL;
}

char *
Depot_GetCollectionPath(collectionname,
			preferencedbp,
			versiondelimiter,
			hintdb)
     char *collectionname;
     PREFERENCEDB *preferencedbp;
     char versiondelimiter;
     HINTDB hintdb;
{
  int version;
  STRINGARRAY *collectionsearchpathlist = NULL;
  char *collectionpath = NULL;

  /* check for path preference */
  collectionpath = Preference_GetString(preferencedbp,
					collectionname,
					"path",
					NULL /* default */ );

  /* if no path preference, use searchpath preference */
  if ((PROGRAM_ErrorNo == E_NULL) && (collectionpath == NULL)) {
    version = Preference_GetInt(preferencedbp,
				collectionname,
				"version",
				-1 /* default */ );
    collectionsearchpathlist =
      Preference_GetStringArray(preferencedbp,
				collectionname,
				"searchpath",
				&DEPOT_DEFAULTSEARCHPATHLIST /* default */,
				0 /* don't allocate */);
	    if ((PROGRAM_ErrorNo == E_NULL)
	&& !StringArray_Empty(collectionsearchpathlist)) {
      /* check for collection on hint cache */
      collectionpath =
	Depot_LocateCollectionPathInHints(collectionname,
					  hintdb,
					  collectionsearchpathlist,
					  versiondelimiter,
					  version);
      /* if no hints for collection in hintdb, search searchpath */
      if ((PROGRAM_ErrorNo == E_NULL) && (collectionpath == NULL)) {
	collectionpath
	  = Collection_LocatePath(collectionname,
			       STRINGARRAY_Values(collectionsearchpathlist),
				  versiondelimiter,
				  version);
      }
    }
  }


  return (PROGRAM_ErrorNo == E_NULL) ? collectionpath : NULL;
}


static char *
Depot_LocateCollectionPathInHints(collectionname,
				  hintdb,
				  searchpathlist,
				  versiondelimiter,
				  version)
     char *collectionname;
     HINTDB hintdb;
     STRINGARRAY *searchpathlist;
     char versiondelimiter;
     int version;
{
  register unsigned i;
  register HINTDB hp;

  Boolean LocatedPath;
  char trialpath[MAXPATHLEN];
  char trialhintpathprefix[MAXPATHLEN];
  char *collectionpath = NULL;
  HINT *hintp;
  HINTDB relatedhintdb;
  char *hintpath = NULL;
  int trialhintpathprefixlen;
  int maxversion, thisversion;
  char *collectionlocation;

  if (version >= 0) {
    i = 0;
    LocatedPath = FALSE;
    while (!LocatedPath && (i < STRINGARRAY_Size(searchpathlist))
	   && (PROGRAM_ErrorNo == E_NULL)) {
      (void) sprintf(trialpath, "%s/%s%c%d",
		     STRINGARRAY_String(searchpathlist, i),
		     collectionname, versiondelimiter, version);
      hintp = HintDB_GetHint(hintdb, trialpath);
      if ((PROGRAM_ErrorNo == E_NULL)
	  && (hintp != NULL)) {
	LocatedPath = TRUE;
	collectionpath = String(trialpath, strlen(trialpath));
      }
    }
  } else {			/* version < 0 */
    i = 0;
    LocatedPath = FALSE;
    while (!LocatedPath && (i < STRINGARRAY_Size(searchpathlist))
	   && (PROGRAM_ErrorNo == E_NULL)) {
      /* look for it in hints for this directory */

      (void) sprintf(trialhintpathprefix, "%s/%s",
		     STRINGARRAY_String(searchpathlist, i), collectionname);
      relatedhintdb = HintDB_GetRelatedHints(hintdb, trialhintpathprefix);

      /* go through the relatedhintdb */
      if ((PROGRAM_ErrorNo == E_NULL) && (relatedhintdb != NULL)) {
	maxversion = -1;
	trialhintpathprefixlen = strlen(trialhintpathprefix);
	for (hp = relatedhintdb;
	     (PROGRAM_ErrorNo == E_NULL) && (*hp != NULL);
	     hp++) {
	  if (*hp != NULL) {
	    /* prefixed by trialhintpathprefix */
	    hintpath = HINT_PathName(*hp); /* don't write to hintpath! */
	    if (PROGRAM_ErrorNo == E_NULL) {
	      if ((*(hintpath + trialhintpathprefixlen) == '\0')
		  && (maxversion < 0)) {
		LocatedPath = TRUE;
		collectionpath = String(trialhintpathprefix, strlen(trialhintpathprefix));
	      } else if ((*(hintpath + trialhintpathprefixlen)
			  == versiondelimiter)
			 && (strchr(hintpath + trialhintpathprefixlen,
				    '/') == NULL)) {
		thisversion = atoi(hintpath +
				   trialhintpathprefixlen + 1);
		if (thisversion > maxversion) {
		  LocatedPath = TRUE;
		  maxversion = thisversion;
		  if (collectionpath != NULL)
		    free(collectionpath);
		  collectionpath = String(hintpath, strlen(hintpath));
		}
	      }
	    }
	  }
	}			/* for loop through list of hints for directory */
	HintDB_Free(relatedhintdb);
      }
      i++;
    }
  }

  if (PROGRAM_ErrorNo == E_NULL) {
    if (LocatedPath)
      collectionlocation = collectionpath;
    else
      collectionlocation = NULL;
  }
  return (PROGRAM_ErrorNo == E_NULL) ? collectionlocation : NULL;
}


DEPOTCONFDB *
Depot_GetCollectionDepotConfDB(collectionname, collectionpath)
     char *collectionname, *collectionpath;
{
  char depotconffilename[MAXPATHLEN];
  FILE *fpread;

  DEPOTCONFDB *coll_confdb = NULL;

  (void) sprintf(depotconffilename, "%s/%s", collectionpath, DEPOTCONFIGFILE);
  if ((fpread = fopen(depotconffilename, "r")) == NULL) {
    if (errno == ENOENT) { /* file doesn\'t exist */
      coll_confdb = DepotConfDB(&DEPOT_DEFAULTDEPOTCONFDB);
      return coll_confdb;
    }
    FatalError(E_FOPENFAILED, "Unable to read: %s\n", depotconffilename);
    return NULL;
  }
  coll_confdb =
    CollectionDepotConfDB_Read(fpread, collectionname);
  (void) fclose(fpread);
  return (PROGRAM_ErrorNo == E_NULL) ? coll_confdb : NULL;
}

/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/022/src/lib/Depot/RCS/Depot_CollectionUtil.c,v $ */
