#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include <ctype.h>

#include "chess.h"
#include "misc.h"
#include "checks.h"
#include "eval.h"
#include "comp.h"
#include "mprocs.h"
#include "moves.h"

#define ROLLOVER_CORRECTION   (double)((1073741824.0/(double)CLOCKS_PER_SEC)*4.0)
#define CMSCORE               (50000000)

extern char gamepath[FILENAME_MAX],path[FILENAME_MAX];
extern char dir_command[10],sep[3];
extern long int rtable[8][8][13],_nhash,_nanaly;
extern double time_spent;
extern int initscore,rollover,prev_rollover,rtable2[8][8][13];
extern hash *hashbank;
extern compdat cdd[MAX_PV];
extern move mvlst[1024],pv[MAX_PV][MAX_PV];
extern int xboardmode,pvlength[MAX_PV];
int depth_searched,pcolor=1,g_side=0,g_mvno=0;

/*  Save a game to .chs format */
void savegame(int sq[8][8],struct gamedat *game,int cp[4]) {
  int a,b;
  char nam[51],nm[FILENAME_MAX],cd_path[FILENAME_MAX];
  FILE *fp;

  if (!game) {fprintf(stderr,"Game Data not Initialised Properly in savegame()!\n");return;}
  strcpy(cd_path,dir_command);
  strcat(cd_path," ");
  strcat(cd_path,gamepath);
  strcat(cd_path,"*.chs");
  printf("\nDirectory of *.chs Files\n\n");
  (void)system(cd_path);
  printf("\nPlease Input Name:-");
  do {
    (void)scanf("%s",nam);
    if (!strstr(nam,".chs")) strcat(nam,".chs");
    if (strlen(nam)>30 || strlen(nam)<5) printf("   Illegal Name!");
  } while (strlen(nam)>30 || strlen(nam)<5);
  strcpy(nm,gamepath);
  strcat(nm,nam);
  if ((fp = fopen(nm,"w"))==NULL)  {
    printf("\n!!! Cannot Open File !!!\n");
    (void)system("cd ..");
    return;
  }
  fprintf(fp,"#COLCHESS SAVEGAME#\n");
  fprintf(fp,"%d",game->proper);
  fprintf(fp,"\n%d",game->mvno);
  fprintf(fp,"\n%d",game->side);
  fprintf(fp,"\n%d",game->skillw);
  fprintf(fp,"\n%d",game->skillb);
  fprintf(fp,"\n%d",game->depth);
  fprintf(fp,"\n%d",game->cdm);
  fprintf(fp,"\n%d",game->death);
  for (a=0;a<4;a++) fprintf(fp,"\n%d",cp[a]);
  fprintf(fp,"\n%d",game->quiesce);
  for (a=0;a<8;a++) {
    fprintf(fp,"\n");
    for (b=0;b<8;b++) {
      fprintf(fp,"%d",sq[a][b]);
      if (b<7) fprintf(fp," ");
    }
  }

  b=((game->mvno)*2)-1;
  if (game->side==WHITE) b--;
  for (a=0;a<b;a++) {
    fprintf(fp,"\n%d,%d %d,%d [%d][%d][%d][%d]",(mvlst+a)->fx,(mvlst+a)->fy,(mvlst+a)->tx,(mvlst+a)->ty,(mvlst+a)->castle,(mvlst+a)->ep,(mvlst+a)->promote,(mvlst+a)->capt);
  }
  (void)fclose(fp);
  printf("\nFile Saved O.K.\n");
}

/*  Load a game in .chs format */
struct gamedat *loadgame(void) {
  int a,b,n;
  char nam[51],nm[FILENAME_MAX],cd_path[FILENAME_MAX],ch;
  FILE *fp;
  struct gamedat *game;
  move *m;

  strcpy(cd_path,dir_command);
  strcat(cd_path," ");
  strcat(cd_path,gamepath);
  strcat(cd_path,"*.chs");
  printf("\nDirectory of *.chs Files\n\n");
  (void)system(cd_path);
  printf("\nPlease Input Name:-");
  do {
	(void)scanf("%s",nam);
	if (!strstr(nam,".chs")) strcat(nam,".chs");
	if (strlen(nam)>30 || strlen(nam)<5) printf("   Illegal Name!");
  } while (strlen(nam)>30 || strlen(nam)<5);
  strcpy(nm,gamepath);
  strcat(nm,nam);
  if ((fp = fopen(nm,"r"))==NULL)  {
    printf("\n!!! Cannot Open File !!!\n");
    (void)system("cd ..");
    return NULL;
  }
  game=malloc(sizeof(struct gamedat));
  assert(game!=NULL);

  (void)fscanf(fp,"%c",&ch);
  if (ch!='#') {
    game->proper=0;
    (void)fclose(fp);
    fprintf(stderr,"SAVEGAME not in ColChess v5.2 or greater format!\n");
    return NULL;
  }
  else (void)fscanf(fp,"COLCHESS SAVEGAME#\n%d\n",&(game->proper));
  (void)fscanf(fp,"%d\n",&(game->mvno));
  (void)fscanf(fp,"%d",&(game->side));
  (void)fscanf(fp,"\n%d",&(game->skillw));
  (void)fscanf(fp,"\n%d",&(game->skillb));
  (void)fscanf(fp,"\n%d",&(game->depth));
  (void)fscanf(fp,"\n%d",&(game->cdm));
  (void)fscanf(fp,"\n%d",&(game->death));

  n=((game->mvno)*2)-1;
  if (game->side==WHITE) n--;

  for (a=0;a<4;a++) (void)fscanf(fp,"\n%d",&(game->cp[a]));
  (void)fscanf(fp,"\n%d",&(game->quiesce));
  for (a=0;a<8;a++) {
	(void)fscanf(fp,"\n");
	for (b=0;b<8;b++) {
	  (void)fscanf(fp,"%d",&(game->sq[a][b]));
          if (b<7) (void)fscanf(fp," ");
	}
  }
  for (a=0;a<n;a++) {
    m=(mvlst+a);
    (void)fscanf(fp,"\n%d,%d %d,%d [%d][%d][%d][%d]",&(m->fx),&(m->fy),&(m->tx),&(m->ty),&(m->castle),&(m->ep),&(m->promote),&(m->capt));
  }
  (void)fclose(fp);
  printf("\nFile Loaded O.K.\n");
  return game;
}

/*  Export a game to .pgn */
void export_game(int mvno,int side) {
  int a,b,n;
  char nam[51],nm[FILENAME_MAX],cd_path[FILENAME_MAX];
  FILE *fp;
   
  strcpy(cd_path,dir_command);
  strcat(cd_path," ");
  strcat(cd_path,gamepath);
  strcat(cd_path,"*.pgn");
  printf("\nDirectory of *.pgn files\n");
  (void)system(cd_path);
  printf("\nPlease Input Name :-");
  do {
    (void)scanf("%s",nam);
    if (!strstr(nam,".pgn")) strcat(nam,".pgn");
    if (strlen(nam)<5 || strlen(nam)>30) printf("   Illegal Name!\n");
  } while (strlen(nam)<5 || strlen(nam)>30);
  strcpy(nm,gamepath);
  strcat(nm,nam);
  if ((fp = fopen(nm,"w"))==NULL) {
    printf("\n!! CANNOT OPEN FILE !!\n");
    return;
  }
  b=(mvno*2)-1;
  if (side==WHITE) b--;
  for (a=0;a<mvno;a++) {
    n=a*2;
    if (a%5==4) fprintf(fp,"\n");
    if (n<b) {
      fprintf(fp,"%d. %c%d-%c%d ",a+1,(char)((mvlst+n)->fx+97),8-((mvlst+n)->fy),(char)((mvlst+n)->tx+97),8-((mvlst+n)->ty));
      n++;
      if (n<b) fprintf(fp,"%c%d-%c%d ",(char)((mvlst+n)->fx+97),8-((mvlst+n)->fy),(char)((mvlst+n)->tx+97),8-((mvlst+n)->ty));
    }
  }
  fprintf(fp,"\n*");
  (void)fclose(fp);
  printf("\nFile Exported O.K.!\n");
}

 /*  Hash a board position.  Store it in 'h' (previously allocated) */
void make_hash(hash *h,board *sq,int cp[4],int side) {
  int n,i;

  h->check=h->check2=0;
  h->vars=(short)(cp[0]+(cp[1]<<1)+(cp[2]<<2)+(cp[3]<<3)+((side-1)<<4));
  for (n=0;n<8;n++) {
    for (i=0;i<8;i++) if (sq->sq[n][i]!=0) {
      h->check+=rtable[n][i][(sq->sq[n][i])+6];
      h->check2+=rtable2[n][i][(sq->sq[n][i])+6];
    }
  }
}

 /*  Add hash 'h' to hash table 'table', with best move 'mv' */
void add_hash(hash_elt *table, move *mv, long int hcheck, int hvars, int hcheck2, int lev, long int score, int cm, int btype) {
  hash *hh;
   
  hh=(hashbank+_nhash);
  hh->check=hcheck;hh->check2=hcheck2;
  hh->vars=(short)hvars;hh->cm=(short)cm;
  hh->depth=(short)lev;hh->score=score;
  if (btype==-1) hh->type=lower_bound;
  else if (btype==0) hh->type=exact;
  else hh->type=upper_bound;
   
   /*  If previous best move is given then include it in the hash entry */
  if (mv && mv->fx>=0) {
    hh->move=mv->fx+(mv->fy<<3)+(mv->tx<<6)+(mv->ty<<9);
    hh->mvars=(short)(mv->ep+1+(abs(mv->promote)<<2)+(mv->castle<<5));
  }
  else hh->move=hh->mvars=0;
   
  hh->next=table->h;
  table->h=(hashbank+_nhash);
}

 /*  See if an entry exists in 'table' with checksums ch and ch2, and variables vars */
 /*  If so, then return the hashed entry */
hash *find_hash(hash_elt *table,long int ch,int vars,int ch2) {
  hash *tb=table->h;

  while (tb!=NULL) {
    if (tb->check==ch && tb->vars==(short)vars && tb->check2==ch2) return tb;
    tb=tb->next; /*  Get next elt. */
  }
  return NULL;
}

 /*  Alter the given hash entry to update it with some better results. */
void alter_hash(hash *h,long int score,int cm,move *mv,int abcutoff,int lev) {
  h->score=score;h->cm=(short)cm;
  h->depth=(short)lev;
  if (abcutoff) {
    if (abcutoff==1) h->type=upper_bound;
    else h->type=lower_bound;
  }
  else h->type=exact;
  if (mv) {
    h->move=mv->fx+(mv->fy<<3)+(mv->tx<<6)+(mv->ty<<9);
    h->mvars=(short)(mv->ep+1+(abs(mv->promote)<<2)+(mv->castle<<5));
  }
}

 /*  Print off prettty dots to show computer progress */
void progress(int beg,int a,int nm) {
  int b;
  if (beg==2 && (a&1)==0) fprintf(stderr,"\b.");
  if (beg==2 && (a&1)==1) fprintf(stderr,"\b\b:");
  if (beg==1) {
    b=(int)((a*10)/nm);
    if (b>9) b=9;
    if (a>0) fprintf(stderr,"\b");
    fprintf(stderr,"%d",b);
  }
}

 /*  Counts the number of each type of piece on the board, and stores it */
 /*  in the array npieces[][] as part of the board structure */
void get_pieces(board *bd) {
  int a,b,p;
  
  for (b=0;b<4;b++) bd->npieces[0][b]=bd->npieces[1][b]=0;
  for (a=0;a<8;a++) {
    for (b=0;b<8;b++) {
      p=bd->sq[a][b];
      if (p>1 && p<6) (bd->npieces[0][p-2])++;
      else if (p<-1 && p>-6) (bd->npieces[1][-(p+2)])++;
    }
  }
}

 /*  Prints off the time a move took, given the number of seconds, 't' */
void print_time(double t) {
  int nmin,nhr;
  double tt=t;
   
  tt/=(double)10;
   
  if (tt<10.0) {
    fprintf(stderr,"%.2f Second",tt);
    if (tt!=1.0) fprintf(stderr,"s");
  }
  else if (tt<60.0) fprintf(stderr,"%.1f Seconds",tt);
  else {
    nmin=(int)tt/60;
    if (nmin>=60) {
      nhr=nmin/60;
      fprintf(stderr,"%d Hour",nhr);
      if (nhr>1) fprintf(stderr,"s");
      fprintf(stderr,", ");
      nmin=nmin%60;
    }
    fprintf(stderr,"%d Minute",nmin);
    if (nmin>1) fprintf(stderr,"s, ");
    else fprintf(stderr,", ");
    tt-=(double)(nmin*60);
    fprintf(stderr,"%.0f Second",tt);
    if ((int)tt!=1) fprintf(stderr,"s");
  }
}

 /*  Get CPU time so far between start and end */
 /*  Also use the time() function if time is getting too long */
double get_time(clock_t start) {
  double rate;
  clock_t end;

  end=clock();
   /* We have had a rollover due to some idiot presuming that a
    * long int was sufficient to store clock ticks for any process. */
  if (prev_rollover==0 && (double)end<0.0) {rollover++;prev_rollover=1;}
  if ((double)end>0.0) prev_rollover=0;
   /*  Time thought so far
    *   Yes we have to do it this way (two divides) to avoid more rollovers! */
  rate = (double)end/(double)CLOCKS_PER_SEC;
  rate -= (double)start/(double)CLOCKS_PER_SEC;
  if (rollover>0) rate += (double)((double)ROLLOVER_CORRECTION * (double)rollover);
  return (rate * 10.0);
}

 /*  Get a number in int format from a char string, starting */
 /*  at character location 'beg' */
int get_num_from_string(char *st,int beg) {
  int n,val=0;
  for (n=beg;n<(int)strlen(st);n++) {
    val*=10;
    val+=(int)st[n]-48;
  }
  return val;
}

 /*  Print off the board in a concise ascii format. */
void print_board(const board *sq) { 
  int a,b,c;

  fprintf(stderr,"\nCurrent Position\n----------------\n");
  for (a=0;a<8;a++) {
	for (b=0;b<8;b++) {
	  c=sq->sq[a][b];
	  if (c==1) fprintf(stderr,"P ");
	  if (c==2) fprintf(stderr,"R ");
	  if (c==3) fprintf(stderr,"N ");
	  if (c==4) fprintf(stderr,"B ");
	  if (c==5) fprintf(stderr,"Q ");
	  if (c==6) fprintf(stderr,"K ");
	  if (c==-1) fprintf(stderr,"p ");
	  if (c==-2) fprintf(stderr,"r ");
	  if (c==-3) fprintf(stderr,"n ");
	  if (c==-4) fprintf(stderr,"b ");
	  if (c==-5) fprintf(stderr,"q ");
	  if (c==-6) fprintf(stderr,"k ");
	  if (c==0) fprintf(stderr,". ");
	}
	if (a==1) fprintf(stderr,"     pP=Pawn");
	if (a==2) fprintf(stderr,"     rR=Rook");
	if (a==3) fprintf(stderr,"     nN=Knight     lower=Black");
	if (a==4) fprintf(stderr,"     bB=Bishop      CAPS=White");
	if (a==5) fprintf(stderr,"     qQ=Queen");
	if (a==6) fprintf(stderr,"     kK=King");
	fprintf(stderr,"\n");
  }
}

 /*  Help on Computer player options */
void help_comp(void) {
  fprintf(stderr,"\n    Computer Player Options");
  fprintf(stderr,"\n    -----------------------\n");
  fprintf(stderr,"\nComp     - Force Computer Player to Move");
  fprintf(stderr,"\nSkill    - Set Computer Skill for Current Side (1-12)");
  fprintf(stderr,"\nDepth    - Set the MINIMUM Computer Search Depth");
  fprintf(stderr,"\nQui      - Set Minimum Quiescence Depth (0-12)");
  fprintf(stderr,"\nHash     - Set Hash Table Size (0-16)");
  fprintf(stderr,"\nDeath    - Toggle Fight to the Death ON/OFF (Computer never resigns)");
  fprintf(stderr,"\nCmove    - Toggle Auto Computer Move ON/OFF");
  fprintf(stderr,"\nAuto     - Set up Computer Automoves");
  fprintf(stderr,"\nLimit    - Set Computer Time Limit. 0 = Unlimited");
  fprintf(stderr,"\nWhite    - Computer Plays White");
  fprintf(stderr,"\nBlack    - Computer Plays Black");
  fprintf(stderr,"\nCoff     - Computer Does NOT Play by Default");
  fprintf(stderr,"\n             Use 'Comp' to Force Computer to Move");
  fprintf(stderr,"\nBoth     - Computer Plays BOTH Sides");
  fprintf(stderr,"\nBook     - Toggle Opening Book ON/OFF");
  fprintf(stderr,"\nLearn    - Toggle Opening Book Learning ON/OFF");
  fprintf(stderr,"\n");
}

 /*  Help on analysis options */
void help_analy(void) {
  fprintf(stderr,"\n    Analysis Options");
  fprintf(stderr,"\n    ----------------\n");
  fprintf(stderr,"\nAnaly      - Give Static Position Analysis (Shortcut 'A')");
  fprintf(stderr,"\nDanger     - Display Danger Squares (Shortcut 'D')");
  fprintf(stderr,"\nTest       - Load & Run an EPD Test");
  fprintf(stderr,"\nEPDanaly   - Analyse an EPD file");
  fprintf(stderr,"\nAnnotate   - Annotate a PGN file");
  fprintf(stderr,"\nStudy      - Study a PGN file and learn new positions");
  fprintf(stderr,"\n               (Learns the first new game position only)");
  fprintf(stderr,"\nSelftest   - Perform Self test on current Opening Book");
  fprintf(stderr,"\n");
}

 /*  Print off help list */
void help(void) {
  fprintf(stderr,"\n    Command List");
  fprintf(stderr,"\n    ------------\n");
  fprintf(stderr,"\nE2-E4/E2E4 - Move a Piece (i.e. from E2 to E4)");
  fprintf(stderr,"\n               Neither method is case sensitive.");
  fprintf(stderr,"\n0-0/0-0-0  - Castle King's Side/Queen's Side");
  fprintf(stderr,"\nEP         - Capture a Pawn 'En-Passant' (can also use co-ordinate form)");
  fprintf(stderr,"\nBoard      - Display the Board (Shortcut 'B')");
  fprintf(stderr,"\nComphelp   - Help on how to use the Computer Player");
  fprintf(stderr,"\nAnalyhelp  - Help on how to use the Specialist Analysis Functions");
  fprintf(stderr,"\nMoves      - Display Move List");
  fprintf(stderr,"\nRetake     - Retake Last Move (Shortcut 'R')");
  fprintf(stderr,"\nGoto       - Goto a Previous Move");
  fprintf(stderr,"\nSet        - Set the Value of a Player's Clock");
  fprintf(stderr,"\nTimer      - Toggle the Timer ON/OFF");
  fprintf(stderr,"\nClock      - Check the Time Remaining");
  fprintf(stderr,"\nLoad       - Load a Previously Saved Game");
  fprintf(stderr,"\nSave       - Save Current Game");
  fprintf(stderr,"\nExport     - Export Current Game to .pgn Format");
  fprintf(stderr,"\nImport     - Import a Game From .pgn Format");
  fprintf(stderr,"\nResign     - Resign the Game");
  fprintf(stderr,"\nFen        - Input a new FEN board position");
  fprintf(stderr,"\nDFen       - Display Current FEN board position");
  fprintf(stderr,"\nReset      - Start a New Game");
  fprintf(stderr,"\nCredits    - Displays the Credits");
  fprintf(stderr,"\nAbout      - Info About This Edition");
  fprintf(stderr,"\nExit       - Exit the Program\n");
}

  /*  Print off game credits */
void credits(void) {
  fprintf(stderr,"\n     Credits");
  fprintf(stderr,"\n     -------");
  fprintf(stderr,"\n\nDesign & Programming : Colin Frayn");
  fprintf(stderr,"\n\n Special Thanks to Paul Kirby, ");
  fprintf(stderr,"without whose invaluable help");
  fprintf(stderr,"\n        ColChess would never have beaten him!\n");
  fprintf(stderr,"\nAnd I am also enormously grateful to Dann Corbit, ");
  fprintf(stderr,"who has given me many good tips,");
  fprintf(stderr,"\n   and who has converted and compiled ColChess countless times!\n");
}

 /*  Print off info. list about this version of ColChess :) */
void about(void) {
  fprintf(stderr,"\n      About ColChess '2000 V"VER);
  fprintf(stderr,"\n      -------------------------");
  fprintf(stderr,"\n\nRelease Date  -  21 Feb 2000");
  fprintf(stderr,"\n\nMajor Enhancements Since ColChess V5.6");
  fprintf(stderr,"\n     Opening Book Finetuned!");
  fprintf(stderr,"\n     Significant Speed Increases!");
  fprintf(stderr,"\n     Computer Timing Much Improved & Corrected!");
  fprintf(stderr,"\n     Draw Support Fixed!");
  fprintf(stderr,"\n     Specialised Endgame Functions!");
  fprintf(stderr,"\n\nNext Version Will Include;");
  fprintf(stderr,"\n     Unknown at This Stage!");
  fprintf(stderr,"\n");
}

 /*  Get the (par+1)'th element of a string of char strings (inp) */
 /*  Convert to an int, and return the value in long int format. */
long int getparam(char *inp,int par) {
  int n,flg=0;
  long int val=0;
   
  for (n=0;n<(int)strlen(inp);n++) {
    if (inp[n]==' ') {
      if (flg==par) return val;
      flg++;
    }
    else if (flg==par) {
      val*=10;
      val+=(int)inp[n]-48;
    }
  }
  return val;
}

 /*  Simple function to estimate how much of the remaining time to spend thinking */
int get_time_limit(int mvno, int left, int opp, int ypts, int opts) {
  int t,mvlft=55-mvno;

  if (mvlft<25) mvlft=25;
  if (opp>left) mvlft++;
  if (ypts+opts<24) mvlft--;
  if (ypts+opts<20) mvlft--;
  if (ypts+opts<16) mvlft--;
  if (ypts+opts<12) mvlft--;
  t=left/mvlft;
  return t;
}

 /*  Load an EPD test suite, and run ColChess on all the positions, */
 /*  keeping track of the results and displaying a summary. */
void perform_epd_test(int skillw,int skillb,int quiesced,int mem, int test) {
  int a,tlimit=5,correct=0,total=0,cm=0,side=1,ok=0,oside=0,count=0,apout=0,i;
  long int modscore;
  char nam[51],nm[FILENAME_MAX],cd_path[FILENAME_MAX];
  char answer[10][20],fen[100]="",ofen[100]="",sidech='w',castle[5]="",ep[3]="";
  char *s,string[512],outfile[FILENAME_MAX],temp[FILENAME_MAX];
  compdat *cd=&cdd[0];
  board *sqq;
  move *best;
  FILE *fp,*fout=NULL;

  strcpy(cd_path,dir_command);
  strcat(cd_path," ");
  strcat(cd_path,path);
  strcat(cd_path,"tests");
  strcat(cd_path,sep);
  strcat(cd_path,"*.epd");
  printf("\nDirectory of EPD Files\n\n");
  (void)system(cd_path);
  printf("\nPlease Input Name:-");
  do {
    (void)scanf("%s",nam);
    if (!strstr(nam,".epd")) strcat(nam,".epd");
    if (strlen(nam)>30 || strlen(nam)<5) printf("   Illegal Name!");
  } while (strlen(nam)>30 || strlen(nam)<5);
  strcpy(nm,path);
  strcat(nm,"tests");
  strcat(nm,sep);
  strcat(nm,nam);
  if ((fp = fopen(nm,"r"))==NULL)  {
    printf("\n!!! Cannot Open File !!!\n");
    return;
  }
  fprintf(stderr,"%s loaded OK!\n",nam);
  fprintf(stderr,"Set Time Limit");
  fprintf(stderr," [%d] :- ",tlimit);
  (void)scanf("%d",&a);
  if (a>43200) {
    a=43200;
    fprintf(stderr,"    (Set at 12 Hours)");
  }
  if (a<1) {
    a=0;
    fprintf(stderr,"    (Unlimited)");
  }
  i=0;
  memset(outfile, 0, sizeof(outfile));
  strncpy(outfile,nam,strlen(nam)-4);
  strcat(outfile,".out");
  fprintf(stderr,"Log to File %s [YES/NO]? ",outfile);
  (void)fscanf(stdin,"%s",temp);
  if (!strcmp(temp,"YES") || !strcmp(temp,"yes")) {
    fprintf(stderr,"Appending to output file!\n");
    apout=1;
    if ((fout = fopen(outfile,"w"))==NULL)  {
      printf("\n!!! Cannot Open File !!!\n");
      (void)fclose(fp);
      return;
    }
  }
  memset(answer, 0, sizeof(answer));
  tlimit=a;
  cd->quiesce=quiesced;
  for (;;) {
    ok=-1;
    for (a=0;a<10;a++) strcpy(answer[a],"-");
    s = fgets(string, (int)sizeof(string), fp);
    if (!s || strlen(s)<16) break;
    count = parseepd(string, fen, &sidech, castle, ep, answer, test);
    if (count==-1) {(void)fclose(fp);return;}
    if (sidech=='w') side=1;
    else side=2;

    total++; 
    strcpy(ofen,fen);
    oside=side;
    sqq=get_board_fen(fen);
    fprintf(stderr,"Test #%d : ",total);
    if (side==WHITE) {
      fprintf(stderr,"White to play\n");
      cd->skill=skillw;
    }
    else {
      fprintf(stderr,"Black to play\n");
      cd->skill=skillb;
    }
    cd->side=side;
    cd->gamestage=pts(1,sqq)+pts(2,sqq);
    for (a=0;a<4;a++) cd->cp[a]=0;
    if (strchr(castle,'K')) cd->cp[0]=1;
    if (strchr(castle,'Q')) cd->cp[1]=1;
    if (strchr(castle,'k')) cd->cp[2]=1;
    if (strchr(castle,'q')) cd->cp[3]=1;    
    cd->last=NULL;
    get_pieces(sqq);
    cd->tlimit=tlimit*10;
    cd->extend=0;
    cd->inchk=chck(sqq,side);
    get_poslist(cd,-1,sqq);
     
    print_board(sqq);
    fprintf(stderr,"Thinking  "); 
    
    best=comp(sqq,mem,1,1,1);     /*  ----  COMP PROCEDURE  ----  */
    
    if (test==1) {
      ok=-1;
      for (a=0;a<count;a++) {
        if (same_move(best,answer[a],sqq)) {
          ok=a;
  	  correct++;
          if (best->cm>0) cm++;
	 break;
        }
      }
      fprintf(stderr,"---Answer");
      if (strcmp(answer[1],"-")) fprintf(stderr,"s");
      fprintf(stderr,": ");
      for (a=0;a<10;a++) if (strcmp(answer[a],"-")) fprintf(stderr,"%s ",answer[a]);
      if (ok>=0) fprintf(stderr,"    **-Correct-**\n");
      else if (best->cm==1 && ok==-1) {fprintf(stderr,"**Different CM Found**\n");correct++;cm++;}
      else fprintf(stderr,"   ??-Incorrect-??\n");
      fprintf(stderr,"Correct So Far: %d[%d]/%d\n\n",correct,cm,total);
    }
    if (apout) {
       /* Print the FEN data to the logfile */
      fprintf(fout,"%s %c %s %s",fen,sidech,castle,ep);
       /* Print out the relevant data to the logfile */
      fprintf(fout," acd %d; acn %ld; acs ",depth_searched,_nanaly);
      if (time_spent<10.0) fprintf(fout,"%.2f; bm ",time_spent/10.0);
      else if (time_spent<100.0) fprintf(fout,"%.1f; bm ",time_spent/10.0);
      else fprintf(fout,"%.0f; bm ",time_spent/10.0);
       /* Print this move in a neat way */
      format_move(fout,best,abs(sqq->sq[best->fy][best->fx]));
       /* Centipawn evaluation is for total position, not just improvement from move */
      if (best->cm==1) modscore=32767-(CMSCORE-best->score);
      else if (best->cm==-1) modscore=-32767+best->score+CMSCORE;
      else {
	if (best->score>(THEORETICAL_WIN-500)) modscore=32000;
	else if (best->score<-(THEORETICAL_WIN-500)) modscore=-32000;
	else modscore=best->score+initscore;
      }
      fprintf(fout,"; ce %ld; pv",modscore);
       /* Print off the PV */
      for (a=0;a<pvlength[0];a++) {
	fprintf(fout," ");
	format_move(fout,&pv[0][a],-1);
      }
      if (pv[0][0].fx==-1) {
	fprintf(fout," ");
        format_move(fout,best,abs(sqq->sq[best->fy][best->fx]));	 
      }
      if (test==1) {
	 /* If this is a test then add a comment to show if the result was good */
	fprintf(fout,"; c0 \"");
        if (ok>=0) fprintf(fout,"**-Correct-**");
        else if (best->score>9000 && ok==-1) fprintf(fout,"**Longer CM Found**");
        else {
	  fprintf(fout,"??-Incorrect-?? (");
          for (a=0;a<10;a++) {
	    if (strcmp(answer[a],"-")) {
	      fprintf(fout,"%s",answer[a]);
	    }
	  }
	  fprintf(fout,")");
	}
	fprintf(fout,"\"");
      }
      fprintf(fout,";\n");
    }
    free(sqq);
    free(best);
  }
  if (test==1) {
    if (total>0) fprintf(stderr,"Result: %d[%d]/%d  (%d%%)[%d%%]\n\n",correct,cm,total,correct*100/total,cm*100/total);
  }
  (void)fclose(fp);
  if (apout) (void)fclose(fout);
}

int parseepd(char *string,char *fen,char *sidech,char *castle,char *ep,char answer[10][20],int test) {
  int count=0,a,i,l=0,len=(int)strlen(string),bm=0;
  char ch;
  
  i=l=0;
  do {
    ch=string[l];l++;
    if (ch!=' ') {fen[i]=ch;i++;}
  } while (ch!=' ' && l<len);
  fen[i]='\x0';
  
  if (l>=len) {fprintf(stderr,"Bad EPD before side to move.");return -1;}
  
  *sidech=string[l];l+=2;
     
  i=0;
  do {
    ch=string[l];l++;
    if (ch!=' ') {castle[i]=ch;i++;}
  } while (ch!=' ' && l<len);
  castle[i]='\x0';

  if (l>=len) {fprintf(stderr,"Bad EPD before E-P square.");return -1;}
   
  i=0;
  do {
    ch=string[l];l++;
    if (ch!=' ') {ep[i]=ch;i++;}
  } while (ch!=' ' && l<len);
  ep[i]='\x0';
   
  if (test==0) return 0;
  if (l>=len) {fprintf(stderr,"Bad EPD before best move.");return -1;}

  do {
    ch=string[l];l++;
    if (ch=='b' || ch=='p') bm=1;
    else if (ch=='m' && bm==1) bm=2;
    else bm=0;
  } while (bm<2 && l<len && ch!='\n');
  l++;
   
  if (l>=len || ch=='\n') {fprintf(stderr,"No best move in EPD!");return -1;}
   
  a=-1;i=0;
  do {
    a++;
    i=0;
    count++;
    do {
      ch=string[l];l++;
      if (ch=='\n') {fprintf(stderr,"Bad Best Move Data in EPD!\n");return -1;}
      if (ch!=' ' && ch!=';') {answer[a][i]=ch;i++;}
    } while (ch!=' ' && ch!=';');
    answer[a][i]='\x0';
  } while (ch!=';');
  return count;
}

 /*  Get a board position from a FEN code */
board *get_board_fen(char *fen) {
  int a=0,x=0,y=0;
  char ch;
  board *bd;
   
  bd=malloc(sizeof(board));
  assert(bd!=NULL);
  for (y=0;y<8;y++) {
    for (x=0;x<8;x++) bd->sq[y][x]=0;
  }
  x=y=0;
  for (a=0;a<(int)strlen(fen);a++) {
    if (fen[a]=='/') {x=0;y++;}
    else {
      if (x>7) fprintf(stderr,"BAD FEN!\n");
      ch=fen[a];
      if ((int)ch>60) {
        if (ch=='P') bd->sq[y][x]=1;
        if (ch=='p') bd->sq[y][x]=-1;
        if (ch=='R') bd->sq[y][x]=2;
        if (ch=='r') bd->sq[y][x]=-2;
        if (ch=='N') bd->sq[y][x]=3;
        if (ch=='n') bd->sq[y][x]=-3;
        if (ch=='B') bd->sq[y][x]=4;
        if (ch=='b') bd->sq[y][x]=-4;
        if (ch=='Q') bd->sq[y][x]=5;
        if (ch=='q') bd->sq[y][x]=-5;
        if (ch=='K') {bd->kpos[0]=x;bd->kpos[1]=y;bd->sq[y][x]=6;}
        if (ch=='k') {bd->kpos[2]=x;bd->kpos[3]=y;bd->sq[y][x]=-6;}
	x++;
      }
      else x+=((int)ch-48);
    }
    if ((a+1)<(int)strlen(fen) && fen[a+1]==' ') break;
  }
  
  return bd;
}

 /*  Tests to see if 'mv' is the same move as the move represented by the */
 /*  char string, 'str'.  'bd' holds the board position before the move */
int same_move(move *mv,char *str,board *bd) {
  int fx=mv->fx,fy=mv->fy,tx=mv->tx,ty=mv->ty;
  int pf=abs(bd->sq[fy][fx]),pt=abs(bd->sq[ty][tx]);
  int a=3;
  char mvstr[10],ch[3];
   
  if (mv==NULL) return 0;
  if (str==NULL) return 0;
if (pf==0) fprintf(stderr,"ERROR 1 IN SAME_MOVE!\n");
  memset(mvstr, 0, sizeof(mvstr));
  if (mv->castle==0) {
    ch[0]=(char)(fx+97);ch[1]='\x0';
    if (pf==2) strcpy(mvstr,"R");
    if (pf==3) strcpy(mvstr,"N");
    if (pf==4) strcpy(mvstr,"B");
    if (pf==5) strcpy(mvstr,"Q");
    if (pf==6) strcpy(mvstr,"K");
    if (pf==1 && pt>0) strcpy(mvstr,ch);
    if (strchr(str,'+')) a++;
    if (strchr(str,'x')) a++;
    if (strchr(str,'#')) a++;
    if (pf>1 && (int)strlen(str)>a && mvstr[0]==str[0]) {
      if ((int)str[1]<60) ch[0]=(char)(56-fy);
      else ch[0]=(char)(97+fx);
      strcat(mvstr,ch);
    }
    if (pt>0) strcat(mvstr,"x");
    if (fx!=tx && pf==1 && pt==0) strcat(mvstr,"x");
    ch[0]=(char)(tx+97);ch[1]=(char)(56-ty);ch[2]='\x0';
    if (pf==1 && pt==0) strcpy(mvstr,ch); 
    else strcat(mvstr,ch);
    if (mv->promote!=0) strcat(mvstr,"=");
    if (abs(mv->promote)==2) strcat(mvstr,"R");
    if (abs(mv->promote)==3) strcat(mvstr,"N");
    if (abs(mv->promote)==4) strcat(mvstr,"B");
    if (abs(mv->promote)==5) strcat(mvstr,"Q");
  }
  else if (mv->castle==1) strcpy(mvstr,"o-o");
  else if (mv->castle==2) strcpy(mvstr,"o-o-o");
  else fprintf(stderr,"ERROR 2 IN SAME_MOVE!");
  if (!strncmp(str,mvstr,strlen(mvstr))) return 1;
  return 0;
}

  /* Format a move and print it to a FILE pointer */
void format_move(FILE *fp, move *mv,int pf) {
  int fx=mv->fx,fy=mv->fy,tx=mv->tx,ty=mv->ty;
  int pt=abs(mv->capt);
  char mvstr[10],ch[3];
   
  if (mv==NULL) return;
  memset(mvstr, 0, sizeof(mvstr));
  if (mv->castle==0) {
    if (pf==-1) {ch[0]=(char)(fx+97);ch[1]=(char)(56-fy);ch[2]='\x0';strcpy(mvstr,ch);}
    ch[0]=(char)(fx+97);ch[1]='\x0';
    if (pf==2) strcpy(mvstr,"R");
    if (pf==3) strcpy(mvstr,"N");
    if (pf==4) strcpy(mvstr,"B");
    if (pf==5) strcpy(mvstr,"Q");
    if (pf==6) strcpy(mvstr,"K");
    if (pf==1 && pt>0) strcpy(mvstr,ch);
    if (pt>0) strcat(mvstr,"x");
    if (fx!=tx && pf==1 && pt==0) strcat(mvstr,"x");
    ch[0]=(char)(tx+97);ch[1]=(char)(56-ty);ch[2]='\x0';
    if (pf==1 && pt==0) strcpy(mvstr,ch); 
    else strcat(mvstr,ch);
    if (mv->promote!=0) strcat(mvstr,"=");
    if (abs(mv->promote)==2) strcat(mvstr,"R");
    if (abs(mv->promote)==3) strcat(mvstr,"N");
    if (abs(mv->promote)==4) strcat(mvstr,"B");
    if (abs(mv->promote)==5) strcat(mvstr,"Q");
    if (mv->gchk) strcat(mvstr,"+");
  }
  fprintf(fp,"%s",mvstr);
}

 /*  Returns '1' if there is too little material left to  */
 /*  checkmate either king */
int is_drawn_game(const board *sq,int gs) {
  if (gs==0) return 1;
  if (gs==3 && (sq->npieces[0][1]==1 || sq->npieces[0][2]==1 || sq->npieces[1][1]==1 || sq->npieces[1][2]==1)) return 1;
  return 0;
}

 /*  Get a position list from the list of previous moves. */
 /*  These functions used for draw by repetition check. */
void get_poslist(compdat *cd,int nm,board *sq) {
  int a,b,c,side,check2=0;
  long int check=0;
  move *mv;
  board *sqq;

  cd->once[0]=NULL;
  cd->once[1]=NULL;
  cd->twice[0]=NULL;
  cd->twice[1]=NULL;
  sqq=malloc(sizeof(board));
  assert(sqq!=NULL);
  memset(sqq->sq, 0, sizeof(sqq->sq));
  for (a=0;a<8;a++) for (b=0;b<8;b++) sqq->sq[a][b]=sq->sq[a][b];
  for (a=nm;a>=0;a--) {
    side=2-(a&1);
    if (a<nm) {
      check=0;
      check2=0;
      for (b=0;b<8;b++) {
        for (c=0;c<8;c++) if (sqq->sq[b][c]!=0) {
          check+=rtable[b][c][(sqq->sq[b][c])+6];
          check2+=rtable2[b][c][(sqq->sq[b][c])+6];
        }
      } 
       /*  Update lists */
      update_poslist(cd,3-side,check,check2);
    }
     /*  Get move data */
    if (a==0) continue;
    mv=(mvlst+a-1);
    if (!mv || mv->fx==-1) continue;
     /*  Move makes draw count reset? */
    if (mv->castle || mv->ep || mv->capt || abs(sqq->sq[mv->ty][mv->tx])==1) {free(sqq);return;}
     /*  Undo this move */
    undo_move(sqq,mv,side);
  }
  free(sqq);
}

 /*  Free a position list */
void free_poslist(poslist *pl) {
  while (pl) {
    free_poslist(pl->next);
    free(pl);
    pl=NULL;
  }
}

 /*  Make and return a copy of a list */
poslist *copy_poslist(poslist *pl) {
  poslist *p=NULL;
  
  if (pl) {
    p=malloc(sizeof(poslist));
    assert(p!=NULL);
    p->check=pl->check;p->check2=pl->check2;
    p->next=copy_poslist(pl->next);
  }
  return p;
}

 /*  Check to see if the position with checksums check and check2 */
 /*  is in the given position list */
int isinposlist(poslist *pl, long int ch, int ch2) {
  poslist *p=pl;
  while (p) {
    if (p->check==ch && p->check2==ch2) return 1;
    p=p->next;
  }
  return 0;
}

 /*  Check to see if the position check,check2 is in the 'once' list */
 /*  and if it is then move it into the 'twice' list.  If not then add */
 /*  it to the 'once' list. */
void update_poslist(compdat *cd,int side, long int ch, int ch2) {
  poslist *p=NULL,*pl=NULL;
  
  p=cd->once[side-1];
  while(p) {
     /*  Found it? */
    if (p->check==ch && p->check2==ch2) {
       /*  Move to the 'twice' list */
      if (pl) pl->next=p->next;
      else cd->once[side-1]=p->next;
      p->next=cd->twice[side-1];
      cd->twice[side-1]=p;
      return;   
    }
    else {pl=p;p=p->next;}
  }
   /*  Position has not arisen before */
  p=malloc(sizeof(poslist));
  assert(p!=NULL);
  p->next=cd->once[side-1];
  p->check=ch;p->check2=ch2;
  cd->once[side-1]=p;
}

 /*  See if the current position is a draw by repetition */
 /*  (slow version - used for human play) */
int test_for_draw(int side,int mvno,board *sqq,int comp) {
  int b,c,nm,isdraw=0,check2=0;
  long int check=0;
  compdat *cd;
   
  for (b=0;b<8;b++) {
    for (c=0;c<8;c++) if (sqq->sq[b][c]!=0) {
      check+=rtable[b][c][(sqq->sq[b][c])+6];
      check2+=rtable2[b][c][(sqq->sq[b][c])+6];
    }
  }
  cd=malloc(sizeof(compdat));
  assert(cd!=NULL);
  nm=(mvno*2)+side-3;
  get_poslist(cd,nm,sqq);
  if (isinposlist(cd->twice[side-1],check,check2)) {
    isdraw=1;
    if (xboardmode==0) {
      if (comp==1) fprintf(stderr,"\nDRAW by repetition!\n");
      else fprintf(stderr,"You May Claim DRAW by repetition!\n");
    }
    else fprintf(stderr,"1/2-1/2 {Draw by repetition}\n");
  }
  free_poslist(cd->once[0]);
  free_poslist(cd->once[1]);
  free_poslist(cd->twice[0]);
  free_poslist(cd->twice[1]);
  free(cd);  
  return isdraw;
}

 /*  Conver the board position into a FEN code string */
char *board_to_fen(board *sq,int side,int cp[4],move *last) {
  int x,y,l=0,i=0;
  char str[80],row[8],*fen=NULL,epsq[3];
  
  strcpy(str,"");
  for (y=0;y<8;y++) {
    i=l=0;
    strcpy(row,"");
    for (x=0;x<8;x++) {
      if (sq->sq[y][x]==0) l++;
      else {
	if (l>0) {row[i]=(char)(l+48);i++;}
	l=0;
	if (sq->sq[y][x]==1) row[i]='P';
	if (sq->sq[y][x]==2) row[i]='R';
	if (sq->sq[y][x]==3) row[i]='N';
	if (sq->sq[y][x]==4) row[i]='B';
	if (sq->sq[y][x]==5) row[i]='Q';
	if (sq->sq[y][x]==6) row[i]='K';
	if (sq->sq[y][x]==-1) row[i]='p';
	if (sq->sq[y][x]==-2) row[i]='r';
	if (sq->sq[y][x]==-3) row[i]='n';
	if (sq->sq[y][x]==-4) row[i]='b';
	if (sq->sq[y][x]==-5) row[i]='q';
	if (sq->sq[y][x]==-6) row[i]='k';
        i++;
      }
    }
    if (l>0) {row[i]=(char)(l+48);i++;}
    strncat(str,row,(unsigned int)i);
    if (y<7) strcat(str,"/");
  }
  if (side==WHITE) strcat(str," w ");
  else strcat(str," b ");
  if (cp[0]) strcat(str,"K");
  if (cp[1]) strcat(str,"Q");
  if (cp[2]) strcat(str,"k");
  if (cp[3]) strcat(str,"q");
  if (!cp[0] && !cp[1] && !cp[2] && !cp[3]) strcat(str,"-");
  if (last==NULL) strcat(str," -");
  else {
    if (last->fx==last->tx && abs(sq->sq[last->ty][last->tx])==1 && abs(last->ty-last->fy)==2) {
      x=last->fx;y=(last->fy+last->ty)/2;
      strcat(str," ");
      epsq[2]='\x0';
      epsq[0]=(char)(x+97);
      epsq[1]=(char)(48+(8-y));
      strcat(str,epsq);
    }
    else strcat(str," -");
  }
  fen=malloc(sizeof(char)*(size_t)(strlen(str)+1));
  assert(fen!=NULL);
  strcpy(fen,str);
  return fen;
}

 /*  Check opening book to see if a position exists */
move *check_opening(board *sq,int side,int cp[4],move *last,openpos *openings) {
  int i=0,a,b,replies[100][7],nr=0,tot=0,choice=0;
  long int repscore[100],total=0;
  char *fen;
  move *best;
  movelst *mvl;
  openpos *op;

  if (!openings) return NULL;
  memset(replies, -1, sizeof(replies));
  memset(repscore, 0, sizeof(repscore));
  fen=board_to_fen(sq,side,cp,last);
  op=openings;
  do {
    i=0;
    if ((strchr(op->fen,'w') && side==WHITE) || (!strchr(op->fen,'w') && side==BLACK)) {
      if (!strcmp(op->fen,fen)) {
        free(fen);
	nr=0;
	mvl=op->moves;
	do {
          replies[nr][0]=(mvl->mv).fx;
          replies[nr][1]=(mvl->mv).fy;
          replies[nr][2]=(mvl->mv).tx;
          replies[nr][3]=(mvl->mv).ty;
          replies[nr][4]=(mvl->mv).castle;
          replies[nr][5]=(mvl->mv).ep;
          replies[nr][6]=(mvl->mv).promote;
          repscore[nr]=(mvl->mv).score;	   
          mvl=mvl->next;
	  nr++;
	} while (mvl);
	total=0;
	for (a=0;a<nr;a++) total+=repscore[a];
	if (total>0) tot=random2(total);
	else tot=0;
	choice=0;
	do {
	  tot-=repscore[choice];
	  choice++;
	} while (tot>=0);
	choice--;
	best=malloc(sizeof(move));
	assert(best!=NULL);
	best->fx=replies[choice][0];best->fy=replies[choice][1];
	best->tx=replies[choice][2];best->ty=replies[choice][3];
	best->castle=replies[choice][4];
	best->ep=replies[choice][5];
	best->promote=replies[choice][6];
	best->cm=best->gchk=0;
	best->capt=sq->sq[best->ty][best->tx];
	best->score=((repscore[choice]*100)/total);
        for (a=0;a<nr;a++) {for (b=0;b<7;b++) replies[a][b]=-1;repscore[a]=0;}
        return best;
      }
    }
    op=op->next;
  } while (op);
  free(fen);
  return NULL;
}

 /*  Check opening book to see if the specified move exists */
int check_opening_exists(board *sq,int side,int cp[4],move *last,move *m,openpos *openings) {
  char *fen;
  movelst *mvl;
  openpos *op;

  if (!openings) return 0;
  fen=board_to_fen(sq,side,cp,last);
  op=openings;
  while (op) {
    if ((strchr(op->fen,'w') && side==WHITE) || (!strchr(op->fen,'w') && side==BLACK)) {
      if (!strcmp(op->fen,fen)) {
        free(fen);
	mvl=op->moves;
        while (mvl) {
	  if (compare(&(mvl->mv),m)==1) {
	    if ((mvl->mv).score>0) return 1;
	    return -1;
          }
	  mvl=mvl->next;
	}
        return 0;
      }
    }
    op=op->next;
  }
  free(fen);
  return 0;
}

 /*  Load openings from book into memory */
openpos *load_openings(void) {
  int i,a,b;
  long int npos=0;
  movelst *mvl=NULL,*oldmvl=NULL;
  char fen[80],temp[1024],ch,y,book_path[FILENAME_MAX];
  openpos *op=NULL,*oldop=NULL,*opall=NULL;
  FILE *fp;

  strcpy(book_path,path);
  strcat(book_path,sep);
  strcat(book_path,"book.dat");
  if ((fp = fopen(book_path,"r"))==NULL) {
    fprintf(stderr,"Failed to load opening book %s\n",book_path);
    return NULL;
  }
  fprintf(stderr,"Loading opening book ... ");
  while (!feof(fp)) {
    npos++;
    (void)fscanf(fp,"%c",&ch);
    if (ch=='#') break;
    oldop=op;
    op=malloc(sizeof(openpos));
    assert(op!=NULL);
    if (oldop) oldop->next=op;
    else opall=op;
    op->next=NULL;
    op->moves=NULL;
    strcpy(fen,"");
    strcpy(temp,"");
    fen[0]=ch;
    i=1;
    do {
      (void)fscanf(fp,"%c",&ch);
      if (ch!='$') {fen[i]=ch;i++;}
    } while (ch!='$');
    fen[i]='\x0';
    i++;
    op->fen=malloc(sizeof(char)*(size_t)i);
    assert(op->fen!=NULL);
    strncpy(op->fen,fen,(size_t)i);
    i=0;
    do {
      (void)fscanf(fp,"%c",&ch);
      if (ch!='\n') {temp[i]=ch;i++;}
    } while (ch!='\n');
    temp[i]='\x0';
    i++;
    op->comments=malloc(sizeof(char)*(size_t)i);
    assert(op->comments!=NULL);
    strncpy(op->comments,temp,(size_t)i);
     
    ch=' ';
    mvl=NULL;
    do {
      do {
	(void)fscanf(fp,"%c",&ch);
      } while (ch==' ');
      if (ch!='\n') {
        oldmvl=mvl;
        mvl=malloc(sizeof(movelst));
	assert(mvl!=NULL);
        mvl->next=NULL;
        if (oldmvl) oldmvl->next=mvl;
        else op->moves=mvl;
        (void)fscanf(fp,"%d%c%d",&a,&y,&b);
        ((mvl->mv).fx)=(int)ch-97;      
        ((mvl->mv).fy)=8-a;      
        ((mvl->mv).tx)=(int)y-97;      
        ((mvl->mv).ty)=8-b;
	(void)fscanf(fp,"%c",&ch);
	if (ch=='[') {
          (void)fscanf(fp,"%d,",&((mvl->mv).castle));
          (void)fscanf(fp,"%d,",&((mvl->mv).ep));
          (void)fscanf(fp,"%d]{",&((mvl->mv).promote));
	}
	else {(mvl->mv).castle=0;(mvl->mv).ep=0;(mvl->mv).promote=0;}
        (void)fscanf(fp,"%ld}",&((mvl->mv).score));
        (void)fscanf(fp,"%c",&ch);
      }
    } while (ch!='\n');
  }
  (void)fclose(fp);
  fprintf(stderr,"%ld Positions Loaded OK\n",npos-1);
  return opall;
}

 /*  Save opening book from memory to disk */
void save_openings(openpos *openings) {
  openpos *op=openings;
  movelst *mvl;
  char book_path[FILENAME_MAX];
  FILE *fp;
   
  strcpy(book_path,path);
  strcat(book_path,sep);
  strcat(book_path,"book.dat");
  if ((fp = fopen(book_path,"w"))==NULL) {
    fprintf(stderr,"Failed to save opening book %s\n",book_path);
    return;
  }
  while (op) {
    fprintf(fp,"%s$",op->fen);
    if (op->comments!=NULL) fprintf(fp,"%s\n",op->comments);
    else fprintf(fp,"\n");
    mvl=op->moves;
    while (mvl) {
      if ((mvl->mv).castle==0) {
        fprintf(fp,"%c%d",(char)(((mvl->mv).fx)+97),8-((mvl->mv).fy));
        fprintf(fp,"%c%d",(char)(((mvl->mv).tx)+97),8-((mvl->mv).ty));
      }
      else {
	if (strchr(op->fen,'w')) {
	  if ((mvl->mv).castle==1) fprintf(fp,"e1g1");
	  else fprintf(fp,"e1c1");
	}
	else {
	  if ((mvl->mv).castle==1) fprintf(fp,"e8g8");
	  else fprintf(fp,"e8c8");
	}
      }
      if ((mvl->mv).castle!=0 || (mvl->mv).ep!=0 || (mvl->mv).promote!=0) fprintf(fp,"[%d,%d,%d]",((mvl->mv).castle),((mvl->mv).ep),((mvl->mv).promote));
      fprintf(fp,"{%ld}",((mvl->mv).score));
      mvl=mvl->next;
      if (mvl) fprintf(fp," ");
    }
    fprintf(fp,"\n");
    op=op->next;
  }
  fprintf(fp,"#END#\n");
  (void)fclose(fp);
}

  /*  Update opening book based on game that has just finished */
void learn_opening(int loseside,int mvno, openpos *openings,int learnboth) {
  int maxmvno=0,n=0,cp[4],pc,side,ok=0,ok1=0,cmove,pt,count=0,goodmove=0;
  openpos *op,*optemp=NULL;
  movelst *mvl,*oldmvl=NULL;
  move *mv=NULL,*last=NULL;
  board *sq;
  char *fen,moves[1024],ch[5];
   
  if (learnboth==0 && mvno<15) {fprintf(stderr,"Game is Dangerously Short (%d moves) - Book Learning Abandoned!\n",mvno);return;} /*  Game is dangerously short - probably a bad one */
  if (mvno>100) {fprintf(stderr,"Warning : Game is Very Long (%d moves) - Possibly not Worth Learning!\n",mvno);} /* Possibly not worth learning this opening */
  maxmvno=min(learn_max_ply,(mvno*2)-3+loseside);
  cp[0]=cp[1]=cp[2]=cp[3]=1;
  if (maxmvno==0) return; /*  No moves :) (or min set to zero) */
  if (!openings) return;
   
  if (learnboth==0) fprintf(stderr,"\nUpdating Opening Book:\n");
  sq=reset_board();
  strcpy(moves,"");
  for (n=0;n<maxmvno;n++) {
    side=(n&1)+1;
    last=mv;
    mv=(mvlst+n);
    if (3-side==loseside || learnboth==1) {
      fen=board_to_fen(sq,side,cp,last);
      op=openings;
      ok1=0;
      optemp=NULL;
      while (!ok1 && op) {
	if (!strcmp(fen,op->fen)) {
	  count=0;
	  goodmove=0;
	  mvl=op->moves;
	  oldmvl=NULL;
          while (!ok1 && mvl) {
	    count++;
	    if ((mvl->mv).score>20) goodmove=1;
	    oldmvl=mvl;
	    if (compare(&(mvl->mv),mv)) {
	      if ((mvl->mv).score>0) {
                if (learnboth==0) fprintf(stderr,"Existing Score Updated!\n");
 	        if (learnboth==0 || ((mvl->mv).score)<100) ((mvl->mv).score)++;
	      }
              else if (learnboth==0) fprintf(stderr,"Existing Move Ignored (Zero Score)!\n");
              ok1=1;
	    }
	    else mvl=mvl->next;
	  }
	  if (!ok1) {
            if (learnboth==0) fprintf(stderr,"New Move Option Added!\n");
	    mvl=malloc(sizeof(movelst));
	    assert(mvl!=NULL);
	    mvl->mv=*mv;
	     /* If scanning opening book & already a good move then ignore this one */
	    if (learnboth==1 && (goodmove==1 || count>4)) (mvl->mv).score=0;
            else (mvl->mv).score=1;
	    mvl->next=NULL;
  	    if (oldmvl) oldmvl->next=mvl;
	    else op->moves=mvl;
	    ok1=ok=1;
	  }
	}
	optemp=op;
	if (!ok1) op=op->next;
      }
      if (!ok1) {
        if (learnboth==0) fprintf(stderr,"New Book Entry Added!\n");
	op=malloc(sizeof(openpos));
        assert(op!=NULL);
	op->next=NULL;
	if (optemp) optemp->next=op;
	else openings=op;
	op->fen=malloc(sizeof(char)*(size_t)(strlen(fen)+1));
	op->comments=malloc(sizeof(char)*(size_t)(8+strlen(moves)));
	assert(op->fen!=NULL);
	assert(op->comments!=NULL);
	strcpy(op->fen,fen);
	strcpy(op->comments,"   // ");
	strcat(op->comments,moves);
	op->moves=malloc(sizeof(movelst));
	assert(op->moves!=NULL);
	mvl=op->moves;
	mvl->next=NULL;
	mvl->mv=*mv;
	(mvl->mv).score=1;
	ok=1;
      }
      free(fen);
    }
    if (ok==1) break;
     /*  Add move to movelist */
    if (side==WHITE) {
      cmove=(n/2)+1;
      if (cmove>1) strcat(moves," ");
      if (cmove<10) {ch[0]=(char)(cmove+48);ch[1]='\x0';}
      else {ch[0]=(char)((cmove/10)+48);ch[1]=(char)((cmove%10)+48);ch[2]='\x0';}
      strcat(moves,ch);
      strcat(moves,".");
    }
    strcat(moves," ");
    if (mv->castle==0) {pc=abs(sq->sq[mv->fy][mv->fx]);pt=abs(sq->sq[mv->ty][mv->tx]);}
    else {pc=6;pt=0;}
    if (mv->castle==0) {
      if (pc==1) {
        if (pt==0 && mv->ep==0) {ch[0]=(char)(mv->tx+97);ch[1]=(char)(56-mv->ty);ch[2]='\x0';}
	else {ch[0]=(char)(mv->fx+97);ch[1]='x';ch[2]=(char)(mv->tx+97);ch[3]=(char)(56-mv->ty);ch[4]='\x0';}
      }
      else {
	if (pc==2) ch[0]='R';
	if (pc==3) ch[0]='N';
	if (pc==4) ch[0]='B';
	if (pc==5) ch[0]='Q';
	if (pc==6) ch[0]='K';
        if (pt==0) {ch[1]=(char)(mv->tx+97);ch[2]=(char)(56-mv->ty);ch[3]='\x0';}
        else  {ch[1]='x';ch[2]=(char)(mv->tx+97);ch[3]=(char)(56-mv->ty);ch[4]='\x0';}
      }
      strcat(moves,ch);
    }
    else {
     if (mv->castle==1) strcat(moves,"0-0");
     if (mv->castle==2) strcat(moves,"0-0-0");
    }
     /*  Do move */
    do_move2(sq,mv,side);
    if (side==WHITE) {  
      if (pc==6) cp[0]=cp[1]=0;
      else if (pc==2) {
	if (mv->fx==0 && mv->fy==7) cp[1]=0;
	if (mv->fx==7 && mv->fy==7) cp[0]=0;
      }
    }
    if (side==BLACK) {
      if (pc==6) cp[2]=cp[3]=0;
      else if (pc==2) {
	if (mv->fx==0 && mv->fy==0) cp[3]=0;
	if (mv->fx==7 && mv->fy==0) cp[2]=0;
      }
    } 
  }
  
  if (learnboth==0) {
    if (ok==0) fprintf(stderr,"No Changes Made!\n");
    fprintf(stderr,"Saving Opening Book - ");
    save_openings(openings);
    fprintf(stderr,"OK!\n");
  }
  free(sq);
}

 /*  Free memory for opening book */
void free_opening(openpos *opening) {
  if (opening) {
    free_opening(opening->next);
    free_mvlst(opening->moves);
    free(opening->fen);
    if (opening->comments) free(opening->comments);
    free(opening);
  }
}

 /* Check if game is a theoretical win for specified side *
  * Returns 1 if it is a win, -1 for a loss, 0 otherwise */
int is_won_game(const board *sq, int side, int ypts, int opts) {
  if (ypts>0 && opts>0) return 0;
   /* You have no pieces */
  if (ypts==0) {
    if (sq->npieces[2-side][0]>0 || sq->npieces[2-side][3]>0 || sq->npieces[2-side][2]>1) return -1;
  }
   /* Opp. has no pieces */
  else {
    if (sq->npieces[side-1][0]>0 || sq->npieces[side-1][3]>0 || sq->npieces[side-1][2]>1) return 1;
  }
  return 0;
}

 /* Tests to see if a file is free from pawns beginning at the given
  * x,y in direction specified by pdir */
int file_free(const board *sq,int x,int y,int pdir) {
  int a;
  for (a=y;a>0 && a<7;a+=pdir) if (abs(sq->sq[a][x])==1) return 0;
  return 1;
}

void scan_book(openpos *openings) {
  int side,mvno,n;
  char nam[51],nm[FILENAME_MAX],cd_path[FILENAME_MAX];
  FILE *fp;
  move mvlstcopy[1024];

  strcpy(cd_path,dir_command);
  strcat(cd_path," ");
  strcat(cd_path,path);
  strcat(cd_path,"*");
  printf("\nDirectory of All Files\n\n");
  (void)system(cd_path);
  printf("\nPlease Input Name:-");
  do {
    (void)scanf("%s",nam);
    if (strlen(nam)>30 || strlen(nam)<3) printf("   Illegal Name!");
  } while (strlen(nam)>30 || strlen(nam)<3);
  strcpy(nm,path);
  strcat(nm,nam);
  if ((fp = fopen(nm,"r"))==NULL)  {
    printf("\n!!! Cannot Open File !!!\n");
    (void)system("cd ..");
    return;
  }

  fprintf(stderr,"\nScanning File ");
  for (n=0;n<1024;n++) mvlstcopy[n]=mvlst[n];
  while (!feof(fp)) {
    load_game_string(fp);
    if ((mvlst+0)->fx<0 || (mvlst+0)->fx>7) {break;}
    side=g_side;
    mvno=g_mvno;
    fprintf(stderr,".");
    learn_opening(side,mvno,openings,1);
  }
  (void)fclose(fp);
  printf("\nBook Scanned O.K.\n");
  save_openings(openings);
  for (n=0;n<1024;n++) mvlst[n]=mvlstcopy[n];
}

void load_game_string(FILE *fp) {
  int fy,ty,fx,tx,castle,side=1,mvno=1,i=0;
  char fxc,txc,ch;
  move *m;
   
  i=0;
  do {
    (void)fscanf(fp,"%c%d%c%d%c",&fxc,&fy,&txc,&ty,&ch);
    fx=(int)(fxc-97);
    tx=(int)(txc-97);
    fy=8-fy;
    ty=8-ty;
    castle=0;
    if (fx==4 && fy==7 && tx==6 && ty==7) castle=1;
    if (fx==4 && fy==0 && tx==6 && ty==0) castle=1;
    if (fx==4 && fy==7 && tx==2 && ty==7) castle=2;
    if (fx==4 && fy==0 && tx==2 && ty==0) castle=2;
    m=(mvlst+i);
    m->fx=fx;m->fy=fy;
    m->tx=tx;m->ty=ty;
    m->castle=castle;
    m->ep=m->promote=m->capt=0;
    side=3-side;
    if (side==WHITE) mvno++;
  } while (ch==' ');
  g_side=side;
  g_mvno=mvno;
}

void perform_self_test(openpos *openings) {
  int limit,ignore,first,n=0,tot,i,score,ignon=0,length;
  openpos *op;
  move *best;
  movelst *mvl;
  compdat *cd=&cdd[0];
  board *sqq;
  char *rest;
  
  fprintf(stderr,"Thinking time (s) : ");
  scanf("%d",&limit);
  fprintf(stderr,"Ignore Entries with > # Total Weighting : ");
  scanf("%d",&ignore);
  fprintf(stderr,"Ignore first # entries : ");
  scanf("%d",&first);
  fprintf(stderr,"Number of Entries to Study : ");
  scanf("%d",&length);
  op=openings;
  
  if (first>0) fprintf(stderr,"Ignoring ");

  while (op && length>0) {
    n++;
    if (n<=first) {fprintf(stderr,".");op=op->next;continue;}
    tot=0;
    mvl=op->moves;
    while (mvl) {tot+=(mvl->mv).score;mvl=mvl->next;}
    if (tot<=ignore) {
      ignon=0;
      length--;
      fprintf(stderr,"\nStudying position #%d = %s\n",n,op->fen);
      mvl=op->moves;
      while (mvl) {printmove(&(mvl->mv));fprintf(stderr,"{%ld} ",(mvl->mv).score);mvl=mvl->next;}
      fprintf(stderr,"\n");
      rest=strchr(op->fen,' ');
      cd->skill=12;
      cd->quiesce=1;
      sqq=get_board_fen(op->fen);
      if (strchr(rest,'w')) cd->side=1;
      else cd->side=2;
      cd->gamestage=pts(1,sqq)+pts(2,sqq);
      for (i=0;i<4;i++) cd->cp[i]=0;
      if (strchr(rest,'K')) cd->cp[0]=1;
      if (strchr(rest,'Q')) cd->cp[1]=1;
      if (strchr(rest,'k')) cd->cp[2]=1;
      if (strchr(rest,'q')) cd->cp[3]=1;    
      cd->last=NULL;
      get_pieces(sqq);
      cd->tlimit=limit*10;
      cd->extend=0; 
      cd->inchk=chck(sqq,cd->side);
      get_poslist(cd,-1,sqq);
      best=comp(sqq,14,1,1,1);
      fprintf(stderr,"Suggested Best Move = ");
      printmove(best);
      fprintf(stderr,"  Move Score = %ld\n",best->score);
      mvl=op->moves;
      while (mvl) {
	if (compare(&(mvl->mv),best)) break;
	mvl=mvl->next;
      }
      if (mvl) {
        score=(((mvl->mv).score)*100)/tot;
	fprintf(stderr,"Book Move! (%d%%)\n",score);
      }
      else {
	fprintf(stderr,"Does Not Agree With Book!\n");
      }
      free(sqq);
      free(best);
    }
    else {
      if (ignon==0) fprintf(stderr,"Ignoring ");
      fprintf(stderr,".");
      ignon=1;
    }
    op=op->next;
  }
  fprintf(stderr,"Study Finished!\n");
}

struct gamedat *import_pgn(void) {
  int i,side=1,mvno=1,x,y,cp[4],num=0,gameno,total;
  char nam[51],nm[FILENAME_MAX],cd_path[FILENAME_MAX],ch;
  FILE *fp;
  move *mlst,*m;
  board *bd;
  struct gamedat *game;

  strcpy(cd_path,dir_command);
  strcat(cd_path," ");
  strcat(cd_path,path);
  strcat(cd_path,"*.pgn");
  printf("\nDirectory of *.pgn Files\n\n");
  (void)system(cd_path);
  printf("\nPlease Input Name:-");
  do {
    (void)scanf("%s",nam);
    if (!strstr(nam,".pgn")) strcat(nam,".pgn");
    if (strlen(nam)>30 || strlen(nam)<5) printf("   Illegal Name!");
  } while (strlen(nam)>30 || strlen(nam)<5);
  strcpy(nm,path);
  strcat(nm,nam);
  if ((fp = fopen(nm,"r"))==NULL)  {
    printf("\n!!! Cannot Open File !!!\n");
    (void)system("cd ..");
    return NULL;
  }
   
   /* Count the games */
  total=0;
  do {
    (void)fscanf(fp,"%c",&ch); 
  } while (ch!='[');
  do {
    do {
      do {
	(void)fscanf(fp,"%c",&ch);
      } while (ch!='\n');
      (void)fscanf(fp,"%c",&ch);
    } while (ch=='[');
    total++;
    do {
      do {
	(void)fscanf(fp,"%c",&ch);
      } while (ch!='\n');
      (void)fscanf(fp,"%c",&ch);
    } while (ch!='[' && !feof(fp));
  } while (ch=='[');
   
   /* Reset fp */
  (void)fclose(fp);
  fp = fopen(nm,"r");
   
   /* Get desired game number */
  if (total>1) {
    do {
      fprintf(stderr,"Load Which Game Number? [1-%d] : ",total);
      (void)fscanf(stdin,"%d",&gameno);
    } while (gameno<1 || gameno>total);
  }
  else gameno=1;
   /* Skip unwanted games */
  for (i=1;i<gameno;i++) {
    do {
      do {
	(void)fscanf(fp,"%c",&ch);
      } while (ch!='\n');
      (void)fscanf(fp,"%c",&ch);
    } while (ch=='[');
    do {
      do {
	(void)fscanf(fp,"%c",&ch);
      } while (ch!='\n');
      (void)fscanf(fp,"%c",&ch);
    } while (ch!='[');
  }
   
  mlst=load_pgn_game(fp,&num);
  (void)fclose(fp);

  bd=reset_board();
  cp[0]=cp[1]=cp[2]=cp[3]=1;
  for (i=0;i<num;i++) {
    m=(mlst+i);
    do_move2(bd,m,side);
    if (m->castle!=0) {
      if (side==WHITE) cp[0]=cp[1]=0;
      else cp[2]=cp[3]=0;
    }
    if (bd->sq[m->ty][m->tx]==6) cp[0]=cp[1]=0;
    if (bd->sq[m->ty][m->tx]==-6) cp[2]=cp[3]=0;
    if (m->fy==7 && m->fx==0) cp[1]=0;
    if (m->fy==7 && m->fx==7) cp[0]=0;
    if (m->fy==0 && m->fx==0) cp[3]=0;
    if (m->fy==0 && m->fx==7) cp[2]=0;
    add_move(side,mvno,m->fx,m->fy,m->tx,m->ty,m->castle,m->ep,m->promote,m->capt);
    side=3-side;
    if (side==WHITE) mvno++;
  }
  game=malloc(sizeof(struct gamedat));
  assert(game!=NULL);
  game->mvno=mvno;
  game->side=side;
  for (x=0;x<8;x++) for (y=0;y<8;y++) game->sq[y][x]=bd->sq[y][x];
  for (i=0;i<4;i++) game->cp[i]=cp[i];
  free(mlst);
  return game;
}

move *load_pgn_game(FILE *fp,int *num) {
  int side=1,mno=1,i=0; /* mno holds the number of _half_-moves */
  int piece,fx=-1,fy=-1,tx=-1,ty=-1,a,opt=0,x,y,bracket_count=0;
  int breakout=0,ok=0;
  char ch,temp[1024],chh;
  move *mlst=NULL,m;
  board *bd,*mp;
   
  chh=' ';
  *num=0;
  do {
    (void)fscanf(fp,"%c",&ch);
    if (ch!='1' && ch!='\n' && !feof(fp)) {
      do { (void)fscanf(fp,"%c",&ch); } while (ch!='\n');
    }
    if (ch=='1') (void)fscanf(fp,"%c",&chh);
  } while ((ch!='1' || chh!='.') && !feof(fp));
  if (feof(fp)) return NULL;
  (void)fscanf(fp,". ");
   
  bd=reset_board();
  mp=reset_board();
   
  i=0;
  memset(temp, 0, sizeof(temp));
  do {
    (void)fscanf(fp,"%c",&ch);
    temp[i]=ch;
    if (ch=='{') bracket_count++;
    if (ch=='}') bracket_count--;
    if (ch=='#' && bracket_count==0) breakout=1;
    if (bracket_count<0) {
      fprintf(stderr,"Bad Bracket Formation in PGN!\n");
    }
    if (bracket_count==0 && ((int)ch<34)) {
      temp[i]=0;
/*
        if (strchr(temp,'{')!=NULL) fprintf(stderr,"Ignored Comment %s\n",temp);
*/
      if (!strchr(temp,'{') && i>1 && !strchr(temp,'.')) {
	ch=temp[0];
	fx=tx=fy=ty=-1;
	if (ch=='R') piece=2; 
	else if (ch=='N') piece=3;
	else if (ch=='B') piece=4;
	else if (ch=='Q') piece=5;
	else if (ch=='K') piece=6;
	else if (ch=='0') piece=7;
	else if (ch=='o') piece=7;
	else if (ch=='O') piece=7;
	else if (ch=='1') piece=0;
	else piece=1;
	   
        m.castle=m.promote=m.ep=0;

	if (piece==1) { /* Pawn move */
	  if (i>3) { /* Pawn capture or promotion */
	    fx=(int)ch-97;
	    if (temp[1]=='x') { /* Capture */
	      tx=(int)temp[2]-97;
	      ty=56-(int)temp[3];
	      if (side==WHITE) fy=ty+1;
	      else fy=ty-1;
	      if (bd->sq[ty][tx]==0) m.ep=1; /* En-passant capture */
	    }
	    else if (temp[2]=='=') { /* Promotion */
	      tx=fx;
	      if (side==WHITE) {ty=0;fy=1;}
	      else {ty=7;fy=6;}
	      if (temp[3]=='R') m.promote=2;
	      if (temp[3]=='N') m.promote=3;
	      if (temp[3]=='B') m.promote=4;
	      if (temp[3]=='Q') m.promote=5;
	      if (side==BLACK) m.promote=-m.promote;
	    }
	    else fprintf(stderr,"Strange Pawn Move in PGN : %s\n",temp);
	  }
	  else { /* Pawn push */
	    fx=tx=(int)ch-97;
	    ty=56-(int)temp[1];
	    if (side==WHITE) {
	      if (bd->sq[ty+1][fx]==0) fy=ty+2;
	      else fy=ty+1;
	    }
	    else {
	      if (bd->sq[ty-1][fx]==0) fy=ty-2;
	      else fy=ty-1;
	    }
	  }
	}

        if (piece>1) { /* Not a pawn move */
	  if (piece==7) { /* Castling move */
	    if (side==WHITE) fy=ty=7;
	    else fy=ty=0;
	    if (i<5) { /* Castle King's side */
	      m.castle=1;
	      fx=4;tx=6;
	    }
	    else { /* Queen's side */
	      m.castle=2;
	      fx=4;tx=2;
	    }
	  }
	  else { /* Not a castling move */
            if (side==BLACK) piece=-piece;
	    if (temp[1]=='x') { /* Capturing move */
	      tx=(int)temp[2]-97;
	      ty=56-(int)temp[3];
	    }
	    else if (temp[2]=='x') { /* Ambiguous capture */
	      tx=(int)temp[3]-97;
	      ty=56-(int)temp[4];
              a=(int)temp[1];
	      if (a<90) { /* Ambiguous rank */
		fy=56-a;
	      }
	      else { /* Ambiguous file */
		fx=a-97;
	      }
	    }
	    else { /* Not a capture */
	      if ((int)temp[2]>90) { /* Ambiguous move */
  	        tx=(int)temp[2]-97;
 	        ty=56-(int)temp[3];
                a=(int)temp[1];
  	        if (a<90) { /* Ambiguous rank */
    		  fy=56-a;
	        }
	        else { /* Ambiguous file */
	  	  fx=a-97;
	        }
	      }
	      else {
	        tx=(int)temp[1]-97;
	        ty=56-(int)temp[2];
	      }
	    }
	     /* Work out which piece moved */
	     
	    opt=bd->sq[ty][tx];
	    bd->sq[ty][tx]=piece;

	     /* Reset possible moves board */
	    memset(mp->sq, 0, sizeof(mp->sq));

	     /* Check where the piece could have come from.
	      * All friendly pieces are marked with '2'. */
	    if (piece==2) mprw(tx,ty,bd->sq,mp);
	    if (piece==3) mpnw(tx,ty,bd->sq,mp);
	    if (piece==4) mpbw(tx,ty,bd->sq,mp);
	    if (piece==5) mpqw(tx,ty,bd->sq,mp);
            if (piece==6) mpkw(tx,ty,bd->sq,mp);
	    if (piece==-2) mprb(tx,ty,bd->sq,mp);
	    if (piece==-3) mpnb(tx,ty,bd->sq,mp);
	    if (piece==-4) mpbb(tx,ty,bd->sq,mp);
	    if (piece==-5) mpqb(tx,ty,bd->sq,mp);
	    if (piece==-6) mpkb(tx,ty,bd->sq,mp);

	    for (y=0;y<8;y++) {
	      for (x=0;x<8;x++) {
		if (mp->sq[y][x]==2 && bd->sq[y][x]==piece) {
		  if ((fx==-1 || fx==x) && (fy==-1 || fy==y)) {
                    bd->sq[y][x]=0;
                    get_pieces(bd);
		    if (piece==6) {bd->kpos[0]=tx;bd->kpos[1]=ty;}
		    if (piece==-6) {bd->kpos[2]=tx;bd->kpos[3]=ty;}
		    if (chck(bd,side)==0) {fy=y;fx=x;} /* Accept this move */
		    if (piece==6) {bd->kpos[0]=x;bd->kpos[1]=y;}
		    if (piece==-6) {bd->kpos[2]=x;bd->kpos[3]=y;}
		    bd->sq[y][x]=piece;
		  }
		}
	      }
	    }
	    bd->sq[ty][tx]=opt;
	    if (fx==-1 || fy==-1) {
	      fprintf(stderr,"Problem in PGN - Move %s unresolved\n",temp);
	    }
	  }
	}

        /* Make up the move */
        m.fx=fx;m.fy=fy;
        m.tx=tx;m.ty=ty;

        /* Do the move */
	if (fx==-1 || fy==-1 || tx==-1 || ty==-1 || piece==0) {
	  ok=0;
	  if (strstr(temp,"1-0")) ok=1;
	  if (strstr(temp,"0-1")) ok=1;
	  if (strstr(temp,"1/2-1/2")) ok=1;
	  if (ok==0) { 
	    fprintf(stderr,"Error in parsing move %s!\n",temp);
	    free(bd);free(mp);
	    *num=0;
	    return NULL;
	  }
	}
        else {
	  do_move2(bd,&m,side);
	   /* Add the move to the list */
/*
	  printmove(&m);
	  fprintf(stderr,"\n");
*/
	  if (mno==1) mlst = malloc(sizeof(move));
	  else mlst = realloc(mlst,sizeof(move)*mno);
	  *(mlst+mno-1)=m;
	  side=3-side;
	  mno++;
	  if (breakout==1) breakout=2;
	}
      }

      memset(temp, 0, sizeof(temp));
      i=0;
    }
    else i++;
  } while (breakout!=2 && ch !='[' && !feof(fp));
  if (breakout==2) while (ch!='\n' && !feof(fp)) (void)fscanf(fp,"%c",&ch); 
  free(bd);
  free(mp);
  *num=mno-1;
  return mlst;
}

 /* Load in a PGN game and annotate it */
void annotate_game(openpos *openings,int book_on,int learn_new) {
  int n,i,side=1,mvno=1,cp[4],num=0,gameno,total,limit,frombook=0;
  int writeout=0, accept=0;
  char nam[51],nm[FILENAME_MAX],cd_path[FILENAME_MAX],ch,*fen;
  FILE *fp=NULL,*opf=NULL;
  move *mlst,*m,*best;
  board *bd;
  compdat *cd=&cdd[0];
  double score=0.0;
  openpos *op=NULL,*optemp=NULL;
  movelst *mvl;
   
  strcpy(cd_path,dir_command);
  strcat(cd_path," ");
  strcat(cd_path,path);
  strcat(cd_path,"*.pgn");
  printf("\nDirectory of *.pgn Files\n\n");
  (void)system(cd_path);
  printf("\nPlease Input Name:-");
  do {
    (void)scanf("%s",nam);
    if (!strstr(nam,".pgn")) strcat(nam,".pgn");
    if (strlen(nam)>30 || strlen(nam)<5) printf("   Illegal Name!");
  } while (strlen(nam)>30 || strlen(nam)<5);
  strcpy(nm,path);
  strcat(nm,nam);
  if ((fp = fopen(nm,"r"))==NULL)  {
    printf("\n!!! Cannot Open File !!!\n");
    (void)system("cd ..");
    return;
  }
   
   /* Count the games */
  total=0;
  fprintf(stderr,"Importing ... ");
  do {
    (void)fscanf(fp,"%c",&ch); 
  } while (ch!='[');
  do {
    do {
      do {
	(void)fscanf(fp,"%c",&ch);
      } while (ch!='\n');
      (void)fscanf(fp,"%c",&ch);
    } while (ch=='[');
    total++;
    do {
      do {
	(void)fscanf(fp,"%c",&ch);
      } while (ch!='\n');
      (void)fscanf(fp,"%c",&ch);
    } while (ch!='[' && !feof(fp));
  } while (ch=='[');
   
   /* Reset fp */
  fprintf(stderr,"OK\n");
  (void)fclose(fp);
  fp = fopen(nm,"r");
   
   /* Get desired game number */
  if (total>1) {
    do {
      fprintf(stderr,"Load Which Game Number? [1-%d, 0=all] : ",total);
      (void)fscanf(stdin,"%d",&gameno);
    } while (gameno<0 || gameno>total);
  }
  else gameno=1;
   /* Skip unwanted games */
  for (i=1;i<gameno;i++) {
    do {
      do {
	(void)fscanf(fp,"%c",&ch);
      } while (ch!='\n');
      (void)fscanf(fp,"%c",&ch);
    } while (ch=='[');
    do {
      do {
	(void)fscanf(fp,"%c",&ch);
      } while (ch!='\n');
      (void)fscanf(fp,"%c",&ch);
    } while (ch!='[');
  }
   
  mlst=load_pgn_game(fp,&num);
  if (!mlst) {
    fprintf(stderr,"Error loading game!");
    (void)fclose(fp);
    return;
  }
  if (gameno>0) fprintf(stderr,"Game Loaded OK!\n");
  else fprintf(stderr,"Game 1 Loaded OK!\n");

  if (learn_new==1) {
    fprintf(stderr,"(A)ccept Given Move or (R)e-evaluate? : ");
    do {
      (void)fscanf(stdin,"%c",&ch);
      ch=toupper(ch);
    } while (ch!='A' && ch!='R');
    if (ch=='A') accept=1;
    else accept=0;
  }

  if (accept==0) {
    fprintf(stderr,"Maximum Thinking Time per Move (s) : ");
    do {
      (void)fscanf(stdin,"%d",&limit);
    } while (limit<1 || limit > 36000);
  }
   
  fprintf(stderr,"Annotate to a file? [Y/N] : ");
  do {
    (void)fscanf(stdin,"%c",&ch);
    ch=toupper(ch);
  } while (ch!='N' && ch!='Y');

  if (ch=='Y') { 
    strcpy(cd_path,dir_command);
    strcat(cd_path," ");
    strcat(cd_path,path);
    strcat(cd_path,"*.ann");
    printf("\nDirectory of *.ann Files\n\n");
    (void)system(cd_path);
    printf("\nPlease Input Name:-");
    do {
      (void)scanf("%s",nam);
      if (!strstr(nam,".ann")) strcat(nam,".ann");
      if (strlen(nam)>30 || strlen(nam)<5) printf("   Illegal Name!");
    } while (strlen(nam)>30 || strlen(nam)<5);
    strcpy(nm,path);
    strcat(nm,nam);
    if ((opf = fopen(nm,"w"))==NULL)  {
      printf("\n!!! Cannot Create File !!!\n");
      (void)system("cd ..");
      (void)fclose(fp);
      return;
    }
    fprintf(opf,"ColChess Game Annotation\n");
    fprintf(opf,"ColChess Version "VER", %d Seconds per Move\n\n",limit);
    writeout=1;
    if (gameno==0) fprintf(opf,"Game 1\n\n");
  }
   
  if (gameno>0) total=1;
  for (gameno=0;gameno<total;gameno++) {
    side=1;mvno=1;
    bd=reset_board();
    cp[0]=cp[1]=cp[2]=cp[3]=1;
    if (gameno>0) {
      if (writeout) {
	opf = fopen(nm,"a");
	fprintf(opf,"\nGame %d\n\n",gameno+1);
      }
      mlst=load_pgn_game(fp,&num);
      if (!mlst) {
        fprintf(stderr,"Error loading game %d!",gameno+1);
	(void)fclose(fp);
	if (writeout==1) (void)fclose(opf);
        return;
      }
      fprintf(stderr,"\n\nGame %d Loaded OK!\n=================",gameno+1);
      if (gameno>9) fprintf(stderr,"=");
      if (gameno>99) fprintf(stderr,"=");
      fprintf(stderr,"\n\n");
    }

    for (i=0;i<num;i++) {
      m=(mlst+i);
      /* Now do the comp analysis  */
      cd->skill=12;
      cd->quiesce=1;
      cd->side=side;
      cd->gamestage=pts(1,bd)+pts(2,bd);
      for (n=0;n<4;n++) cd->cp[n]=cp[n];
      if (i>0) cd->last=(mlst+i-1);
      else cd->last=NULL;
      get_pieces(bd);
      cd->tlimit=limit*10;
      cd->extend=0; 
      cd->inchk=chck(bd,side);
      get_poslist(cd,-1,bd);
      fprintf(stderr,"\nNow Analysing ");
      if (side==WHITE) fprintf(stderr,"White's Move Number %d; ",mvno);
      if (side==BLACK) fprintf(stderr,"Black's Move Number %d; ",mvno);
      printmove(m);
      fprintf(stderr,"\n");
      print_board(bd);
      if (writeout==1) {
        if (side==WHITE) fprintf(opf,"%d. ",mvno);
        else {
	  fprintf(opf,"   ");
	  if (mvno>9) fprintf(opf," ");
	  if (mvno>99) fprintf(opf," ");
        }
        fprintf(opf,"%c%d%c%d  ",(char)(m->fx+97),8-m->fy,(char)(m->tx+97),8-m->ty);
      }
      best=NULL;

       /* Do search */
      frombook=0;
      if (openings && book_on) frombook=check_opening_exists(bd,side,cp,cd->last,m,openings);
      if (frombook==0) {
        if (openings && book_on) best=check_opening(bd,side,cp,cd->last,openings);
	if (accept==0) {
          if (!best) best=comp(bd,14,1,1,1);
          else frombook=1;
	}
	else {
	  if (best) frombook=1;   /* Position found in opening book, move _not_ found */
	}
      }
     
      if (best) {
        score=(double)best->score;
        score/=100.0;
      }
      if (!best || compare(best,m)==1) {
        if (frombook==0) {
	  if (accept==0) fprintf(stderr,"ColChess Agrees with the move played!\n");
	  else fprintf(stderr,"ColChess Does not Know this Position!\n");
	}
        else if (frombook==1) fprintf(stderr,"ColChess found this move in its opening book!\n");
        else fprintf(stderr,"ColChess found this move in its opening book with a zero score!\n");
        if (writeout==1) {
 	  if (frombook==0) fprintf(opf,"(%.2f)\n",score);
	  else if (frombook==1) fprintf(opf,"(Book Move)\n");
	  else fprintf(opf,"(Zero Score Book Move)\n");
        }
      } 
      else {
        fprintf(stderr,"ColChess Disagrees with the move played!\n");
        fprintf(stderr,"Suggested Best Move = ");
        printmove(best);
        if (frombook==0) fprintf(stderr,"  (%.2f)\n",score);
        else if (frombook==1) fprintf(stderr,"  (Book Move)\n");
        else  fprintf(stderr,"  (Zero Score Book Move)\n");
        if (writeout==1) {
	  fprintf(opf,"{ %c%d%c%d ",(char)(best->fx+97),8-best->fy,(char)(best->tx+97),8-best->ty);
	  if (frombook==0) fprintf(opf," (%.2f) }\n",score);
	  else if (frombook==1) fprintf(opf," (Book Move) }\n");
	  else fprintf(opf," (Zero Score Book Move) }\n");
        }
      }
    
       /* Learn position if new & using 'study' mode */
      if (frombook==0 && learn_new==1) {
	fprintf(stderr,"Learning New Position\n");
        fen=board_to_fen(bd,side,cp,cd->last);
	if (writeout==1) {
  	  fprintf(opf,"%s\n",fen);
	  fprintf(opf," - Adding to Opening Book!\n");
	}
	op=openings;
	while (op) {
	  optemp=op;
	  op=op->next;
	}
	op=malloc(sizeof(openpos));
        assert(op!=NULL);
	op->next=NULL;
	if (optemp) optemp->next=op;
	else openings=op;
	op->fen=malloc(sizeof(char)*(size_t)(strlen(fen)+1));
	op->comments=NULL;
	assert(op->fen!=NULL);
	strcpy(op->fen,fen);
	op->moves=malloc(sizeof(movelst));
	assert(op->moves!=NULL);
	mvl=op->moves;
	mvl->next=NULL;
	if (accept==0) mvl->mv=*best; /* Move was reevaluated by ColChess */
	else mvl->mv=*m;         /* Accepting existing move */
	(mvl->mv).score=1;
	fprintf(stderr,"Saving Opening Book!\n");
        save_openings(openings);
	fprintf(stderr,"Opening Book Saved OK!\n");
	free(fen);
	if (best) free(best);
	break;
      }
       /* Do the actual move and proceed */

      if (best) free(best);
     
      do_move2(bd,m,side);
      if (m->castle!=0) {
        if (side==WHITE) cp[0]=cp[1]=0;
        else cp[2]=cp[3]=0;
      }
      if (bd->sq[m->ty][m->tx]==6) cp[0]=cp[1]=0;
      if (bd->sq[m->ty][m->tx]==-6) cp[2]=cp[3]=0;
      if (m->fy==7 && m->fx==0) cp[1]=0;
      if (m->fy==7 && m->fx==7) cp[0]=0;
      if (m->fy==0 && m->fx==0) cp[3]=0;
      if (m->fy==0 && m->fx==7) cp[2]=0;
      side=3-side;
      if (side==WHITE) mvno++;
    }
    free(mlst);
    free(bd);
     /* Make sure that work so far is stored in case of ^C
      * File is reopened for appending if necessary */
    if (writeout==1) (void)fclose(opf);
  }
  (void)fclose(fp);
}

/* Short, cute, and totally unrigorous. */
void free_mvlst(movelst *p) {
  while(p) {
    free_mvlst(p->next);
    free(p);
    p=NULL;
  }
}
