#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <assert.h>

#include "chess.h"
#include "misc.h"
#include "checks.h"
#include "comp.h"
#include "moves.h"
#include "mprocs.h"
#include "eval.h"

#ifdef _WIN32
#include <signal.h>
#else
#include <sys/signal.h>
#endif

#define randomize()    srand((unsigned)time(NULL))

#ifdef _WIN32 /*  WINDOWS MODE **** */
char path[FILENAME_MAX] = ""; /*  <--- PATH TO CHESS DIRECTORY FOR WINDOWS. USE DOUBLE SLASHES!! */
char dir_command[10] = "dir";
char sep[3] = "\\";

#else         /*  UNIX MODE **** */
char path[FILENAME_MAX] = "./"; /*  <--- PATH TO CHESS DIRECTORY FOR UNIX */
char dir_command[10] = "ls";
char sep[3] = "/";
#endif

#ifdef _WIN32
const char ColChessVersion[] = "ColChess Version "VER".  Windows version, possibly modified by Dann Corbit.";
#else
const char ColChessVersion[] = "ColChess Version "VER".  Official Linux version by Colin Frayn.";
#endif


char gamepath[FILENAME_MAX];
/*  Position score weights */
int posdat[6][8][8];
int posdat_pawn[8][8];
/*  Random number table for hashing function */
long int rtable[8][8][13];
/* Xboard mode ON/OFF */
int xboardmode=0;
/*  Precalculated data */
int centre_board[8][8],rank_bonus[8],rank_bonus_half[8],early_queen_penalty[8];
int piece_value[7],piece_value_10[7],piece_value_100[7];
int halved[1000],squared[20],rtable2[8][8][13];
int file_bonus[8],file_bonus2[8],corner_board[8][8],initscore;
tosquare temptsq[30];
board defsq,defsqop;
compdat cdd[MAX_PV];
move mvlst[1024];

int main (int argc, char **argv) {
  int sq[8][8],clock_left[2],timeron=0,post=0,proper=1,cdepth=0,hint=0;
  int oside=1,side=1,mvno=1,automv=0,quiesced=1,cp[4],cmvno,cside,epx=0,epy=0;
  int a=0,b,c,cdm=0,death=0,depth=5,state=0,pawn=1,learn_book_on=1;
  int skillw=12,skillb=12,ocdm=0,posn=0,tlimit=10,prom=0,book_on=1,computer_go=1;
  int fx=0,fy=0,tx=0,ty=0,ptm,tmt,nm,np,mem=14,computer_side=0,frombook=0;
  long int time_temp;
  struct gamedat *game;
#ifdef _WIN32
  miscdata inidata;
#endif
  board *mp,*sqq,*temp;
  move *best,*mv=NULL,*pm;
  compdat *cd;
  openpos *openings;
  char input[100],ch,*chh,*pathloc=NULL,tempch[100],*winloc=NULL;
  time_t beg,end;
   
#ifdef _WIN32
  fprintf(stderr,"Attempting to load .ini file colchess.ini\n");
  inidata=load_ini_file(NULL);
  if (inidata.ok!=1) {
    winloc = getenv("windir");
    if (winloc != NULL) {
      fprintf(stderr,"Attempting to load .ini file %s\\colchess.ini\n",winloc);
      inidata=load_ini_file(winloc);
    }
  }
  a=inidata.ok;
  if (a==1) {
    mem=inidata.hash;
    learn_book_on=inidata.learn;    
    book_on=inidata.book;
  }
#endif
  if (a==0 || winloc==NULL) {
    pathloc = getenv("COLCHESS_PATH");
    if (pathloc != NULL) {
      fprintf(stderr,"Using Path %s\n",pathloc);
      if (strlen(pathloc) < sizeof(path)) strcpy(path, pathloc);
    }
  }
    
  clock_left[0]=clock_left[1]=300;
  strcpy(gamepath,path);
  strcat(gamepath,"games");
  strcat(gamepath,sep);
  beg=time(NULL);

   /* Output the current version number and type */
  (void)puts(ColChessVersion);
   
   /*  Help on command line options. */
  if (argc>1 && (!strcmp(argv[1],"?") || !strcmp(argv[1],"HELP") || !strcmp(argv[1],"help"))) {
    fprintf(stderr,"Command Line Options;\n---------------------\n");
    fprintf(stderr,"CHESS Q<quiescence depth> H<hash size> D<search depth> L<Time limit>\n");
    fprintf(stderr,"SW<White Player Skill> SB<Black Player Skill> DEATH WHITE BLACK BOTH\nLEARN NOLEARN BOOK NOBOOK\n\n");
    exit(0);
  }
  /*  Setup */
  randomize();
  mp=malloc(sizeof(board));
  sqq=malloc(sizeof(board));
  if (!mp || !sqq) {fprintf(stderr,"Memory Problems in chess.c!\n");exit(0);}

  temp=reset_board();
  memset(sqq->sq, 0, sizeof(sqq->sq));
  memset(mp->sq, 0, sizeof(mp->sq));
  memset(sq, 0, sizeof(sq));
  for (a=0;a<8;a++) for (b=0;b<8;b++) sq[a][b]=temp->sq[a][b];
  free(temp);

   /*  Does just what it says :) */
  setup_precalculated_data();

   /* Load in the opening book */
  openings=load_openings();
   
   /*  Setup castling permissions */
  cp[0]=cp[1]=cp[2]=cp[3]=1;

   /*  Parse command line options */
  if (argc>1) {
    for (a=1;a<argc;a++) {
      if (toupper((unsigned char)argv[a][0])=='Q') {quiesced=get_num_from_string(argv[a],1);if (quiesced<0) quiesced=0;if (quiesced>30) quiesced=30;}
      else if (toupper((unsigned char)argv[a][0])=='H') {mem=get_num_from_string(argv[a],1);if (mem<0) mem=0;if (mem>16) mem=16;}
      else if (strlen(argv[a])<4 && toupper((unsigned char)argv[a][0])=='D') {depth=get_num_from_string(argv[a],1);if (depth<2) depth=2;if (depth>40) depth=40;}
      else if (toupper((unsigned char)argv[a][0])=='L') {tlimit=get_num_from_string(argv[a],1);if (tlimit<0) tlimit=0;if (tlimit>3600) tlimit=3600;}
      else if (toupper((unsigned char)argv[a][0])=='S') {
        if (toupper((unsigned char)argv[a][1])=='W') {skillw=get_num_from_string(argv[a],2);if (skillw<0) skillw=0;if (skillw>12) skillw=12;}
        else if (toupper((unsigned char)argv[a][1])=='B') {skillb=get_num_from_string(argv[a],2);if (skillb<0) skillb=0;if (skillb>12) skillb=12;}
        else fprintf(stderr,"%s - Unrecognised option!\n",argv[a]);
      }
      else if (!strcmp(argv[a],"death") || !strcmp(argv[a],"DEATH")) death=1;
      else if (!strcmp(argv[a],"white") || !strcmp(argv[a],"WHITE")) computer_side=1;
      else if (!strcmp(argv[a],"black") || !strcmp(argv[a],"BLACK")) computer_side=2;
      else if (!strcmp(argv[a],"both") || !strcmp(argv[a],"BOTH")) computer_side=3;
      else if (!strcmp(argv[a],"learn") || !strcmp(argv[a],"LEARN")) learn_book_on=1;
      else if (!strcmp(argv[a],"nolearn") || !strcmp(argv[a],"NOLEARN")) learn_book_on=0;
      else if (!strcmp(argv[a],"book") || !strcmp(argv[a],"BOOK")) book_on=1;
      else if (!strcmp(argv[a],"nobook") || !strcmp(argv[a],"NOBOOK")) book_on=0;
      else fprintf(stderr,"%s - Unrecognised Option!\n",argv[a]);
    }
  }

  if (xboardmode==0) {
    fprintf(stderr,"\n\nWelcome to ColChess 2000!\n"); /*  Welcome the next victim :) */
    fprintf(stderr,"-------------------------\n");
  }
  setbuf(stdin, NULL);
   
   
                      /*  ---===   BEGIN MAIN LOOP!   ===---  */

  for (;;) {
    /*  Setup board *sqq using values of sq */
    for (a=0;a<8;a++) {
      for (b=0;b<8;b++) {
  	sqq->sq[a][b]=sq[a][b];
	if (sq[a][b]==6) {sqq->kpos[0]=b;sqq->kpos[1]=a;}
	if (sq[a][b]==-6) {sqq->kpos[2]=b;sqq->kpos[3]=a;}
      }
    }
    get_pieces(sqq); /*  Count how many of each piece we have on the board */

    /*  Time Control */
	
    if (timeron && side!=oside) {
      end=time(NULL);
      clock_left[2-side]-=(end-beg);
      beg=time(NULL);
      oside=side;
      if (xboardmode==0) {
        fprintf(stderr,"White: %ds  Black: %ds\n",clock_left[0],clock_left[1]);
        if (clock_left[1]<=0) fprintf(stderr,"BLACK is out of time!\n");
        if (clock_left[0]<=0) fprintf(stderr,"WHITE is out of time!\n");
        if (clock_left[0]<=0 || clock_left[1]<=0) {
          timeron=0;computer_side=0;
        }
      }
    }

     /*  Print prompt & info. */
    if (xboardmode==0) {
      fprintf(stderr,"\n[%d]",mvno);
      if (side==WHITE) fprintf(stderr,"W");
      if (side==BLACK) fprintf(stderr,"B");
    }
        state=0;
        if (mvno>1 || side==BLACK) {
	  pm=(mvlst+(mvno*2)+side-4);
	  if (pm->fx>=0 && abs(sq[pm->ty][pm->tx])==1 && abs(pm->fy-pm->ty)==2) {epx=pm->fx;epy=(pm->ty+pm->fy)/2;}
          else {epx=epy=-1;}
	}
        else {epx=epy=-1;}
	if (chck(sqq,side)!=0) {
          state=1;
	  if (chkmte(side,sqq,-1,epx,epy)==1) {
	    if (xboardmode==0) fprintf(stderr,"#"); /*  If currently in CM */
	    automv=0;state=2;computer_side=0;
	  }
          else if (xboardmode==0) fprintf(stderr,"+"); /*  If currently in check */
	}
	if (state==0 && chkmte(side,sqq,-1,epx,epy)==1) {
	  if (xboardmode==0) fprintf(stderr,"$"); /*  If in stalemate - I guessed the symbol I should use :) */
	  automv=0;state=3;computer_side=0;
	}
        if (xboardmode==0) fprintf(stderr,":>"); /*  Print a friendly prompt :) */

        /*  Get input string */
        if (computer_go==0 || (side!=computer_side && computer_side!=3)) {
	  if (automv==0) (void)fgets(input,MAX_INPUT_LENGTH,stdin); /*  Get input from user */
	  input[strlen(input)-1]='\x0';   /*  Sort out end of line characters */
	  if (automv>0) strcpy(input,"COMP"); /*  Still automoving */
          if (!strcmp(input,"")) continue; /*  NULL input */
	}
        else strcpy(input,"COMP"); /*  Computer's turn */
         /* Suggest Hint */
        if (!strcmp(input,"HINT") || !strcmp(input,"hint")) {strcpy(input,"COMP");hint=1;}
        else hint=0;
        if (!strcmp(input,"UNDO") || !strcmp(input,"undo")) strcpy(input,"RETAKE");
         /* Clear the input stream */
        (void)fflush(stdin);

        /*  If castling Queen's side. Ladies first. */
	if (!strcmp("0-0-0",input) || !strcmp("o-o-o",input) || (side==WHITE && cp[1]==1 && (!strcmp("e1c1",input) || !strcmp("E1C1",input))) || (side==BLACK && cp[3]==1 && (!strcmp("e8c8",input) || !strcmp("E8C8",input)))) {
	  if (side==WHITE) {
	    if (castleok(1,2,sqq,cp[0],cp[1])==1) {
              sq[7][4]=0;sq[7][2]=6;
	      sq[7][0]=0;sq[7][3]=2;
	      add_move(1,mvno,4,7,2,7,2,0,0,0);
              if (xboardmode==1) computer_go=1;
	      cp[0]=cp[1]=0;
	      side=2;
	    }
	    else error(1);
	    continue;
	  }
	  else if (side==BLACK) {
	    if (castleok(2,2,sqq,cp[2],cp[3])==1) {
              sq[0][4]=0;sq[0][2]=-6;
	      sq[0][0]=0;sq[0][3]=-2;
	      add_move(2,mvno,4,0,2,0,2,0,0,0);
	      cp[2]=cp[3]=0;
	      side=1;mvno++;
	    }
	    else error(1);
	    continue;
	  }
	  if (xboardmode==0) fprintf(stderr,"Castling QS");
	}
        /*  .. or castling King's side. */
	if (!strcmp(input,"0-0") || !strcmp(input,"o-o") || (side==WHITE && cp[0]==1 && (!strcmp("e1g1",input) || !strcmp("E1G1",input))) || (side==BLACK && cp[2]==1 && (!strcmp("e8g8",input) || !strcmp("E8G8",input)))) {
          if (side==WHITE) {
            if (castleok(1,1,sqq,cp[0],cp[1])==1) {
              add_move(1,mvno,4,7,6,7,1,0,0,0);
              if (xboardmode==1) computer_go=1;
              sq[7][4]=0;sq[7][6]=6;
              sq[7][7]=0;sq[7][5]=2;
              cp[0]=cp[1]=0;
              side=2;
            }
            else error(1);
            continue;
	  }
	  if (side==BLACK) {
	    if (castleok(2,1,sqq,cp[2],cp[3])==1) {
              sq[0][4]=0;sq[0][6]=-6;
	      sq[0][7]=0;sq[0][5]=-2;
	      add_move(2,mvno,4,0,6,0,1,0,0,0);
	      cp[2]=cp[3]=0;
              side=1;mvno++;
	    }
	    else error(1);
	    continue;
	  }
	  if (xboardmode==0) fprintf(stderr,"Castling KS");
	}
        /*  Automatic En-passant capture */
        if (!strcmp("EP",input) || !strcmp("ep",input)) {
          a=(mvno*2)-2;
          if (side==WHITE) a--;
          if (a<0) {error(4);continue;}
          mv=(mvlst+a);
          a=en_passant_possible(sqq,mv,side);
          if (a==-1) {error(4);continue;}
          if (a==2) {
            fprintf(stderr,"    Take From Which File? (%c or %c)  ",(char)(mv->tx+64),(char)(mv->tx+66));
            do {
	      (void)scanf("%c",&ch);
              ch=(char)toupper((unsigned char)ch);
            } while ((int)ch!=mv->tx+64 && (int)ch!=mv->tx+66);
            if ((int)ch==mv->tx+64) a=0;
            else a=1;
          }
          if (a==0) fx=mv->tx-1;
          if (a==1) fx=mv->tx+1;
          tx=mv->tx;
          fy=side+2;
          if (side==WHITE) {ty=2;pawn=1;}
          if (side==BLACK) {ty=5;pawn=-1;}
  	  sqq->sq[ty][tx]=pawn;
	  sqq->sq[fy][fx]=0;
          if (a==0) sqq->sq[fy][fx+1]=0;
          if (a==1) sqq->sq[fy][fx-1]=0;
          b=chck(sqq,side);
	  if (b!=0) {error(3);continue;}
	  sq[ty][tx]=pawn;
	  sq[fy][fx]=0;
          if (a==0) {sq[fy][fx+1]=0;a--;}
          if (a==1) sq[fy][fx-1]=0;
  	  add_move(side,mvno,fx,fy,tx,ty,0,a,0,-pawn);

	  a=test_check(3-side,sqq,-1,-1);
	  if (a==1 && learn_book_on && proper==1) learn_opening(3-side,mvno,openings,0);
				  
	  side=3-side;
	  if (side==WHITE) mvno++;
          fprintf(stderr,"You Capture En-Passant!\n");
          continue;
        }
        /*  Get help list */
	if (!strcmp("HELP",input) || !strcmp("help",input)) {
	  help();
	  continue;
	}
         /*  Computer commands help */
        if (!strcmp("COMPHELP",input) || !strcmp("comphelp",input)) {
	  help_comp();
	  continue;
	}
         /*  Analysis help */
        if (!strcmp("ANALYHELP",input) || !strcmp("analyhelp",input)) {
	  help_analy();
	  continue;
	}
        /*  About the game */
	if (!strcmp("ABOUT",input) || !strcmp("about",input)) {
	  about();
	  continue;
	}
        /*  Show credits */
	if (!strcmp("CREDITS",input) || !strcmp("credits",input)) {
	  credits();
	  continue;
	}
        /*  Show board */
	if (!strcmp("BOARD",input) || !strcmp("board",input) || !strcmp("b",input) || !strcmp("B",input)) {
	  print_board(sqq);
	  continue;
	}
        /*  List danger squares (sum up total number of pieces */
        /*      defending each piece and subtract the number of */
        /*      pieces attacking them.) */
	if (!strcmp("DANGER",input) || !strcmp("danger",input) || !strcmp(input,"d") || !strcmp(input,"D")) {
	  fprintf(stderr,"\nDanger Squares\n--------------\n");
	  temp=sdef(side,sqq);
	  for (a=0;a<8;a++) {
		for (b=0;b<8;b++) {
		  if (temp->sq[a][b]>-1) fprintf(stderr," ");
		  fprintf(stderr,"%d ",temp->sq[a][b]);
		}
                fprintf(stderr,"\n");
	  }
	  free(temp);
	  continue;
	}
	/*  Computer player side options */
        if (!strcmp(input,"WHITE") || !strcmp(input,"white")) {computer_side=1;fprintf(stderr,"ColChess Plays White!\n");continue;}
        if (!strcmp(input,"BLACK") || !strcmp(input,"black")) {computer_side=2;fprintf(stderr,"ColChess Plays Black!\n");continue;}
        if (!strcmp(input,"COFF") || !strcmp(input,"coff")) {computer_side=0;fprintf(stderr,"Computer Autoplay OFF!\n");continue;}
        if (!strcmp(input,"BOTH") || !strcmp(input,"both")) {computer_side=3;fprintf(stderr,"ColChess Playing Both Sides!\n");continue;}
        /*  Give up.  ColChess was too good for you :) */
	if (!strcmp(input,"RESIGN") || !strcmp("resign",input)) {
		if (side==WHITE) fprintf(stderr,"\nWhite Resigns!");
		else fprintf(stderr,"\nBlack Resigns!");
		fprintf(stderr,"\n\nPress A Key!");
 	        free(mp);
            	free_opening(openings);
		free(sqq);
		bye();
		return 1;
	}
	/*  Leave the game without the dishonour of resignation */
	if (!strcmp(input,"EXIT") || !strcmp("exit",input) || !strcmp(input,"QUIT") || !strcmp("quit",input)) {
	   free(mp); free(sqq);
	   free_opening(openings);
	   bye();return 1;}
        /*  Reset the game and start again.  Useful if you're losing :) */
	if (!strcmp(input,"RESET") || !strcmp("reset",input) || !strcmp(input,"NEW") || !strcmp("new",input)) {
	  temp=reset_board();
	  memset(mp->sq, 0, sizeof(mp->sq));
          for (a=0;a<8;a++) {
            for (b=0;b<8;b++) {
              sq[a][b]=temp->sq[a][b];
            }
          }
          free(temp);
          cp[0]=cp[1]=cp[2]=cp[3]=1;
          mvno=side=1;
	  clock_left[0]=clock_left[1]=300;
	  proper=1;depth=5;
          timeron=0;
          if (xboardmode==1) {computer_side=2;computer_go=1;}
          else fprintf(stderr,"\nNew Game!\n---------\n");
          continue;
        }
        /*  Input FEN position */
	if (!strcmp(input,"FEN") || !strcmp("fen",input)) {
	  fprintf(stderr,"Input FEN Position: "); 
	  (void)fscanf(stdin,"%s",input);
	  temp=get_board_fen(input);
	  memset(mp->sq, 0, sizeof(mp->sq));
          for (a=0;a<8;a++) {
            for (b=0;b<8;b++) {
              sq[a][b]=temp->sq[a][b];
            }
          }
          free(temp);
          cp[0]=cp[1]=cp[2]=cp[3]=0;
	  mvno=1;

	  fprintf(stderr,"\n(W)hite or (B)lack? ");
          a=0;
	  do {
	    (void)scanf("%c",&ch);
            ch=(char)toupper((unsigned char)ch);
	    if (ch=='W') a=1;
	    if (ch=='B') a=2;
	  } while (a!=1 && a!=2);

	  fprintf(stderr,"\nInput Castling Permissions [KQkq or -] : ");
	  do {
	    (void)scanf("%s",tempch);
	  } while ((int)strlen(tempch)==0 || (int)strlen(tempch)>4);
	  if (strchr(tempch,'K')) cp[0]=1;
	  if (strchr(tempch,'Q')) cp[1]=1;
	  if (strchr(tempch,'k')) cp[2]=1;
	  if (strchr(tempch,'q')) cp[3]=1;

          side=a;
	  if (side==BLACK) mvlst[0].fx=-1;
	  clock_left[0]=clock_left[1]=300;
	  proper=0;
          continue;
        }
        /*  Save the game */
	if (!strcmp(input,"SAVE") || !strcmp("save",input)) {
	  game=malloc(sizeof(struct gamedat));
          if (game==NULL) {fprintf(stderr,"ERROR IN SAVE_GAME!\n");exit(0);}
	  game->mvno=mvno;
	  game->cdm=cdm;
	  game->skillw=skillw;
	  game->skillb=skillb;
	  game->death=death;
	  game->side=side;
          game->depth=depth;
          game->quiesce=quiesced;
	  game->proper=proper;
	  savegame(sq,game,cp);
	  free(game);
	  continue;
	}
        /*  Load a saved game */
	if (!strcmp(input,"LOAD") || !strcmp("load",input)) {
	  game=loadgame();
	  if (game) {
	    mvno=game->mvno;
	    cdm=game->cdm;
	    skillw=game->skillw;
	    skillb=game->skillb;
	    death=game->death;
	    side=game->side;
            depth=game->depth;
	    quiesced=game->quiesce;
	    proper=game->proper;
            for (a=0;a<8;a++) {
              for (b=0;b<8;b++) sq[a][b]=(game->sq[a][b]);
	    }
            for (a=0;a<4;a++) cp[a]=(game->cp[a]);
            nm=(mvno*2)-1;
            if (side==WHITE) nm--;
	    free(game);
	  }
	  else proper=0;
	  continue;
	}
        /*  Load a saved game from PGN format*/
	if (!strcmp(input,"IMPORT") || !strcmp("import",input)) {
	  game=import_pgn();
	  if (game) {
	    mvno=game->mvno;
	    cdm=death=0;
	    skillw=skillb=12;
	    side=game->side;
            depth=5;
	    quiesced=1;
	    proper=1;
            for (a=0;a<8;a++) {
              for (b=0;b<8;b++) sq[a][b]=(game->sq[a][b]);
	    }
            for (a=0;a<4;a++) cp[a]=(game->cp[a]);
	    free(game);
	  }
	  else proper=0;
	  continue;
	}
	/*  Export game to .pgn format */
        if (!strcmp(input,"EXPORT") || !strcmp(input,"export")) {
	  export_game(mvno,side);
          continue;
	}
	/*  Annotate a PGN game to a file */
        if (!strcmp(input,"ANNOTATE") || !strcmp(input,"annotate")) {
	  annotate_game(openings,book_on,0);
          continue;
	}
	/*  Learn Openings from (a) PGN game(s) */
        if (!strcmp(input,"STUDY") || !strcmp(input,"study")) {
	   /* OK so it uses the same function ;) */
	  annotate_game(openings,book_on,1);
          continue;
	}
        /*  Load in and cannibalise someone else's opening book :) */
        if (!strcmp(input,"SCANBOOK") || !strcmp(input,"scanbook")) {
	  scan_book(openings);
          continue;
	}
        /*  Set the computer skill level.  Separate for white and black. */
	if (!strcmp(input,"SKILL") || !strcmp(input,"skill")) {
	  fprintf(stderr,"    New Skill Level");
	  if (side==WHITE) fprintf(stderr," [%d] :- ",skillw);
	  if (side==BLACK) fprintf(stderr," [%d] :- ",skillb);
	  (void)scanf("%d",&a);
	  if (a>12) {
	    a=12;
	    fprintf(stderr,"    (Set at 12)");
	  }
          if (a<1) {
	    a=1;
	    fprintf(stderr,"    (Set at 1)");
	  }
	  if (side==WHITE) skillw=a;
	  if (side==BLACK) skillb=a;
	  continue;
	}
	/*  Setup hash table size.  Value is 0 (OFF) or */
	/*     log2 of size in kb. */
	if (!strcmp(input,"HASH") || !strcmp(input,"hash")) {
	  fprintf(stderr,"    Set Hash Table Size (0-16)");
	  fprintf(stderr," [%d] :- ",mem);
	  (void)scanf("%d",&a);
	  if (a>16) {
	    a=16;
	    fprintf(stderr,"    (Set at 16)");
	  }
          if (a<0) {
	    a=0;
	    fprintf(stderr,"    (Set at 0)");
	  }
          mem=a;
          if (a>0) fprintf(stderr,"\nSize %d = %d kbyte(s)",a,1<<a);
          else fprintf(stderr,"\nHash Table OFF");
	  continue;
	}
	/*  Set user-defined time limit for moves. */
	if (!strcmp(input,"LIMIT") || !strcmp(input,"limit")) {
	  fprintf(stderr,"    Set Time Limit");
	  fprintf(stderr," [%d] :- ",tlimit);
	  (void)scanf("%d",&a);
	  if (a>43200) {
	    a=43200;
	    fprintf(stderr,"    (Set at 12 Hours)");
	  }
          if (a<1) {
	    a=0;
	    fprintf(stderr,"    (Unlimited)");
	  }
	  tlimit=a;
	  continue;
	}
        /*  Set the computer quiescence depth. */
        /*  Note - capture quiescence depth is always infinite. */
	if (!strcmp(input,"QUI") || !strcmp(input,"qui")) {
	  fprintf(stderr,"    New Quiescence Depth [%d] :- ",quiesced);
	  (void)scanf("%d",&a);
	  if (a>30) {
	    a=30;
	    fprintf(stderr,"    (Set at 30)");
	  }
          if (a<0) {
	    a=0;
	    fprintf(stderr,"    (Set at 0)");
	  }
          quiesced=a;
	  continue;
	}
        /*  Make the computer move automatically when it has thought */
        /*     of a move. */
	if (!strcmp(input,"CMOVE") || !strcmp(input,"cmove")) {
	  cdm=1-cdm;
	  if (cdm==1) fprintf(stderr,"Computer Automatic Move ON");
	  if (cdm==0) fprintf(stderr,"Computer Automatic Move OFF");
	  continue;
	}
        /*  Toggle fight to the death.  If ON then computer doesn't resign. */
	if (!strcmp(input,"DEATH") || !strcmp(input,"death")) {
	  death=1-death;
	  if (death==1) fprintf(stderr,"Fight To The Death ON");
	  if (death==0) fprintf(stderr,"Fight To The Death OFF");
	  continue;
	}
        /*  Toggle Opening book ON/OFF */
	if (!strcmp(input,"BOOK") || !strcmp(input,"book")) {
	  book_on=1-book_on;
	  if (book_on) fprintf(stderr,"Opening Book ON");
	  else fprintf(stderr,"Opening Book OFF");
	  continue;
	}
         /*  Turn Opening Book Learning ON/OFF */
        if (!strcmp(input,"LEARN") || !strcmp(input,"learn")) {
	  learn_book_on=1-learn_book_on;
	  if (learn_book_on==1) fprintf(stderr,"Opening Book Learning ON");
	  else fprintf(stderr,"Opening Book Learning OFF");
	  continue;
	}
        /*  Set computer minimum search depth (in ply) */
	/*       Computer ALWAYS searches at least this deep. */
        if (!strncmp(input,"DEPTH",5) || !strncmp(input,"depth",5)) {
	  if (strlen(input)>5) {
	    depth=(int)getparam(input,1);
	    continue;
	  }
	  fprintf(stderr,"    New Search Depth [%d] :- ",depth);
	  (void)scanf("%d",&a);
	  if (a>40) {
            a=40;
	    fprintf(stderr,"    (Set at 40)");
	  }
	  if (a<2) {
	    a=2;
	    fprintf(stderr,"    (Set at 2)");
	  }
          depth=a;
	  continue;
        }
        /*  Setup computer automoves. */
       if (!strcmp("AUTO",input) || !strcmp("auto",input)) {
	 fprintf(stderr,"How Many Automoves? - ");
	 (void)scanf("%d",&automv);
	 ocdm=cdm;
	 cdm=1;
	 continue;
       }
        /*  Make the computer think of a move. */
       if (!strcmp("COMP",input) || !strcmp("comp",input)) {
         if (state==2) {fprintf(stderr,"You are in Checkmate!\n");continue;}
         if (state==3) {fprintf(stderr,"You are in Stalemate!\n");continue;}
	 if (xboardmode==0) fprintf(stderr,"Thinking ");
         np=0;
         for (a=0;a<8;a++) {
           for (b=0;b<8;b++) if (sq[a][b]!=0) np++;
         }
         cd=&cdd[0];
         if (side==WHITE) cd->skill=skillw;
         else cd->skill=skillb;
         if (timeron) cdepth=1;
         else cdepth=depth-1;
	 if (hint==1) cdepth=1;
         cd->side=side;
         cd->quiesce=quiesced;
         cd->gamestage=pts(1,sqq)+pts(2,sqq);
         for (a=0;a<4;a++) cd->cp[a]=cp[a];
         if (mvno==1 && side==WHITE) cd->last=NULL;
         else cd->last=(mvlst+((mvno*2)-4+side));
         get_pieces(sqq);
         get_poslist(cd,(mvno*2)-3+side,sqq);
         cd->inchk=chck(sqq,side);
         cd->extend=0;
         if (timeron) {
	   cd->tlimit=get_time_limit(mvno,clock_left[side-1]*10,clock_left[2-side]*10,pts(side,sqq),pts(3-side,sqq));
	   if (pts(side,sqq)>pts(3-side,sqq) && clock_left[side-1]<60 && clock_left[side-1]>10) cd->tlimit++;
	 }
         else cd->tlimit=tlimit*10;
	 if (hint==1) cd->tlimit=0;
         best=NULL;
	 frombook=0;

	 if (openings && book_on) best=check_opening(sqq,side,cp,cd->last,openings);
	 if (!best) best=comp(sqq,mem,post,0,cdepth);     /*  ----  COMP PROCEDURE  ----  */
         else frombook=1;

	 free_poslist(cd->once[0]);
	 free_poslist(cd->once[1]);
	 free_poslist(cd->twice[0]);
	 free_poslist(cd->twice[1]);
         if (xboardmode==1) {
	   if (hint==0 && death==0 && frombook==0 && cd->tlimit>10 && choose_resign(best->score,sqq,cd)) {fprintf(stderr,"resign\n");cdm=automv=0;computer_side=0;}
           else {
             if (hint==0 && best->fx==-1) fprintf(stderr,"1/2-1/2 {Insufficient Material}\n");
	     else {
	       if (hint==0) fprintf(stderr,"move ");
	       else fprintf(stderr,"Hint: ");
	       printmove2(best);
	       if (hint==0 && frombook) fprintf(stderr," <book %ld%%>",best->score);
	       fprintf(stderr,"\n");
	     }
	   }
	 }
	 else {
	   fprintf(stderr,"\nBest = ");
            /*  Want to resign? */
	   if (death==0 && frombook==0 && depth>4 && choose_resign(best->score,sqq,cd)) {fprintf(stderr,"Resign!\n");cdm=automv=0;computer_side=0;}
	   else if (best->fx==-1) {fprintf(stderr,"Draw!\n");cdm=automv=0;computer_side=0;}
	   else printmove(best);
	   if (frombook) fprintf(stderr," <Book %ld%%>\n",best->score);
	 }
	 if (cdm==1 || side==computer_side || computer_side==3) {  /*  Do move if req'd */
	   temp=do_move(sqq,best,side);
	   if (side==WHITE) {
	     if (best->castle!=0) cp[0]=cp[1]=0;
	     else {
	       if (best->fy==7 && best->fx==4) {cp[0]=cp[1]=0;}
	       if (best->fx==0 && best->fy==7) cp[1]=0;
      	       if (best->fx==7 && best->fy==7) cp[0]=0;
	     }
	   }
	   if (side==BLACK) {
             if (best->castle!=0) cp[2]=cp[3]=0;
	       else {
	         if (best->fy==0 && best->fx==4) {cp[2]=cp[3]=0;}
	         if (best->fx==0 && best->fy==0) cp[3]=0;
         	 if (best->fx==7 && best->fy==0) cp[2]=0;
	       }
	     }
       	     for (a=0;a<8;a++) {
	       for (b=0;b<8;b++) sqq->sq[a][b]=sq[a][b]=temp->sq[a][b];
	     }
	     for (a=0;a<4;a++) sqq->kpos[a]=temp->kpos[a];
	     free(temp);
	      /* Add move to the move list */
	     add_move(side,mvno,best->fx,best->fy,best->tx,best->ty,best->castle,best->ep,best->promote,best->capt);
	     side=3-side;
             if (side==WHITE) mvno++;
             /* Set up en-passant square */
             if (abs(sq[best->ty][best->tx])==1 && abs(best->fy-best->ty)==2) {epx=best->fx;epy=(best->ty+best->fy)/2;}
             else {epx=epy=-1;}
	      /* Test to see if move gives check/CM */
	     if (test_check(side,sqq,epx,epy)==1) {
	       automv=0;cdm=0;timeron=0;
               if (learn_book_on && proper==1) learn_opening(side,mvno,openings,0);
	     }
	       /* See if move gives draw */
             if (test_for_draw(side,mvno,sqq,1)==1) {computer_side=0;automv=0;}
	     if (xboardmode==0) {
	       print_board(sqq);  /*  Printout board after computer has moved. */
	       fprintf(stderr,"\n");
	     }
	  }
	  free(best);
	  if (automv==1) cdm=ocdm; /*  If computer has finished automoving then set cdm back to what it was */
	  if (automv>0) automv--; /*  Reduce the number of automoves left */
	  continue;
	}
        /*  RETAKE - Take back the last move. */
        /*  GOTO - go to the specified move. */
        /*    Both will screw up if board has been tampered with and a proper move */
	/*    list has not been generated.  Disabled in tournament games. */
	if ((!strcmp("RETAKE",input) || !strcmp("retake",input) || !strcmp("GOTO",input) || !strcmp("goto",input) || !strcmp(input,"r") || !strcmp(input,"R"))) {
	  if (timeron) {
	    fprintf(stderr,"This option is disabled in TIMER mode!\n");
	    continue;
	  }
	  if (mvno==1 && side==WHITE) {
	    fprintf(stderr,"First Move!!\n");
	    continue;
	  }
	  cmvno=mvno;
	  cside=side;
	  if (!strcmp("RETAKE",input) || !strcmp("retake",input) || !strcmp(input,"r") || !strcmp(input,"R")) {
	    c=1;
            if (side==WHITE) mvno--;
	    side=3-side;
	  }
	  else {
            c=0;
	    fprintf(stderr,"\nGoto Which Move? - ");
	    (void)scanf("%d",&a);
	    if (a<1 || a>mvno) {fprintf(stderr,"Illegal Entry");continue;}
	    mvno=a;
            fprintf(stderr,"\n(W)hite or (B)lack? ");
            a=0;
	    do {
	      (void)scanf("%c",&ch);
              ch=(char)toupper((unsigned char)ch);
	      if (ch=='W') a=1;
	      if (ch=='B') a=2;
  	    } while (a!=1 && a!=2);
  	    side=a;
	  }
          while (cside!=side || cmvno!=mvno) {
	    undo_move(sqq,(mvlst+(cmvno*2)-4+cside),3-cside);
	    cside=3-cside;
	    if (cside==BLACK) cmvno--;
	  }
	  for (b=0;b<8;b++) { /*  Setup sq properly */
	    for (c=0;c<8;c++) sq[b][c]=sqq->sq[b][c];
	  }
	  print_board(sqq); /*  Show new board position */
	  continue;
	}
        /*  Analyse position using various methods.  Includes defsc() static evaluation function */
	if (!strcmp("ANALY",input) || !strcmp("analy",input) || !strcmp(input,"a") || !strcmp(input,"A")) {
          cd=&cdd[0];
	  for (a=0;a<8;a++) for (b=0;b<8;b++) defsqop.sq[a][b]=defsq.sq[a][b]=0;
	  cd->side=side;
          cd->gamestage=pts(1,sqq)+pts(2,sqq);
	  cd->pts[0]=pts(1,sqq);
	  cd->pts[1]=pts(2,sqq);
	  for (a=0;a<4;a++) cd->cp[a]=cp[a];
	  memset(sqq->pawnfile, 0, sizeof(sqq->pawnfile));
	  for (a=0;a<8;a++) {
	    for (b=0;b<8;b++) {
	      if (sqq->sq[a][b]==1) sqq->pawnfile[b][0]++;
	      if (sqq->sq[a][b]==-1) sqq->pawnfile[b][1]++;
	    }
	  }
          posn=defsc(side,sqq,cd); /*  Calc. static position evaluation */
	  /*  Display results */
	  fprintf(stderr,"\nPosition on Turn %d\n",mvno);
	  fprintf(stderr,"\nYour Points = %d",pts(side,sqq));
	  fprintf(stderr,"\nOpp. Points = %d\n",pts(3-side,sqq));
	  fprintf(stderr,"\nOverall Position Score = %.2f Pawns",(double)(posn)/100.0);
	  fprintf(stderr,"\n\nOverall Position Analysis = ");
	  if (posn<-200 || posn>200) fprintf(stderr,"Very ");
	  if (abs(posn)< 50 && abs(posn)>30) fprintf(stderr,"Slightly ");
	  if (posn<-100) fprintf(stderr,"Bad\n");
	  if (posn>=-100 && posn<-30) fprintf(stderr,"Unfavourable\n");
	  if (posn>=-30 && posn<30) fprintf(stderr,"Even Sides\n");
	  if (posn>=30 && posn<100) fprintf(stderr,"Favourable\n");
	  if (posn>=100) fprintf(stderr,"Good\n");
	  continue;
	}
        /*  Generate move list so between specified limits. */
	if (!strcmp("MOVES",input) || !strcmp("moves",input)) {
	  fprintf(stderr,"\nStart at - ");
	  (void)scanf("%d",&a);
	  do {
		fprintf(stderr,"\nFinish at - ");
		(void)scanf("%d",&b);
		if (a<1) a=1;
		if (b>mvno) b=mvno;
		if (a>mvno) a=mvno;
		if (a>b) fprintf(stderr,"Illegal Entry");
	  } while (a>b);

	  for (c=a;c<=b;c++) {
		fprintf(stderr,"\n %d. ",c);
		if (c<100) fprintf(stderr," ");
		if (c<10) fprintf(stderr," ");
		if (c==mvno && side==WHITE) fprintf(stderr,".......");
		else {
		  mv=(mvlst+((c-1)*2));
		  printmove(mv);
		  fprintf(stderr,"       ");
		  if (c==mvno) fprintf(stderr,".......");
		  else {
			mv=(mvlst+((c*2)-1));
			printmove(mv);
		  }
		}
	  }
	  fprintf(stderr,"\n");
	  continue;
        }
        if (!strcmp("TEST",input) || !strcmp("test",input)) {
	  perform_epd_test(skillw,skillb,quiesced,mem,1);
          continue;
	}
        if (!strcmp("SELFTEST",input) || !strcmp("selftest",input)) {
	  perform_self_test(openings);
          continue;
	}
        if (!strcmp("EPDANALY",input) || !strcmp("epdanaly",input)) {
	  perform_epd_test(skillw,skillb,quiesced,mem,0);
          continue;
	}
        if (!strcmp("DFEN",input) || !strcmp("dfen",input)) {
	  if ((mvno>1 || side==BLACK)) mv=(mvlst+(mvno*2-4+side));
	  else mv=NULL;
          chh=board_to_fen(sqq,side,cp,mv);
	  fprintf(stderr,"FEN is: %s\n",chh);
	  free(chh);
          continue;
	}
     
        /*  --  TOURNAMENT MODE TIME CONTROLS          --  */
     
       if (!strcmp("TIMER",input) || !strcmp("timer",input)) {
	  timeron=1-timeron;
          if (timeron) fprintf(stderr,"TIMER ON!\n");
	  else fprintf(stderr,"TIMER OFF!\n");
	  if (timeron) beg=time(NULL);
	  else {
            end=time(NULL);
            clock_left[side-1]-=(end-beg);
	  }
	  continue;
	}
       if (!strcmp("CLOCK",input) || !strcmp("clock",input)) {
	  if (timeron) {
            end=time(NULL);
            clock_left[side-1]-=(end-beg);
	    beg=time(NULL);
	  }
	  fprintf(stderr,"TIMER");
	  if (timeron) fprintf(stderr," ON  ");
	  else fprintf(stderr," OFF  ");
	  fprintf(stderr,"White: %ds  Black: %ds\n",clock_left[0],clock_left[1]);
	  continue;
	}
       if (!strcmp("SET",input) || !strcmp("set",input)) {
          fprintf(stderr,"Which Player [W/B]? - ");
          do { 
	    (void)scanf("%c",&ch);
	    ch=(char)toupper((unsigned char)ch);
	  } while (ch!='W' && ch!='B');
	  fprintf(stderr,"New Clock Value - ");
	  (void)scanf("%d",&a);
	  if (ch=='W') clock_left[0]=a;
	  else clock_left[1]=a;
	  fprintf(stderr,"New Clock Value Set!\n");
	  continue;
	}

        /*  --  SPECIAL XBOARD COMMANDS  --  */
     
        if (!strcmp(input,"HARD") || !strcmp(input,"hard")) {  /*  Pondering ON signal */
          continue;
        }
        if (!strcmp(input,"EASY") || !strcmp(input,"easy")) { /*  Pondering OFF signal */
          continue;
        }
        if (!strcmp(input,"POST") || !strcmp(input,"post")) { /*  Post Current Thinking */
	  post=1;
          continue;
        }
        if (!strcmp(input,"NOPOST") || !strcmp(input,"nopost")) { /*  Post Current Thinking */
	  post=0;
          continue;
        }
        if (!strcmp(input,"EDIT") || !strcmp(input,"edit")) { /*  Edit Mode */
	  continue;
	}
        if (!strcmp(input,"COMPUTER") || !strcmp(input,"computer")) { /*  No idea what this does */
	  continue;
	}
        if (!strcmp(input,"FORCE") || !strcmp(input,"force")) { /*  Force to move */
	  computer_go=0;
	  computer_side=0;
          continue;
        }
        if (!strcmp(input,"RANDOM") || !strcmp(input,"random")) { /*  Force to move */
	  randomize();
          continue;
        }
        if (!strcmp(input,"ANALYZE") || !strcmp(input,"analyze")) { /*  Force to move */
           fprintf(stderr,"Error (unknown command): analyze");
	   continue;
        }
        if (!strcmp(input,"GO") || !strcmp(input,"go")) { /*  Timer ON signal */
	  timeron=1;
	  computer_go=1;
          continue;
        }
        if (!strcmp(input,"XBOARD") || !strcmp(input,"xboard")) { /*  Set XBOARD mode on */
	  depth=2;
	  book_on=1;
	  xboardmode=1;
          (void)signal(SIGINT, SIG_IGN);
	  fprintf(stderr,"\n");
	  continue;
        }
        if (!strncmp(input,"LEVEL",5) || !strncmp(input,"level",5)) { /*  Set time control stuff */
          continue;
	}
        if (!strncmp(input,"RESULT",6) || !strncmp(input,"result",6)) { /*  Game result */
	  if (strstr(input,"resigns") || strstr(input,"RESIGNS")) {
            if (learn_book_on && proper==1) {
	      if (strstr(input,"WHITE") || strstr(input,"white") || strstr(input,"White")) learn_opening(1,mvno-1,openings,0);
	      else learn_opening(2,mvno-1,openings,0);
	    }
	  }
          continue;
	}
	if (!strncmp(input,"TIME",4) || !strncmp(input,"time",4) || !strncmp(input,"OTIM",4) || !strncmp(input,"otim",4)) {
	  timeron=1;
	  time_temp=getparam(input,1);
	  b=(int)(time_temp/100);
	  if (b>0 && b<30000) {
	    if (input[0]=='T' || input[0]=='t') clock_left[2-side]=b;
            else clock_left[side-1]=b;
	  }
	  continue;
        }
        if (!strncmp(input,"SD",2) || !strncmp(input,"sd",2)) { /*  Set depth */
	  b=(int)getparam(input,1);
          if (b>2 && b<50) depth=b;
	  continue;
        }
        if (!strncmp(input,"ST",2) || !strncmp(input,"st",2)) { /*  Set time limit */
	  time_temp=getparam(input,1);
	  if (time_temp>1 && time_temp<=3600) tlimit=(int)time_temp;
	  continue;
        }

     
        /*  --  SPELLING MISTAKE? Just for a larf :))  --  */

        if (!strcmp(input,"DPETH") || !strcmp(input,"dpeth")) {
          fprintf(stderr,"I think perhaps you mean 'depth'?\n");
          continue;
        }
        if (!strcmp(input,"SVAE") || !strcmp(input,"svae")) {
          fprintf(stderr,"I think perhaps you mean 'save'?\n");
          continue;
        }
        if (!strcmp(input,"CMOP") || !strcmp(input,"cmop")) {
          fprintf(stderr,"I think perhaps you mean 'comp'?\n");
          continue;
        }
        /*  --  MUD COMMAND? :)  --  */

        if (!strcmp(input,"SCORE") || !strcmp(input,"score")) {
          fprintf(stderr,"I think you've been playing too much MUD! :)\n");
          continue;
        }
        if (!strcmp(input,"KILL") || !strcmp(input,"kill")) {
          fprintf(stderr,"I think you've been playing too much MUD! :)\n");
          continue;
        }
   
        /*  --  INPUT IS NOT A SPECIAL COMMAND!  --  */

        /*  Is input illegal command? */
        
        /* Possibly overspilled FEN data */
     
        if (!strcmp(input,"bm")) continue;
        if (input[0]=='-') continue;
        if (strchr(input,';')) continue;
     
        /*  Input wrong length */
	if (strlen(input)<4 || strlen(input)>5) {error(2);continue;}
        prom=0;
        if (strlen(input)==5 && input[2]!='-') { /*  Is either error or promotion */
	  if (toupper((unsigned char)input[4])=='R') prom=2;
	  else if (toupper((unsigned char)input[4])=='N') prom=3;
	  else if (toupper((unsigned char)input[4])=='B') prom=4;
	  else if (toupper((unsigned char)input[4])=='Q') prom=5;
	  else {error(2);continue;}
	}

        /*  No? Must be just a normal move. */
        /*  Probably. */

	fx=((int)toupper((unsigned char)input[0])-65);
	fy=56-(int)input[1];
	if (strlen(input)==5 && prom==0) {tx=((int)toupper((unsigned char)input[3])-65);ty=56-(int)input[4];}
        else {tx=((int)toupper((unsigned char)input[2])-65);ty=56-(int)input[3];}
   
	ptm=sq[fy][fx]; /*  Identifier of piece moving */
	tmt=sq[ty][tx]; /*  Target square */

        /*  Check that move is legal */
	if (fx<0 || fx>7 || fy<0 || fy>7) {error(1);continue;}
	if (tx<0 || tx>7 || ty<0 || ty>7) {error(1);continue;}
	if (side==WHITE && ptm<0) {error(1);continue;}
	if (side==BLACK && ptm>0) {error(1);continue;}
	if (ptm==0) {error(1);continue;}
	if (ptm<-6 || ptm>6) {error(1);continue;}

        memset(mp->sq, 0, sizeof(mp->sq));

	if (side==WHITE) {  /*  Put a 1 in all places piece can move to */
	  if (ptm==1) mppw(fx,fy,sq,mp);
	  if (ptm==2) mprw(fx,fy,sq,mp);
	  if (ptm==3) mpnw(fx,fy,sq,mp);
	  if (ptm==4) mpbw(fx,fy,sq,mp);
	  if (ptm==5) mpqw(fx,fy,sq,mp);
	  if (ptm==6) mpkw(fx,fy,sq,mp);
	}
	if (side==BLACK) {
	  if (ptm==-1) mppb(fx,fy,sq,mp);
	  if (ptm==-2) mprb(fx,fy,sq,mp);
	  if (ptm==-3) mpnb(fx,fy,sq,mp);
	  if (ptm==-4) mpbb(fx,fy,sq,mp);
	  if (ptm==-5) mpqb(fx,fy,sq,mp);
	  if (ptm==-6) mpkb(fx,fy,sq,mp);
	}

	a=-1;
        if (abs(ptm)==1) {
          a=(mvno*2)-2;
          if (side==WHITE) a--;
          if (a>=0) {
	    mv=(mvlst+a);
            a=en_passant_possible(sqq,mv,side);
	    if (a==0 || a==2) {
	      if (side==WHITE) mp->sq[2][fx+1]=1;
	      else mp->sq[5][fx+1]=1;
	    }
	    if (a==1 || a==2) {
	      if (side==WHITE) mp->sq[2][fx-1]=1;
	      else mp->sq[5][fx-1]=1;
	    }
	  }
	  else a=-1;
	}
        /*  Check that piece can actually move to specified place */
	if (mp->sq[ty][tx]!=1) {error(1);continue;}

        /*  Alter king positions if necessary ready for check test. */
	if (ptm==6) {sqq->kpos[0]=tx;sqq->kpos[1]=ty;}
	if (ptm==-6) {sqq->kpos[2]=tx;sqq->kpos[3]=ty;}
        /*  Temporarily move piece */
	sqq->sq[ty][tx]=ptm;
	sqq->sq[fy][fx]=0;
        if (tx!=fx && tmt==0 && a!=-1) { /*  En-passant */
	  if (side==WHITE) sqq->sq[3][tx]=0;
	  else sqq->sq[4][tx]=0;
	}
        if (prom!=0 && side==WHITE) sqq->sq[ty][tx]=prom;
        else if (prom!=0 && side==BLACK) sqq->sq[ty][tx]=-prom;
   
        /*  If move leaves you in check then that's naughty :) */
	if (chck(sqq,side)!=0) {
	  sqq->sq[ty][tx]=tmt;sqq->sq[fy][fx]=ptm;
	  if (a!=-1 && side==WHITE) sqq->sq[3][tx]=-1;
	  else if (a!=-1 && side==BLACK) sqq->sq[4][tx]=1;
	  error(3);continue;
	}
        /*   Permanently(!) move piece */
	sq[ty][tx]=ptm;
	sq[fy][fx]=0;
        if (tx!=fx && tmt==0 && a!=-1) { /*  En-passant */
	  if (side==WHITE) sq[3][tx]=0;
	  else sq[4][tx]=0;
	}
     
        /*  Set castling permissions if necessary. */
	if (ptm==2 && fx==0) cp[1]=0;
	if (ptm==2 && fx==7) cp[0]=0;
	if (ptm==-2 && fx==0) cp[3]=0;
	if (ptm==-2 && fx==7) cp[2]=0;
	if (ptm==6) cp[0]=cp[1]=0;
	if (ptm==-6) cp[2]=cp[3]=0;

        c=prom;
	if (side==BLACK) c=-c;
        /*  Pawn promotion? */
	if (prom==0 && side==WHITE && ptm==1 && ty==0) c=quap(1);
	if (prom==0 && side==BLACK && ptm==-1 && ty==7) c=quap(2);
        if (c!=0) sq[ty][tx]=c;
         /*  Add move to list */
	if (tx!=fx && tmt==0 && a!=-1) { /*  If en-passant */
          if (fx<tx) a=-1;
	  else a=1;
	}
	else a=0;

        if (xboardmode==1) computer_go=1;
	add_move(side,mvno,fx,fy,tx,ty,0,a,abs(c),tmt);
         /* Set up en-passant square */
	if (abs(ptm)==1 && abs(fy-ty)==2) {epx=fx;epy=(ty+fy)/2;}
        else {epx=epy=-1;}
         /*  Test to see if you put opponent in check. */
	a=test_check(3-side,sqq,epx,epy);
        if (learn_book_on && a==1 && proper==1) learn_opening(3-side,mvno,openings,0);
        if (a==1) timeron=0;
	side=3-side; /*  Over to your opponent */
	if (side==WHITE) mvno++; /*  Increment move if black has just moved. */
         /*  See if position is drawn by repetition */
        a=test_for_draw(side,mvno,sqq,0);
        if (a==1) {computer_side=0;automv=0;}
  }
  free(mp);
  free(sqq);
  free_opening(openings);
  return 1;
}

void error(int err) {    /*  All purpose error display routine :) */
  if (err==1) fprintf(stderr,"    Illegal Move!\n");
  if (err==2) fprintf(stderr,"    Unknown Command!\n");
  if (err==3) fprintf(stderr,"    That Would Leave You In Check!\n");
  if (err==4) fprintf(stderr,"    No En-Passant Capture Possible!\n");
}

int quap(int side) {  /*  Human player promotes a pawn */
  int flg;
  char ch;

  fprintf(stderr,"\nChoose a Piece.[Q,R,B,N]..");
  do {
	flg=0;
	(void)scanf("%c",&ch);
	ch=(char)toupper((unsigned char)ch);
	if (ch=='Q' || ch=='R' || ch=='B' || ch=='N') flg=1;
  } while (flg==0);
  if (side==WHITE) {
	if (ch=='Q') return 5;
	if (ch=='R') return 2;
	if (ch=='B') return 4;
	if (ch=='N') return 3;
  }
  if (side==BLACK) {
	if (ch=='Q') return -5;
	if (ch=='R') return -2;
	if (ch=='B') return -4;
	if (ch=='N') return -3;
  }
  fprintf(stderr,"ERROR IN QUAP");
  return 0;
}

int test_check(int side,board *sq,int epx,int epy) {  /*  Test to see if opponent is in check(mate) */
  get_pieces(sq);
  if (chck(sq,side)==0) return 0;
  if (side==WHITE && chkmte(1,sq,-1,epx,epy)==1) {
    if (xboardmode==0) fprintf(stderr,"\nCHECKMATE!!");
    else fprintf(stderr,"\n0-1 {Black mates}\n");
    return 1;
  }
  if (side==BLACK && chkmte(2,sq,-1,epx,epy)==1) {
    if (xboardmode==0) fprintf(stderr,"\nCHECKMATE!!");
    else fprintf(stderr,"\n1-0 {White mates}\n");
    return 1;
  }
  if (xboardmode==0) fprintf(stderr,"\nCheck!");
  return 0;
}

void bye(void) { /*  Leave ColChess */
  fprintf(stderr,"\n\nThanks for Playing ColChess 2000!\n");
  fprintf(stderr,"Play Again Soon!\n\n\n");
}

board *reset_board(void) {  /*  Reset board to game start. */
  int a;
  board *temp;

  temp=malloc(sizeof(board));
  if (!temp) {fprintf(stderr,"Memory problem in reset_board!\n");exit(0);}
   
  memset(temp->sq, 0, sizeof(temp->sq));
  temp->kpos[0]=4;
  temp->kpos[1]=7;
  temp->kpos[2]=4;
  temp->kpos[3]=0;
  for (a=0;a<8;a++) {temp->sq[1][a]=-1;temp->sq[6][a]=1;}
  temp->sq[0][0]=temp->sq[0][7]=-2;
  temp->sq[0][1]=temp->sq[0][6]=-3;
  temp->sq[0][2]=temp->sq[0][5]=-4;
  temp->sq[0][3]=-5;
  temp->sq[0][4]=-6;
  temp->sq[7][0]=temp->sq[7][7]=2;
  temp->sq[7][1]=temp->sq[7][6]=3;
  temp->sq[7][2]=temp->sq[7][5]=4;
  temp->sq[7][3]=5;
  temp->sq[7][4]=6;

  return temp;
}

void load_pos_data(void) { /*  Load in the position score data */
  int a,b,c,sh=0,sc=0;
  FILE *f;
  char filename[FILENAME_MAX];

  strcpy(filename,path);
  strcat(filename,"chessdat.dat");
  if ((f=fopen(filename,"r"))==NULL) { /*  Open file */
    fprintf(stderr,"ERROR LOADING DATA FILE!\n");
    exit(0);
  }

  (void)fscanf(f,"CHESS DATA FILE\n\n");
  for (a=0;a<6;a++) {  /*  Read in data */
    (void)fscanf(f,"-----\n");
    for (b=0;b<8;b++) {
      for (c=0;c<7;c++) (void)fscanf(f,"%d,",&posdat[a][b][c]);
      (void)fscanf(f,"%d\n",&posdat[a][b][7]);
    }
    (void)fscanf(f,"SHIFT=%d\nSCALE=%d\n",&sh,&sc);
    for (b=0;b<8;b++) for (c=0;c<8;c++) posdat[a][b][c]=((posdat[a][b][c]+sh)*sc);
  }
  (void)fclose(f);
  for (a=0;a<8;a++) for (b=0;b<8;b++) posdat_pawn[a][b]=posdat[0][a][b];
}

void setup_precalculated_data(void) {
  int a,b,c;
   
  load_pos_data();  /*  Load position data for pieces from chessdat.dat */
   /*  Bonus for advanced pawns depending on game stage */
  rank_bonus[7]=rank_bonus[6]=0;rank_bonus[5]=0;
  rank_bonus[4]=4;rank_bonus[3]=8;
  rank_bonus[2]=24;rank_bonus[1]=48;
  rank_bonus[0]=100;
   
  memset(centre_board, 0, sizeof(centre_board));
  memset(corner_board, 0, sizeof(corner_board));
  for (a=0;a<8;a++) {
    rank_bonus_half[a]=rank_bonus[a]/2;
    for (b=0;b<8;b++) {
      if (a==3 || a==4) (centre_board[a][b])++;
      if (b==3 || b==4) (centre_board[a][b])++;
      if (a>1 && a<6) (centre_board[a][b])++;
      if (b>1 && b<6) (centre_board[a][b])++;
      if (a==0 || a==7) (centre_board[a][b])--;
      if (b==0 || b==7) (centre_board[a][b])--;
      if (a==0 || a==7) (corner_board[a][b])+=2;
      if (b==0 || b==7) (corner_board[a][b])++;
      if (b<2 && b>5) (corner_board[a][b])++;
      corner_board[a][b]-=centre_board[a][b];
      corner_board[a][b]*=2;
      for (c=0;c<13;c++) rtable[a][b][c]=(long)(rand()%10000000); /*  Long integer random table for hash function */
      for (c=0;c<13;c++) rtable2[a][b][c]=(int)(rand()%10000); /*  Integer random table for hash function */
    }
  }
  for (a=0;a<8;a++) { /*  Penalty for bringing queen out early */
     /*  This is used if no pieces have been taken so far */
    early_queen_penalty[a]=(7-a)*a*2;
  }
  piece_value[0]=0;
  piece_value[1]=1;
  piece_value[2]=5;
  piece_value[3]=piece_value[4]=3;
  piece_value[5]=9;
  piece_value[6]=1000;
  for (a=0;a<6;a++) {piece_value_10[a]=piece_value[a]*10;}
  for (a=0;a<6;a++) {piece_value_100[a]=piece_value[a]*100;}
  piece_value_10[6]=0;
  piece_value_100[6]=0;
  file_bonus[0]=file_bonus[7]=6;
  file_bonus[1]=file_bonus[6]=3;
  file_bonus[2]=file_bonus[5]=0;
  file_bonus[3]=file_bonus[4]=-2;
  for (a=0;a<8;a++) file_bonus2[a]=file_bonus[a]*2;
  for (a=0;a<1000;a++) halved[a]=a/2;
  for (a=0;a<20;a++) squared[a]=a*a;
}

#ifdef _WIN32
miscdata load_ini_file(char *winloc) {
  FILE *fp;
  char inipath[FILENAME_MAX];
  int hsize;
  miscdata retdat;

  if (winloc) {   
    strcpy(inipath,winloc);
    strcat(inipath,"\\colchess.ini");
  }
  else strcpy(inipath,"colchess.ini");
  
  if ((fp=fopen(inipath,"r"))==NULL) {
    retdat.ok=0;
    return retdat;
  }
  memset(path, 0, sizeof(path));
  fscanf(fp,"COLCHESS_PATH=%s\n",path);
  fprintf(stderr,"Using path %s\n",path);
  fscanf(fp,"HASH=%d [0 to 16]\n",&(retdat.hash));
  fscanf(fp,"BOOK=%d [0 or 1]\n",&(retdat.book));
  fscanf(fp,"LEARN=%d [0 or 1]\n",&(retdat.learn));
  fprintf(stderr,"colchess.ini loaded OK!\n");
  hsize=1<<retdat.hash;
  if (retdat.hash==0) hsize=0;
  fprintf(stderr,"Hash %dk,  book ",hsize);
  if (retdat.book==0) fprintf(stderr,"OFF,  learn ");
  else fprintf(stderr,"ON,  learn ");
  if (retdat.learn==0) fprintf(stderr,"OFF\n");
  else fprintf(stderr,"ON\n");
  (void)fclose(fp);
  retdat.ok=1;
  return retdat;
}
#endif
