
/* Copyright (c) 1999, 2000, 2001, 2002, 2003, 2004 Thomas Runge (coto@core.de)
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h> 
#include <sys/param.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <netdb.h>
#include <fcntl.h>
#include <limits.h>
#include <sys/socket.h>
#include <sys/signal.h>
#include <netinet/in.h>
#include <errno.h>
#include "main.h"
#include "server.h"
#include "log.h"

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif
#ifndef MIN
#define MIN(x, y) ((x) < (y) ? (x) : (y))
#endif
#ifndef BUFLEN
#define BUFLEN 128
#endif

#define TITLESTATIC "Static Webcam"
#define TITLESTREAM "Streaming Webcam"

#define HTTPOK "HTTP/1.0 200 OK\r\n"

#define DOCTYPE \
"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \
 \"http://www.w3.org/TR/html4/transitional.dtd\">"

#define CHECK_STREAMPAGE \
"<br><br>Check out the <a href=\"stream\">streaming cam</a>."

#define STATICPAGE HTTPOK"\
Content-type: text/html; charset=iso-8859-1\r\n\r\n\
"DOCTYPE"\r\n\
<html><head><title>%s</title>\
<link rel=\"stylesheet\" type=\"text/css\" href=\"coto.css\">\
<meta http-equiv=Refresh content=\"%d\"></head>\
<body><br><img src=\"/static/image\" width=\"%d\" height=\"%d\" alt=\"[Static WebCam Picture]\">\
<br><br>(Image updates every %d second%s)%s\
<br><br><small>page generated by \
<a href=\"http://core.de/~coto/projects/"PROGRAM"/\">"PROGRAM"</a>.\
</small></body></html>\r\n"

#define STREAMPAGE HTTPOK"\
Content-type: text/html; charset=iso-8859-1\r\n\r\n\
"DOCTYPE"\r\n\
<html><head><title>%s</title>\
<link rel=\"stylesheet\" type=\"text/css\" href=\"coto.css\"></head>\
<body><br><img src=\"/stream/image\" width=\"%d\" height=\"%d\" alt=\"[Streaming WebCam Picture]\">\
<br><br>Check out the <a href=\"static\">static cam</a>.\
<br><br><small>page generated by \
<a href=\"http://core.de/~coto/projects/"PROGRAM"/\">"PROGRAM"</a>.\
</small></body></html>\r\n"

#define ERRORPAGE "\
HTTP/1.0 404 Not Found\r\n\
Content-type: text/html; charset=iso-8859-1\r\n\r\n\
"DOCTYPE"\r\n\
<html><head><title>Error</title>\
<link rel=\"stylesheet\" type=\"text/css\" href=\"coto.css\">\
</head><body>\
<h1>Sorry, unknown request.</h1><br>\
Generated by \
<a href=\"http://core.de/~coto/projects/"PROGRAM"/\">"PROGRAM"</a>.\
</body></html>\r\n"

#define ROBOTSTXTPAGE HTTPOK"\
Content-type: text/html; charset=iso-8859-1\r\n\r\n\
User-agent: *\r\n\
Disallow: /\r\n"

#define RESPAGE "\
HTTP/1.0 503 Service Unavailable\r\n\
Retry-After: 120\r\n\
Content-type: text/html; charset=iso-8859-1\r\n\r\n\
"DOCTYPE"\r\n\
<html><head><title>"PROGRAM" overloaded</title>\
<link rel=\"stylesheet\" type=\"text/css\" href=\"coto.css\">\
</head><body>\
<h1>Sorry, this server is too busy. Please try again later!</h1><br>\
Generated by \
<a href=\"http://core.de/~coto/projects/"PROGRAM"/\">"PROGRAM"</a>.\
</body></html>\r\n"

#define BOUNDARY PROGRAM

#define REPHEADER HTTPOK\
"Content-type: multipart/x-mixed-replace;boundary="BOUNDARY"\r\n"
#define TYPEHEADER "Content-type: image/jpeg\r\n"
#define LENHEADER  "Content-length: %d\r\n"
#define STATICHEADER HTTPOK""TYPEHEADER""LENHEADER"\r\n"
#define STREAMHEADER TYPEHEADER""LENHEADER"\r\n"
#define STREAMFOOTER "\r\n--"BOUNDARY"\r\n"
#define CSSHEADER    HTTPOK"Content-type: text/css; charset=iso-8859-1\r\n"LENHEADER"\r\n"

#include "css.h"

#define BREAK(x) if(!(x)) break;

static int isock;
static struct client *clients;
static int conns, streamer;
static char *static_page = NULL;
static char *stream_page = NULL;
static char *css_page    = NULL;
static char *css_header  = NULL;
static size_t static_len = 0;
static size_t stream_len = 0;
static size_t css_len    = 0;
static size_t css_h_len  = 0;

static int server_write(struct webcam *cam,
								int d, const void *buf, size_t nbytes);
static int read_line(int fd, char *ptr, ssize_t maxlen);
static void read_client_msg(struct webcam *cam, struct client *cl);
static void check_clients(struct webcam *cam);
static void client_accept(struct webcam *cam);
static void client_disconnect(struct webcam *cam, struct client *cl);
static int send_image(struct webcam *cam, int d);
static int send_external(struct webcam *cam, char *filename, int d);
static int send_file(struct webcam *cam, char *filename, int d);
static void send_error(struct webcam *cam, int d);
static void send_data(struct webcam *cam);
static int get_filesize(const char *fname);

/* TODO */
#if 0
static void send_header(struct webcam *cam, int d,
						int retcode, const char *sretcode,
						const char *content_type, int content_len,
						int expire_now)
{
	char buf[1024];
	char dbuf[128], ebuf[128];
	time_t ttime;

	ttime = time(NULL);
	strftime(dbuf, sizeof(dbuf), RFC1123DATE, localtime(&ttime));
	if(expire_now == TRUE)
		sprintf(ebuf, dbuf);
	else
	{
		ttime += 60*60*24; /* one day */
		strftime(ebuf, sizeof(ebuf), RFC1123DATE, localtime(&ttime));
	}

	snprintf(buf, sizeof(buf), "HTTP/1.0 %d %s\r\nServer: %s/%s\r\nContent-type: %s\r\nContent-length: %d\r\nDate: %s\r\nExpires: %s\r\nLast-Modified: %s\r\nConnection: close\r\n", retcode, sretcode, PROGRAM, VERSION, content_type, content_len, dbuf, ebuf, dbuf);

	server_write(cam, d, buf, strlen(buf));
}
#endif

static void reset_client(struct client *cl)
{
	cl->sock   = -1;
	cl->ip     = 0;
	cl->port   = 0;
	cl->btrans = 0;
	cl->ctime  = 0;
	cl->action = ACTION_UNKNOWN;
	cl->can_write   = FALSE;
	cl->got_header  = FALSE;
	cl->stream_init = FALSE;
	if(cl->req != NULL)
	{
		free(cl->req);
		cl->req = NULL;
	}
	if(cl->f_req != NULL)
	{
		free(cl->f_req);
		cl->f_req = NULL;
	}
	if(cl->browser != NULL)
	{
		free(cl->browser);
		cl->browser = NULL;
	}
	if(cl->referer != NULL)
	{
		free(cl->referer);
		cl->referer = NULL;
	}
}

static void setnonblocking(int sock)
{
	int opts = fcntl(sock, F_GETFL);
	if(opts < 0)
		error("fcntl(F_GETFL)", strerror(errno));
	opts = (opts | O_NONBLOCK);
	if(fcntl(sock, F_SETFL, opts) < 0)
		error("fcntl(F_SETFL)", strerror(errno));
}

static int server_write(struct webcam *cam,
								int d, const void *buf, size_t nbytes)
{
	ssize_t ret, sent;

	sent = 0;
	while(sent != nbytes)
	{
		ret = write(clients[d].sock, buf+sent, nbytes-sent);
		if(ret < 0)
		{
			if(errno != EAGAIN)
			{
				if(cam->prefs->verbose)
					dlog(__FILE__, "write: %s (client gone)\n", strerror(errno));
				client_disconnect(cam, &clients[d]);
				return(FALSE);
			}
		}
		else
		{
			if((cam->prefs->verbose) && (ret != nbytes-sent))
				dlog(__FILE__, "short send, %d bytes out of %d (left: %d)\n",
						ret, nbytes, nbytes-sent);
			sent += ret;
			cam->btrans       += ret;
		}
	}
	return(TRUE);
}

void server_init(struct webcam *cam)
{
	struct sockaddr_in serv_addr;
	struct protoent *tcp;
	int reuse_addr = 1;
	int i;

	if(cam->prefs->verbose)
		dlog(__FILE__, "init server, port: %d\n", cam->prefs->server_port);

	clients = (struct client*)malloc(cam->prefs->server_maxconn*sizeof(struct client));
	if(clients == NULL)
	{
		fprintf(stderr, "clients malloc: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	for(i = 0; i < cam->prefs->server_maxconn; i++)
	{
		clients[i].sock  = -1;
		clients[i].ip    = 0;
		clients[i].port  = 0;
		clients[i].ctime = 0;
		clients[i].can_write   = FALSE;
		clients[i].got_header  = FALSE;
		clients[i].stream_init = FALSE;
		clients[i].req     = NULL;
		clients[i].f_req   = NULL;
		clients[i].browser = NULL;
		clients[i].referer = NULL;
	}
	conns = 0;
	streamer = 0;
	cam->btrans = 0;

	tcp = getprotobyname("tcp");
	if(tcp == NULL)
		error("Can't get protocol numbers for TCP", "Exiting.");

	isock = socket(AF_INET, SOCK_STREAM, tcp->p_proto);
	if(isock < 0) 
		error("Can't create socket", strerror(errno));

	setsockopt(isock, SOL_SOCKET, SO_REUSEADDR, &reuse_addr,
				sizeof(reuse_addr));

	memset((char*) &serv_addr, 0, sizeof(serv_addr));
	serv_addr.sin_family      = AF_INET;
	serv_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	serv_addr.sin_port        = htons(cam->prefs->server_port);
	if(bind(isock, (struct sockaddr *) &serv_addr, sizeof(struct sockaddr)) < 0) 
		error("Can't bind to port", strerror(errno));
	if(listen(isock, cam->prefs->server_maxconn << 1) < 0)
		error("Can't listen to bound socket", strerror(errno));

	asprintf(&static_page, STATICPAGE, cam->prefs->title,
				cam->prefs->sleep, cam->cols, cam->rows,
				cam->prefs->sleep,
				(cam->prefs->sleep == 1 ? "" : "s"),
				cam->prefs->server_dostream ? CHECK_STREAMPAGE : "");
	static_len = strlen(static_page);

	asprintf(&css_page, CSS_STRING,
				cam->prefs->col_back,
				cam->prefs->col_fore,
				cam->prefs->col_link,
				cam->prefs->col_hover,
				cam->prefs->col_fore,
				cam->prefs->col_fore
			);
	css_len = strlen(css_page);

	asprintf(&css_header, CSSHEADER, css_len);
	css_h_len = strlen(css_header);

	if(cam->prefs->server_dostream)
	{
		asprintf(&stream_page, STREAMPAGE, cam->prefs->title,
					cam->cols, cam->rows);
		stream_len = strlen(stream_page);
	}

	if(cam->prefs->verbose)
		dlog(__FILE__, "server up and running\n");
}

static void check_clients(struct webcam *cam)
{
	fd_set in_set, out_set, exc_set;
	int i, s;
	struct timeval timeout;

	timeout.tv_sec  = cam->prefs->sleep;
	timeout.tv_usec = 0;

	if(cam->prefs->verbose)
		dlog(__FILE__, "check streamer (remaining: %d)\n", streamer);

	FD_ZERO(&in_set);
	FD_ZERO(&out_set);
	FD_ZERO(&exc_set);
	FD_SET(isock, &in_set);

	for(i = 0; i < cam->prefs->server_maxconn; i++)
	{
		if(clients[i].sock != -1)
		{
			if(clients[i].got_header == TRUE)
				FD_SET(clients[i].sock, &out_set);
			else
				FD_SET(clients[i].sock, &in_set);
			FD_SET(clients[i].sock, &exc_set);
		}
	}

	s = select(FD_SETSIZE, &in_set, &out_set, &exc_set, &timeout);
	if(s < 0)
	{
		if(errno == EINTR)
			return;
		else
			error("Failed to select", strerror(errno));
	}

	if(FD_ISSET(isock, &in_set))
		client_accept(cam);

	if(s > 0)
	{
		for(i = 0; i < cam->prefs->server_maxconn; i++)
		{
			if((clients[i].sock != -1) && FD_ISSET(clients[i].sock, &exc_set))
			{
				if(cam->prefs->verbose)
					dlog(__FILE__, "socket exception from %d.%d.%d.%d:%d\n",
							MAKEIP(clients[i].ip), clients[i].port);
				client_disconnect(cam, &clients[i]);
			}
			if((clients[i].sock != -1) && FD_ISSET(clients[i].sock, &in_set))
			{
				if(cam->prefs->verbose)
					dlog(__FILE__, "client %d is ready to read\n", i);
				read_client_msg(cam, &clients[i]);
			}
			if((clients[i].sock != -1) && FD_ISSET(clients[i].sock, &out_set))
			{
				if(cam->prefs->verbose)
					dlog(__FILE__, "client %d is ready to write\n", i);
				clients[i].can_write = TRUE;
			}
			else
				clients[i].can_write = FALSE;
		}
	}

	for(i = 0; i < cam->prefs->server_maxconn; i++)
	{
		if(	(clients[i].sock != -1) &&
			(clients[i].got_header == FALSE) &&
			(clients[i].ctime < (time(NULL) - CONN_TIMEOUT)))
		{
			client_disconnect(cam, &clients[i]);
			if(cam->prefs->verbose)
				dlog(__FILE__, "client %d timeout\n", i);
		}
	}
}

static void client_accept(struct webcam *cam)
{
	struct sockaddr_in cli_addr;
	int clen, found;
	int nsock, i;

	found = FALSE;
	clen = sizeof(struct sockaddr_in);
	nsock = accept(isock, (struct sockaddr *) &cli_addr, &clen);
	if(nsock < 0) 
		error("Failure on accept", strerror(errno));

	if(cam->prefs->verbose)
		dlog(__FILE__, "connection from %d.%d.%d.%d:%d\n",
				MAKEIP(cli_addr.sin_addr.s_addr),
				ntohs(cli_addr.sin_port));

	for(i = 0; i < cam->prefs->server_maxconn; i++)
	{
		if(clients[i].sock == -1)
		{
			setnonblocking(nsock);
			clients[i].sock  = nsock;
			clients[i].ip    = cli_addr.sin_addr.s_addr;
			clients[i].port  = cli_addr.sin_port;
			clients[i].ctime = time(NULL);
			found = TRUE;
			conns++;
			if(cam->prefs->verbose)
			{
				dlog(__FILE__, "found free slot for new client: %d (socket: %d)\n", i, nsock);
				dlog(__FILE__, "conns   : %d\n", conns);
				dlog(__FILE__, "streamer: %d\n", streamer);
			}
			break;
		}
	}

	if(found == FALSE)
	{
		struct client errcl;

		/* no free slots */
		errcl.browser = NULL;
		errcl.referer = NULL;
		errcl.req     = NULL;
		errcl.f_req   = NULL;
		reset_client(&errcl);

		errcl.retcode = 503;
		errcl.sock    = nsock;
		errcl.ip      = cli_addr.sin_addr.s_addr;
		errcl.port    = cli_addr.sin_port;
		log_access(cam, &errcl, LOG_GONE);

		write(nsock, RESPAGE, strlen(RESPAGE));
		cam->btrans += strlen(RESPAGE);
		close(nsock);
		if(cam->prefs->verbose)
			dlog(__FILE__, "found no free slot for new client\n");
	}
}

static void client_disconnect(struct webcam *cam, struct client *cl)
{
	if(cl->sock != -1)
		close(cl->sock);
	if(cl->stream_init == TRUE)
		streamer--;
	log_access(cam, cl, LOG_GONE);
	reset_client(cl);
	conns--;
	if(cam->prefs->verbose)
	{
		dlog(__FILE__, "conns   : %d\n", conns);
		dlog(__FILE__, "streamer: %d\n", streamer);
	}
}

void server_finish(struct webcam *cam)
{
	int i;

	for(i = 0; i < cam->prefs->server_maxconn; i++)
	{
		if(clients[i].sock != -1)
		{
			log_access(cam, &clients[i], LOG_SHUTDOWN);
			close(clients[i].sock);
			reset_client(&clients[i]);
		}
	}

	if(static_page != NULL)
		free(static_page);
	if(css_page != NULL)
		free(css_page);
	if(css_header != NULL)
		free(css_header);
	if(stream_page != NULL)
		free(stream_page);
}

int server_work(struct webcam *cam)
{
	check_clients(cam);
	send_data(cam);
	return(streamer);
}

static int send_image(struct webcam *cam, int d)
{
	return(send_file(cam, cam->prefs->filename, d));
}

static int send_external(struct webcam *cam, char *filename, int d)
{
	char fn[MAXPATHLEN+1];

	printf("external: %s\n", clients[d].req);
	snprintf(fn, sizeof(fn), "%s/%s", cam->prefs->rootdir, filename);
	printf("sending : %s\n", fn);
	return(send_file(cam, fn, d));
}

static int send_file(struct webcam *cam, char *filename, int d)
{
	FILE *fp;
	size_t nbytes;
	int success;
	char buf[1024];

	success = TRUE;
	if((fp = fopen(filename, "r")) == NULL)
	{
		dlog(__FILE__, "server fopen: %s\n", strerror(errno));
		return(FALSE);
	}

	if(cam->prefs->verbose)
		dlog(__FILE__, "opened file \"%s\" for sending to client %d\n", filename, d);

	while(success && (nbytes = fread(buf, 1, sizeof(buf), fp)) > 0)
	{
		if(clients[d].can_write == FALSE)
			break;
		if(cam->prefs->verbose)
			dlog(__FILE__, "sending packet to %d.%d.%d.%d:%d (%d bytes)\n", MAKEIP(clients[d].ip), clients[d].port, nbytes);
		if(!server_write(cam, d, buf, nbytes))
			success = FALSE;
		clients[d].btrans += nbytes;
	}
	fclose(fp);
	if(cam->prefs->verbose)
		dlog(__FILE__, "closed served file \"%s\"\n", cam->prefs->filename);

	return(success);
}

/* open jpeg, read from disk and send to client(s) */
static void send_data(struct webcam *cam)
{
	int i;
	char buf[1024];

	if(conns < 1)
		return;

	for(i = 0; i < cam->prefs->server_maxconn; i++)
	{
		clients[i].retcode = 200;
	}

	for(i = 0; i < cam->prefs->server_maxconn; i++)
	{
		if(clients[i].sock == -1 || clients[i].can_write == FALSE)
		{
			if(cam->prefs->verbose)
			{
				if(clients[i].sock == -1)
					dlog(__FILE__, "client %d: sock == -1\n", i);
				else
					dlog(__FILE__, "client %d: can_write == FALSE\n", i);
			}
			continue;
		}

		if(clients[i].got_header == FALSE)
		{
			if(cam->prefs->verbose)
				dlog(__FILE__, "client didn't send header, yet: %d.%d.%d.%d:%d\n",
							MAKEIP(clients[i].ip), clients[i].port);
			continue;
		}

		if(clients[i].can_write == TRUE)
		{
			int len;
			switch(clients[i].action)
			{
				case ACTION_STREAM:
					if(cam->prefs->verbose)
						dlog(__FILE__, "sending to client %d\n", i);
					snprintf(buf, sizeof(buf), STREAMHEADER,
								get_filesize(cam->prefs->filename));
					if(clients[i].stream_init == FALSE)
					{
						BREAK(server_write(cam, i, REPHEADER, strlen(REPHEADER)));
						BREAK(server_write(cam, i, STREAMFOOTER, strlen(STREAMFOOTER)));
						clients[i].stream_init = TRUE;
						streamer++;
						if(cam->prefs->verbose)
							dlog(__FILE__, "streamer: %d\n", streamer);
					}
					BREAK(server_write(cam, i, buf, strlen(buf)));
					BREAK(send_image(cam, i));
					BREAK(server_write(cam, i, STREAMFOOTER, strlen(STREAMFOOTER)));
					break;
				case ACTION_STATIC:
					snprintf(buf, sizeof(buf), STATICHEADER,
								get_filesize(cam->prefs->filename));
					if(server_write(cam, i, buf, strlen(buf)))
					{
						send_image(cam, i);
						client_disconnect(cam, &clients[i]);
					}
					break;
				case ACTION_STATICPAGE:
					if(cam->prefs->verbose)
						dlog(__FILE__, "sending static page to %d.%d.%d.%d:%d (%d bytes)\n", MAKEIP(clients[i].ip), clients[i].port, static_len);
					if(server_write(cam, i, static_page, static_len))
					{
						clients[i].btrans += static_len;
						client_disconnect(cam, &clients[i]);
					}
					break;
				case ACTION_STREAMPAGE:
					if(cam->prefs->verbose)
						dlog(__FILE__, "sending stream page to %d.%d.%d.%d:%d (%d bytes)\n", MAKEIP(clients[i].ip), clients[i].port, stream_len);
					if(server_write(cam, i, stream_page, stream_len))
					{
						clients[i].btrans += stream_len;
						client_disconnect(cam, &clients[i]);
					}
					break;
				case ACTION_CSS:
					if(cam->prefs->verbose)
						dlog(__FILE__, "sending css to %d.%d.%d.%d:%d (%d bytes)\n", MAKEIP(clients[i].ip), clients[i].port, css_len);
					if(server_write(cam, i, css_header, css_h_len))
					{
						if(server_write(cam, i, css_page, css_len))
						{
							clients[i].btrans += css_len;
							client_disconnect(cam, &clients[i]);
						}
					}
					break;
				case ACTION_ROBOT:
					len = strlen(ROBOTSTXTPAGE);
					if(cam->prefs->verbose)
						dlog(__FILE__, "sending robots.txt to %d.%d.%d.%d:%d (%d bytes)\n", MAKEIP(clients[i].ip), clients[i].port, len);
					if(server_write(cam, i, ROBOTSTXTPAGE, len))
					{
						clients[i].btrans += len;
						client_disconnect(cam, &clients[i]);
					}
					break;
				case ACTION_EXTERNAL:
					// TODO send header!
					if(send_external(cam, clients[i].req, i) == FALSE)
						send_error(cam, i);
					else
						client_disconnect(cam, &clients[i]);
					break;
				case ACTION_UNKNOWN:
				default:
					send_error(cam, i);
					break;
			}
		}
	}
}

static void send_error(struct webcam *cam, int i)
{
	int len = strlen(ERRORPAGE);

	clients[i].retcode = 404;
	if(cam->prefs->verbose)
		dlog(__FILE__, "sending errorpage to %d.%d.%d.%d:%d (%d bytes)\n", MAKEIP(clients[i].ip), clients[i].port, len);
	if(server_write(cam, i, ERRORPAGE, len))
		clients[i].btrans += len;
	client_disconnect(cam, &clients[i]);
}

static void read_client_msg(struct webcam *cam, struct client *cl)
{
	char buf[BUFLEN];
	ssize_t res;

	if(cam->prefs->verbose)
		dlog(__FILE__, "msg from %d.%d.%d.%d:%d following:\n",
				MAKEIP(cl->ip), cl->port);

	while((res = read_line(cl->sock, buf, BUFLEN-1)) > 0)
	{
		trim(buf);
		if(cam->prefs->verbose)
		{
			dlog(__FILE__, "\"%s\"\n", buf);
		}
		if(!strncmp(buf, "GET ", 4))
		{
			if(cl->req != NULL)
				free(cl->req);
			if(cl->f_req != NULL)
				free(cl->f_req);
			cl->f_req = strdup(buf);
			cl->req   = strdup(trim(strtok(buf+4, " \t\r\n")));
			cl->action = ACTION_UNKNOWN;
			if(!strcmp(cl->req, "/"))
				cl->action = ACTION_STATICPAGE;
			if(!strcmp(cl->req, "/static"))
				cl->action = ACTION_STATICPAGE;
			if(!strcmp(cl->req, "/stream"))
				cl->action = ACTION_STREAMPAGE;
			if(!strcmp(cl->req, "/coto.css"))
				cl->action = ACTION_CSS;
			if(!strcmp(cl->req, "/static/image"))
				cl->action = ACTION_STATIC;
			if(!strcmp(cl->req, "/stream/image"))
				cl->action = ACTION_STREAM;
			if(!strcmp(cl->req, "/robots.txt"))
				cl->action = ACTION_ROBOT;
			if(cl->action == ACTION_UNKNOWN && cam->prefs->use_external)
				cl->action = ACTION_EXTERNAL;

			if(cam->prefs->server_dostream == FALSE)
			{
				if(	cl->action == ACTION_STREAM ||
					cl->action == ACTION_STREAMPAGE)
				{
					cl->action = ACTION_UNKNOWN;
					if(cam->prefs->verbose)
						dlog(__FILE__, "Client requested currently switched off stream page.\n");
				}
			}
		}
		if(!strncmp(buf, "User-Agent: ", 12))
		{
			if(cl->browser != NULL)
				free(cl->browser);
			cl->browser = strdup(trim(buf+12));
		}
		if(!strncmp(buf, "Referer: ", 9))
		{
			if(cl->referer != NULL)
				free(cl->referer);
			cl->referer = strdup(trim(buf+9));
		}
		if((cl->req != NULL) && strlen(buf) == 0)
		{
			cl->got_header = TRUE;
			log_access(cam, cl, LOG_NEW);
			break;
		}
		strcat(buf, "\0");
	}

	if(cam->prefs->verbose)
	{
		dlog(__FILE__, "(msg end)\n");
		if(res > 0)
		{
			dlog(__FILE__, "Got header: %s\n", cl->got_header ? "true" : "false");
			dlog(__FILE__, "RequestURI: %s\n", cl->req);
			if(cl->browser)
				dlog(__FILE__, "Browser   : %s\n", cl->browser);
			if(cl->referer)
				dlog(__FILE__, "Referer   : %s\n", cl->referer);
		}
		if(res < 0)
			dlog(__FILE__, "read() error: %s\n", strerror(errno));
	}

	if(res == 0)
	{
		if(cam->prefs->verbose)
			dlog(__FILE__, "client waving a \"goodbye\"\n");
		client_disconnect(cam, cl);
	}
}

static ssize_t read_line(int fd, char *ptr, ssize_t maxlen)
{
	ssize_t n, rc;
	char c;

	for(n = 1; n < maxlen; n++)
	{
		if((rc = read(fd, &c, 1)) == 1)
		{
			*ptr++ = c;
			if(c == '\n')
				break;
		}
		else if(rc == 0)
		{
			if(n == 1)
				return(0); /* EOF, no data read */
			else
				break;
		}
		else 
			return(-1); /* error */
	}

	*ptr = '\0';
	return(n);
}

static int get_filesize(const char *fname)
{
	struct stat sst;

	if(stat(fname, &sst) < 0)
		error("stat", strerror(errno));

	return(sst.st_size);
}

