
/* Copyright (c) 1999, 2000, 2001, 2002, 2003, 2004 Thomas Runge (coto@core.de)
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/syslimits.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <expat.h>
#include "prefs.h"
#include "cams.h"
#include "main.h"

#define XMLBUFLEN 25
#define BUFSTEP   64

static char *catchbuf;
static int catchbuflen;
static int catchbufmax;
static int depth;
static int in_program, in_pic, in_cam;
static struct webcam_data *ccam;

static void prefs_print(struct prefs_t *prefs);

static int raisebuf()
{
	char *p;

	catchbufmax += BUFSTEP;
	p = realloc(catchbuf, catchbufmax);
	if(p == NULL)
	{
		dlog(__FILE__, "resizebuf: %s\n", strerror(errno));
		free(catchbuf);
		catchbuf = NULL;
		catchbuflen = 0;
		catchbufmax = 0;
		return(FALSE);
	}
	catchbuf = p;

	return(TRUE);
}

static void char_handler(void *userdata,
					const XML_Char *s, int len)
{
	int i;

	for(i = 0; i < len; i++)
	{
		catchbuflen++;
		while(catchbuflen >= catchbufmax)
			if(raisebuf() == FALSE)
				break;
		catchbuf[catchbuflen-1] = s[i];
	}
}

static void start_handler(void *userdata,
					const XML_Char *name, const XML_Char **atts)
{
	struct webcam *cam;
	struct prefs_t *prefs;

	cam   = (struct webcam *)userdata;
	prefs = cam->prefs;
	depth++;

	if((depth == 1) && !strcmp(PROGRAM, name))
	{
		in_program = TRUE;
	}

	if(in_program == TRUE)
	{
		if((depth == 1) && (!strcasecmp(name, PROGRAM)))
		{
			int i = 0;

			while(atts[i] != NULL)
			{
				if(!strcasecmp(atts[i], "cam"))
				{
					prefs->cam = strdup(atts[++i]);
				}
				if(!strcasecmp(atts[i], "sleep"))
				{
					prefs->sleep = atoi(atts[++i]);
				}
				if(!strcasecmp(atts[i], "daemon"))
				{
					if(!strcasecmp(atts[++i], "true"))
						prefs->daemon = TRUE;
					else
						prefs->daemon = FALSE;
				}
				if(!strcasecmp(atts[i], "verbose"))
				{
					prefs->verbose = atoi(atts[++i]);
				}
				if(!strcasecmp(atts[i], "keepold"))
				{
					if(!strcasecmp(atts[++i], "true"))
						prefs->keepold = TRUE;
					else
						prefs->keepold = FALSE;
				}
				i++;
			}
		}

		if((depth == 2) && (!strcasecmp(name, "httpserver")))
		{
			int i = 0;

			while(atts[i] != NULL)
			{
				if(!strcasecmp(atts[i], "enabled"))
				{
					if(!strcasecmp(atts[++i], "true"))
						prefs->server_enabled = TRUE;
					else
						prefs->server_enabled = FALSE;
				}
				if(!strcasecmp(atts[i], "port"))
				{
					prefs->server_port = atoi(atts[++i]);
				}
				if(!strcasecmp(atts[i], "stream"))
				{
					if(!strcasecmp(atts[++i], "true"))
						prefs->server_dostream = TRUE;
					else
						prefs->server_dostream = FALSE;
				}
				if(!strcasecmp(atts[i], "maxconn"))
				{
					prefs->server_maxconn = atoi(atts[++i]);
				}
				if(!strcasecmp(atts[i], "external"))
				{
					if(!strcasecmp(atts[++i], "true"))
						prefs->use_external = TRUE;
					else
						prefs->use_external = FALSE;
				}
				if(!strcasecmp(atts[i], "rootdir"))
				{
					prefs->rootdir = strdup(atts[++i]);
				}
				i++;
			}
		}

		if((depth == 2) && (!strcasecmp(name, "css")))
		{
			int i = 0;

			while(atts[i] != NULL)
			{
				if(!strcasecmp(atts[i], "foreground"))
				{
					prefs->col_fore = strdup(atts[++i]);
				}
				if(!strcasecmp(atts[i], "background"))
				{
					prefs->col_back = strdup(atts[++i]);
				}
				if(!strcasecmp(atts[i], "link"))
				{
					prefs->col_link = strdup(atts[++i]);
				}
				if(!strcasecmp(atts[i], "hover"))
				{
					prefs->col_hover = strdup(atts[++i]);
				}
				i++;
			}
		}

		if((depth == 2) && (!strcasecmp(name, "log")))
		{
			int i = 0;

			while(atts[i] != NULL)
			{
				if(!strcasecmp(atts[i], "enabled"))
				{
					if(!strcasecmp(atts[++i], "true"))
						prefs->log_enabled = TRUE;
					else
						prefs->log_enabled = FALSE;
				}
				if(!strcasecmp(atts[i], "filename"))
				{
					prefs->logfile = strdup(atts[++i]);
				}
				i++;
			}
		}

		if((depth == 2) && (!strcasecmp(name, "cam")))
		{
			int i = 0;

			while(atts[i] != NULL)
			{
				if(!strcasecmp(atts[i], "name"))
				{
					if(!strcasecmp(atts[++i], prefs->cam))
					{
						struct webcam_data *tcam;
						int j = 0;

						while(supported_cams[j] != NULL)
						{
							tcam = supported_cams[j++]();
							if(!strcmp(tcam->name, prefs->cam))
							{
								in_cam = TRUE;
								ccam = tcam;
								ccam->setprefs(cam);
								break;
							}
						}
					}
				}
				i++;
			}
		}

		if((depth == 2) && (!strcasecmp(name, "picture")))
		{
			in_pic = TRUE;
		}
	}
}

static void end_handler(void *userdata,
					const XML_Char *name)
{
	struct webcam *cam;
	struct prefs_t *prefs;
	char *tbuf;

	cam   = (struct webcam *)userdata;
	prefs = cam->prefs;

	catchbuf[catchbuflen] = '\0';
	tbuf = trim(catchbuf);

	if((depth == 1) && !strcmp(PROGRAM, name))
	{
		in_program = FALSE;
	}

	if(in_program == TRUE)
	{
		if((depth == 2) && (!strcasecmp(name, "picture")))
		{
			in_pic = FALSE;
		}

		if((depth == 3) && (in_pic == TRUE))
		{
			if(!strcasecmp(name, "message"))
			{
				if(strlen(tbuf) > 0)
					prefs->msg = strdup(tbuf);
			}
			if(!strcasecmp(name, "font"))
			{
				if(strlen(tbuf) > 0)
					prefs->fontfile = strdup(tbuf);
			}
			if(!strcasecmp(name, "filename"))
			{
				if(strlen(tbuf) > 0)
					prefs->filename = strdup(tbuf);
			}
			if(!strcasecmp(name, "width"))
			{
				if(strlen(tbuf) > 0)
					prefs->width = atoi(tbuf);
			}
			if(!strcasecmp(name, "height"))
			{
				if(strlen(tbuf) > 0)
					prefs->height = atoi(tbuf);
			}
			if(!strcasecmp(name, "quality"))
			{
				if(strlen(tbuf) > 0)
					prefs->quality = atoi(tbuf);
				if(prefs->quality < 0 || prefs->quality > 100)
				{
					dlog(__FILE__, "quality out of range, assuming 75%%\n");
					prefs->quality = 75;
				}
			}
			if(!strcasecmp(name, "title"))
			{
				if(strlen(tbuf) > 0)
					prefs->title = strdup(tbuf);
			}
		}

		if((depth == 2) && (!strcasecmp(name, "cam")))
		{
			in_cam = FALSE;
		}

		if((depth == 3) && (in_cam == TRUE))
		{
			ccam->addpref(prefs, (char*)name, tbuf);
		}
	}

	depth--;
	catchbuf[0] = '\0';
	catchbuflen = 0;
}

struct webcam_data *prefs_read(char *filename, struct webcam *cam)
{
	int fd;
	XML_Parser parser;

	catchbuf    = NULL;
	catchbuflen = 0;
	catchbufmax = 0;
	depth       = 0;
	ccam        = NULL;
	raisebuf();

	fd = open(filename, O_RDONLY, 0);
	if(fd == -1)
	{
		dlog(__FILE__, "Failed to open config file (%s): %s\n", filename, strerror(errno));
		exit(EXIT_FAILURE);
	}

	parser = XML_ParserCreate("UTF-8");
	XML_SetUserData(parser, cam);
	XML_SetElementHandler(parser, start_handler, end_handler);
	XML_SetCharacterDataHandler(parser, char_handler);

	for(;;)
	{
		ssize_t bytes_read;
		void *buf = XML_GetBuffer(parser, XMLBUFLEN);
		if(buf == NULL)
		{
			dlog(__FILE__, "XML_GetBuffer failed\n");
			break;
		}

		bytes_read = read(fd, buf, XMLBUFLEN);
		if(bytes_read == -1)
		{
			dlog(__FILE__, "read: %s\n", strerror(errno));
			break;
		}

		if(XML_ParseBuffer(parser, bytes_read, bytes_read == 0) == XML_STATUS_ERROR)
		{
			dlog(__FILE__, "XML_ParseBuffer failed\n");
			break;
		}

		if(bytes_read == 0)
			break;
	}

	XML_ParserFree(parser);
	close(fd);

	if(catchbuf != NULL)
		free(catchbuf);
	catchbuf = NULL;
	catchbuflen = 0;
	catchbufmax = 0;

	if(cam->prefs->verbose)
		prefs_print(cam->prefs);

	return(ccam);
}

static void prefs_print(struct prefs_t *prefs)
{
	dlog(__FILE__, "prefs, width: %d\n", prefs->width);
	dlog(__FILE__, "prefs, height: %d\n", prefs->height);
	dlog(__FILE__, "prefs, keepold: %d\n", prefs->keepold);
	dlog(__FILE__, "prefs, sleep: %d\n", prefs->sleep);
	dlog(__FILE__, "prefs, quality: %d\n", prefs->quality);
	dlog(__FILE__, "prefs, server_enabled: %d\n", prefs->server_enabled);
	dlog(__FILE__, "prefs, server_port: %d\n", prefs->server_port);
	dlog(__FILE__, "prefs, server_dostream: %d\n", prefs->server_dostream);
	dlog(__FILE__, "prefs, server_maxconn: %d\n", prefs->server_maxconn);
	dlog(__FILE__, "prefs, daemon: %d\n", prefs->daemon);
	dlog(__FILE__, "prefs, verbose: %d\n", prefs->verbose);
	dlog(__FILE__, "prefs, external: %d\n", prefs->use_external);
	dlog(__FILE__, "prefs, rootdir: %s\n", prefs->rootdir);
	dlog(__FILE__, "prefs, cam: %s\n", prefs->cam);
	dlog(__FILE__, "prefs, log_enabled: %d\n", prefs->log_enabled);
	dlog(__FILE__, "prefs, logfile: %s\n", prefs->logfile);
	dlog(__FILE__, "prefs, filename: %s\n", prefs->filename);
	dlog(__FILE__, "prefs, fontfile: %s\n", prefs->fontfile);
	dlog(__FILE__, "prefs, msg: %s\n", prefs->msg);
	dlog(__FILE__, "prefs, title: %s\n", prefs->title);
	dlog(__FILE__, "prefs, foreground: %s\n", prefs->col_fore);
	dlog(__FILE__, "prefs, background: %s\n", prefs->col_back);
	dlog(__FILE__, "prefs, link      : %s\n", prefs->col_link);
	dlog(__FILE__, "prefs, hover     : %s\n", prefs->col_hover);
}

