
/* Copyright (c) 1999, 2000, 2001, 2002, 2003, 2004 Thomas Runge (coto@core.de)
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include "main.h"
#include "log.h"
#include "server.h"

void log_init(struct webcam *cam)
{
	cam->log_fp = NULL;
	if(cam->prefs->logfile == NULL)
		cam->prefs->logfile = strdup(DEFLOG);

	if(cam->prefs->log_enabled)
	{
		if((cam->log_fp = fopen(cam->prefs->logfile, "a")) == NULL)
			error("fopen logfile", strerror(errno));
	}
	log_access(cam, NULL, LOG_START);
}

// see http://httpd.apache.org/docs/logs.html#common
// or http://www.webcom.com/help/clog.shtml
// LogFormat "%h %l %u %t \"%r\" %>s %b \"%{Referer}i\" \"%{User-agent}i\"" combined
void log_access(struct webcam *cam, struct client *cl, int action)
{
	if(cam->log_fp != NULL)
	{
		char dbuf[32];
		time_t tp;
		struct tm *lt;
		char *cp;

		tp = time(NULL);
		lt = localtime(&tp);
		cp = asctime(lt);
		cp[strlen(cp)-1] = '\0';
		switch(action)
		{
			case LOG_START:
				if(cam->prefs->verbose)
					dlog(__FILE__, "logging log start\n");
				/* fprintf(cam->log_fp, "START;%s\n", cp); */
				break;
			case LOG_END:
				if(cam->prefs->verbose)
					dlog(__FILE__, "logging log end, transferred: %ld bytes\n",
										cam->btrans);
				/* fprintf(cam->log_fp, "END;%s;%ld\n", cp, cam->btrans); */
				break;
			case LOG_NEW:
				if(cam->prefs->verbose)
					dlog(__FILE__, "logging new client\n");
				/*
				fprintf(cam->log_fp, "NEW;%s;%d.%d.%d.%d:%d;\"%s\";\"%s\";%s\n",
						cp, MAKEIP(cl->ip), ntohs(cl->port), cl->req,
						cl->browser ? cl->browser : "",
						cl->referer ? cl->referer : "");
				*/
				break;
			case LOG_GONE:
				if(cam->prefs->verbose)
					dlog(__FILE__, "logging left client\n");
				strftime(dbuf, sizeof(dbuf), "%d/%b/%Y:%H:%M:%S %z", lt);
				//fprintf(cam->log_fp, "GONE;%s;%d.%d.%d.%d:%d;%ld\n",
				//		cp, MAKEIP(cl->ip), ntohs(cl->port), cl->btrans);
				fprintf(cam->log_fp, "%d.%d.%d.%d - - [%s] \"%s\" %d %ld \"%s\" \"%s\"\n",
						MAKEIP(cl->ip), dbuf, cl->f_req, cl->retcode, cl->btrans,
						cl->referer ? cl->referer : "-",
						cl->browser ? cl->browser : "-");
				break;
			case LOG_SHUTDOWN:
				if(cam->prefs->verbose)
					dlog(__FILE__, "logging shutdown\n");
				/*
				fprintf(cam->log_fp, "SHUTDOWN;%s;%d.%d.%d.%d:%d;%ld\n",
						cp, MAKEIP(cl->ip), ntohs(cl->port), cl->btrans);
				*/
				break;
			default:
				error("unsupported LOG action", "Exiting.");
				break;
		}
		fflush(cam->log_fp);
	}
}

void log_error(struct webcam *cam, const char *error)
{
	if(cam->log_fp != NULL)
	{
		time_t tp;
		struct tm *lt;
		char *cp;

		tp = time(NULL);
		lt = localtime(&tp);
		cp = asctime(lt);
		cp[strlen(cp)-1] = '\0';

		fprintf(cam->log_fp, "ERROR;%s;%s\n", cp, error);
		fflush(cam->log_fp);
	}
}

void log_finish(struct webcam *cam)
{
	log_access(cam, NULL, LOG_END);
	if(cam->log_fp != NULL)
		fclose(cam->log_fp);
}

