
/* Copyright (c) 1999, 2000, 2001, 2002, 2003, 2004 Thomas Runge (coto@core.de)
 *
 * Based on Omnivisions documentation for the ov511+ and ov7620 chips.
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/param.h>
#include <sys/time.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <time.h>
#include <dev/usb/usb.h>
#include "main.h"
#include "cam_ov511.h"
#include "ov511reg.h"
#include "ov7620reg.h"

#define FMT_YUV422 1
#define FMT_YUV420 2
#define FMT_YUV400 3

/* device needs some time to settle down */
#define SETTLE 20
/* max number of ugen devices to check */
#define MAXUGEN 15
/* max number of USB buffer */
#define UBUFSIZE 961

static void cam_ov511_setprefs(struct webcam *cam);
static void cam_ov511_addpref(struct prefs_t *prefs, char *key, char *value);
static void cam_ov511_startup(struct webcam *cam);
static void cam_ov511_finish(struct webcam *cam);
static void cam_ov511_capture_pic(struct webcam *cam);
static void cam_ov511_getline(struct webcam *cam, int line);

static int usb_read(struct webcam *cam, uint16_t reg);
static void usb_write(struct webcam *cam, uint16_t reg, uint16_t val);
static int usb_do_request(struct webcam *cam,
							uint8_t type, uint16_t reg, uint16_t val);
static uint8_t sccb_read(struct webcam *cam, uint8_t reg);
static void sccb_write(struct webcam *cam, uint8_t reg, uint8_t val);
static void read_data(struct webcam *cam);
static void init_device(struct webcam *cam);
static int test_device(struct webcam *cam);

static struct webcam_data cam_ov511 =
	{
		"ov511", 3, JCS_YCbCr,
		cam_ov511_setprefs, cam_ov511_addpref, cam_ov511_startup,
		cam_ov511_finish, cam_ov511_capture_pic, cam_ov511_getline
	};

/* prefs */
static int type;
static int unit;
static int brightness;
static int contrast;
static int format;
static int is_large;

static uint8_t sccb_readid;
static uint8_t sccb_writeid;
static uint8_t win_hmin, win_hmax, win_vmin, win_vmax;

struct webcam_data *cam_ov511_init()
{
	return(&cam_ov511);
}

static void cam_ov511_setprefs(struct webcam *cam)
{
	cam->ctrl     = -1;
	cam->isoc     = -1;
	cam->ubufsize = 0;
	cam->ubuf     = NULL;
	cam->buf      = NULL;

	format     = FMT_YUV422;
	unit       = -1;
	brightness = -1;
	contrast   = -1;
	is_large   = FALSE;
	type       = TYPE_UNKNOWN;
	sccb_readid  = 0x0;
	sccb_writeid = 0x0;
	win_hmin = win_hmax = win_vmin = win_vmax = 0;
}

static void cam_ov511_addpref(struct prefs_t *prefs, char *key, char *value)
{
	char *s = trim(value);

	if(!strcasecmp(key, "unit"))
	{
		if(strlen(s) > 0)
			unit = atoi(s);
	}
	if(!strcasecmp(key, "brightness"))
	{
		if(strlen(s) > 0)
			brightness = atoi(s);
	}
	if(!strcasecmp(key, "contrast"))
	{
		if(strlen(s) > 0)
			contrast = atoi(s);
	}
}

static void cam_ov511_startup(struct webcam *cam)
{
	char ctldevice[FILENAME_MAX];
	int fmtw, fmth;
	int camstart, camend, n;

	fmtw = fmth = 1;

	if(unit == -1)
	{
		camstart = 0;
		camend   = MAXUGEN;
	}
	else
	{
		if(cam->prefs->verbose)
			dlog(__FILE__, "user specified unit %d for cam.\n", unit);
		camstart = unit;
		camend   = unit+1;
	}

	/* get the device */
	for(cam->unit = camstart; cam->unit < camend; cam->unit++)
	{
		snprintf(ctldevice, FILENAME_MAX, "/dev/ugen%d.00", cam->unit);

		if(cam->prefs->verbose)
			dlog(__FILE__, "testing %s\n", ctldevice);

		if((cam->ctrl = open(ctldevice, O_RDWR)) < 0)
		{
			if(errno == EBUSY)
				fprintf(stderr, "device \"%s\" is busy, another instance of %s already running?\n", ctldevice, PROGRAM);
			continue;
		}

		if(test_device(cam) == TRUE)
		{
			if(cam->prefs->verbose)
				dlog(__FILE__, "found cam at %s\n", ctldevice);
			break;
		}
	}
	if(cam->ctrl == -1)
		error("No ov511+ based webcam found", "Exiting");

	if(	(contrast != -1) &&
		((contrast < 0) ||
		(contrast > 127)))
	{
		dlog(__FILE__, "contrast out of range (0 <= x <= 127)\n");
		exit(EXIT_FAILURE);
	}

	if(	(brightness != -1) &&
		((brightness < 0) ||
		(brightness > 255)))
	{
		dlog(__FILE__, "brightness out of range (0 <= x <= 255)\n");
		exit(EXIT_FAILURE);
	}

	if(cam->prefs->verbose)
		dlog(__FILE__, "specified width: %d  height: %d\n", cam->cols, cam->rows);

	/* 640x480 doesn't fit into internal memory as YUV422 */
	if((format == FMT_YUV422) &&
		((cam->cols * cam->rows << 1) > (1 << 19)))
	{
		format = FMT_YUV420;
	}

	switch(format)
	{
		case FMT_YUV422:
			fmtw = 16;
			fmth = 8;
			break;
		case FMT_YUV420:
			fmtw = 32;
			fmth = 16;
			break;
		case FMT_YUV400:
			fmtw = 32;
			fmth = 8;
			break;
		default:
			error("Unknown capture format", "Exiting");
	}

	/* alloc USB transfer buffer */
	cam->ubufsize = UBUFSIZE;
	cam->ubuf = (char*)malloc(cam->ubufsize);
	if(cam->ubuf == NULL)
		error("malloc usb buf", strerror(errno));
	if(cam->prefs->verbose)
		dlog(__FILE__, "ubufsize: %d\n", cam->ubufsize);

	cam->cols = cam->prefs->width  - (cam->prefs->width % fmtw);
	cam->rows = cam->prefs->height - (cam->prefs->height % fmth);

	/* if scaled down image will work at 422 */
	if((format == FMT_YUV420) &&
		((cam->cols * cam->rows << 1) <= (1 << 19)))
	{
		format = FMT_YUV422;
	}

	if(cam->prefs->verbose)
		dlog(__FILE__, "using width: %d  height: %d\n", cam->cols, cam->rows);

	init_device(cam);

	/* 422: 2x; 420: 1.5x; 400: 1x */
	cam->bufsize = cam->cols * cam->rows * 2 + 960;
	cam->buf = (char*)malloc(cam->bufsize);
	if(cam->buf == NULL)
		error("malloc image data", strerror(errno));
	if(cam->prefs->verbose)
		dlog(__FILE__, "bufsize: %d\n", cam->bufsize);
	cam->round = 0;

	// get the n-th. pic, as the cam has to settle down first
	// TODO better play with AWB modes
	for(n = 0; n < SETTLE; n++)
		read_data(cam);
}

static void cam_ov511_finish(struct webcam *cam)
{
	/* finally reset device */
	usb_write(cam, OV511_SYS_RST, RST_ALL);

	/* and shutdown system */
	usb_write(cam, OV511_SYS_EN_SYS, 0x00);

	/* and close down */
	close(cam->isoc);

	/* free resources */
	free(cam->buf);
	cam->buf = NULL;
	free(cam->ubuf);
	cam->ubuf = NULL;
	close(cam->ctrl);
	cam->ctrl = -1;
}

static void init_device(struct webcam *cam)
{
	struct usb_alt_interface alt;
	uint8_t data;
	int winx, winy, winw, winh, i, hscale, vscale;
	char buf[FILENAME_MAX+1];

	/* reset device */
	usb_write(cam, OV511_SYS_RST, RST_ALL);
	usb_write(cam, OV511_SYS_RST, 0);

	/* initialize system */
	usb_write(cam, OV511_SYS_EN_SYS, 0x01);

	usb_write(cam, OV511_FIFO_PKSZ, 0x01);
	usb_write(cam, OV511_FIFO_PKFMT, 0x00);
	usb_write(cam, OV511_SYS_RST, RST_FLUSH | RST_FIFO |
						RST_OMNICE | RST_DRAM | RST_CAM);
	usb_write(cam, OV511_SYS_RST, 0);

	/* disable snap button */
	usb_write(cam, OV511_SYS_SNAP, 0x00);

	/* disable OmniCE */
	usb_write(cam, OV511_OMNI_CTRL, 0x00);

	if(cam->prefs->verbose)
		dlog(__FILE__, "checking sensor...\n");

	i = 0;
	while((type = sccb_types[i].type) != TYPE_UNKNOWN)
	{
		/* set SCCB slave read/write adresses */
		sccb_readid  = sccb_types[i].read;
		sccb_writeid = sccb_types[i].write;

		usb_write(cam, OV511_SCCB_SRA, sccb_readid);
		usb_write(cam, OV511_SCCB_SID, sccb_writeid);

		if(cam->prefs->verbose)
			dlog(__FILE__, "trying %s\n", sccb_types[i].txt);

		if(sccb_read(cam, OV7620_MANH) == 0x7F)
		{
			if(sccb_read(cam, OV7620_MANL) == 0xA2)
			{
				if(cam->prefs->verbose)
					dlog(__FILE__, "found supported sensor\n");

				if(cam->cols > sccb_types[i].maxw)
				{
					snprintf(buf, sizeof(buf),
						"max: %d", sccb_types[i].maxw);
					error("width too big for sensor", buf);
				}
				if(cam->rows > sccb_types[i].maxh)
				{
					snprintf(buf, sizeof(buf),
						"max: %d", sccb_types[i].maxh);
					error("height too big for sensor", buf);
				}
				if(cam->cols < sccb_types[i].minw)
				{
					snprintf(buf, sizeof(buf),
						"min: %d", sccb_types[i].minw);
					error("width too small for sensor", buf);
				}
				if(cam->rows < sccb_types[i].minh)
				{
					snprintf(buf, sizeof(buf),
						"min: %d", sccb_types[i].minh);
					error("height too small for sensor", buf);
				}

				win_hmin = sccb_types[i].hmin;
				win_hmax = sccb_types[i].hmax;
				win_vmin = sccb_types[i].vmin;
				win_vmax = sccb_types[i].vmax;

				if(	cam->prefs->width  > sccb_types[i].largew ||
					cam->prefs->height > sccb_types[i].largeh)
				{
					is_large = TRUE;
				}

				break;
			}
			else
			{
				if(cam->prefs->verbose)
					dlog(__FILE__, "No %s found (L)\n", sccb_types[i].txt);
			}
		}
		else
		{
			if(cam->prefs->verbose)
				dlog(__FILE__, "No %s found\n", sccb_types[i].txt);
		}
		i++;
	}

	if(type == TYPE_UNKNOWN)
		error("No supported sensor found", "Exiting.");

	sccb_write(cam, OV7620_COMA, COMA_RST);
	/* wait for reset finished */
	while(sccb_read(cam, OV7620_COMA) & COMA_RST);

	if(is_large)
		sccb_write(cam, OV7620_CLK, 0x06);
	else
		sccb_write(cam, OV7620_CLK, 0x01);

	data = sccb_read(cam, OV7620_COMA);
	sccb_write(cam, OV7620_COMA, data | COMA_AWB | COMA_AGC);

	data = sccb_read(cam, OV7620_COMC);
	if(is_large)
		sccb_write(cam, OV7620_COMC, data & ~COMC_SIZ);
	else
		sccb_write(cam, OV7620_COMC, data | COMC_SIZ);

	data = sccb_read(cam, OV7620_COMK);
	sccb_write(cam, OV7620_COMK, data | COMK_AWB | COMK_ASM);

	/* these magic values improve image quality a lot, taken
	 * from Linux driver. Thank you, Mark!
	 */
	sccb_write(cam, OV6620_CPP,  0xa0);
	sccb_write(cam, OV6620_BIAS, 0xd2);
	sccb_write(cam, OV6620_COMK, 0x8b);
	sccb_write(cam, OV6620_COML, 0x40);
	sccb_write(cam, OV6620_COMM, 0x39);
	sccb_write(cam, OV6620_COMM, 0x3c);
	sccb_write(cam, OV6620_COMM, 0x24);
	sccb_write(cam, 0x4a, 0x80);
	sccb_write(cam, 0x4b, 0x80);
	sccb_write(cam, OV6620_YMXA, 0xd2);
	sccb_write(cam, OV6620_ARL,  0xc1);
	sccb_write(cam, OV6620_YMXB, 0x04);

	/* AEW Auto Exposure White Pixel Ratio */
	sccb_write(cam, OV7620_AEW, 0x10); 

	/* AEC Auto Exposure Black Pixel Ratio */
	sccb_write(cam, OV7620_AEB, 0x8a); 

	/* Common control H */
	if(format == FMT_YUV400)
		sccb_write(cam, OV7620_COMH, COMH_PSC | COMH_BW);
	else
		sccb_write(cam, OV7620_COMH, COMH_PSC);

	if(type == TYPE_OV7620)
	{
		/* Signal Process Control B */
		sccb_write(cam, OV7620_SPCB, SPCB_MOD | SPCB_BL1);

		/* Color Space Selection */
		sccb_write(cam, OV7620_YUV, YUV_UV1 | YUV_UVC | YUV_ASC);

		/* Cheat contrast with gamma */
		if(contrast == -1)
		{
			sccb_write(cam, OV7620_YGAM, 0);
			if(cam->prefs->verbose)
				dlog(__FILE__, "selecting auto-contrast\n");
		}
		else
		{
			data = contrast << 1;
			sccb_write(cam, OV7620_YGAM, YGAM_ENABLE | data);
			if(cam->prefs->verbose)
				dlog(__FILE__, "selecting contrast: %d\n", contrast);
		}
	}

	/* Auto brightness enabled */
	if(brightness == -1)
	{
		sccb_write(cam, OV7620_COMJ, COMJ_FIX | COMJ_BRGHT);
		if(cam->prefs->verbose)
			dlog(__FILE__, "selecting auto-brightness\n");
	}
	else
	{
		sccb_write(cam, OV7620_COMJ, COMJ_FIX);
		sccb_write(cam, OV7620_BRT, brightness);
		if(cam->prefs->verbose)
			dlog(__FILE__, "selecting brightness: %d\n", brightness);
	}

	/* center window */
	hscale = (is_large ? 2:1);
	vscale = (is_large ? 1:0);
	if(type == TYPE_OV6620)
		hscale--;

	winw = cam->cols >> hscale;
	winh = cam->rows >> vscale;
	winx = win_hmin + ((win_hmax - win_hmin - winw) >> 1);
	winy = win_vmin + ((win_vmax - win_vmin - winh) >> 1);

	//printf("width: %d  height: %d\n", cam->cols, cam->rows);
	//printf("IS0 HS: 0x%02x  HE: 0x%02x  VS: 0x%02x  VE: 0x%02x\n", sccb_read(cam, OV7620_HS), sccb_read(cam, OV7620_HE), sccb_read(cam, OV7620_VS), sccb_read(cam, OV7620_VE));
	//printf("MAX HS: 0x%02x  HE: 0x%02x  VS: 0x%02x  VE: 0x%02x\n", win_hmin, win_hmax, win_vmin, win_vmax);

	if(type == TYPE_OV6620)
	{
		/* wrong 6620 docs, cheating */
		winw += 2;
		winh += 2;
	}

	sccb_write(cam, OV7620_HS, winx);
	sccb_write(cam, OV7620_HE, winx + winw);
	sccb_write(cam, OV7620_VS, winy);
	sccb_write(cam, OV7620_VE, winy + winh);

	//printf("IS1 HS: 0x%02x  HE: 0x%02x  VS: 0x%02x  VE: 0x%02x\n", sccb_read(cam, OV7620_HS), sccb_read(cam, OV7620_HE), sccb_read(cam, OV7620_VS), sccb_read(cam, OV7620_VE));

	sccb_write(cam, OV7620_FRARH, 0x04);

	/* make sure, we get complete frames */
	usb_write(cam, OV511_DRAM_ENFC, 0xFF);

	usb_write(cam, OV511_CAM_PXCNT, (cam->cols >> 3) - 1);
	usb_write(cam, OV511_CAM_SNPX,  (cam->cols >> 3) - 1);
	usb_write(cam, OV511_CAM_LNCNT, (cam->rows >> 3) - 1);
	usb_write(cam, OV511_CAM_SNLN,  (cam->rows >> 3) - 1);
	usb_write(cam, OV511_CAM_PXDV, 0x00);
	usb_write(cam, OV511_CAM_SNPD, 0x00);
	usb_write(cam, OV511_CAM_LNDV, 0x00);
	usb_write(cam, OV511_CAM_SNLD, 0x00);

	/* select 8/16 bit data format, 8bit for b/w, 16bit for color */
	if(format == FMT_YUV400)
	{
		usb_write(cam, OV511_CAM_M400, 0x00);
		usb_write(cam, OV511_CAM_SN400, 0x00);
	}
	else
	{
		usb_write(cam, OV511_CAM_M400, 0x01);
		usb_write(cam, OV511_CAM_SN400, 0x01);
	}

	/* select YUV format and enable filter */
	data = M420_YFIR;
	if(format == FMT_YUV420)
		data |= M420_YUV;
	usb_write(cam, OV511_CAM_M420, data);

	data = SN420_YFIR;
	if(format == FMT_YUV420)
		data |= M420_YUV;
	usb_write(cam, OV511_CAM_SN420, data);

	/* select FIFO packet size */
	usb_write(cam, OV511_FIFO_PKSZ, (cam->ubufsize-1) >> 5);

	/* select FIFO packet format */
	usb_write(cam, OV511_FIFO_PKFMT, FMT_ENCE | FMT_ENPKNO);

	/* switch to alternative 7 (which uses 961 bytes) */
	alt.uai_interface_index = 0;
	alt.uai_alt_no = 7;
	if(ioctl(cam->ctrl, USB_SET_ALTINTERFACE, &alt) < 0)
		error("USB_SET_ALTINTERFACE", strerror(errno));

	/* reset device (required after fifo changes) */
	usb_write(cam, OV511_SYS_RST, RST_FLUSH | RST_SCCB |
									RST_FIFO | RST_OMNICE |
									RST_DRAM | RST_CAM);
	usb_write(cam, OV511_SYS_RST, 0);

	/* we are ready to read data from isochronous endpoint (#1) */
	snprintf(buf, FILENAME_MAX, "/dev/ugen%d.01", cam->unit);
	if(cam->prefs->verbose)
		dlog(__FILE__, "ISOC: %s\n", buf);
	if((cam->isoc = open(buf, O_RDONLY)) < 0)
		error("open USB syn device", strerror(errno));
}

static void cam_ov511_capture_pic(struct webcam *cam)
{
	/* take picture */
	read_data(cam);
}

static void cam_ov511_getline(struct webcam *cam, int line)
{
	char *img;
	char *p;
	int bpl;		/* blocks per line */
	int bw;			/* block width */
	int cbr;		/* current block row */
	int cbc;		/* current block column */
	int pkgsize;	/* package size */
	int i;

	cbr = line >> 3;
	img = cam->buf;
	p   = cam->copybuf;
	switch(cam->format)
	{
		case FMT_YUV422:
				bw  = 16;
				pkgsize = 256;
				break;
		case FMT_YUV420:
				bw  = 32;
				pkgsize = 384;
				break;
		case FMT_YUV400:
				bw  = 32;
				pkgsize = 256;
				break;
		default:
				dlog(__FILE__, "unsupported format: %d\n", cam->format);
				exit(EXIT_FAILURE);
				break;
	}
	bpl = cam->cols / bw;

	for(i = 0; i < cam->cols; i++)
	{
		char *py, *pu, *pv;
		int pkg;					/* current package */
		int pline;					/* line inside the current package */
		int yoff;					/* Y offset inside package */
		int uoff;					/* U/V offset */
		unsigned char uval, vval;	/* U and V values */

		uoff  = 0;
		cbc   = i / bw;
		pkg   = cbr * bpl + cbc;
		pline = line % 8;
		uval = vval = 0x80;

		/* get start of H, U and V values inside the current package */
		pu = img + (pkg * pkgsize);
		pv = pu+64;
		py = pv+64;

		if(cam->format == FMT_YUV400)
			py = pu;

		/* decode luminance */
		yoff = (pline << 3) + (i % 8);
		if((i%bw) >= 8)
			yoff += 64;
		if((i%bw) >= 16)
			yoff += 64;
		if((i%bw) >= 24)
			yoff += 64;
		*p++ = py[yoff];

		/* decode color information */
		switch(cam->format)
		{
			case FMT_YUV420:
				pkg -= cbc;			/* start of the row */
				if(cbr % 2)			/* start of even row */
					pkg -= bpl;
				uoff = i >> 4;		/* which packet has color info? */
				pkg += uoff;		/* add to start of even row */
				pu = img + (pkg * pkgsize);	/* here it is! */
				pv = pu+64;

				uoff = (pline << 2) + ((i % 16) >> 1);
				if(line % 2)
					uoff += 4;
				uoff += ((i % 16) >> 1);
				uval = pu[uoff];
				vval = pv[uoff];
				break;
			case FMT_YUV422:
				uoff = (pline << 3) + ((i % 16) >> 1);
				uval = pu[uoff];
				vval = pv[uoff];
				break;
			case FMT_YUV400:
				break;
			default:
				error("Unsupported format to decode", "Exiting.");
				break;
		}
		*p++ = uval;
		*p++ = vval;

		//printf("line: %d i: %d  pline: %d  pkg: %d  cbc: %d  cbr: %d  yoff: %d  uoff: %d\n", line, i, pline, pkg, cbc, cbr, yoff, uoff);
	}
}

static int test_device(struct webcam *cam)
{
	struct usb_device_info devinfo;

	if(ioctl(cam->ctrl, USB_GET_DEVICEINFO, &devinfo) < 0)
		error("USB_GET_DEVICEINFO", strerror(errno));

	if(cam->prefs->verbose)
	{
		dlog(__FILE__, "device : /dev/ugen%d\n", cam->unit);
		dlog(__FILE__, "vendor : 0x%x\n", devinfo.udi_vendorNo);
		dlog(__FILE__, "product: 0x%x\n", devinfo.udi_productNo);
	}

	//EyeToy if(devinfo.udi_vendorNo != 0x054C || devinfo.udi_productNo != 0x0155)
	if(devinfo.udi_vendorNo != 0x05A9 || devinfo.udi_productNo != 0xA511)
		return(FALSE);

	return(TRUE);
}

static int usb_do_request(struct webcam *cam,
							uint8_t type, uint16_t reg, uint16_t val)
{
	struct usb_ctl_request req;
	unsigned char data[2];

	req.ucr_request.bmRequestType = type;
	req.ucr_request.bRequest = 2;
	*data = val;

	USETW(req.ucr_request.wValue, 0);
	USETW(req.ucr_request.wIndex, reg);
	USETW(req.ucr_request.wLength, 1);
	req.ucr_data = data;
	req.ucr_flags = 0;
	req.ucr_actlen = 0;

	if(ioctl(cam->ctrl, USB_DO_REQUEST, &req) < 0)
		return(-1);

	return(data[0] & 0xFF);
}

static int usb_read(struct webcam *cam, uint16_t reg)
{
	int ret = usb_do_request(cam, UT_READ_VENDOR_INTERFACE, reg, 0);
	if(ret == -1)
		error("USB read", strerror(errno));
	return(ret);
}

static void usb_write(struct webcam *cam, uint16_t reg, uint16_t val)
{
	int ret = usb_do_request(cam, UT_WRITE_VENDOR_INTERFACE, reg, val);
	if(ret == -1)
		error("USB write", strerror(errno));
}

static uint8_t sccb_read(struct webcam *cam, uint8_t reg)
{
	uint8_t val;
	int status;

	/* Wait for bus being idle. */
	while(!usb_read(cam, OV511_SCCB_CTRL) & SCCBR_IDLE)
		if(cam->prefs->verbose)
			dlog(__FILE__, "sccb_read1, waiting for bus idle\n");

	/* 2 byte dummy write cycles (In order to set sub address
	 * of SCCB slave device for the next coming read cycle)
	 */

	/* writes to sub address (SMA, register 43h) */
	usb_write(cam, OV511_SCCB_SMA, reg);

	/* Writes to control bits to select write cycle and
	 * launch SCCB cycles (TYPE, STARTSCCB, register 40h)
	 */
	usb_write(cam, OV511_SCCB_CTRL, SCCBW_START | SCCBW_TYPEW);

	/* Wait for bus being idle. */
	//TODO really?
	while(!usb_read(cam, OV511_SCCB_CTRL) & SCCBR_IDLE)
		if(cam->prefs->verbose)
			dlog(__FILE__, "sccb_read2, waiting for bus idle\n");

	/* 2 byte read cycles */

	/* writes to sub address (SRA, register 44h) */
	usb_write(cam, OV511_SCCB_SRA, sccb_readid);

	while(1)
	{
		/* writes to control bits to select read cycle and
		 * launch SCCB cycles (TYPE, STARTSCCB, register 40h)
		 */
		usb_write(cam, OV511_SCCB_CTRL, SCCBW_START | SCCBW_TYPER);

		/* reads from status bits (TMOUT, NOACK, IDLE, register 40h) */
		while(!(status = usb_read(cam, OV511_SCCB_CTRL)) & SCCBR_IDLE)
			if(cam->prefs->verbose)
				dlog(__FILE__, "sccb_read3, waiting for bus idle\n");

		//if(status & SCCBR_TMOUT)
		//	if(cam->prefs->verbose)
		//		dlog(__FILE__, "sccb_read5, bus timeout\n");
		if(status & SCCBR_NOACK)
		{
			if(cam->prefs->verbose)
				dlog(__FILE__, "sccb_read4, no bus ack\n");
		}
		else
			break;
	}
    
	/* reads from data port (SIO-0, register 45h) */
	val = usb_read(cam, OV511_SCCB_SIO0);

	/* Since a multi-byte cycle overwrites its original subaddress,
	 * if a read cycle follows immediately to a multi-byte cycle,
	 * it is necessary to insert a single byte write cycle that
	 * provides a new subaddress.
	 */
	usb_write(cam, OV511_SCCB_SMA, reg); // ZZZ really required?
	usb_write(cam, OV511_SCCB_CTRL, SCCBW_START | SCCBW_TYPER);
            
	return(val);
}

static void sccb_write(struct webcam *cam, uint8_t reg, uint8_t val)
{
    /* 3 byte write cycles */
    /* writes to sub address (SWA, register 42h) */
    usb_write(cam, OV511_SCCB_SWA, reg);

    /* writes to data port (SIO-0, register 45h) */
    usb_write(cam, OV511_SCCB_SIO0, val);

    /* launch SCCB cycles (TYPE, STARTSCCB, register 40h) */
    usb_write(cam, OV511_SCCB_CTRL, SCCBW_START);
}

static void read_data(struct webcam *cam)
{
	int is_eof, is_capt, pk_exp, got_start, offset, offset2;
	unsigned char pk_is;

	is_eof  = is_capt = got_start = FALSE;
	pk_exp  = pk_is = 0;
	offset  = 0;
	offset2 = 0;


	if(cam->prefs->verbose > 1)
		dlog(__FILE__, "\n");

	while((is_eof == FALSE) || (got_start == FALSE))
	{
		int n, img_off = 0;
		char *p = cam->ubuf;

		n = read(cam->isoc, cam->ubuf, cam->ubufsize);
		if(n == -1)
			perror("isoc read");

		cam->round++;

		if(n != cam->ubufsize)
		{
			dlog(__FILE__, "Failed to read image (%d/%d bytes) in round %d; %ld bytes transfered\n", n, cam->ubufsize, cam->round, cam->btrans);

			camreset(cam);
			return;
		}

		if(is_capt == TRUE)
		{
			pk_exp++;
			pk_is = p[cam->ubufsize-1];
			if(pk_exp != pk_is)
				got_start = FALSE;
		}

		if(cam->prefs->verbose > 1)
			dlog(__FILE__, "receiving packet: %d, expected: %d %s\n",
					pk_is, pk_exp, (pk_is == pk_exp ? "" : "X"));

		if(	(!p[0] && !p[1] && !p[2] && !p[3] && !p[4] &&
			 !p[5] && !p[6] && !p[7]) && (p[8] & 0x08))
		{
			img_off = 9;

			if((is_eof = p[8] & 0x80)) /* end */
			{
				if(got_start)
				{
					int w = p[9];
					int h = p[10];
					int pformat = p[8] & 0x30;

					if(cam->prefs->verbose > 1)
						dlog(__FILE__, "img width : %d  height: %d\n", (w+1)<<3, (h+1)<<3);
					switch(pformat)
					{
						case 0x0:
							cam->format = FMT_YUV422;
							if(cam->prefs->verbose > 1)
								dlog(__FILE__, "format: YUV 422\n");
							break;
						case 0x10:
							cam->format = FMT_YUV420;
							if(cam->prefs->verbose > 1)
								dlog(__FILE__, "format: YUV 420\n");
							break;
						case 0x30:
						case 0x20:
							cam->format = FMT_YUV400;
							if(cam->prefs->verbose > 1)
								dlog(__FILE__, "format: YUV 400\n");
							break;
						default:
							dlog(__FILE__, "Unknown format: 0x%x\n", pformat);
							break;
					}
					img_off = 11;
					is_capt = FALSE;
				}
			}
			else
			{
				pk_exp    = 0;
				offset    = 0;
				offset2   = 0;
				is_capt   = TRUE;
				got_start = TRUE;
			}

			if(cam->prefs->verbose > 1)
				dlog(__FILE__, "SOF/EOF: %s\n", is_capt ? "Start" : "End");

		}

		if(pk_is == pk_exp && is_capt)
		{
			int l = cam->ubufsize-img_off-1;
			if(offset + l > cam->bufsize)
			{
				offset2 += l;
				dlog(__FILE__, "data doesn't fit buffer length, %d > %d\n",
							offset+l, cam->bufsize);
			}
			else
			{
				memmove(cam->buf+offset, p+img_off, l);
				offset  += l;
				offset2 += l;
			}
		}

		if(pk_exp == 255)
			pk_exp = 0;
	}
	if(cam->prefs->verbose > 1)
		dlog(__FILE__, "\ncollected %d bytes, should be %d.\n",
					offset, offset2);
}

