/* vim: set ft=objc ts=4 et sw=4 nowrap: */
/*
 *  WorkInProgress.m
 *  
 *  Copyright (c) 2004
 *
 *  Author: Andreas Heppel <aheppel@web.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <Foundation/Foundation.h>
#include <AppKit/AppKit.h>

#include "Functions.h"
#include "Constants.h"

#include "WorkInProgress.h"

static NSString *nibName = @"WorkInProgress";
#define PI_LENGTH 2

@implementation WorkInProgressIndicator

/** Display the next step
 * 
 * The method simply increases the indicator's value by one and redisplays
 * the indicator. If the value reaches the maximum it is wrapped.
 *
 * The method is called by the indeterminate indicator's timer function.
 */
- (void) animate:(id)sender
{
    double value;
    static BOOL forth = YES;
    
    if (![self isIndeterminate])
        return;

    if (forth) {
        value = [self doubleValue] + 1;
        if (value == [self maxValue] - PI_LENGTH)
            forth = NO;
    } else {
        value = [self doubleValue] - 1;
        if (value == [self minValue])
            forth = YES;
    }

    [self setDoubleValue: value];
}

/** Draw the indicator
 * 
 * The method does basically the same as NSProgressIndicator's drawRect method.
 * It is a bit simplified, as our indicator can never be vertical, its value can
 * never be larger than max and it always has a bezel.
 * We use blue colour here.
 */
- (void)drawRect:(NSRect)rect
{
    NSRect	r;
    double value, min, max;

    [super drawRect: rect];

    // Calculate the inside rect to be drawn
	NSSize borderSize = _sizeForBorderType (NSBezelBorder);
	r = NSInsetRect(_bounds, borderSize.width, borderSize.height);

    value = [self doubleValue];
    min = [self minValue];
    max = [self maxValue];

    r.origin.x = NSWidth(r) * (value - min) / (max - min);
    r.size.width = NSWidth(r) * PI_LENGTH / (max - min);
    r = NSIntersectionRect(r, rect);

    if (!NSIsEmptyRect(r)) {
//        [[NSColor colorWithCalibratedRed: 0.03 green: 0.20 blue: 0.61 alpha: 1.] set];
        [[NSColor controlShadowColor] set];
        NSRectFill(r);
    }
}


@end


@implementation WorkInProgress

- (void)dealloc
{
    TEST_RELEASE (win);
    [super dealloc];
}

- (id)init
{
    self = [super init];

    if (self) {
        if ([NSBundle loadNibNamed: nibName owner: self] == NO) {
            logToConsole(MessageStatusError, [NSString stringWithFormat:
                                _(@"Common.loadNibFail"), nibName]);
            DESTROY (self);
            return self;
        } else {
            NSRect wframe = [win frame];
            NSRect scrframe = [[NSScreen mainScreen] frame];
            NSRect winrect = NSMakeRect((scrframe.size.width - wframe.size.width) / 2,
                              (scrframe.size.height - wframe.size.height) / 2,
                               wframe.size.width,
                               wframe.size.height);
      
            [win setFrame: winrect display: NO];
        }			
    }
  
    return self;
}

- (void)startAnimationWithString: (NSString *)string
                        appName: (NSString *)appname;
{
    if (win) {
        [win setTitle: appname];
        [textField setStringValue: string];

        [progInd setDoubleValue: 0.];
        [progInd setIndeterminate: YES];
            [progInd setAnimimationDelay: 1./6.];
	    [progInd setControlTint: NSProgressIndicatorPreferredThickness];
        [progInd startAnimation: self];

        if ([win isVisible] == NO) {
            [win orderFrontRegardless];
        }
    }
}

- (void)stopAnimation
{
    [progInd stopAnimation: self];
 
    [win close];
}

- (BOOL)windowShouldClose:(id)sender
{
    [progInd stopAnimation: self];
 
    return YES;
}

@end
