/* vim: set ft=objc ts=4 nowrap: */
/* All Rights reserved */

#include <AppKit/AppKit.h>
#include <AppKit/NSHelpManager.h>

#include "ToolPanel.h"
#include "ToolSelector.h"
#include "AppController.h"
#include "Constants.h"
#include "Functions.h"

#include <Burn/ExternalTools.h>

static ToolPanel *singleInstance = nil;


@interface ToolPanel (Private)
- (void) getAvailableDrives;
- (void) getAvailableTools;
@end


@implementation ToolPanel (Private)


//
// other methods
//

- (void) getAvailableDrives
{
	NSString *temp;
	NSArray *drives = nil;

	[devicePopUp removeAllItems];

	drives = getAvailableDrives();

	/*
	 * If we really got something from the bundle we use it,
	 * otherwise we fall back to reading /proc/scsi/scsi.
	 */
	if (drives && [drives count]) {
		int i;
		for (i = 0; i < [drives count]; i++) {
			[devicePopUp addItemWithTitle: [drives objectAtIndex: i]];
		}
	} else {
		/*
		 * If we didn't find or get any burner devices, we put a
		 * dummy entry into the popup.
		 */
		[devicePopUp addItemWithTitle: NOT_FOUND];
	}

	temp = [[[NSUserDefaults standardUserDefaults] objectForKey: @"SelectedTools"] objectForKey: @"BurnDevice"];
	if ([temp length]) {
		[devicePopUp selectItemWithTitle: temp];
	} else {
		[devicePopUp selectItemAtIndex: 0];
	}

	[self deviceChanged: self];
}



- (void) getAvailableTools
{
	int i;
	NSArray *bundles;
	id tool;

	[burnToolPopUp removeAllItems];
	[isoToolPopUp removeAllItems];
	[ripToolPopUp removeAllItems];

	bundles = [[AppController appController] allBundles];

	for (i = 0; i < [bundles count]; i++) {
		tool = [bundles objectAtIndex: i];
		if ([[(id)tool class] conformsToProtocol: @protocol(Burner)]) {
			[burnToolPopUp addItemWithTitle: [(id<BurnTool>)tool name]];
		} else if ([[(id)tool class] conformsToProtocol: @protocol(AudioConverter)]) {
			if ([[(id<AudioConverter>)tool fileType] isEqualToString: @"cd"])
				[ripToolPopUp addItemWithTitle: [(id<BurnTool>)tool name]];
		} else if ([[(id)tool class] conformsToProtocol: @protocol(IsoImageCreator)]) {
			[isoToolPopUp addItemWithTitle: [(id<BurnTool>)tool name]];
		}
	}

	if ([burnToolPopUp numberOfItems] == 0) {
		[burnToolPopUp addItemWithTitle: _(@"ToolPanel.empty")];
	}
	if ([isoToolPopUp numberOfItems] == 0) {
		[isoToolPopUp addItemWithTitle: _(@"ToolPanel.empty")];
	}
	if ([ripToolPopUp numberOfItems] == 0) {
		[ripToolPopUp addItemWithTitle: _(@"ToolPanel.empty")];
	}
}

@end

@implementation ToolPanel


- (id) init
{
	[self initWithNibName: @"ToolPanel"];
	return self;
}


- (id) initWithNibName: (NSString *) nibName;
{
	if (singleInstance) {
		[self dealloc];
	} else {
		self = [super init];
		if (![NSBundle loadNibNamed: nibName owner: self]) {
			logToConsole(MessageStatusError, [NSString stringWithFormat:
								_(@"Common.loadNibFail"), nibName]);
		} else {
			view = [panel contentView];
			[view retain];

			singleInstance = self;
		}
	}
	return singleInstance;
}

- (void) dealloc
{
	/*
	 * Make the compiler shut up.
	 */
    [super dealloc];
}

- (void) awakeFromNib
{
	burnToolPopUp = [toolTable burnToolPopUp];
	[burnToolPopUp setTarget: self];
	[burnToolPopUp setAction: @selector(toolChanged)];

	isoToolPopUp = [toolTable isoToolPopUp];
	[isoToolPopUp setTarget: self];
	[isoToolPopUp setAction: @selector(toolChanged)];

	ripToolPopUp = [toolTable ripToolPopUp];
	[ripToolPopUp setTarget: self];
	[ripToolPopUp setAction: @selector(toolChanged)];

	[toolTable setTarget: self action: @selector(toolChanged)];

	[self initializeFromDefaults];
}


//
// PreferencesModule methods
//

- (NSImage *) image
{
	NSBundle *aBundle;
	
	aBundle = [NSBundle bundleForClass: [self class]];
	
	return AUTORELEASE([[NSImage alloc] initWithContentsOfFile:
					[aBundle pathForResource: @"iconGeneral" ofType: @"tiff"]]);
}

- (NSString *) title
{
	return _(@"ToolPanel.title");
}

- (NSView *) view
{
	return view;
}

- (BOOL) hasChangesPending
{
	return YES;
}


- (void) initializeFromDefaults
{
	NSString *temp;

	[self getAvailableDrives];
	[self getAvailableTools];
	[toolTable getAvailableTools];

	temp = [[[NSUserDefaults standardUserDefaults] objectForKey: @"SelectedTools"] objectForKey: @"RipSW"];
	if (temp && [temp length]) {
		[ripToolPopUp selectItemWithTitle: temp];
	} else {
		[ripToolPopUp selectItemAtIndex: 0];
	}

	temp = [[[NSUserDefaults standardUserDefaults] objectForKey: @"SelectedTools"] objectForKey: @"BurnSW"];
	if (temp && [temp length]) {
		[burnToolPopUp selectItemWithTitle: temp];
	} else {
		[burnToolPopUp selectItemAtIndex: 0];
	}

	temp = [[[NSUserDefaults standardUserDefaults] objectForKey: @"SelectedTools"] objectForKey: @"ISOSW"];
	if (temp && [temp length]) {
		[isoToolPopUp selectItemWithTitle: temp];
	} else {
		[isoToolPopUp selectItemAtIndex: 0];
	}

	[[NSNotificationCenter defaultCenter]
		postNotificationName: ToolChanged
		object: nil
		userInfo: nil];
}

- (void) saveChanges
{
	NSMutableDictionary *selectedTools = [[NSUserDefaults standardUserDefaults]
													objectForKey: @"SelectedTools"];
	[selectedTools setObject: [burnToolPopUp titleOfSelectedItem]
					  forKey: @"BurnSW"];
	[selectedTools setObject: [isoToolPopUp titleOfSelectedItem]
					  forKey: @"ISOSW"];
	[selectedTools setObject: [ripToolPopUp titleOfSelectedItem]
					  forKey: @"RipSW"];
	[selectedTools setObject: [devicePopUp titleOfSelectedItem]
					  forKey: @"BurnDevice"];

	[toolTable saveChanges];
}



- (void) toolChanged: (id)sender
{
	[self saveChanges];

	if (sender == burnToolPopUp) 
		[self getAvailableDrives];
	
	[[NSNotificationCenter defaultCenter]
		postNotificationName: ToolChanged
		object: nil
		userInfo: nil];
}


- (void) deviceChanged: (id)sender
{
	NSMutableDictionary *selectedTools = [[NSUserDefaults standardUserDefaults]
													objectForKey: @"SelectedTools"];
	[selectedTools setObject: [devicePopUp titleOfSelectedItem]
					  forKey: @"BurnDevice"];
}

//
// class methods
//
+ (id) singleInstance
{
	if (singleInstance == nil) {
		singleInstance = [[ToolPanel alloc] init];
	}

	return singleInstance;
}

@end
