/* vim: set ft=objc ts=4 et sw=4 nowrap: */
/*
 *  ParametersWindowController.m
 *
 *  Copyright (c) 2004
 *
 *  Author: Andreas Heppel <aheppel@web.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ParametersWindowController.h"

#include "AppController.h"
#include "Constants.h"
#include "Functions.h"
#include "ProjectWindowController.h"

#include <Burn/PreferencesModule.h>
#include <Burn/ExternalTools.h>

#include "ParametersWindow.h"

/**
 * <p>standardModules contains the list of class names
 * for preferences classes. The preferences objects are created
 * and inserted into the panel in the order of their appearance
 * in standardModules.</p>
 */
static NSString *standardModules[] = {
    @"GeneralParameters",
    nil
};


@implementation ParametersWindowController

- (id) initWithWindowNibName: (NSString *)windowNibName
{
	ParametersWindow *parametersWindow;

	parametersWindow = [[ParametersWindow alloc] initWithContentRect: NSMakeRect(250,250,372,415)
						 styleMask: NSTitledWindowMask
						 backing: NSBackingStoreRetained
						 defer: NO];

	self = [super initWithWindow: parametersWindow];
	
	[parametersWindow layoutWindow];
	[parametersWindow setDelegate: self];

	// We link our outlets
	matrix = [parametersWindow matrix];
	box = [parametersWindow box];

	RELEASE(parametersWindow);

	// We set our window title
	[[self window] setTitle: _(@"ParametersWindowController.title")];

	// We maintain an array of opened modules
	allModules = [[NSMutableDictionary alloc] initWithCapacity: 2];

	// We initialize our matrix with the standard modules
	[self initializeWithStandardModules];

	// We then add our additional modules
	[self initializeWithOptionalModules];

	// We select the first cell in our matrix
	[matrix selectCellAtRow: 0 column: 0];
	[self handleCellAction: matrix];

	// We finally set our autosave window frame name and restore the one from the user's defaults.
	[[self window] setFrameAutosaveName: @"ParametersWindow"];
	[[self window] setFrameUsingName: @"ParametersWindow"];

	return self;
}


//
//
//
- (void) dealloc
{
	RELEASE(allModules);

	[super dealloc];
}


//
// action methods
//
- (void) cancelClicked: (id) sender
{
    logToConsole(MessageStatusWarning,
                 _(@"ParametersWindowController.cancelled"));
    [NSApp stopModalWithCode: NSCancelButton];
    [self close];
}


- (void) okClicked: (id) sender
{
	[self saveParameters];

    [NSApp stopModalWithCode: NSOKButton];
    [self close];
}


- (void) handleCellAction: (id)sender
{	
	id aModule;
	
	aModule = [allModules objectForKey: [[matrix selectedCell] title]];

	if (aModule) {
		[self addModuleToView: aModule];
	} else {
		logToConsole(MessageStatusError, [NSString stringWithFormat:
					_(@"Common.loadBundleFail"), [[matrix selectedCell] title]]);
	}
}

//
// other methods
//
- (void) saveParameters
{
	NSArray *allNames;
	id<PreferencesModule> aModule;
	int i;

	allNames = [allModules allKeys];

	for (i = 0; i < [allNames count]; i++) {
		aModule = [allModules objectForKey: [allNames objectAtIndex: i]];
		[aModule saveChanges];
	}

	[[NSUserDefaults standardUserDefaults] synchronize];
}


- (void) addModuleToView: (id<PreferencesModule>)aModule
{	
	if (aModule == nil) {
		return;
	}

	if ([box contentView] != [aModule view]) {
		[box setContentView: [aModule view]];
		[box setTitle: [aModule title]];
	}
}


//
//
//
- (void) initializeWithStandardModules
{
    int i = 0;
    BOOL done = NO;

    while (!done) {
        NSString *className = standardModules[i++];

        if (className) {
            Class class = NSClassFromString(className);
        	id<PreferencesModule> module = [class singleInstance];
            NSButtonCell *buttonCell;
            int column;

        	if (!module) {
	        	logToConsole(MessageStatusError, [NSString stringWithFormat:
		        				_(@"Common.initModuleFail"), className]);
    		    return;
    	    }

			// We add our column
			[matrix addColumn];
			column = ([matrix numberOfColumns] - 1);

			[allModules setObject: module forKey: [module title]];
	  
			buttonCell = [matrix cellAtRow: 0 column: column];
	  
			[buttonCell setTag: column];
			[buttonCell setTitle: [module title]];
			[buttonCell setFont: [NSFont systemFontOfSize: 8]];
			[buttonCell setImage: [module image]];

    	    RELEASE((id<NSObject>)module);
        } else {
            done = YES;
        }
    }
}



//
//
//
- (void) initializeWithOptionalModules
{
	int i;
	id<PreferencesModule> aModule;
	NSArray *bundles;

	bundles = [[AppController appController] allBundles];
	for (i = 0; i < [bundles count]; i++) {
		id aBundle;
      
		aBundle = [bundles objectAtIndex: i];

		// We get our Preferences module and we add it to our matrix.
		aModule = [aBundle parameters];
      
		if (aModule) {
			NSButtonCell *aButtonCell;
			int column;

			// We add our column
			[matrix addColumn];
			column = ([matrix numberOfColumns] - 1);

			[allModules setObject: aModule forKey: [aModule title]];
	  
			aButtonCell = [matrix cellAtRow: 0 column: column];
	  
			[aButtonCell setTag: column];
			[aButtonCell setTitle: [aModule title]];
			[aButtonCell setFont: [NSFont systemFontOfSize: 8]];
			[aButtonCell setImage: [aModule image]];
		}
	}

	[matrix sizeToCells];
	[matrix setNeedsDisplay: YES];
}


//
// access/mutation methods
//


//
// class methods
//

@end
