/* vim: set ft=objc ts=4 nowrap: */
/*
 *  TrackInspector.m
 *
 *  Copyright (c) 2002-2005
 *  
 *  Author: Andreas Heppel <aheppel@web.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <Foundation/Foundation.h>
#include <AppKit/AppKit.h>

#include "AppController.h"
#include "TrackInspector.h"
#include "Functions.h"
#include "Constants.h"

#include <unistd.h>
#include <sys/types.h>


typedef enum {
	ttDataFile = 0,
	ttAudioCD,
	ttAudioWav,
	ttAudioAu,
	ttLast
} typeTag;

@implementation TrackInspector

// we need to keep the track info even is the inspector
// is closed
static NSArray *allTracks = nil;


- (id)init
{
	self = [super init];
	
	if(self) {
		// load the gui...
		if (![NSBundle loadNibNamed: @"TrackInspector" owner: self]) {
			logToConsole(MessageStatusError, [NSString stringWithFormat:
								_(@"Common.loadNibFail"), @"TrackInspector"]);
			[self dealloc];
			return nil;
		} else {
			// initialize all member variables...
    		fileIcon = [NSImage imageNamed: @"iconDelete.tiff"];
   			multiIcon = [NSImage imageNamed: @"iconMultiSelection.tiff"];
			RETAIN(multiIcon);
			[typeIcon setImage: fileIcon];

			trackCount = 1;

			[self setTracks: allTracks];

			[[NSNotificationCenter defaultCenter] addObserver: self
			   selector: @selector(tracksChanged:)
			   name: TrackSelectionChanged
			   object: nil];
		}
	}		
	return self;
}

- (void)dealloc
{

	[[NSNotificationCenter defaultCenter] removeObserver: self
							name: TrackSelectionChanged
							object: nil];
	RELEASE(multiIcon);
	[super dealloc];
}

- (void) awakeFromNib
{
	int i;
	NSArray *fTypes = [[AppController appController] registeredFileTypes];

	maxTrackType = ttLast + [fTypes count];
	additionalTypes = [NSMutableArray new];

    [typePopUp removeAllItems];

	[typePopUp addItemWithTitle: _(@"Common.data")];
	[typePopUp addItemWithTitle: @"CD Audio"];
	[typePopUp addItemWithTitle: @"Audio (wav)"];
	[typePopUp addItemWithTitle: @"Audio (au)"];

	for (i = 0; i < [fTypes count]; i++) {
			NSString *s = [NSString stringWithFormat: @"Audio (%@)",
					 		[fTypes objectAtIndex: i]];
			[typePopUp addItemWithTitle: s];
			s = [NSString stringWithFormat: @"audio:%@",
					 		[fTypes objectAtIndex: i]];
			[additionalTypes addObject: s];
	}

	for (i = ttDataFile; i < maxTrackType; i++) {
		NSMenuItem *item = [typePopUp itemAtIndex: i];
		if (item != nil)
			[item setTag: i];
	}

	[typePopUp setEnabled: NO];
	[typePopUp setAutoenablesItems: NO];
}

- (void) typeChanged: (id)sender
{
	NSMenuItem *item = [typePopUp selectedItem];
	int sel = [item tag];

	switch (sel) {
	case ttDataFile:
		[curTrack setType: @"data"];
		break;
	case ttAudioCD:
		[curTrack setType: @"audio:cd"];
		break;
	case ttAudioWav:
		[curTrack setType: @"audio:wav"];
		break;
	case ttAudioAu:
		[curTrack setType: @"audio:au"];
		break;
	default:
		[curTrack setType: [additionalTypes objectAtIndex: sel - ttLast]];
		break;
	}
	[self setImage];
}

- (void) tracksChanged: (id)notification
{
	[self setTracks: [[notification userInfo] objectForKey: @"Tracks"]];
}

- (void) removeTracks
{
	if (allTracks != nil) {
		[allTracks release];
		allTracks = nil;
	}
	fileIcon = [NSImage imageNamed: @"iconDelete.tiff"];
	[typeIcon setImage: fileIcon];

	[nameField setStringValue: _(@"Common.noTracks")];
	[sourceField setStringValue: @""];
	[sizeField setStringValue: @""];
	[durationField setStringValue: @""];

	[typePopUp setEnabled: NO];
}

- (void) setTracks: (NSArray *)tracks
{
	int i;
	if (!tracks || [tracks count] == 0) {
		[self removeTracks];
		return;
	}

	// if we got a new track array we get rid of the old one
	if(![tracks isEqualToArray: allTracks]) {
		[allTracks release];
		allTracks = [tracks retain];
	}

	trackCount = [allTracks count];

	curTrack = [allTracks objectAtIndex: 0];

	/*
	 * First, disable all options. In case of a single track
	 * selection the matching options will be re-enabled.
	 * In case of a multiple track selection they won't.
	 */
	for (i = ttDataFile; i < maxTrackType; i++) {
		[[typePopUp itemAtIndex: i] setEnabled: NO];
	}
	
	[typePopUp setEnabled: YES];

	if (trackCount == 1) {   // Single Selection
		long duration = [curTrack duration];
		double size = [curTrack size];
		NSString *sizeFormat;

		// calculate the size in kB or MB
		if (size < (1024.*1024.*0.25)) {
			size = size/1024.;
			sizeFormat = _(@"Common.kB");
		} else {
			size = size/1024./1024.;
			sizeFormat = _(@"Common.MB");
		}

		[nameField setStringValue: [curTrack description]];
		[sourceField setStringValue: [curTrack source]];
		[sizeField setStringValue: [NSString stringWithFormat: sizeFormat, size]];
		[durationField setStringValue: framesToString(duration)];

		// set the type information according to the track type
		if ([[curTrack type] isEqual: @"audio:cd"]) {
			[[typePopUp itemAtIndex: ttAudioCD] setEnabled: YES];
			[typePopUp selectItemAtIndex: ttAudioCD];
		} else if ([[curTrack type] isEqual: @"audio:wav"]) {
			[[typePopUp itemAtIndex: ttDataFile] setEnabled: YES];
			[[typePopUp itemAtIndex: ttAudioWav] setEnabled: YES];
			[typePopUp selectItemAtIndex: ttAudioWav];
		} else if ([[curTrack type] isEqual: @"audio:au"]) {
			[[typePopUp itemAtIndex: ttDataFile] setEnabled: YES];
			[[typePopUp itemAtIndex: ttAudioAu] setEnabled: YES];
			[typePopUp selectItemAtIndex: ttAudioAu];
		} else if ([[curTrack type] isEqual: @"data"]) {
			NSString *ext = [[[curTrack source] pathExtension] lowercaseString];

			// data could be converted to almost everything
			[[typePopUp itemAtIndex: ttDataFile] setEnabled: YES];
			if ([ext isEqual: @"wav"]) {
				[[typePopUp itemAtIndex: ttAudioWav] setEnabled: YES];
			} else if ([ext isEqual: @"au"]) {
				[[typePopUp itemAtIndex: ttAudioAu] setEnabled: YES];
			} else {
				NSString *type = [NSString stringWithFormat: @"audio:%@", ext];
				unsigned i = [additionalTypes indexOfObject: type];
				if (i != NSNotFound) {
					[[typePopUp itemAtIndex: (ttLast + i)] setEnabled: YES];
				}
			}
			[typePopUp selectItemAtIndex: ttDataFile];
		} else if ([[curTrack type] isEqual: @"dir"]) {
			[[typePopUp itemAtIndex: ttDataFile] setEnabled: YES];
			[typePopUp selectItemAtIndex: ttDataFile];
		} else {
			unsigned i = [additionalTypes indexOfObject: [curTrack type]];

			[[typePopUp itemAtIndex: ttDataFile] setEnabled: YES];

			if (i != NSNotFound) {
				[[typePopUp itemAtIndex: (ttLast + i)] setEnabled: YES];
				[typePopUp selectItemAtIndex: (ttLast + i)];
			} else {
				[typePopUp selectItemAtIndex: ttDataFile];
			}
		}
	} else {	   // Multiple Selection
		int i;
		long duration = 0;
		double size = 0.;
		NSString *sizeFormat;

		// if more than one track is selected, we display the total size
		// and time. Track type selection/display will not be possible
		[nameField setStringValue: [NSString stringWithFormat: _(@"Common.nrTracks"), trackCount]];
		[sourceField setStringValue: @""];

		for (i = 0; i < trackCount; i++) {
			duration += [(Track*)[allTracks objectAtIndex: i] duration];
			size += [(Track*)[allTracks objectAtIndex: i] size];
		}

		// calculate the size in kB or MB
		if (size < (1024.*1024.*0.25)) {
			size = size/1024.;
			sizeFormat = _(@"Common.kB");
		} else {
			size = size/1024./1024.;
			sizeFormat = _(@"Common.MB");
		}

		[sizeField setStringValue: [NSString stringWithFormat: sizeFormat, size]];
		[durationField setStringValue: framesToString(duration)];
		[typePopUp setEnabled: NO];
	}
	[self setImage];
}

- (void) setImage
{
	if (trackCount == 1) {   // Single Selection
		if ([[curTrack type] isEqual: @"audio:cd"])
			fileIcon = [NSImage imageNamed: @"iconAudioCD.tiff"];
		else
			fileIcon = [[NSWorkspace sharedWorkspace] iconForFile: [curTrack source]];
	} else {	   // Multiple Selection
		fileIcon = multiIcon;
	}
	[typeIcon setImage: fileIcon];
}

- (void)deactivate: (NSView *)view
{
	/* This gives the view back to its original parent */
	[(NSWindow*)window setContentView: view];
}

- (NSString *)inspectorName
{
	return _(@"TrackInspector.name");
}

- (NSString *)winname
{
	return [window title];
}

- (id) window
{
	return window;
}


@end
