/* vim: set ft=objc ts=4 nowrap: */
/*
 *	BurnProgress.m
 *
 *	Copyright (c) 2002
 *
 *	Author: Andreas Heppel <aheppel@web.de>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "BurnProgress.h"

#include "Constants.h"
#include "Functions.h"

static int showdet = 0;

/**
 * <p>The class BurnProgress implements a panel with three indicators
 * for overall and track progress and the buffer level.</p>
 * <p>Each progress indicator can be set up individually, i.e.
 * the can have different ranges and different values. The labels
 * can be set to display arbitray strings. All settings will be made
 * from outside the panels code using the public interface.</p>
 * <p>The panel has a abort button to interrupt the current process.
 * This button can be enabled or disabled depending on the caller's needs.</p>
 * <p>BurnProgress is used to display the grabbing and burning processes,
 * using one progress indicator for the whole process and the other for the
 * current track.</p>
 */
@implementation BurnProgress



//
//
//
- (void) dealloc
{
	RELEASE(abort);
	RELEASE(entireLabel);
	RELEASE(trackLabel);
	RELEASE(bufferLabel);
	RELEASE(entireProgress);
	RELEASE(trackProgress);
	RELEASE(bufferIndicator);
	[super dealloc];
}

- (void) layoutWindow
{
	NSView *content;
	NSSize contentSize;

	showdet = 0;
	content = [self contentView];

	[self setHidesOnDeactivate: NO];
	[self setTitle: _(@"BurnProgress.prepTracks")];

	box = [[NSBox alloc] initWithFrame: NSMakeRect(5,50,390,171)];
	[box setBorderType: NSNoBorder];
	[box setTitle: @""];
	[content addSubview: box];

   	MAKE_LABEL(entireLabel, NSMakeRect(0,130,380,TextFieldHeight), @"", 'l', NO, box);

	[self setEntireLabel: _(@"BurnProgress.prepProcess")];

	entireProgress = [[NSProgressIndicator alloc] initWithFrame: NSMakeRect(0,110,380,15)];
	[entireProgress setBezeled: YES];
	[entireProgress setIndeterminate: NO];
	[entireProgress setControlTint: NSProgressIndicatorPreferredThickness];
	[box addSubview: entireProgress];

    MAKE_LABEL(trackLabel, NSMakeRect(0,75,380,TextFieldHeight), @"", 'l', NO, box);

	trackProgress = [[NSProgressIndicator alloc] initWithFrame: NSMakeRect(0,55,380,15)];
	[trackProgress setBezeled: YES];
	[trackProgress setIndeterminate: NO];
	[trackProgress setControlTint: NSProgressIndicatorPreferredThickness];
	[box addSubview: trackProgress];

    MAKE_LABEL(bufferLabel, NSMakeRect(0,20,380,TextFieldHeight), _(@""), 'l', NO, box);

	bufferIndicator = [[NSProgressIndicator alloc] initWithFrame: NSMakeRect(0,0,380,15)];
	[bufferIndicator setBezeled: YES];
	[bufferIndicator setIndeterminate: NO];
	[bufferIndicator setControlTint: NSProgressIndicatorPreferredThickness];
	[box addSubview: bufferIndicator];

	abort = [[NSButton alloc] initWithFrame: NSMakeRect(310,10,80,ButtonHeight)];
	[abort setAutoresizingMask: NSViewMinXMargin | NSViewMaxYMargin];
	[abort setButtonType: NSMomentaryPushButton];
	[abort setTitle: _(@"Common.abort")];
	[abort setTarget: [self windowController]];
	[abort setAction: @selector(abortClicked:)];
	[abort setFont: [NSFont systemFontOfSize: 0]];
	[content addSubview: abort];

	details = [[NSButton alloc] initWithFrame: NSMakeRect(10,10,80,ButtonHeight)];
	[details setAutoresizingMask: NSViewMinXMargin | NSViewMaxYMargin];
	[details setButtonType: NSMomentaryPushButton];
	[details setTitle: _(@"Common.console")];
	[details setTarget: self];
	[details setAction: @selector(showConsole:)];
	[details setFont: [NSFont systemFontOfSize: 0]];
	[content addSubview: details];
}


//
// action methods
//

- (IBAction) showConsole: (id)sender
{
	[sharedConsole() showWindow: self];
}


//
// access/mutation methods
//

- (NSProgressIndicator *)entireProgress
{
	return entireProgress;
}

- (NSProgressIndicator *)trackProgress
{
	return trackProgress;
}

- (NSProgressIndicator *)bufferIndicator
{
	return bufferIndicator;
}


/**
 * <p>Enables or disables the 'Cancel' button.</p>
 * <br />
 * <strong>Inputs</strong><br />
 * <deflist>
 * <term>enable</term>
 * <desc>Sets whether the 'Cancel' button is to be enabled (YES) or
 * disabled (NO).</desc>
 * </deflist>
 */
- (void) enableAbort: (BOOL) enable
{
	[abort setEnabled: enable];
}

/**
 * <p>Sets a ne caption for the 'Cancel' button.</p>
 * <br />
 * <strong>Inputs</strong><br />
 * <deflist>
 * <term>newText</term>
 * <desc>The new caption for the 'Cancel' button.</desc>
 * </deflist>
 */
- (void) setAbortText: (NSString *)newText
{
	[abort setTitle: newText];
}

/**
 * <p>Sets the text for one of the labels.</p>
 * <br />
 * <strong>Inputs</strong><br />
 * <deflist>
 * <term>newText</term>
 * <desc>The new text for the label.</desc>
 * <term>which</term>
 * <desc>Tells the panel which label to change.</desc>
 * </deflist>
 */
- (void) setEntireLabel: (NSString *)newText
{
	[entireLabel setStringValue: newText];
}

- (void) setTrackLabel: (NSString *)newText
{
	[trackLabel setStringValue: newText];
}

- (void) setBufferLabel: (NSString *)newText
{
	[bufferLabel setStringValue: newText];
}

/**
 * <p>Sets the range for one of the progress indicators and resets it.</p>
 * <br />
 * <strong>Inputs</strong><br />
 * <deflist>
 * <term>which</term>
 * <desc>Tells the panel which progress indicator to change.</desc>
 * <term>from</term>
 * <desc>The lower value for the progress indicator.</desc>
 * <term>to</term>
 * <desc>The upper value for the progress indicator.</desc>
 * </deflist>
 */
- (void) initProgressBar: (EnumProgressBars)which From: (double)from To: (double)to
{
	NSProgressIndicator *progInd;

	switch (which) {
	case progressEntire:
		progInd = entireProgress;
		break;
	case progressTrack:
		progInd = trackProgress;
		break;
	case progressBuffer:
		progInd = bufferIndicator;
		break;
	}

	[progInd setMinValue: from];
	[progInd setMaxValue: to];
	[self resetProgressBar: which];
}

/**
 * <p>Sets the progress value for one of the progress indicators.</p>
 * <br />
 * <strong>Inputs</strong><br />
 * <deflist>
 * <term>which</term>
 * <desc>Tells the panel which progress indicator to change.</desc>
 * <term>value</term>
 * <desc>The new progress value.</desc>
 * </deflist>
 */
- (void) setProgressBar: (EnumProgressBars)which toValue: (double)value
{
	NSProgressIndicator *progInd;

	switch (which) {
	case progressEntire:
		progInd = entireProgress;
		break;
	case progressTrack:
		progInd = trackProgress;
		break;
	case progressBuffer:
		progInd = bufferIndicator;
		break;
	}

	[progInd setDoubleValue: value];
}

/**
 * <p>Resets one of the progress indicators to zero (0).</p>
 * <br />
 * <strong>Inputs</strong><br />
 * <deflist>
 * <term>which</term>
 * <desc>Tells the panel which progress indicator to change.</desc>
 * </deflist>
 */
- (void) resetProgressBar: (EnumProgressBars)which
{
	NSProgressIndicator *progInd;

	switch (which) {
	case progressEntire:
		progInd = entireProgress;
		break;
	case progressTrack:
		progInd = trackProgress;
		break;
	case progressBuffer:
		progInd = bufferIndicator;
		break;
	}

	[progInd setDoubleValue: [progInd minValue]];
}

/**
 * <p>Sets one of the progress indicators to its maximum value.</p>
 * <br />
 * <strong>Inputs</strong><br />
 * <deflist>
 * <term>which</term>
 * <desc>Tells the panel which progress indicator to change.</desc>
 * </deflist>
 */
- (void) setProgressBarToMax: (EnumProgressBars)which
{
	NSProgressIndicator *progInd;

	switch (which) {
	case progressEntire:
		progInd = entireProgress;
		break;
	case progressTrack:
		progInd = trackProgress;
		break;
	case progressBuffer:
		progInd = bufferIndicator;
		break;
	}

	[progInd setDoubleValue: [progInd maxValue]];
}

@end
