/* vim: set ft=objc et sw=4 ts=4 nowrap: */
/*
 *  CDrecordSettingsView.m
 *
 *  Copyright (c) 2002
 *
 *  Author: Andreas Heppel <aheppel@web.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.    See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <AppKit/AppKit.h>
#include "CDrecordSettingsView.h"
#include "Functions.h"
#include "Constants.h"

#ifdef _
#undef _
#endif

#define _(X) \
    [[NSBundle bundleForClass: [self class]] localizedStringForKey:(X) value:@"" table:nil]

static CDrecordSettingsView *singleInstance = nil;


@implementation CDrecordSettingsView

- (id) init
{
    return [self initWithNibName: @"Settings"];
}

- (id) initWithNibName: (NSString *) nibName
{
    if (singleInstance) {
        [self dealloc];
    } else {
        self = [super init];

        if (![NSBundle loadNibNamed: nibName owner: self]) {
            NSLog (@"CDrecord: Could not load nib \"%@\".", nibName);
            [self dealloc];
        } else {
            view = [window contentView];
            [view retain];

            // We get our defaults for this panel
            [self initializeFromDefaults];

            singleInstance = self;
        }
    }

    return singleInstance;
}


- (void) dealloc
{
    singleInstance = nil;
    RELEASE(view);

    [super dealloc];
}


- (void) chooseClicked: (id)sender
{
    NSArray *fileToOpen;
    NSOpenPanel *oPanel;
    NSString *dirName;
    NSString *fileName;
    int result;

    dirName = [programTextField stringValue];
    fileName = [dirName lastPathComponent];
    dirName = [dirName stringByDeletingLastPathComponent];

    oPanel = [NSOpenPanel openPanel];
    [oPanel setAllowsMultipleSelection: NO];
    [oPanel setCanChooseDirectories: NO];
    [oPanel setCanChooseFiles: YES];

    result = [oPanel runModalForDirectory:dirName file:fileName types:nil];
  
    if (result == NSOKButton) {
        fileToOpen = [oPanel filenames];

        if ([fileToOpen count] > 0) {
            fileName = [fileToOpen objectAtIndex:0];
            [programTextField setStringValue:fileName];
        }
    }
}


//
// access methods
//

- (NSImage *) image
{
    NSBundle *aBundle;
    
    aBundle = [NSBundle bundleForClass: [self class]];
    
    return AUTORELEASE([[NSImage alloc] initWithContentsOfFile:
                    [aBundle pathForResource: @"iconCDrecord" ofType: @"tiff"]]);
}

- (NSString *) title
{
    return _(@"cdrecord");
}

- (NSView *) view
{
    return view;
}

- (BOOL) hasChangesPending
{
    return YES;
}


//
//
//
- (void) initializeFromDefaults
{
    NSString *temp;
    NSMutableDictionary *parameters =
            [[NSUserDefaults standardUserDefaults] objectForKey: @"CDrecordParameters"];

    temp = [parameters objectForKey: @"Program"];
    if (!temp) {
        temp = which(@"cdrecord");
        if (temp) {
            [parameters setObject: temp forKey: @"Program"];
        }
    }
    if (temp) {
        [programTextField setStringValue: temp];
    }
}


/*
 * saveChanges checks the values for the programs and displays an alert panel
 * for any program not defined or not executable. The user may then decide
 * to either not save the missing program and thus keep the old value or to
 * save the invalid value anyway.
 */
- (void) saveChanges
{
    NSString *cdrecord;
    NSMutableDictionary *parameters =
            [[NSUserDefaults standardUserDefaults] objectForKey: @"CDrecordParameters"];

    if (!parameters) {
        parameters = [NSMutableDictionary dictionary];
        [[NSUserDefaults standardUserDefaults] setObject: parameters
                                                  forKey: @"CDrecordParameters"];
    }

    cdrecord = [programTextField stringValue];

    if (!checkProgram(cdrecord)) {
        NSRunAlertPanel(@"CDrecord.bundle",
                        [NSString stringWithFormat:
                                _(@"Program for %@ not defined or not executable. %@ may not run correctly."),
                                @"cdrecord", @"CDrecord.bundle"],
                        _(@"OK"), nil, nil);
    }

    [parameters setObject: cdrecord forKey: @"Program"];

    [[NSUserDefaults standardUserDefaults] synchronize];
}


//
// class methods
//
+ (id) singleInstance
{
    if (!singleInstance) {
        singleInstance = [[CDrecordSettingsView alloc] initWithNibName: @"Settings"];
    }

    return singleInstance;
}


@end
