/* vim: set ft=objc et sw=4 ts=4 nowrap: */
/*
 *    BlankPanel.m
 *
 *    Copyright (c) 2002
 *
 *    Author: Andreas Heppel <aheppel@web.de>
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.    See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "BlankPanel.h"

#include "Constants.h"
#include "Functions.h"
#include "AppController.h"

static BlankPanel *sharedPanel = nil;
static BOOL blanking = NO;


@implementation BlankPanel


- (id) init
{
    [self initWithNibName: @"BlankPanel"];
    return self;
}


- (id) initWithNibName: (NSString *) nibName;
{
    if (sharedPanel) {
        [self dealloc];
    } else {
        self = [super init];
        sharedPanel = self;
        if (![NSBundle loadNibNamed: nibName owner: self]) {
            logToConsole(MessageStatusError, [NSString stringWithFormat:
                                _(@"Common.loadNibFail"), nibName]);
        } else {
            [panel setExcludedFromWindowsMenu: YES];

            [[NSNotificationCenter defaultCenter] addObserver: self
                                   selector: @selector(burnerInUse:)
                                   name: BurnerInUse
                                   object: nil];

              [panel setFrameAutosaveName: @"BlankPanel"];
            [panel setFrameUsingName: @"BlankPanel"];
        }
    }
    return sharedPanel;
}

- (void) dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver: self
                           name: BurnerInUse
                           object: nil];
}


- (void) close
{
    [progressLabel setStringValue: @""];
    [super close];
}


- (void) activate
{
    [panel orderFront: self];
}

- (BOOL) windowShouldClose: (id)sender
{
    return !blanking;
}

//
// action methods
//
- (void) fastBlank: (id) sender
{
    [NSThread detachNewThreadSelector: @selector(blankThread:)
                            toTarget: self
                            withObject: fastButton];
}

- (void) completeBlank: (id) sender
{
    [NSThread detachNewThreadSelector: @selector(blankThread:)
                            toTarget: self
                            withObject: completeButton];
}

- (void) burnerInUse: (id)sender
{
    if ([[[sender userInfo] objectForKey: @"InUse"] isEqualToString: @"YES"]) {
        [fastButton setEnabled: NO];
        [completeButton setEnabled: NO];
    } else {
        [fastButton setEnabled: YES];
        [completeButton setEnabled: YES];
    }
}

- (void) blankThread: (id)anObject
{
    EBlankingMode value;
    id<Burner> burner;
    NSDictionary *tools;
    id pool;
    pool = [NSAutoreleasePool new];

    if (![[AppController appController] lockBurner]) {
        logToConsole(MessageStatusError, _(@"Common.burnerLocked"));
        goto blank_end;
    }

    blanking = YES;

    if (anObject == completeButton)
        value = fullBlank;
    else
        value = fastBlank;

    [progressLabel setTextColor: [NSColor blueColor]];
    [progressLabel setStringValue: _(@"BlankPanel.inProcess")];

    tools = [[NSUserDefaults standardUserDefaults] objectForKey: @"SelectedTools"];
    burner = [[AppController appController] bundleForKey: [tools objectForKey: @"BurnSW"]];

    if (burner) {
        if ([burner blankCDRW: value
               withParameters: [[NSUserDefaults standardUserDefaults] dictionaryRepresentation]]) {
            [progressLabel setTextColor: [NSColor greenColor]];
            [progressLabel setStringValue: _(@"BlankPanel.success")];
            logToConsole(MessageStatusInfo, _(@"BlankPanel.success"));
        } else {
            [progressLabel setTextColor: [NSColor redColor]];
            [progressLabel setStringValue: _(@"BlankPanel.error")];
            logToConsole(MessageStatusError, _(@"BlankPanel.error"));
        }
    }

    [[AppController appController] unlockBurner];

    blanking = NO;

blank_end:
    RELEASE(pool);
    [NSThread exit];
}


//
// class methods
//
+ (id) sharedPanel
{
    if (sharedPanel == nil) {
        sharedPanel = [[BlankPanel alloc] init];
    }

    return sharedPanel;
}


@end
