/*
Bullet Continuous Collision Detection and Physics Library Maya Plugin
Copyright (c) 2008 Walt Disney Studios
 
This software is provided 'as-is', without any express or implied warranty.
In no event will the authors be held liable for any damages arising
from the use of this software.
Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it freely,
subject to the following restrictions:
 
1. The origin of this software must not be misrepresented; you must
not claim that you wrote the original software. If you use this
software in a product, an acknowledgment in the product documentation
would be appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must
not be misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.
 
Written by: Nicola Candussi <nicola@fluidinteractive.com>
*/

//bt_convex_hull_shape.h

#ifndef DYN_BT_CONVEX_HULL_SHAPE_H
#define DYN_BT_CONVEX_HULL_SHAPE_H

#include <vector>

#include "convex_hull_shape_impl.h"
#include "bt_collision_shape.h"

class bt_convex_hull_shape_t: public bt_collision_shape_t, public convex_hull_shape_impl_t 
{
public:
    virtual void gl_draw(size_t draw_style) {

        if(m_vertices.empty() || m_indices.empty()) return;

        glPushMatrix();
        glTranslatef(m_center[0], m_center[1], m_center[2]);
        float angle;
        vec3f axis;
        q_to_axis_angle(m_rotation, axis, angle);
        glRotatef(rad2deg(angle), axis[0], axis[1], axis[2]);
        glEnableClientState(GL_VERTEX_ARRAY);
        glEnableClientState(GL_NORMAL_ARRAY);
        if(draw_style & collision_shape_t::kDSSolid) {
            glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
        } else {
            glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
        }
        glVertexPointer(3, GL_FLOAT, 0, &(m_vertices[0]));
        glNormalPointer(GL_FLOAT, 0, &(m_normals[0]));
        glDrawElements(GL_TRIANGLES, m_indices.size(), GL_UNSIGNED_INT, &(m_indices[0]));
        glDisableClientState(GL_VERTEX_ARRAY);
        glDisableClientState(GL_NORMAL_ARRAY);
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
        glPopMatrix();
    }

    virtual void set_scale(vec3f const& s1) {
 //       shape()->setLocalScaling(btVector3(s[0], s[1], s[2]));
		btVector3 s(s1[0],s1[1],s1[2]);
		
		float delSq = 0.f;
		for(int i = 0; i < 3; i++)
		{
			float del = s[i] - m_cachedScaling[i];
			delSq += del * del;
		}
		if (delSq > FLT_EPSILON)
		{
//			btAssert(shape()->getType()==
			btCompoundShape* compound =(btCompoundShape*)shape();
			btConvexHullShape* child = (btConvexHullShape*)compound->getChildShape(0);
/*
			btTransform scaleTransf;
			scaleTransf.setIdentity();
			scaleTransf.getBasis()[0][0] = (s/m_cachedScaling)[0];
			scaleTransf.getBasis()[1][1] = (s/m_cachedScaling)[1];
			scaleTransf.getBasis()[2][2] = (s/m_cachedScaling)[2];
			btTransform combinedTr = tr.inverse() * (scaleTransf * (tr * vtx));
*/
			const btTransform& tr = compound->getChildTransform(0);
			for (int i=0;i<child->getNumPoints();i++)
			{
				btVector3 oldPoint = child->getUnscaledPoints()[i];
				btVector3 parentPoint = tr*oldPoint;
				btVector3 parentPointScaled = parentPoint*(s/m_cachedScaling);
				btVector3 childPoint = tr.inverse()*parentPointScaled;
				child->getUnscaledPoints()[i]=childPoint;
			}
			m_cachedScaling = s;
		}

        update();
    }

    virtual void get_scale(vec3f& s) {
//        const btVector3& scale = shape()->getLocalScaling();
//        s = vec3f(scale.x(), scale.y(), scale.z());
        s = vec3f(m_cachedScaling.x(), m_cachedScaling.y(), m_cachedScaling.z());
    }

    virtual float volume()                  { return m_volume;  }
    virtual vec3f const& local_inertia()    { return m_local_inertia;  }
    virtual vec3f const& center()           { return m_center; }
    virtual quatf const& rotation()         { return m_rotation;  }

protected:
    friend class bt_solver_t;

    bt_convex_hull_shape_t(vec3f const* vertices, size_t num_vertices,
                           vec3f const* normals,
                            unsigned int const *indices, size_t num_indices): 
                        bt_collision_shape_t(),
                        m_normals(normals, normals + num_vertices),
                        m_indices(indices, indices + num_indices)
    { 
        m_volume = ::volume(vertices, (int*)indices, num_indices);
        m_center = center_of_mass(vertices, (int*)indices, num_indices);
        mat3x3f I = inertia(vertices, (int*)indices, num_indices, m_center);
        m_rotation = diagonalizer(I);

        mat3x3f Q, Qinv; 
        q_to_mat(m_rotation, Q); 
        q_to_mat(qconj(m_rotation), Qinv);
        
        //D = trans(Q) * I * Q;
        m_local_inertia = diag(prod(trans(Q), mat3x3f(prod(I, Q))));

        m_vertices.resize(num_vertices);
        for(size_t i = 0; i < m_vertices.size(); ++i) {
            m_vertices[i] = prod(Qinv, vertices[i] - m_center);
        }

        m_ch_shape.reset(new btConvexHullShape((float const*)&(m_vertices[0]), num_vertices, sizeof(vec3f)));
        btCompoundShape *compound_shape = new btCompoundShape;
//		btTransform childTransf;
//		childTransf.setIdentity();
//		childTransf.setOrigin(btVector3(m_center[0], m_center[1], m_center[2]));
  //      compound_shape->addChildShape(childTransf, m_ch_shape.get());

///*
        compound_shape->addChildShape(btTransform(btQuaternion(m_rotation[1],
                                                               m_rotation[2],
                                                               m_rotation[3],
                                                               m_rotation[0]),
                                                  btVector3(m_center[0],
                                                            m_center[1],
                                                            m_center[2])),
                                      m_ch_shape.get());
//*/
        set_shape(compound_shape);
		m_cachedScaling[0] = m_cachedScaling[1] = m_cachedScaling[2] = 1.f;
    }

    void update()
    {
        //apply the scaling
        btVector3 const& scale = m_ch_shape->getLocalScaling();
        btVector3 const* points = m_ch_shape->getUnscaledPoints();
        for(int i = 0; i < m_ch_shape->getNumPoints(); ++i) {
            m_vertices[i] = vec3f(scale.x() * points[i].x(), scale.y() * points[i].y(), scale.z() * points[i].z()); 
        }
        m_volume = ::volume(&(m_vertices[0]), (int*)&(m_indices[0]), m_indices.size());
        mat3x3f I = inertia(&(m_vertices[0]), (int*)&(m_indices[0]), (int)m_indices.size(), vec3f(0, 0, 0));
        //std::cout << I << std::endl;
        //m_rotation = diagonalizer(I);
        //std::cout << rotation << std::endl;
        //the rotation shouldn't change from scaling

        mat3x3f Q, Qinv; 
        q_to_mat(m_rotation, Q); 
        q_to_mat(qconj(m_rotation), Qinv);

        //D = Q * I * trans(Q);
        m_local_inertia = diag(prod(trans(Q), mat3x3f(prod(I, Q))));
    }

private:
    shared_ptr<btConvexHullShape>   m_ch_shape;
    std::vector<vec3f>              m_vertices;
    std::vector<vec3f>              m_normals;
    std::vector<unsigned int>       m_indices; 

    float                           m_volume;
    vec3f                           m_center;
    quatf                           m_rotation;
    vec3f                           m_local_inertia;
	btVector3						m_cachedScaling;
};

#endif
