# -*- perl -*-

# Copyright (c) 2003 by Jeff Weisberg
# Author: Jeff Weisberg <argus @ tcp4me.com>
# Date: 2003-Jan-23 21:23 (EST)
# Function: Distributed Argus Redundancy Protocol
#
# $Id: DARP.pm,v 1.30 2008/06/25 16:03:17 jaw Exp $

package DARP;
@ISA = qw(Configable);

BEGIN {
    eval {
	require Digest::MD5;
	require Digest::HMAC;
        Digest::MD5->import('md5');
        Digest::HMAC->import('hmac_hex');
	
	$::HAVE_DARP = 1;
    };
}

use DARP::Conf;
use DARP::Master;
use DARP::Slave;
use DARP::Service;
use DARP::Watch;
use DARP::Misc;
use Argus::Encode;

use strict qw(refs vars);
use vars qw(@ISA $doc $mode $info);

$mode  = 'disabled';
$info  = undef;

$doc = {
    package => __PACKAGE__,
    file    => __FILE__,
    isa     => [],
    versn => '3.3',
    html  => 'darp',
    methods => {},
    conf => {
	quotp => 1,
	bodyp => 1,
    },
    fields => {
	tag     => {},
	slaves  => {},
	masters => {},
	debug  => {
	    descr => 'send all sorts of gibberish to the log',
	    attrs => ['inherit', 'config'],
	},
	port => {
	    descr => 'TCP port to listen on for slave connections',
	    attrs => ['config'],
	    default => 2055,	# XXX not officially assigned
	},
	timeout => {
	    descr => 'maximum time to wait for input from slave',
	    attrs => ['config'],
	    default => 120,
	},

    },
};

sub config {
    my $me = shift;
    my $cf = shift;
    
    $me->{tag} = $me->{name};
    $me->init_from_config( $cf, $doc, '' );
    $me->cfcleanup();
    
    $me->{status} = 'up';
    $info = $me;

    $me->{masters} ||= [];
    $me->{slaves}  ||= [];
    
    $me->{all} = [ $me, @{$me->{masters}}, @{$me->{slaves}} ];

    if( @{$me->{masters}} && @{$me->{slaves}} ){
	# It's a lot like life
	# This play between the sheets
	# With you on top
	# And me underneath
	# Forget all about equality
	# Let's play master and servant
	#   -- Depeche Mode, Master and Servant
	::loggit( "DARP running in mixed master/slave mode", 0 );
	$mode = 'master/slave';
    }
    elsif( @{$me->{masters}} ){
	# We cannot all be masters
	#   -- Shakespeare, Othello
	::loggit( "DARP running in slave mode", 0 );
	$mode = 'slave';
    }
    elsif( @{$me->{slaves}} ){
	# Let every man be master
	#   -- Shakespeare, Macbeth
	::loggit( "DARP running in master mode", 0 );
	$mode = 'master';
    }
    else{
	# I climbed a high rock to reconnoitre,
	# but could see no sign neither of man nor cattle,
	# only some smoke rising from the ground.
	#   -- Homer, Odyssey
	
	# no master specified, no slaves specified, wtf?
	$cf->warning( "DARP enabled but not configured" );
	$mode = 'enabled';
    }
    
    DARP::Master->create( $me ) if @{$me->{slaves}};
    DARP::Service->enable();
    DARP::Misc->enable();

    $me;
}

sub unique { 'DARP' };

sub readconfig {
    my $cf  = shift;
    my $mom = shift;
    my $me  = __PACKAGE__->new;
    my( $nomoredata );
    
    my $line = $cf->nextline();
    my($tag) = $line =~ /^DARP\s+\"(.*)\"/i;

    $me->{name} = $tag;
    $me->{parents} = [ $mom ] if $mom;
    $me->cfinit($cf, $tag, 'DARP');
    $me->{masters} = [];
    $me->{slaves}  = [];
    
    while( defined($_ = $cf->nextline()) ){
	print STDERR "  gotline: $_\n" if $::opt_d;
	if( /^\s*\}/ ){
	    # done
	    last;
	}
	
	elsif( /^master\s+\"(.*)\"\s+\{/i ){
	    my $tag = $1;
	    $nomoredata = 1;
	    $cf->ungetline( "DARP_Slave __DARP {" );
	    my $x = Service::readconfig($cf, $me, {tag => $tag, darp => 1} );
	    push @{$me->{masters}},  $x;
	    push @{$me->{children}}, $x;
	}

	elsif( /^slave /i ){
	    $nomoredata = 1;
	    $cf->ungetline( $_ );
	    eval {
		my $x = DARP::Conf::readconfig($cf, $me);
		push @{ $me->{slaves} }, $x;
		push @{$me->{children}}, $x;
	    };
	}
	
	elsif( /:/ ){
	    my($k, $v) = split /:[ \t]*/, $_, 2;
	    
	    if( $nomoredata ){
		$me->warning( "additional data not permitted here (ignored)" );
		next;
	    }
	    $cf->warning( "redefinition of parameter '$k'" )
	    	  if defined $me->{config}{$k};
	    $me->{config}{$k} = $v;
	}
	
	else{
	    eval{ $cf->error( "invalid entry in config file: '$_'" ); };
	    $cf->eat_block() if /\{\s*$/;
	}
    }

    $me->config( $cf );
    
}

sub gen_confs {

    my $r = '';
    if( $info ){
	$r = "\n" . $info->gen_conf();
    }
    $r;
}

# used in communication protocol
sub calc_hmac {
    my $key = shift;
    my %p   = @_;

    my $t = '';
    foreach my $k (sort keys %p){
	next unless defined $p{$k};
	next if $k eq 'hmac';
	$t .= "$k: $p{$k}; ";
    }
    # print STDERR "hm: $t\n";
    hmac_hex($t, md5($key), \&md5);
}

sub taglist {
    my $tags = shift;
    my %tags;
    
    # print STDERR "darp taglist\n";
    
    foreach my $t ( split /\s+/, $tags ){
	
	if( $t eq '*' ){
	    foreach my $s ( @{$DARP::info->{all}} ){
		my $t = $s->{name};
		$tags{$t} = 1;
	    }
	}
	elsif( $t eq 'SLAVES' ){
	    foreach my $s ( @{$DARP::info->{slaves}} ){
		my $t = $s->{name};
		$tags{$t} = 1;
	    }

	    # slaves includes me, if I have masters
	    $tags{$DARP::info->{tag}} = 1
		if @{ $DARP::info->{masters} };
	}
	elsif( $t eq 'MASTERS' ){
	    foreach my $s ( @{$DARP::info->{masters}} ){
		my $t = $s->{name};
		$tags{$t} = 1;
	    }

	    # masters includes me, if I have slaves
	    $tags{$DARP::info->{tag}} = 1
		if @{ $DARP::info->{slaves} };
	}
	else{
	    $tags{$t} = 1;
	}
    }

    keys %tags;
}

################################################################
sub cmd_darpstatus {
    my $ctl = shift;
    my $param = shift;

    $ctl->ok();

    if( $DARP::info->{tag} ){
	foreach my $d ( @{$info->{all}} ){
	    my $t = encode( $d->{name} );
	    $ctl->write( "$t:\t$d->{type}:$d->{status}\n" );
	}
	$ctl->final();
    }else{
	$ctl->bummer(500, 'DARP not enabled');
    }	
}

sub cmd_darp_tag {
    my $ctl = shift;

    if( $DARP::info->{tag} ){
	$ctl->ok();
	my $t = encode($DARP::info->{tag});
	$ctl->write( "self: $t\n" );
	$ctl->final();
    }else{
	$ctl->bummer(500, 'DARP not enabled');
    }
}

    
sub cmd_darp_master_info {
    my $ctl = shift;
    my $param = shift;

    $ctl->ok();
    
    if( $DARP::info->{tag} ){
	foreach my $d ( @{$DARP::info->{masters}} ){
	    my $tag = encode( $d->{name} );
	    # RSN - remove backwards compat tcp::hostname
	    my $adr = encode($d->{ip}{hostname} || $d->{tcp}{hostname});
	    
	    $ctl->write( "$tag: $adr\n" );
	}
	$ctl->final();
    }else{
	$ctl->bummer(500, 'DARP not enabled');
    }
}

################################################################
Doc::register( $doc );
Control::command_install( 'darp_status',     \&cmd_darpstatus,     "show darp status summary" );
Control::command_install( 'darp_mytag',      \&cmd_darp_tag,       "what is my tag" );
Control::command_install( 'darp_masters',    \&cmd_darp_master_info, "list masters info" );

1;
