# -*- perl -*-

# Copyright (c) 2003 by Jeff Weisberg
# Author: Jeff Weisberg <argus @ tcp4me.com>
# Created: 2003-Dec-03 17:44 (EST)
# Function: override various Service methods
#
# $Id: DARP::Service.pm,v 1.27 2007/12/31 22:20:34 jaw Exp $

package DARP::Service;

use strict qw(refs vars);
use vars qw($doc);

my %tag_mode_dfl =
(
 distributed => 'SLAVES',
 failover    => '*',
 redundant   => '*',
 );

$doc = {
    package => __PACKAGE__,
    file    => __FILE__,
    isa     => [],
    versn => '3.3',
    html  => 'darp',
    methods => {},

    fields => {
      darp::darp_mode => {
	  descr => 'what DARP mode to use',
	  attrs => ['config', 'inherit'],
	  vals  => ['none', 'failover', 'distributed', 'redundant'],
	  default => 'none',
      },
      darp::darp_tags => {
	  descr => 'list of DARP tags specfying who should monitor the service',
	  attrs => ['config', 'inherit'],
      },
      darp::darp_gravity => {
	  descr => 'multi monitor decision algorithm',
	  attrs => ['config', 'inherit'],
	  vals  => ['up', 'down', 'vote', 'self', 'ietf'],
	  default => 'down',
      },
      darp::darp_on_web => {
	  descr => 'display DARP data on web page',
	  attrs => ['config', 'inherit', 'bool'],
	  default => 'yes',
      },

      darp::tags     => {},
      darp::statuses => {},
      darp::synced   => { descr => 'status has been synchronized with master' },
	
    },
};

sub init {
    my $me = shift;
    my $cf = shift;
    
    $me->init_from_config( $cf, $doc, 'darp' );
    $me->{darp}{darp_tags} ||= $tag_mode_dfl{ $me->{darp}{darp_mode} };

    my @tags = DARP::taglist( $me->{darp}{darp_tags} );

    $me->{darp}{tags}{$_} = 1 foreach @tags;

    $me->{darp}{darp_tags} = join ' ', sort @tags;
    $me->Service::init( $cf, @_ );	# MonEl::init
    
    $me->{darp}{statuses}{ $DARP::info->{tag} } = $me->{status}
    if $DARP::info && $DARP::info->{tag};

    $me;
}

sub pre_start_check {
    my $me = shift;
    
    $me->debug( 'DARP pre-start check' );

    # check if:
    # mode = none
    # mode = dist|redund & tags{self}
    # mode = fail & tags{self} & ( self=master | master=down )

    my $mode = $me->{darp}{darp_mode} || 'none';
    my $self = $DARP::info->{tag};

    my $test = 0;

    # does this server run this test?
    $test = 1 unless $DARP::info && $DARP::info->{tag};
    $test = 1 if $mode eq 'none';
    $test = 1 if $mode eq 'distributed' && $me->{darp}{tags}{$self};
    $test = 1 if $mode eq 'redundant'   && $me->{darp}{tags}{$self};

    if( $mode eq 'failover' && $me->{darp}{tags}{$self} ){
	# I am master | all masters down

	$test = 1 if $DARP::info->{slaves} && @{$DARP::info->{slaves}};

	if( !$test ){
	    my $upp;
	    foreach my $m ( @{$DARP::info->{masters}} ){
		$upp = 1 if $m->{status} eq 'up';
	    }
	    $test = 1 unless $upp; # none are up
	}
    }
    
    if( $test ){
	$me->Service::pre_start_check( @_ );
    }else{
	$me->debug( 'DARP - skipping test' );
	$me->reschedule();
    }
}

# sub done {
#     my $me = shift;
#     
#     $me->debug( 'DARP done' );
#     $me->Service::done( @_ );
# }

sub update {
    my $me = shift;
    my $st = shift;
    my $sv = shift;

    $me->debug("darp update"); #XXX
    unless( $DARP::info && $DARP::info->{tag} ){
	# handle common case, and leave
	return $me->Service::update( $st, $sv, @_ );
    }
    
    my $sst = $st;
    my $ost = $me->{darp}{statuses}{ $DARP::info->{tag} };
    
    # update my own status
    $me->{darp}{statuses}{ $DARP::info->{tag} } = $st
	if $st;

    my $mode = $me->{darp}{darp_mode} || 'none';
    my $grav = $me->{darp}{darp_gravity};

    $me->debug("darp update: $mode, $grav, $st"); #XXX
    
    # failover+none - status = my status
    if( $DARP::info->{slaves} && @{$DARP::info->{slaves}}
	&& ($mode eq 'distributed' || $mode eq 'redundant')
	&& $grav ne 'self' ){
	
    	# check gravity, mode, status - calc new status

	my( %stat );
	if( $grav eq 'ietf' ){

	    # look at me and my slaves, skip those not up at the moment,
	    # nb: masters will aways be unk unless s+m
	    foreach my $d ( $DARP::info, @{ $DARP::info->{slaves} } ){
		my $t = $d->{name};
		next unless $me->{darp}{tags}{$t};
		next unless $d->{status} eq 'up';
		
		my $s  = $me->{darp}{statuses}{$t} || 'unk';
		$stat{$s} ++;
	    }
	}else{
	    foreach my $t ( keys %{ $me->{darp}{tags} } ){
		my $s = $me->{darp}{statuses}{$t} || 'unk';
		$stat{$s} ++;
	    }
	}

	if( $grav eq 'down' ){
	    $st = $stat{down} ? 'down' : 'up';
	}elsif( $grav eq 'up' ){
	    $st = $stat{up} ? 'up' : 'down';
	}elsif( $grav eq 'vote' || $grav eq 'ietf' ){
	    $st = $stat{up} >= $stat{down} ? 'up' : 'down';
	}
    }

    # NB: remote might not provide severity, try to avoid severity flap.
    if( $st eq 'down' ){
	$sv ||= $me->{currseverity};
	$sv = $me->{severity} if $sv eq 'clear';
    }else{
	$sv = 'clear';
    }
    
    # update self first, then masters
    $me->Service::update( $st, $sv, @_ );

    if( $DARP::info->{masters} ){
	foreach my $m ( @{ $DARP::info->{masters}} ){
	    # RSN - should be param, to be able to send aggr status instead
	
	    if( $sst ne $ost || !$me->{darp}{synced}{ $m->{tag} } ){
		
		if( ($mode ne 'none') && $m && ($m->{darps}{state} eq 'up') ){
		    # RPC the update to master - we send actual local status, not aggr status
		    
		    # Masters, I have to tell a tale of woe,
		    #   -- William Morris, The Earthly Paradise
		    
		    $me->debug( "DARP Slave - sending update to '$m->{tag}'" );
		    # RSN - name mapping
		    $m->send_command( 'darp_update',
				      object   => $me->unique(),
				      status   => $sst,
				      severity => $sv,
				      ) if $sst;
		    
		    # sync flag set in DARP_Slave, when we rcv rspns
		    # $me->{darp}{synced}{$m->{tag}} = $^T;
		}
	    }
	}
    }
    
}

sub webpage {
    my $me   = shift;
    my $fh   = shift;
    my $topp = shift;

    $me->Service::webpage($fh, $topp);

    return unless $me->{darp};
    return unless $me->{darp}{darp_on_web};
    return unless $me->{darp}{tags};
    
    print $fh "<!-- start of darp web -->\n";
    print $fh "<HR>\n<B>DARP Info</B>\n<TABLE CLASS=DARP>\n";
    # print $fh "<TR><TD COLSPAN=2>mode</TD><TD>$me->{darp}{darp_mode}</TD></TR>\n";

    foreach my $d ( $DARP::info, @{$DARP::info->{slaves}} ){
	my $t = $d->{name};
	next unless $me->{darp}{tags}{$t};	# only those monitoring this object
	my $s = $me->{darp}{statuses}{$t} || 'unknown';
	my $sv = $me->{ovstatus} eq 'down' ? 'critical' : 'major';
	my $c = MonEl::web_color($s, $sv );
	next if $d->{darpc} && $d->{darpc}{hidden};

	my $l = $d->{darpc}{label} || $t;
	
	# embolden this host
	$l = "<B>$l</B>" if $t eq $DARP::info->{tag};

	# make link if url configured
	if( $d->{darpc} && $d->{darpc}{remote_url} ){
	    $l = "<A HREF=\"$d->{darpc}{remote_url}?object="
		. $me->filename() . ";func=page\">$l</A>";
	}

	print $fh "<TR><TD>$l </TD><TD>status</TD><TD BGCOLOR=\"$c\">$s</TD></TR>\n";
    }
    
    print $fh "</TABLE>\n";
    print $fh "<!-- end of darp web -->\n";
    
}

sub about_more {
    my $me = shift;
    my $ctl = shift;

    $me->Service::about_more($ctl);
    $me->more_about_whom($ctl, 'darp', 'darp::statuses', 'darp::synced');
}

################################################################
# graph data tagging
sub graph_tag {
    my $me = shift;

    return unless $DARP::info;
    return unless $DARP::info->{tag};
    return unless $me->{darp};
    return unless $me->{darp}{darp_mode};
    return if $me->{darp}{darp_mode} eq 'none';

    return $DARP::info->{tag};
}

################################################################
# initialize
sub enable {
    no strict;
    foreach my $srvc (qw(DataBase Ping Prog TCP UDP Argus::Compute)){
	unshift @{ "${srvc}::ISA" }, __PACKAGE__;
    }
}
################################################################
Doc::register( $doc );


1;
