// aliados.cpp
// Revision 17-may-2005

#include "cpm.h"

#include "realconsole.h"
#include "directory.h"

#include <iostream>
#include <fstream>
#include <string>
#include <stdexcept>
#include <memory>
#include <vector>
#include <map>
#include <algorithm>
#include <iterator>

#include <stdlib.h>


using std::cin;
using std::cout;
using std::cerr;
using std::endl;
using std::flush;
using std::string;
using std::runtime_error;
using std::auto_ptr;
using std::vector;
using std::map;
using std::copy;
using std::back_inserter;


namespace aliados_impl_main {


const string version (VERSION);

void showversion ()
{
	cout << "aliados " << version <<
		" (C) 2004-2005 Julian Albo" << endl;
}

class Exit {
public:
	Exit (int value) : n (value) { }
	int retcode () { return n; }
private:
	int n;
};

class InvalidOption : public runtime_error {
public:
	InvalidOption (const string & str) :
		runtime_error ("Invalid option: " + str)
	{
	}
};

class NeedParameter : public runtime_error {
public:
	NeedParameter (const string & str) :
		runtime_error ("Option " + str + " needs a parameter")
	{
	}
};

class NeedParameters : public runtime_error {
public:
	NeedParameters (const string & str, size_t /*n*/) :
		runtime_error ("Option " + str + " needs parameters")
	{
	}
};

class InvalidParameter : public runtime_error {
public:
	InvalidParameter (const string & str) :
		runtime_error ("Inavlid parameter in option " + str)
	{
	}
};


class Op {
public:
	Op (int argc_n, char * * const argv_n);

	bool rawin () const { return flag_rawin; }
	bool rawout () const { return flag_rawout; }
	bool wskeys () const { return flag_wskeys; }
	string printer () const { return name_printer; }

	string program () const { return name_program; }
	string command () const { return command_line; }
	const vector <string> & args () const { return names_args; }
	void setdisks (CPM & cpm) const;

	const CpmOptions & getcpmoptions () const;
private:
	int argc;
	char * * const argv;

	bool flag_rawin;
	bool flag_rawout;
	bool flag_wskeys;
	bool flag_argdisk;
	bool flag_autodisk;
	string name_printer;
	string name_program;
	string command_line;
	mutable vector <string> names_args;

	typedef map <char, string> diskmap_t;
	diskmap_t diskmap;

	CpmOptions options;

	typedef bool (Op::* handle_option_t) (int & n);
	typedef map <string, handle_option_t> mapoption_t;
	static mapoption_t mapoption;

	static const string op_version;
	static const string op_raw;
	static const string op_rawin;
	static const string op_rawout;
	static const string op_prefix;
	static const string op_tpaend;
	static const string op_wskeys;
	static const string op_printer;
	static const string op_disk;
	static const string op_defdisk;
	static const string op_argdisk;
	static const string op_autodisk;
	static const string op_command;

	class OptionMapInit {
		friend class Op;
		OptionMapInit ();
	};
	friend class OptionMapInit;
	OptionMapInit mapinit;

	void setargdisks (CPM & cpm) const;

	handle_option_t gethandle (const char * str);

	bool h_default_option (int & n);
	bool h_end_options (int & n);
	bool h_option_d (int & n);
	bool h_option_s (int & n);
	bool h_option_z (int & n);
	bool h_option_version (int & n);
	bool h_option_raw (int & n);
	bool h_option_rawin (int & n);
	bool h_option_rawout (int & n);
	bool h_option_wskeys (int & n);
	bool h_option_prefix (int & n);
	bool h_option_tpaend (int & n);
	bool h_option_printer (int & n);
	bool h_option_disk (int & n);
	bool h_option_defdisk (int & n);
	bool h_option_argdisk (int & n);
	bool h_option_autodisk (int & n);
	bool h_option_command (int & n);
};


const string Op::op_version  ("--version");
const string Op::op_raw      ("--raw");
const string Op::op_rawin    ("--rawin");
const string Op::op_rawout   ("--rawout");
const string Op::op_tpaend   ("--tpaend");
const string Op::op_prefix   ("--prefix");
const string Op::op_wskeys   ("--wskeys");
const string Op::op_printer  ("--printer");
const string Op::op_disk     ("--disk");
const string Op::op_defdisk  ("--defdisk");
const string Op::op_argdisk  ("--argdisk");
const string Op::op_autodisk ("--autodisk");
const string Op::op_command  ("--command");

Op::mapoption_t Op::mapoption;

Op::OptionMapInit::OptionMapInit ()
{
	Op::mapoption ["--"]=            & Op::h_end_options;
	Op::mapoption ["-c"]=            & Op::h_option_command;
	Op::mapoption ["-d"]=            & Op::h_option_d;
	Op::mapoption ["-s"]=            & Op::h_option_s;
	Op::mapoption ["-v"]=            & Op::h_option_version;
	Op::mapoption ["-z"]=            & Op::h_option_z;
	Op::mapoption [Op::op_version]=  & Op::h_option_version;
	Op::mapoption [Op::op_raw]=      & Op::h_option_raw;
	Op::mapoption [Op::op_rawin]=    & Op::h_option_rawin;
	Op::mapoption [Op::op_rawout]=   & Op::h_option_rawout;
	Op::mapoption [Op::op_wskeys]=   & Op::h_option_wskeys;
	Op::mapoption [Op::op_prefix]=   & Op::h_option_prefix;
	Op::mapoption [Op::op_tpaend]=   & Op::h_option_tpaend;
	Op::mapoption [Op::op_printer]=  & Op::h_option_printer;
	Op::mapoption [Op::op_disk]=     & Op::h_option_disk;
	Op::mapoption [Op::op_defdisk]=  & Op::h_option_defdisk;
	Op::mapoption [Op::op_argdisk]=  & Op::h_option_argdisk;
	Op::mapoption [Op::op_autodisk]= & Op::h_option_autodisk;
	Op::mapoption [Op::op_command]=  & Op::h_option_command;
}


Op::Op (int argc_n, char * * const argv_n) :
	argc (argc_n),
	argv (argv_n),
	flag_rawin (false),
	flag_rawout (false),
	flag_wskeys (false),
	flag_argdisk (false),
	flag_autodisk (false)
{
	int n;
	for (n= 1; n < argc; ++n)
	{
		if ( (this->*gethandle (argv [n] ) ) (n) )
			break;
	}

	if (n < argc)
	{
		if (command_line.empty () )
		{
			name_program= argv [n];
			++n;
		}
		copy (argv + n, argv + argc, back_inserter (names_args) );
	}
}

Op::handle_option_t Op::gethandle (const char * str)
{
	mapoption_t::const_iterator it (mapoption.find (str) );
	if (it != mapoption.end () )
		return it->second;
	else
	{
		if (str [0] == '-')
			throw InvalidOption (str);
		else
			return & Op::h_default_option;
	}
}

bool Op::h_default_option (int & /*n*/)
{
	return true;
}

bool Op::h_end_options (int & /*n*/)
{
	return true;
}

bool Op::h_option_d (int & /*n*/)
{
	options.setdebugsystem (true);
	options.setdebugz80 (true);
	return false;
}

bool Op::h_option_s (int & /*n*/)
{
	options.setdebugsystem (true);
	return false;
}

bool Op::h_option_z (int & /*n*/)
{
	options.setdebugz80 (true);
	return false;
}

bool Op::h_option_version (int & /*n*/)
{
	showversion ();
	throw Exit (0);
}

bool Op::h_option_raw (int & /*n*/)
{
	flag_rawin= true;
	flag_rawout= true;
	return false;
}

bool Op::h_option_rawin (int & /*n*/)
{
	flag_rawin= true;
	return false;
}

bool Op::h_option_rawout (int & /*n*/)
{
	flag_rawout= true;
	return false;
}

bool Op::h_option_wskeys (int & /*n*/)
{
	flag_wskeys= true;
	return false;
}

bool Op::h_option_prefix (int & n)
{
	if (++n >= argc)
		throw NeedParameter (op_prefix);
	string prefarg= argv [n];
	char prefix;
	switch (prefarg.size () )
	{
	case 0:
		prefix= '\0';
		break;
	case 1:
		prefix= prefarg [0];
		break;
	default:
		throw runtime_error ("Prefix must be one char");
	}
	options.setprefix (prefix);
	return false;
}

bool Op::h_option_tpaend (int & n)
{
	if (++n >= argc)
		throw NeedParameter (op_tpaend);
	char * aux;
	unsigned long ultpaend= strtoul (argv [n], & aux, 0);
	if (* aux != '\0')
		throw InvalidParameter (op_tpaend);
	options.settpaend (static_cast <unsigned short> (ultpaend) );
	return false;
}

bool Op::h_option_printer (int & n)
{
	if (++n >= argc)
		throw NeedParameter (op_printer);
	name_printer= argv [n];
	return false;
}

bool Op::h_option_disk (int & n)
{
	if (++n >= argc -1)
		throw NeedParameters (op_disk, 2);
	string disk= argv [n];
	string path= argv [++n];
	string::size_type l= disk.size ();
	if (! (l == 1 || (l == 2 && disk [1] == ':') ) )
		throw runtime_error ("Invalid disk");
	diskmap [disk [0]  ]= path;
	return false;
}

bool Op::h_option_defdisk (int & n)
{
	if (++n >= argc)
		throw NeedParameter (op_tpaend);
	string disk= argv [n];

	string::size_type l= disk.size ();
	if (! (l == 1 || (l == 2 && disk [1] == ':') ) )
		throw runtime_error ("Invalid default disk");
	options.setdefaultdisk (disk [0] );
	return false;
}

bool Op::h_option_argdisk (int & /*n*/)
{
	flag_argdisk= true;
	return false;
}

bool Op::h_option_autodisk (int & /*n*/)
{
	flag_autodisk= true;
	return false;
}

bool Op::h_option_command (int & n)
{
	if (++n >= argc)
		throw NeedParameter (op_command);
	if (! command_line.empty () )
		throw runtime_error ("Multiple commands not allowed");
	command_line= argv [n];
	return false;
}

void Op::setargdisks (CPM & cpm) const
{
	map <string, char> dir_used;
	for (size_t i= 0, l= names_args.size (); i < l; ++i)
	{
		string & arg= names_args [i];
		string::size_type pos= arg.rfind ('/');
		if (pos != string::npos)
		{
			const string dir= arg.substr (0, pos);
			map <string, char>::iterator it= dir_used.find (dir);
			char diskletter;
			if (it != dir_used.end () )
			{
				diskletter= it->second;
			}
			else
			{
				diskletter= cpm.newdisk (dir);
				dir_used [dir]= diskletter;
			}
			arg= string (1, diskletter) + ':' +
				arg.substr (pos + 1);
		}
	}
}

void Op::setdisks (CPM & cpm) const
{
	if (flag_autodisk)
	{
		FindFile ff ("./", "disk[A-P]", true);
		string disk;
		while (ff.get (disk) == FindFile::Result::Ok)
		{
			if (disk.size () == 5)
			{
				char letter= disk [4];
				if (letter >= 'A' && letter <= 'P')
					cpm.setdisk (letter, disk);
			}
		}
	}
	for (diskmap_t::const_iterator it= diskmap.begin ();
		it != diskmap.end ();
		++it)
	{
		cpm.setdisk (it->first, it->second);
	}
	if (flag_argdisk)
		setargdisks (cpm);
}

const CpmOptions & Op::getcpmoptions () const
{
	return options;
}


int doit (int argc, char * * argv)
{
	const Op options (argc, argv);

	RealConsole console;

	auto_ptr <Printer> pprinter;
	const string printerfile= options.printer ();
	if (printerfile.empty () )
		pprinter.reset (new SpoolPrinter);
	else
		pprinter.reset (new FilePrinter (printerfile) );

	CPM cpm (console, * pprinter, options.getcpmoptions () );

	bool rawin= options.rawin ();
	bool rawout= options.rawout ();
	if (rawin)
		console.rawin ();
	if (rawout)
		console.rawout ();

	bool wskeys= options.wskeys ();
	if (wskeys)
		console.wskeys ();

	options.setdisks (cpm);

	string program= options.program ();
	string command= options.command ();

	if (program.empty () && command.empty () )
	{
		showversion ();
		cpm.interactive ();
	}
	else
	{
		// Build arguments.

		const vector <string> & args= options.args ();
		string cpmargs;
		for (size_t i= 0, l= args.size (); i < l; ++i)
		{
			string aux= args [i];
			if (! aux.empty () )
			{
				cpmargs+= ' ';
				cpmargs+= aux;
			}
		}
		if (command.empty () )
			cpm.runprogram (program, cpmargs);
		else
		{
			if (! cpmargs.empty () )
			{
				command+= ' ';
				command+= cpmargs;
			}
			cpm.runcommandline (command);
		}
	}
	return 0;
}


} // namespace aliados_impl_main


int main (int argc, char * * argv)
{
	using namespace aliados_impl_main;

	try
	{
		return doit (argc, argv);
	}
	catch (Exit & e)
	{
		return e.retcode ();
	}
	catch (std::exception & e)
	{
		cerr << "ERROR: " << e.what () << endl;
	}
	catch (const std::string & str)
	{
		cerr << "ERROR: " << str << endl;
	}
	catch (const char * str)
	{
		cerr << "ERROR: " << str << endl;
	}
	catch (...)
	{
		cerr << "UNEXPECTED ERROR CONDITION" << endl;
	}
	return 1;
}


// End of aliados.cpp
