#include <gtk/gtk.h>
#include <gconf/gconf-client.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <FLAC/metadata.h>
#include <string.h>
#include <sys/stat.h>
#include <assert.h>
#include <gst/gst.h>
#include <libintl.h>
#include <locale.h>
#include "config.h"

#define _(x) gettext(x)

//#define DEBUG
//#define MAX_EXTLEN 6
#define DATAFILE(x) (DATADIR #x)
#define ART_WIDTH 350
#define ICON_WIDTH 128
#define CANTEEN_CHOICES 4
#define GCONFDIR "/apps/albumplayer/"
#define GCONFKEY(x) (GCONFDIR #x)

static GtkTargetEntry targets[] = {
  { "text/uri-list", 0, 0 }
};

enum {
  COL_ID,
  COL_NR,
  COL_NAME,
  COL_SECS,
  N_COLS
};

enum {
  CAN_ICON,
  CAN_INFO,
  CAN_FILENAME,
  CAN_COLS
};

FLAC__StreamMetadata *sheet = NULL;
FLAC__StreamMetadata info;
FLAC__StreamMetadata *tags = NULL;
int cur_track = 0;
int loaded = 0;
int playing = 0;
int have_artwork = 0;
GstElement *pipeline, *source;
GRand *gen;
GConfClient *gconf;
GtkBuilder *builder;
GdkPixbuf *questionmark;
GdkPixbuf *coverlayer;
gchar *last_canteen = NULL;
int sched_seek = -1;

GdkPixbuf* scale_width(GdkPixbuf* pb, float nw) {
  double width = gdk_pixbuf_get_width(pb);
  double height = gdk_pixbuf_get_height(pb);
  double ar = height / width;
  GdkPixbuf* res = gdk_pixbuf_scale_simple(pb, nw, nw*ar, GDK_INTERP_BILINEAR);
  return res;
}

FLAC__byte* get_comment_md(FLAC__byte* name, FLAC__StreamMetadata *md) {  
    FLAC__StreamMetadata_VorbisComment_Entry* first =
      md->data.vorbis_comment.comments;
    FLAC__uint32 comments = md->data.vorbis_comment.num_comments;
    int i = 0;
    
    for(; i<comments; i++) {
      char* value = strchr(first[i].entry,'=') + 1;
      if(g_str_has_prefix(first[i].entry, name))
	return value;
    }
    return NULL;
}

FLAC__byte* get_comment(FLAC__byte* name) {
  if(tags) {
      return get_comment_md(name, tags);
  } else return NULL;
}

FLAC__uint64 get_track_offset(unsigned tr) {
  if(sheet) {
    assert(tr < sheet->data.cue_sheet.num_tracks);
    return sheet->data.cue_sheet.tracks[tr].offset;
  } else return 0;
}

FLAC__uint64 get_track_offset_s(unsigned tr) {
  return get_track_offset(cur_track) / info.data.stream_info.sample_rate;
}

unsigned sample_rate() {
  return info.data.stream_info.sample_rate;
}

FLAC__uint64 total_samples() {
  return info.data.stream_info.total_samples;
}

FLAC__uint64 get_track_seconds(unsigned tr) {
  if(sheet) {
    assert(tr < sheet->data.cue_sheet.num_tracks);
    FLAC__uint64 samples = get_track_offset(tr + 1) - get_track_offset(tr);
    return samples / sample_rate();
  } else {
    return total_samples() / sample_rate();
  }
}

FLAC__byte get_track_type(unsigned tr) {
  if(sheet) {
    assert(tr < sheet->data.cue_sheet.num_tracks);
    return sheet->data.cue_sheet.tracks[tr].type;
  } else return 0;
}

FLAC__byte get_track_number(unsigned tr) {
  if(sheet) {
    assert(tr < sheet->data.cue_sheet.num_tracks);
    return sheet->data.cue_sheet.tracks[tr].number;
  } else return 1;
}

FLAC__byte get_tracks() {
  if(sheet) {
    FLAC__byte tr = sheet->data.cue_sheet.num_tracks;
//    if(get_track_type(tr-1) == 0) return tr - 1;
//    else return tr;
    return tr - 1;
  } else return 1;
}

FLAC__byte* get_track_tag(unsigned tr, char *what) {
  char* query_string = g_strdup_printf("%s(%02d)", what, get_track_number(tr));
  FLAC__byte* try = NULL;

  if((try = get_comment(query_string))) goto finished;

  g_free(query_string);
  query_string = g_strdup_printf("%s(%d)", what, get_track_number(tr));
  
  if((try = get_comment(query_string))) goto finished;
  
  g_free(query_string);
  query_string = g_strdup_printf("%s[%d]", what, get_track_number(tr));
  if((try = get_comment(query_string))) goto finished;
  
  if(!g_strcmp0(what,"TITLE"))
    g_message(_("The %s tag is unset. Using %s instead."),
	      query_string, what);

  try = get_comment(what);

finished:
  g_free(query_string);
  return try;
}

int set_info(const char* filename) {
  int ret = FLAC__metadata_get_streaminfo(filename, &info);
  return ret;
}

FLAC__StreamMetadata* set_sheet(const char* filename) {
  if(sheet)  FLAC__metadata_object_delete(sheet);
  if(!FLAC__metadata_get_cuesheet(filename, &sheet))
    g_message(_("File does not include cue sheet."));
  return sheet;
}

FLAC__StreamMetadata* set_tags(const char* filename) {
  if(tags)  FLAC__metadata_object_delete(tags);
  if(!FLAC__metadata_get_tags(filename, &tags))
    g_message(_("File has no tags set."));
  return tags;
}

GdkPixbuf* get_flac_artwork(const char* filename) {
  FLAC__StreamMetadata *res = NULL;
  GdkPixbufLoader* loader = gdk_pixbuf_loader_new();
  GdkPixbuf* ret = NULL;
  if(FLAC__metadata_get_picture(filename, &res, -1, NULL, NULL, -1, -1, -1, -1)) {
    FLAC__StreamMetadata_Picture pic = res->data.picture;
    gdk_pixbuf_loader_write(loader, pic.data, pic.data_length, NULL);
    gdk_pixbuf_loader_close(loader, NULL);
    if(ret = gdk_pixbuf_loader_get_pixbuf(loader)) g_object_ref(ret);
    FLAC__metadata_object_delete(res);
  } else {
    gdk_pixbuf_loader_close(loader, NULL);
    g_message(_("File does not contain cover art."));
  }
  g_object_unref(loader);

  return ret;
}

int detail_mode() {
  GtkToggleAction *acti = GTK_TOGGLE_ACTION(gtk_builder_get_object(builder, "toggle_detail"));
  return gtk_toggle_action_get_active(acti);
}

void toggle_view(GtkWidget *action) {
  if(!loaded) return;
  gboolean active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
  GtkWidget *detail = GTK_WIDGET(gtk_builder_get_object(builder, "detail"));
  if(!active) gtk_widget_hide(detail);
  else gtk_widget_show(detail);
  gtk_window_resize(GTK_WINDOW(gtk_builder_get_object(builder, "player")),
		    20, 20);
  gconf_client_set_bool(gconf, GCONFKEY(show_detail), active, NULL);
}

/*GdkPixbuf* get_folder_artwork(const char* dir) {
  const char** name = &artwork_files[0];
  GdkPixbuf* res = NULL;
  char* fname = calloc(sizeof(char), strlen(*name) + MAX_EXTLEN);

  while(*name) {
    const char** ext = &image_exts[0];
    while(*ext) {
      strcpy(fname, *name); strcat(fname, *ext);
      res = gdk_pixbuf_new_from_file(fname, NULL);
      if(res) goto found;
      ext++;
    }
    name++;
  }

  if(!res) res = gdk_pixbuf_new_from_file(BASEFILE(empty.png), NULL);
found:
  free(fname);
  return res;
}
*/

int quit_cb(gpointer data) {
  gtk_main_quit();
  return 0;
}

void dnd_init() {
  GtkWidget *artwork = GTK_WIDGET(gtk_builder_get_object(builder, "player"));
  gtk_drag_dest_set(artwork, GTK_DEST_DEFAULT_ALL, targets, G_N_ELEMENTS(targets), GDK_ACTION_COPY);
}

void update_artwork(const char* selected) {
  GdkPixbuf *aw_flac;

  GtkWidget *artwork = GTK_WIDGET(gtk_builder_get_object(builder, "artwork"));
  GtkWidget *aw_frame = GTK_WIDGET(gtk_builder_get_object(builder, "aw_frame"));

  have_artwork = 0;
  if(selected) aw_flac = get_flac_artwork(selected);
  if(selected && aw_flac) {
    GdkPixbuf* aw_scaled = gdk_pixbuf_scale_simple(aw_flac, ART_WIDTH, ART_WIDTH, GDK_INTERP_BILINEAR);
    gtk_image_set_from_pixbuf(GTK_IMAGE(artwork), aw_scaled);
    gtk_widget_show(aw_frame);
    have_artwork = 1;
    g_object_unref(aw_scaled);
    g_object_unref(aw_flac);
  } else gtk_widget_hide(aw_frame);
  gtk_window_resize(GTK_WINDOW(gtk_builder_get_object(builder, "player")),
		    20, 20);
}

char* seconds_to_time(int secs) {
  char* res_time = g_strdup_printf("%d:%02d", secs / 60, secs % 60);
  return res_time;
}

gchar* format_seconds (GtkScale *scale,
                       gdouble   value)
{
  if(loaded)
    return g_strdup_printf ("%s/%s", seconds_to_time(value), seconds_to_time(get_track_seconds(cur_track)));
  else return g_strdup("");
 }

void cur_bold_func(GtkTreeViewColumn *col, GtkCellRenderer *cell,
		   GtkTreeModel *model, GtkTreeIter *it, gpointer data) {
  GValue tr = {0};
  gtk_tree_model_get_value(GTK_TREE_MODEL(model), it, COL_ID, &tr);

  g_object_set(G_OBJECT(cell), "ellipsize", PANGO_ELLIPSIZE_MIDDLE, NULL);
  
  if(g_value_get_int(&tr) == cur_track) {
    g_object_set(G_OBJECT(cell), "weight", 800, NULL);
    g_object_set(G_OBJECT(cell), "style", PANGO_STYLE_ITALIC, NULL);
  } else {
    g_object_set(G_OBJECT(cell), "weight", 400, NULL);
    g_object_set(G_OBJECT(cell), "style", PANGO_STYLE_NORMAL, NULL);
  }
}

void update_np() {
  GtkWidget* seekbar = GTK_WIDGET(gtk_builder_get_object(builder, "seekbar"));
  GtkWidget* tool_canteen = GTK_WIDGET(gtk_builder_get_object(builder, "tool_canteen"));
  GtkWidget* tool_back = GTK_WIDGET(gtk_builder_get_object(builder, "tool_back"));
  GtkWidget* tool_play = GTK_WIDGET(gtk_builder_get_object(builder, "tool_play"));
  GtkWidget* tool_next = GTK_WIDGET(gtk_builder_get_object(builder, "tool_next"));
  
  if(loaded) {
    GtkWidget* al_lab = GTK_WIDGET(gtk_builder_get_object(builder, "album"));
    GtkWidget* ti_lab = GTK_WIDGET(gtk_builder_get_object(builder, "track"));
    char *artist = get_track_tag(cur_track, "ARTIST");
    char* album = get_track_tag(cur_track, "ALBUM");
    char* disc_no = get_track_tag(cur_track, "DISCNUMBER");
    char* title = get_track_tag(cur_track, "TITLE");
    char* al_str = g_markup_printf_escaped(_("on <i>%s</i> from <i>%s</i>"),
					   album ? album : _("an unknown album"), artist ? artist : _("an unknown artist"));
    char* ti_str = g_markup_printf_escaped(_("Track %d of %d: <i>%s</i>"),
					   get_track_number(cur_track),
					   get_track_number(get_tracks()-1), title ? title : _("Unknown Title"));
    if(disc_no) {
      char* ti_tmp = al_str;
      al_str = g_strdup_printf(_("%s, CD %s"), ti_tmp, disc_no);
      g_free(ti_tmp);
    }
    gtk_label_set_markup(GTK_LABEL(al_lab), al_str);
    gtk_label_set_markup(GTK_LABEL(ti_lab), ti_str);

    g_free(al_str);
    g_free(ti_str);

    gtk_range_set_range(GTK_RANGE(seekbar), 0, get_track_seconds(cur_track));
    gtk_widget_queue_draw(GTK_WIDGET(seekbar));
    gtk_widget_queue_draw(GTK_WIDGET(gtk_builder_get_object(builder, "tracklist")));
  }
  GtkWidget* detail = GTK_WIDGET(gtk_builder_get_object(builder, "detail"));
  if(loaded && detail_mode()) gtk_widget_show(detail);
  else gtk_widget_hide(detail);

  gtk_widget_set_sensitive(tool_next, loaded);
  gtk_widget_set_sensitive(tool_play, loaded);
  gtk_widget_set_sensitive(tool_back, loaded);
  gtk_widget_set_sensitive(seekbar, loaded);
  gtk_widget_set_sensitive(tool_canteen, last_canteen != NULL);
}

guint64 pos_within_track(int tr) {
  GstFormat fmt = GST_FORMAT_TIME;
  guint64 pos;
  gst_element_query_position(pipeline, &fmt, &pos);
  int secs = GST_TIME_AS_SECONDS(pos);

  return secs;
}

void pause() {
  GtkWidget* pausebutton = GTK_WIDGET(gtk_builder_get_object(builder, "tool_play"));

  gst_element_set_state(pipeline, GST_STATE_PAUSED);
  gtk_tool_button_set_stock_id(GTK_TOOL_BUTTON(pausebutton), GTK_STOCK_MEDIA_PLAY);
  playing = 0;
}

void play() {
  GtkWidget* pausebutton = GTK_WIDGET(gtk_builder_get_object(builder, "tool_play"));

  gst_element_set_state(pipeline, GST_STATE_PLAYING);
  gtk_tool_button_set_stock_id(GTK_TOOL_BUTTON(pausebutton), GTK_STOCK_MEDIA_PAUSE);
  playing = 1;
}

GtkListStore *get_tracklist() {
  return GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(gtk_builder_get_object(builder, "tracklist"))));
}

GtkListStore *get_canteen() {
  return GTK_LIST_STORE(gtk_icon_view_get_model(GTK_ICON_VIEW(gtk_builder_get_object(builder, "canteen"))));
}

void clear_tracklist() {
  gtk_list_store_clear(get_tracklist());
}

void canteen_init() {
  GtkWidget *list = GTK_WIDGET(gtk_builder_get_object(builder, "canteen"));
  GtkListStore *model = gtk_list_store_new(CAN_COLS, GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_STRING);
  gtk_icon_view_set_model(GTK_ICON_VIEW(list), GTK_TREE_MODEL(model));
}

void clear_canteen() {
  gtk_list_store_clear(get_canteen());
}

void add_to_tracklist(int tr) {
  GtkListStore *model = get_tracklist();
  GtkTreeIter iter;
  gtk_list_store_append(model, &iter);

  gchar* time = seconds_to_time(get_track_seconds(tr));
  gtk_list_store_set(model, &iter, COL_ID, tr, COL_NR, get_track_number(tr), COL_NAME, get_track_tag(tr, "TITLE"), COL_SECS, time, -1);
  g_free(time);
}

void set_status(const char* ti, const char* al) {
  GtkWidget* al_lab = GTK_WIDGET(gtk_builder_get_object(builder, "album"));
  GtkWidget* ti_lab = GTK_WIDGET(gtk_builder_get_object(builder, "track"));
  gtk_label_set_text(GTK_LABEL(al_lab), al);
  gtk_label_set_text(GTK_LABEL(ti_lab), ti);
}

void canteen_close() {
  GtkWidget *artwork = GTK_WIDGET(gtk_builder_get_object(builder, "artwork"));
  GtkWidget *aw_frame = GTK_WIDGET(gtk_builder_get_object(builder, "aw_frame"));
  GtkWidget *canteen = GTK_WIDGET(gtk_builder_get_object(builder, "canteen_box"));
  if(have_artwork) gtk_widget_show(artwork);
  else gtk_widget_hide(aw_frame);
  gtk_widget_hide(canteen);
  gtk_window_resize(GTK_WINDOW(gtk_builder_get_object(builder, "player")),
		    20, 20);
}

gboolean is_empty(const gchar *str) {
  for(int i = 0; str[i]; i++)
    if(!g_ascii_isspace(str[i])) return false;
  return true;
}

int choose_rand_nonempty(int n, int k, int* notin, int notin_s) {
  int die = g_rand_int_range(gen, 0, n);
  if(memchr(notin, die, sizeof(int)*notin_s)) return choose_rand_nonempty(n, k, notin, notin_s);
  return die;
}

int* choose_k(int n, int k, int* length) {
  int i = 0;
  *length = k > n ? n : k;
  int* dest = calloc(sizeof(int), *length);
  if(k >= n) {
    for(i = 1; i <= n; i++) dest[i-1] = i-1;
    return dest;
  }
  for(i = 1; i <= *length; i++) dest[i-1] = choose_rand_nonempty(n, k, dest, i-1);
  return dest;
}

gchar** get_nonempty(gchar **lines, int length, int* newlength) {
  int nlength = 0;
  gchar ** ret = calloc(sizeof(gchar*), length);
  
  for(int i = 0; i < length; i++)
    if(is_empty(lines[i])) g_free(lines[i]);
    else ret[nlength++] = lines[i];
  *newlength = nlength;
  g_free(lines);
  return ret;
}

gchar* rel_path(const gchar* base, const gchar* filename) {
  if(g_path_is_absolute(filename)) return g_strdup(filename);
  else return g_build_filename(base, filename, NULL);
}

void add_to_canteen(const char* filename) {
  GtkListStore *model = get_canteen();
  GdkPixbuf *artwork = get_flac_artwork(filename);
  
  if(!artwork) {
    artwork = questionmark;
    g_object_ref(artwork);
  }
  
  GdkPixbuf *aw_scaled = gdk_pixbuf_scale_simple(artwork, ICON_WIDTH, ICON_WIDTH, GDK_INTERP_BILINEAR);
  gdk_pixbuf_composite(coverlayer, aw_scaled, 0, 0, ICON_WIDTH, ICON_WIDTH, 0, 0, ICON_WIDTH / 200.0, ICON_WIDTH / 200.0, GDK_INTERP_BILINEAR, 255);
  gchar* album = NULL, *artist = NULL, *discno = NULL;
  
  g_object_unref(artwork);

  FLAC__StreamMetadata* md;
  if(FLAC__metadata_get_tags(filename, &md)) {
    album = get_comment_md("ALBUM", md);
    artist = get_comment_md("ARTIST", md);
    discno = get_comment_md("DISCNUMBER", md);
    if(album) album = g_strdup(album);
    if(artist) artist = g_strdup(artist);
    FLAC__metadata_object_delete(md);
  }

  if(!album) album = g_path_get_basename(filename);
  if(!artist) artist = g_strdup(_("Unknown Artist"));
  gchar* tooltip = g_strdup_printf(_("%s by %s"), album, artist);

  if(discno) {
    gchar* ti_tmp = tooltip;
    tooltip = g_strdup_printf(_("%s, CD %s"), ti_tmp, discno);
    g_free(ti_tmp);
  }

  GtkTreeIter iter;
  gtk_list_store_append(model, &iter);
  
  gtk_list_store_set(model, &iter, CAN_ICON, aw_scaled, CAN_INFO, tooltip, CAN_FILENAME, filename, -1);
  g_free(tooltip);
  g_free(album);
  g_free(artist);
  g_object_unref(aw_scaled);
}

void canteen(const char* filename) {
  int len, ret;
  int* dest = NULL;
  gchar *contents = NULL;
  static gchar **lines = NULL;
  static int length = 0;
  
  if(lines && filename == last_canteen) goto skip;

  if(lines) g_strfreev(lines);
  if(!g_file_get_contents(filename, &contents, NULL, NULL)) return;
  lines = g_strsplit(contents, "\n", 0);
  g_free(contents);

  length = 0; for(; lines[length]; length++);
  lines = get_nonempty(lines, length, &length);

skip:
  dest = choose_k(length, CANTEEN_CHOICES, &ret);
  if(ret == 0) goto clean;

  GtkWidget *artwork = GTK_WIDGET(gtk_builder_get_object(builder, "artwork"));
  GtkWidget *canteen = GTK_WIDGET(gtk_builder_get_object(builder, "canteen_box"));
  gtk_widget_hide(artwork);
  gtk_widget_show(canteen);
  gtk_widget_show(GTK_WIDGET(gtk_builder_get_object(builder, "aw_frame")));
  clear_canteen();
  
  gchar* m3u_path = g_path_get_dirname(filename);
  for(int i = 0; i < ret; i++) {
    gchar* filepath = rel_path(m3u_path, lines[dest[i]]);
    add_to_canteen(filepath);
    g_free(filepath);
  }
  g_free(m3u_path);

  if(filename != last_canteen) {
    gchar* tmp = g_strdup(filename);
    g_free(last_canteen);
    last_canteen = tmp;
    if(last_canteen) gconf_client_set_string(gconf, GCONFKEY(last_canteen), last_canteen, NULL);
  }
clean:
  free(dest);
}

void seek_to_sec(int second) {
  int real_sec = get_track_offset_s(cur_track) + second;
  gst_element_seek_simple(pipeline, GST_FORMAT_TIME, GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT, real_sec * GST_SECOND);
}

int jump_to_track(int tr, gboolean seek) {
  if(0 <= tr && tr < get_tracks()){
    if(get_track_type(tr) == 1) return 0;
    cur_track = tr;
    if(seek) seek_to_sec(0);
    update_np();
    return 1;
  }
  return 1;
}

void previous_cb() {
  if(pos_within_track(cur_track) - get_track_offset_s(cur_track) > 3)
    jump_to_track(cur_track, 1);
  else if(jump_to_track(cur_track - 1, 1) == 0) previous_cb();
}

void next_cb() {
  if(jump_to_track(cur_track + 1, 1) == 0) next_cb();
}

void stop() {
  pause();
  gst_element_set_state(pipeline, GST_STATE_NULL);
  jump_to_track(0, 0);
}

void activate_cb(GtkTreeView *playlist, GtkTreePath *path) {
  GtkTreeIter current;
  GtkTreeModel *model = gtk_tree_view_get_model(playlist);
  GValue tr = {0};
  gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &current, path);
  gtk_tree_model_get_value(GTK_TREE_MODEL(model), &current, COL_ID, &tr);
  jump_to_track(g_value_get_int(&tr), 1);
}

void seek_cb (gpointer seekbar) {
  sched_seek = gtk_range_get_value(GTK_RANGE(seekbar));
}

void pause_cb(gpointer seekbar) {
  if(!playing) play();
  else pause();
}

gboolean progress_timeout_cb(gpointer seekbar) {
  if(loaded) {
    g_signal_handlers_block_by_func(seekbar, seek_cb, seekbar);

    if(sched_seek >= 0) {
      seek_to_sec(sched_seek);
//      gtk_range_set_value(GTK_RANGE(seekbar), sched_seek);
      sched_seek = -1;
    } else {
      int secs = pos_within_track(cur_track);
      if(secs > get_track_offset_s(cur_track) + get_track_seconds(cur_track)) jump_to_track(cur_track + 1, 0);
      
      gtk_range_set_value(GTK_RANGE(seekbar), secs - get_track_offset_s(cur_track));
    }
    gtk_widget_queue_draw(GTK_WIDGET(seekbar));
    g_signal_handlers_unblock_by_func(seekbar, seek_cb, seekbar);
  }
  return true;
}

void tracklist_init() {
  GtkWidget *list = GTK_WIDGET(gtk_builder_get_object(builder,
							   "tracklist"));
  GtkListStore *model = gtk_list_store_new(N_COLS, G_TYPE_INT,
			     G_TYPE_INT, G_TYPE_STRING, G_TYPE_STRING);
  GtkCellRenderer *renderer = gtk_cell_renderer_text_new();

  gtk_tree_view_set_model(GTK_TREE_VIEW(list), GTK_TREE_MODEL(model));

  GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes(_("Nr."), renderer, "text", COL_NR, NULL);
  gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_column_set_fixed_width(column, 32);
  gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);

  column = gtk_tree_view_column_new_with_attributes(_("Title"), renderer, "text", COL_NAME, NULL);
  gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_column_set_expand(column, 1);
  gtk_tree_view_column_set_cell_data_func(column, renderer, cur_bold_func, NULL, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);

  GValue align = {0};
  g_value_init(&align, G_TYPE_FLOAT);
  g_value_set_float(&align, 1.0);

  renderer = gtk_cell_renderer_text_new();
  g_object_set_property(G_OBJECT(renderer), "xalign", &align);
  column = gtk_tree_view_column_new_with_attributes(_("Length"), renderer, "text", COL_SECS, NULL);
  gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);
  gtk_tree_view_column_set_fixed_width(column, 40);
}

void filters_init() {
  GtkWidget *opendlg = GTK_WIDGET (gtk_builder_get_object (builder, "opendialog"));
  GtkFileFilter *filter = gtk_file_filter_new();
  gtk_file_filter_add_pattern(filter, "*.flac");
  gtk_file_filter_add_pattern(filter, "*.m3u");
  gtk_file_filter_set_name(filter, _("Albums (*.flac) and album lists (*.m3u)")); 
  gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(opendlg), filter);

  filter = gtk_file_filter_new();
  gtk_file_filter_add_pattern(filter, "*.*");
  gtk_file_filter_set_name(filter, _("All files"));
  gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(opendlg), filter);
}

static gboolean
bus_call (GstBus     *bus,
          GstMessage *msg,
          gpointer    data)
{
  switch (GST_MESSAGE_TYPE (msg)) {

    case GST_MESSAGE_EOS:
      stop();
      break;

    case GST_MESSAGE_ERROR: {
      gchar  *debug;
      GError *error;

      gst_message_parse_error (msg, &error, &debug);
      g_free (debug);

      g_printerr (_("Error: %s\n"), error->message);
      g_error_free (error);

      stop();
      break;
    }
    default: break;
  }

  return TRUE;
}

void reset(const char* status1, const char* status2) {
  update_artwork(NULL);
  if(playing) stop();
  loaded = 0;
  update_np();
  set_status(status1, status2);
}

void load_file(const char* selected) {
  gboolean exist = g_file_test(selected, G_FILE_TEST_IS_REGULAR);
  if(exist && g_str_has_suffix(selected, ".m3u")) {
    canteen(selected);
    return;
  }

  loaded = 0;
  if(pipeline) {stop(); gst_object_unref(pipeline);}

  if(!exist) {
    reset(_("The following file does not exist:"), selected);
    loaded = 0;
    return;
  }

  if(g_str_has_suffix(selected, ".flac") && set_info(selected)) {
    update_artwork(selected);
    set_tags(selected);
    set_sheet(selected);
    cur_track = 0;
    loaded = 1;
    update_np();

    GstElement *decoder, *dest;
  
    pipeline = gst_pipeline_new("my-pipeline");
    source = gst_element_factory_make("filesrc", "source");
    decoder = gst_element_factory_make("flacdec", "filter");
    dest = gst_element_factory_make("autoaudiosink", "sink");
    
    GstBus* bus = gst_pipeline_get_bus (GST_PIPELINE (pipeline));
    gst_bus_add_watch (bus, bus_call, NULL);
    gst_object_unref (bus);

    gst_bin_add_many(GST_BIN(pipeline), source, decoder, dest, NULL);
    if(!gst_element_link_many(source, decoder, dest, NULL)) 
      g_warning(_("Could not initialize GStreamer."));

    g_object_set(G_OBJECT(source), "location", selected, NULL);
    gst_element_set_state(pipeline, GST_STATE_PAUSED);

    clear_tracklist();
    int i;
    for(i=0; i<get_tracks(); i++) add_to_tracklist(i);
      
    play();
  } else {
    reset(_("Unknown file format."), "");
   }
}

void canteen_refr() {
  load_file(last_canteen);
}

void canteen_activate(GtkIconView *canteen, GtkTreePath *path) {
  GtkTreeIter current;
  GtkTreeModel *model = gtk_icon_view_get_model(canteen);
  GValue tr = {0};
  gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &current, path);
  gtk_tree_model_get_value(GTK_TREE_MODEL(model), &current, CAN_FILENAME, &tr);

  load_file(g_value_get_string(&tr));
  canteen_close();
}

void load_cb(GtkWidget *selector) {
  char *selected;
  gtk_widget_hide(selector);

  selected = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(selector));
  if(selected) load_file(selected);
  g_free(selected);
}

void drop_handler(GtkWidget *widget, GdkDragContext *context,
                  gint x, gint y, GtkSelectionData *sdata,
                  guint info, guint t, gpointer data) {
  gchar **uris = g_uri_list_extract_uris(sdata->data);
  char *fname = g_filename_from_uri(uris[0], NULL, NULL);
  load_file(fname);
  g_strfreev(uris);
  g_free(fname);
}


int
main (int argc, char *argv[])
{
  GtkWidget *window, *seekbar, *aw_frame, *opendlg;
  
  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  textdomain(PACKAGE);
  bind_textdomain_codeset(PACKAGE, "UTF-8");

// Initialization
  gen = g_rand_new();
  gst_init (&argc, &argv);
  gtk_init (&argc, &argv);
  gconf_init (argc, argv, NULL);

  gconf = gconf_client_get_default();

  builder = gtk_builder_new ();
  gtk_builder_add_from_file (builder, DATAFILE(/albumplayer.xml), NULL);
  gtk_builder_connect_signals (builder, NULL);      

  dnd_init();
  tracklist_init();
  canteen_init();
  filters_init();

  window = GTK_WIDGET (gtk_builder_get_object (builder, "player"));
  seekbar = GTK_WIDGET (gtk_builder_get_object (builder, "seekbar"));
  
  questionmark = gdk_pixbuf_new_from_file(DATAFILE(/questionmark.png), NULL);
  coverlayer = gdk_pixbuf_new_from_file(DATAFILE(/coverlayer.png), NULL);
  
// Configure gui
  GtkToggleAction *detail = GTK_TOGGLE_ACTION(gtk_builder_get_object(builder, "toggle_detail"));
  int show_detail = gconf_client_get_bool(gconf, GCONFKEY(show_detail), NULL);
  last_canteen = gconf_client_get_string(gconf, GCONFKEY(last_canteen), NULL);
  gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(detail), show_detail);

  GtkWidget* tool_canteen = GTK_WIDGET(gtk_builder_get_object(builder, "tool_canteen"));
  if(last_canteen) gtk_widget_set_sensitive(tool_canteen, last_canteen != NULL);
  
  if(argc > 1) load_file(argv[1]);
  g_timeout_add (500, (GSourceFunc)progress_timeout_cb, seekbar);

// Setup main window
//  GdkGeometry hints;
//  hints.max_height = -1;
//  hints.max_width = 2000;
//  gtk_window_set_geometry_hints(GTK_WINDOW(window), GTK_WIDGET(window),
// 				&hints, GDK_HINT_MAX_SIZE);

  GList *iconlist = NULL;
  GdkPixbuf *player48 = gdk_pixbuf_new_from_file(DATAFILE(/record.png), NULL);
  GdkPixbuf *player24 = gdk_pixbuf_new_from_file(DATAFILE(/record24.png), NULL);

  iconlist = g_list_append (iconlist, player48);
  iconlist = g_list_append (iconlist, player24);

  gtk_window_set_icon_list(GTK_WINDOW(window), iconlist);

  gtk_widget_show (window);

  gtk_main ();

// Tear down
  if(loaded) {
    gst_element_set_state(pipeline, GST_STATE_NULL);
    gst_object_unref(GST_OBJECT(pipeline));
  }
  g_object_unref (G_OBJECT (builder));
  g_object_unref (G_OBJECT (gconf));


  return 0;
}
