////////////////////////////////////////////////////////////////////////
/*! 
 *  \file   test.cpp
 *  \brief  TiffIO plugin self-test
 *  \author JD Gascuel
 * 
 * Copyright JD Gascuel, 2005.
 * 
 * Jean-Dominique.Gascuel@imag.fr
 *
 * This software is a computer program whose purpose is to add TIFF image
 * read/write capabilities to the Qt library, hence to any Qt application.
 * 
 * This software is governed by the CeCILL  license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info". 
 * 
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty  and the software's author,  the holder of the
 * economic rights,  and the successive licensors  have only  limited
 * liability. 
 * 
 * In this respect, the user's attention is drawn to the risks associated
 * with loading,  using,  modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean  that it is complicated to manipulate,  and  that  also
 * therefore means  that it is reserved for developers  and  experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or 
 * data to be ensured and,  more generally, to use and operate it in the 
 * same conditions as regards security. 
 * 
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 * TiffIO is based on the libtiff 3.7.1, with the following copyright :
 *      Copyright (c) 1988-1997 Sam Leffler
 *      Copyright (c) 1991-1997 Silicon Graphics, Inc.
 * See the libtiff-3.7.1/ subdirectory.
 */
////////////////////////////////////////////////////////////////////////
// File History:
//
// $Id: test.cpp,v 1.21 2006/02/10 14:03:16 gascuel Exp $
//
// 2005-02-01 : Creation.
// 2005-06-20 : added JPEG test (one image).

#include <stdio.h>
#include <stdlib.h>

#include <qglobal.h>

#if QT_VERSION < 0x040000 //------------ Qt 3.x ------------------------------
#   include <qapplication.h> 
#   include <qimage.h>
#   include <qdir.h>
#   include <qimage.h>
#else                     //------------ Qt 4.x ------------------------------
#   include <QApplication>
#   include <QDir>
#   include <QImage>
#   include <QtCore/qplugin.h>
#endif

//----------------------------------------------------------------------------
//
// Fake handler to make sure an error occured while testing unsupported files.
//
static int qFatals = 0, qWarnings = 0, qDebugs = 0;
static void
fakeMsgHandler(QtMsgType type, const char *)
{
	// fprintf(stdout, "Trapped message = '%s'\n", msg);
    switch( type ) {
#if QT_VERSION >= 0x040000
        case QtSystemMsg: // aka. QtCriticalMsg
#endif
        case QtFatalMsg:   qFatals++;   break;
        case QtWarningMsg: qWarnings++; break;
        case QtDebugMsg:   qDebugs++;   break;
    }
}

//-------------------------------------------------------------------------
// Make sure handler is loaded.
//
// On Qt4 : we have a way to ask for the plugin.
// For Qt3, this is done by hand inside TiffIO_Qt3.cpp.

#ifdef QT_STATICPLUGIN
Q_IMPORT_PLUGIN(TIFFIOPlugin)
#endif

//----------------------------------------------------------------------------
//
//
int main(int argc, char *argv[])
{
    QApplication app(argc, argv, false);

#ifdef NO_JPEG
    fprintf(stdout, "*** NO JPEG SUPPORT ***\n");
#else
    fprintf(stdout, "    jpeg-in-tiff supported\n");
#endif
#ifdef NO_ZLIB
    fprintf(stdout, "*** NO ZLIB SUPPORT ***\n");
#else
    fprintf(stdout, "    zip compression support\n");
#endif
#ifdef NO_LZW
    fprintf(stdout, "*** NO LZW SUPPORT ***\n");
#else
    fprintf(stdout, "    lzw compression support\n");
#endif

    //-------------------------------------------------------------------------
    // Get access to a writable directory.

    QString tmp = getenv("TMP");
#if defined(WIN32)
    if( tmp.isEmpty() ) tmp = getenv("TEMP");
    if( tmp.isEmpty() ) tmp = "C:/Temp";
    QDir::current().mkdir(tmp);   // In case it does not exists...
#elif defined(UNIX)
    if( tmp.isEmpty() ) tmp = "/tmp";
#else
#error Should provide for a writable temporary directory
#endif
    tmp += "/";

    //-------------------------------------------------------------------------
    // A bunch of various combination of formats, all created with PhotoShop.
    struct test {
        char *name;             // File path, inside Examples/
        int w, h, depth;        // Image size.
        unsigned int ll, ul, lr, ur;     // Values of the 4 corner pixels, ll = (0,0)
    } examples[] = {
        // NAME                                 W    H    depth LL (redish)      UL (yellow)       LR (blue)        UR (greenish)

			{"Tiff-100x100-rgb24-none-mac.tif",      100, 100, 32,   qRgb(254,0,93),  qRgb(254,254,0),  qRgb(1,0,254),   qRgb(1,254,95)},
			{"Tiff-100x100-rgb24-none-pc.tif",       100, 100, 32,   qRgb(254,0,93),  qRgb(254,254,0),  qRgb(1,0,254),   qRgb(1,254,95)},
			{"Tiff-100x100-rgb24-lsb2msb.tif",       100, 100, 32,   qRgb(254,0,93),  qRgb(254,254,0),  qRgb(1,0,254),   qRgb(1,254,95)},
			{"Tiff-100x100-rgb24-msb2lsb.tif",       100, 100, 32,   qRgb(254,0,93),  qRgb(254,254,0),  qRgb(1,0,254),   qRgb(1,254,95)},
			{"Tiff-100x100-rgb24-separate.tif",      100, 100, 32,   qRgb(254,0,93),  qRgb(254,254,0),  qRgb(1,0,254),   qRgb(1,254,95)},
			{"Tiff-100x100-rgb24-tiled.tif",         100, 100, 32,   qRgb(254,0,93),  qRgb(254,254,0),  qRgb(1,0,254),   qRgb(1,254,95)},

			{"Tiff-100x100-rgb16-none-pc.tif",       100, 100, 32,   qRgb(252,0,126),  qRgb(252,251,0),  qRgb(1,0,252),   qRgb(1,251,126)},

            {"Tiff-100x100-gray8-none.tif",          100, 100, 8,    qRgb(131,131,131), qRgb(247,247,247), qRgb(71,71,71), qRgb(219,219,219)},
			{"Tiff-100x100-gray8-pack.tif",          100, 100, 8,    qRgb(131,131,131), qRgb(247,247,247), qRgb(71,71,71), qRgb(219,219,219)},

			{"Tiff-2480x3496-faxg4.tif",             2480, 3496, 1,  qRgb(255,255,255), qRgb(255,255,255), qRgb(255,255,255), qRgb(255,255,255)},

#ifndef NO_LZW
			{"Tiff-100x100-cmnj8-lzw.tif",           100, 100, 32,   qRgb(255,3,139), qRgb(239,255,10), qRgb(29,58,255), qRgb(101,255,4)},
			{"Tiff-100x100-lab8-lzw.tif",            100, 100, 32,   qRgb(244,0,85),  qRgb(255,233,0),  qRgb(73,86,147), qRgb(135,191,53)},
			{"Tiff-100x100-lab16-lzw.tif",           100, 100, 32,   qRgb(255,0,113), qRgb(255,249,0),  qRgb(0,0,223),   qRgb(90,250,110)},
			{"Tiff-100x100-rgb24-lzw-mac.tif",       100, 100, 32,   qRgb(254,0,93),  qRgb(254,254,0),  qRgb(1,0,254),   qRgb(1,254,95)},
			{"Tiff-100x100-rgb24-lzw-pc.tif",        100, 100, 32,   qRgb(254,0,93),  qRgb(254,254,0),  qRgb(1,0,254),   qRgb(1,254,95)},
			{"Tiff-100x100-rgb16-lzw-mac.tif",       100, 100, 32,   qRgb(252,0,126), qRgb(252,251,0),  qRgb(1,0,252),   qRgb(1,251,126)},
			{"Tiff-100x100-color64-lzw-mac.tif",     100, 100, 8,    qRgb(247,18,84), qRgb(241,241,4),  qRgb(11,11,243), qRgb(20,246,85)},
			{"Tiff-100x100-color64-lzw-pc.tif",      100, 100, 8,    qRgb(247,18,84), qRgb(241,241,4),  qRgb(11,11,243), qRgb(20,246,85)},
			{"Tiff-100x100-gray8-lzw-mac.tif",       100, 100, 8,    qRgb(131,131,131), qRgb(247,247,247), qRgb(71,71,71), qRgb(219,219,219)},
			{"Tiff-100x100-gray16-lzw-pc.tif",       100, 100, 8,    qRgb(130,130,130), qRgb(247,247,247), qRgb(71,71,71), qRgb(220,220,220)},
			{"Tiff-100x100-bitmap-lzw-pc.tif",       100, 100, 1,    qRgb(255,255,255), qRgb(255,255,255),    qRgb(0,0,0), qRgb(255,255,255)},
#endif

#ifndef NO_ZLIB
			{"Tiff-100x100-rgb24-zip-mac.tif",       100, 100, 32,   qRgb(254,0,93),  qRgb(254,254,0),  qRgb(1,0,254),   qRgb(1,254,95)},
			{"Tiff-100x100-rgb24-zip-pc.tif",        100, 100, 32,   qRgb(254,0,93),  qRgb(254,254,0),  qRgb(1,0,254),   qRgb(1,254,95)},
			{"Tiff-100x100-rgb16-zip-pyrm.tif",      100, 100, 32,   qRgb(252,0,126),  qRgb(252,251,0),  qRgb(1,0,252),   qRgb(1,251,126)},
			{"Tiff-100x100-gray8-zip-pc.tif",        100, 100, 8,    qRgb(131,131,131), qRgb(247,247,247), qRgb(71,71,71), qRgb(219,219,219)},
			{"Tiff-100x100-gray16-zip-mac.tif",      100, 100, 8,    qRgb(130,130,130), qRgb(247,247,247), qRgb(71,71,71), qRgb(220,220,220)},
			{"Tiff-100x100-bitmap-zip-pc.tif",       100, 100, 1,    qRgb(255,255,255), qRgb(255,255,255),    qRgb(0,0,0), qRgb(255,255,255)},
#endif

#if !defined(NO_LZW) && !defined(NO_ZLIB)
			{"Tiff-100x100-rgb16Lzw-layersZip.tif",  100, 100, 32,   qRgb(252,0,126),  qRgb(252,251,0),  qRgb(1,0,252),   qRgb(1,251,126)},
#endif
            
#ifndef NO_JPEG
			{"Tiff-100x100-gray8-jpeg.tif",          100, 100, 8,    qRgb(130,130,130), qRgb(246,246,246), qRgb(70,70,70), qRgb(219,219,219)},
			{"Tiff-100x100-rgb24-jpeg-YCrCb.tif",    100, 100, 32,   qRgb(255,0,94),  qRgb(255,254,3),  qRgb(2,1,253),   qRgb(3,252,96)},
			{"Tiff-100x100-rgb24-jpeg-RGB.tif",    100, 100, 32,   qRgb(254,0,93),  qRgb(254,254,0),  qRgb(2,0,254),   qRgb(2,254,96)},
#endif
			{0, 0, 0, 0, 0, 0, 0, 0}
    };

    int error = 0;
    for(test *t = examples; t->name; t++)
    {
        // ---------------------------------------------------------- Read test
        QImage img( QString("Examples/%1").arg(t->name), "TIFF" );

        if( img.isNull() )
            fprintf(stderr, "%d: Cannot load image %s\n", ++error, t->name);
        else if( img.width() != t->w || img.height() != t->h || img.depth() != t->depth )
            fprintf(stderr, "%d: Wrong header for %s. %dx%dx%d should be %dx%dx%d\n",
                ++error, t->name,
                img.width(), img.height(), img.depth(),
                t->w, t->h, t->depth);
        else
        {
            if( img.pixel(0,0) != t->ll )
                fprintf(stderr, "%d: %s wrong pixel value.\n  ll %d,%d,%d,%d should be %d,%d,%d,%d\n",
                    ++error, t->name,
                    qRed(img.pixel(0,0)), qGreen(img.pixel(0,0)), qBlue(img.pixel(0,0)), qAlpha(img.pixel(0,0)),
                    qRed(t->ll),          qGreen(t->ll),          qBlue(t->ll),          qAlpha(t->ll));

            if( img.pixel(0,t->h-1) != t->ul)
                fprintf(stderr, "%d: %s wrong pixel value.\n  ul %d,%d,%d,%d should be %d,%d,%d,%d\n",
                    ++error, t->name,
                    qRed(img.pixel(0,t->h-1)), qGreen(img.pixel(0,t->h-1)), qBlue(img.pixel(0,t->h-1)), qAlpha(img.pixel(0,t->h-1)),
                    qRed(t->ul),               qGreen(t->ul),               qBlue(t->ul),               qAlpha(t->ul));

            if( img.pixel(t->w-1,0) != t->lr)
                fprintf(stderr, "%d: %s wrong pixel value.\n  lr %d,%d,%d,%d should be %d,%d,%d,%d\n",
                    ++error, t->name,
                    qRed(img.pixel(t->w-1,0)), qGreen(img.pixel(t->w-1,0)), qBlue(img.pixel(t->w-1,0)), qAlpha(img.pixel(t->w-1,0)),
                    qRed(t->lr),               qGreen(t->lr),               qBlue(t->lr),               qAlpha(t->lr));

            if( img.pixel(t->w-1,t->h-1) != t->ur )
                fprintf(stderr, "%d: %s wrong pixel value.\n  ur %d,%d,%d,%d should be %d,%d,%d,%d\n",
                    ++error, t->name,
                    qRed(img.pixel(t->w-1,t->h-1)), qGreen(img.pixel(t->w-1,t->h-1)), qBlue(img.pixel(t->w-1,t->h-1)), qAlpha(img.pixel(t->w-1,t->h-1)),
                    qRed(t->ur),                    qGreen(t->ur),                    qBlue(t->ur),                    qAlpha(t->ur));

            // --------------------------------------------------------- write test
            QString name = tmp+t->name;
            if( !  img.save(name, "TIFF") )
            {
                fprintf(stderr, "%d: Unable to save to '%s'", ++error,
#if QT_VERSION < 0x040000
                    name.latin1());
#else
                    name.toLatin1().data());
#endif
                continue;
            }
            QImage reload(name);
            if( img != reload )
            {
                if( img.width() != reload.width() || img.height() != reload.height() )
                    fprintf(stderr, "%d: %s write test failed, images size differ: %dx%d should be %dx%d\n",
                        ++error, t->name,
                        reload.width(), reload.height(), img.width(), img.height());
                else if( img.depth() != reload.depth() || img.numColors() != reload.numColors() )
                    fprintf(stderr, "%d: %s write test failed, images mode differ: %d/%d should be %d/%d\n",
                        ++error, t->name,
                        reload.depth(), reload.numColors(), img.depth(), img.numColors());
                else
                    fprintf(stderr, "%d: %s write test failed, images pixels differ\n", ++error, t->name);
            }
            //QDir::current().remove(name);
        }
    }


    //----------------------------------------------------------------------------
    // A bunch of images that are not supported, but should not crash the library,
    // Just generates qFatal()/qWarning().
    static char *unsuportedExamples[] = {
        "Tiff-100x100-cmnj16-lzw.tif",

#ifdef NO_LZW
		"Tiff-100x100-cmnj8-lzw.tif",
		"Tiff-100x100-lab8-lzw.tif",
		"Tiff-100x100-lab16-lzw.tif",
		"Tiff-100x100-rgb24-lzw-mac.tif",
		"Tiff-100x100-rgb24-lzw-pc.tif",
		"Tiff-100x100-rgb16-lzw-mac.tif",
		"Tiff-100x100-color64-lzw-mac.tif",
		"Tiff-100x100-color64-lzw-pc.tif",
		"Tiff-100x100-gray8-lzw-mac.tif",
		"Tiff-100x100-gray16-lzw-pc.tif",
		"Tiff-100x100-bitmap-lzw-pc.tif",
#endif
#ifdef NO_ZLIB
		"Tiff-100x100-rgb24-zip-mac.tif",
		"Tiff-100x100-rgb24-zip-pc.tif",
		"Tiff-100x100-rgb16-zip-pyrm.tif",
		"Tiff-100x100-gray8-zip-pc.tif",
		"Tiff-100x100-gray16-zip-mac.tif",
		"Tiff-100x100-bitmap-zip-pc.tif",
#endif
#if defined(NO_LZW) || defined(NO_ZLIB)
		"Tiff-100x100-rgb16Lzw-layersZip.tif",
#endif
#ifdef NO_JPEG
		"Tiff-100x100-gray8-jpeg.tif",
		"Tiff-100x100-rgb24-jpeg-YCrCb.tif",
		"Tiff-100x100-rgb24-jpeg-RGB.tif",
#endif
        0
    };

    qInstallMsgHandler(fakeMsgHandler);
    for(char **t = unsuportedExamples; *t; ++t)
    {
        qFatals = qWarnings = qDebugs = 0;
        QImage img( QString("Examples/%1").arg(*t), "TIFF" );
        if( ! img.isNull() )
            fprintf(stderr, "%d: Should not have been able to load unsupported image %s\n", ++error, *t);
        if( qFatals || !qWarnings )
            fprintf(stderr, "%d: Failed to generate warning(s) when loading unsupported image %s\n", ++error, *t);
    }
    qInstallMsgHandler(0);

	if( error == 0 )
		fprintf(stdout, "All %d images checked ok\n",
			sizeof examples/sizeof(test) + sizeof unsuportedExamples/sizeof(char*));

    return -error;
}
