////////////////////////////////////////////////////////////////////////
/*! 
 *  \file   TiffIO_Qt4.cpp
 *  \brief  Plugin to enable TIFF I/O in Qt 3.x and 4.x.
 *  \author JD Gascuel
 * 
 * Copyright JD Gascuel, 2005.
 * 
 * Jean-Dominique.Gascuel@imag.fr
 *
 * This software is a computer program whose purpose is to add TIFF image
 * read/write capabilities to the Qt library, hence to any Qt application.
 * 
 * This software is governed by the CeCILL  license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info". 
 * 
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty  and the software's author,  the holder of the
 * economic rights,  and the successive licensors  have only  limited
 * liability. 
 * 
 * In this respect, the user's attention is drawn to the risks associated
 * with loading,  using,  modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean  that it is complicated to manipulate,  and  that  also
 * therefore means  that it is reserved for developers  and  experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or 
 * data to be ensured and,  more generally, to use and operate it in the 
 * same conditions as regards security. 
 * 
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 * TiffIO is based on the libtiff 3.7.2, with the following copyright :
 *      Copyright (c) 1988-1997 Sam Leffler
 *      Copyright (c) 1991-1997 Silicon Graphics, Inc.
 * See the libtiff-3.7.2/ subdirectory.
 */
////////////////////////////////////////////////////////////////////////
// File History:
//
// $Id: TiffIO_Qt4.cpp,v 1.9 2006/02/06 19:40:41 gascuel Exp $
//
// 2005-09-24 : Creation for Qt 4.0 port.
// 2005-10-12 : Minor updates for gcc/mingw and Qt 4.0.1 by Christian Ehrlicher.
//

#include <qglobal.h>
#if QT_VERSION < 0x040000
#error This file should compile only with Qt 4.x or upper. There is an error in the .pri/qmake processing.
#endif

#include <QImage>
#include <QImageIOHandler>
#include <QImageIOPlugin>

extern void commonReadTIFF (QImageIOHandler* handler, QImage& img, int& status);
extern void commonWriteTIFF(QImageIOHandler* handler, const QImage& img, int& status);

////////////////////////////////////////////////////////////////////////

bool checkTIFFHeader(QIODevice *device)
{
    char head[4];
    quint64 start = device->pos();      // Store current pointer position.
    device->read(head, 4);              // Read first four bytes.
    device->seek(start);                // Seek back to begin of file.

    return head[0]=='M' && head[1]=='M' &&  head[2]==0x00 && (head[3]==0x2a || head[3]==0x2b)
       ||  head[0]=='I' && head[1]=='I' && (head[2]==0x2a || head[2]==0x2b) && head[3]==0x00
       ||  head[0]=='P' && head[1]=='D' &&  head[2]==0x00 && (head[3]==0x2a || head[3]==0x2b)
       ||  head[0]=='D' && head[1]=='P' && (head[2]==0x2a || head[2]==0x2b) && head[3]==0x00
       ;   
}

////////////////////////////////////////////////////////////////////////

class QtTiffImageIOHandler : public QImageIOHandler
{
    int _status;                        //! Remember last read/write success or failure.
public:

    QtTiffImageIOHandler(QIODevice *device)
    {
        setDevice(device);
    }

    virtual QByteArray name() const
    {
        return "tiff";
    }

    virtual bool supportsOption (ImageOption option ) const
    {
        if(option == Endianness) return true;
        return false;
    }

    virtual bool canRead() const
    {
	if( !device() )
	{
		qWarning("QtTiffImageIOHandler::canRead() called with no device");
		return false;
	}

         if( checkTIFFHeader(device()) )
         {
            return true;
        }
        return false;
    }

    virtual bool read(QImage* img)
    {
        setFormat("tiff");
        commonReadTIFF(this, *img, _status);
        return _status == 0;
    }

    virtual bool write(const QImage& img)
    {
        commonWriteTIFF(this, img, _status);
        return _status == 0;
    }
};

////////////////////////////////////////////////////////////////////////

class TIFFIOPlugin : public QImageIOPlugin
{
public:
    virtual QStringList keys() const
    {
        QStringList list;
            list += "tif";
            list += "tiff";
        return  list;
    }

    virtual Capabilities capabilities(QIODevice *device, const QByteArray &format) const
    {
        // qDebug("capabilities(%p, %s)", device, format.data());
 
        if( !format.isEmpty()
         && format != "tif"
         && format != "tiff"
        )
            return 0;

        Capabilities cap = 0;

        // Called without a particular file, just check what the plugin can do...
        if( ! device )
            cap |= Capabilities(CanRead|CanWrite);

        if( device && device->isReadable() && checkTIFFHeader(device) )
            cap |= CanRead;
 
        if( device && device->isWritable() )
            cap |= CanWrite;

        return cap;
    }

    virtual QImageIOHandler* create(QIODevice *device, const QByteArray & = QByteArray()) const
    {
        // qDebug("Creating Qt4 handler for TIFF QImages");
        return new QtTiffImageIOHandler(device);
    }
};

////////////////////////////////////////////////////////////////////////

Q_EXPORT_PLUGIN(TIFFIOPlugin);
