////////////////////////////////////////////////////////////////////////
/*! 
 *  \file   TiffIO.cpp
 *  \brief  Plugin to enable TIFF I/O in Qt 3.x and 4.x.
 *  \author JD Gascuel
 * 
 * Copyright JD Gascuel, 2005.
 * 
 * Jean-Dominique.Gascuel@imag.fr
 *
 * This software is a computer program whose purpose is to add TIFF image
 * read/write capabilities to the Qt library, hence to any Qt application.
 * 
 * This software is governed by the CeCILL  license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info". 
 * 
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty  and the software's author,  the holder of the
 * economic rights,  and the successive licensors  have only  limited
 * liability. 
 * 
 * In this respect, the user's attention is drawn to the risks associated
 * with loading,  using,  modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean  that it is complicated to manipulate,  and  that  also
 * therefore means  that it is reserved for developers  and  experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or 
 * data to be ensured and,  more generally, to use and operate it in the 
 * same conditions as regards security. 
 * 
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 * TiffIO is based on the libtiff 3.7.2, with the following copyright :
 *      Copyright (c) 1988-1997 Sam Leffler
 *      Copyright (c) 1991-1997 Silicon Graphics, Inc.
 * See the libtiff-3.7.2/ subdirectory.
 */
////////////////////////////////////////////////////////////////////////
// File History:
//
// $Id: TiffIO.cpp,v 1.22 2006/02/09 23:46:21 gascuel Exp $
//
// 2004-11-03 : Creation.
//              Support reading of RGBA and RGB bits true color images
//              with up to 16 bits per component (automatically downscaled 
//              to 3x8 bits colors), up to 16 bits gray images (in MINISWHITE 
//              or MINISBLACK format, downscaled to 8-bit grays), 
//              Color palette with up to 8 bits.
//              Support all Paletted/RGB/Gray TIFF encoding scheme (in 
//              particular LZW and ZIP), 
//              except JPEG compression, and LAB or CMYK color spaces.
//
//              Support writting of Qt images with 8-bpp, and 24 and 32-bpp.
//              Always use LZW compression. Automatically switches to MINISBLACK
//              when saving 8-bit all gray image. Use PALETTE mode for 8-bit images,
//              and RGBA mode for 24/32 bit images.
//
// 2004-11-04 : Simplified, and added support for 8-bit LAB and CYMK images.
//              Added support for non-file IO, so .gz images is working.
//              Added support for fileBigEndians as well as littleEndians file.
//
// 2004-11-22 : FIX : regexp should start with a ^, or wrong image could
//              be declared to be TIFF images.
//
// 2004-05-27 : FIX from Till Oliver Knoll, when calling QtTIFFOpen on a memory
//              stream.
// 2005-01-31 : FIX more compatibility issues on HP-UX and AIX, 
//              thanks to Till Oliver Knoll.
//
// 2005-09-24 : Initial port to Qt 4.0.1
//
// 2006-01-06 : FIX bit order bug in bitmaps (this different from qtBigEndian).

#include "tiffio.h"

#include <qglobal.h>

#if QT_VERSION < 0x040000
#   include <qimage.h>
#   include <qiodevice.h>
#   define HANDLER_CLASS QImageIO
#   define HANDLER_DEVICE ioDevice
#else
#   include <QString>
#   include <QImage>
#   include <QImageIOHandler>
#   define HANDLER_CLASS QImageIOHandler
#   define HANDLER_DEVICE device
#   define Q_UINT32 quint32
#endif

#define GETDEVICE                                               \
    HANDLER_CLASS *io = reinterpret_cast<HANDLER_CLASS*>(fd);   \
    Q_ASSERT(io);                                               \
    QIODevice* dev = io->HANDLER_DEVICE();                      \
    Q_ASSERT(dev);

////////////////////////////////////////////////////////////////////////
/*
 * TIFF Library Qt-specific Routines.  Adapted from tif_win32.c, 
 * Sam Leffler/SGI 1997, and tif_unix 4/5/95 by
 * Scott Wagner (wagner@itek.com), Itek Graphix, Rochester, NY USA
 */

static tsize_t
QtReadProc(thandle_t fd, tdata_t buf, tsize_t size)
{
    GETDEVICE
    if( dev->isReadable() )
#if QT_VERSION < 0x040000
        return (tsize_t) dev->readBlock((char *)buf, size);
#else
        return (tsize_t) dev->read((char *)buf, size);
#endif
    else
        return 0;
}

static tsize_t
QtWriteProc(thandle_t fd, tdata_t buf, tsize_t size)
{
    GETDEVICE
    if( dev->isWritable() )
#if QT_VERSION < 0x040000
        return (tsize_t) dev->writeBlock((char *)buf, size);
#else
        return (tsize_t) dev->write((char *)buf, size);
#endif
    else
        return 0;
}

static toff_t
QtSizeProc(thandle_t fd)
{
    GETDEVICE

    // Beware : there is a bug in QT, size() re-read what
    // disk says, and do not take into account any buffered
    // write pending...
    //   (QFile uses fwrite, that do buffering over FILE.
    //    But the fh is a private member we do not have access to).
#if QT_VERSION < 0x040000
    if( dev->isWritable() )
        dev->flush();
#else
    // Stupid folks from Trolltech suppressed QIODevice::flush()
    // from Qt4...
    // So we have to find a kludge to work around that:
    if( dev->isWritable() && ! (dev->openMode() & QIODevice::Unbuffered) )
    {
        dev->close();
        // Re-open in unbuffered mode. Do it in ReadWrite, so data
        // already written is not truncated. And not in Appen mode
        // too, that will clobber any further seek...
        dev->open( QIODevice::ReadWrite | QIODevice::Unbuffered );
    }
#endif
    
    toff_t pos  = (toff_t) dev->size();
    return pos;
}

static toff_t
QtSeekProc(thandle_t fd, toff_t offset, int where)
{
    GETDEVICE

    switch(where) {
	case SEEK_CUR:
#if QT_VERSION < 0x040000
        offset += dev->at();
#else
        offset += dev->pos();
#endif
		break;
	case SEEK_END:
        offset = QtSizeProc(fd) - offset;
		break;
    }

    if(
#if QT_VERSION < 0x040000
        dev->at(offset)
#else
        dev->seek(offset)
#endif
        )
        return offset;
    else
    	return 0;
}

/*
 * Closing the file just needs to get the right pointer, and close the right QIODevice.
 */
static int
QtCloseProc(thandle_t fd)
{
    GETDEVICE
    dev->close();
	return 0;
}

/*
 * Open a TIFF file descriptor for read/writing.
 * Note that TIFFFdOpen and TIFFOpen recognise the character 'u' in the mode
 * string, which forces the file to be opened unmapped.
 */
TIFF*
QtTIFFOpen(HANDLER_CLASS *io, char *mode)
{
    // From Qt 4.x, there is no way we can retrieve the fileName of a given QIODevice.
    // (which is meaningfull for array buffer, or sockets, but a pity for good old
    //  disk files).
    char *fileName = "QImage";
	TIFF* tif = TIFFClientOpen(fileName, mode, (thandle_t)io,
			QtReadProc, QtWriteProc,
			QtSeekProc, QtCloseProc, QtSizeProc,
			0,  0);
	return tif;
}

////////////////////////////////////////////////////////////////////////

static void QtTIFFErrorHandler(const char* module, const char* fmt, va_list ap)
{
    char buffer[1024];
#ifdef WIN32
    // Windows people are stupid. As usual:
    _vsnprintf(buffer, sizeof buffer, fmt, ap);
#else
    vsnprintf(buffer, sizeof buffer, fmt, ap);
#endif

    // TIFF codec should never crash the main App. Just generate warnings !
    qWarning(QString("%1: FATAL error : %2")
        .arg(module)
        .arg(buffer)
#if QT_VERSION < 0x040000
        .latin1()
#else
        .toLatin1()
#endif
    );
}

static void QtTIFFWarningHandler(const char* module, const char* fmt, va_list ap)
{
    char buffer[1024];
#ifdef WIN32
    // Windows people are stupid. As usual:
    _vsnprintf(buffer, sizeof buffer, fmt, ap);
#else
    vsnprintf(buffer, sizeof buffer, fmt, ap);
#endif

    qWarning(QString("%1: Warning : %2")
        .arg(module)
        .arg(buffer)
#if QT_VERSION < 0x040000
        .latin1()
#else
        .toLatin1()
#endif
    );
}

////////////////////////////////////////////////////////////////////////

extern TIFFDisplay display_sRGB;
static float   refWhite[3];
static TIFFCIELabToRGB* cielab = 0;
 
void initCIELabConversion(TIFF* tif)
{
	float   *whitePoint;

	if( !cielab)
    {
		cielab = (TIFFCIELabToRGB *)_TIFFmalloc(sizeof(TIFFCIELabToRGB));
        Q_CHECK_PTR(cielab);
    }

	TIFFGetFieldDefaulted(tif, TIFFTAG_WHITEPOINT, &whitePoint);
	refWhite[1] = 100.0F;
	refWhite[0] = whitePoint[0] / whitePoint[1] * refWhite[1];
	refWhite[2] = (1.0F - whitePoint[0] - whitePoint[1])
		      / whitePoint[1] * refWhite[1];
	if (TIFFCIELabToRGBInit(cielab, &display_sRGB, refWhite) < 0)
        qWarning("Failed to initialize CIE L*a*b*->RGB conversion state.");
}

////////////////////////////////////////////////////////////////////////

#define TIFF2QT(c)  qRgba(TIFFGetR(c), TIFFGetG(c), TIFFGetB(c), TIFFGetA(c))

void commonReadTIFF(HANDLER_CLASS* handler, QImage& img, int& status)
{
    TIFFSetErrorHandler(QtTIFFErrorHandler);
    TIFFSetWarningHandler(0);                           // Strip warnings while readind TIFF's dir.
        TIFF *tif = QtTIFFOpen(handler, "rBm");         // LZW bogus in H or L mode...
    TIFFSetWarningHandler(QtTIFFWarningHandler);

    if( tif )
    {
        uint32 width=0, height=0;
        uint16 bps=0, spp=0, mode=0;

        int success = TIFFGetField(tif, TIFFTAG_IMAGEWIDTH,      &width)
                   && TIFFGetField(tif, TIFFTAG_IMAGELENGTH,     &height)
                   && TIFFGetField(tif, TIFFTAG_BITSPERSAMPLE,   &bps)
                   && TIFFGetField(tif, TIFFTAG_SAMPLESPERPIXEL, &spp)
                   && TIFFGetField(tif, TIFFTAG_PHOTOMETRIC,     &mode)
                   ;

        // Some images don't have fill order tag, so default to <<big endian>>.
        uint16 fillOrder=FILLORDER_MSB2LSB;
        (void) TIFFGetField(tif, TIFFTAG_FILLORDER, &fillOrder);

        // Ask Qt about qtBigEndian and word size of the platform :
#if QT_VERSION < 0x040000
        bool qtBigEndian;
        int wordSize;
        qSysInfo( &wordSize, &qtBigEndian );
#else
        bool qtBigEndian = QSysInfo::BigEndian;
#endif

        // Should we swab the file ?
        bool fileBigEndian = TIFFIsBigEndian(tif);
        bool badIndians = fileBigEndian != qtBigEndian;

        if( (spp==3) || (spp==4) )
        {
#if QT_VERSION < 0x040000
            img.create(width, height, 32);
#else
            img = QImage(width, height, (spp==4) ? QImage::Format_ARGB32 : QImage::Format_RGB32);
#endif

            if( img.isNull() )
            {
                qWarning("Cannot allocate 32 bit image %dx%d", (int)width, (int)height);
                return;
            }
 
            // Helpfull function, that handle any LAB/CYMK color conversion as well: stick with it !
            // (But only when in 8 bps ... too bad.
            if( bps <= 8 || (bps<=16 && mode==PHOTOMETRIC_RGB) )
            {
                success &= TIFFReadRGBAImageOriented(tif, width, height, (uint32*)img.bits(), ORIENTATION_TOPLEFT);
                if( success && (TIFF2QT(0x11223344) != 0x11223344) )
                    for(uint i=0; i<width; i++)
                        for(uint j=0; j<height; ++j)
                            img.setPixel(i,j, TIFF2QT( img.pixel(i,j) ));
            }
            else if( bps <= 16 )
            {
                // When there is 16bits by pixels, TIFFReadRGBAImageOriented() knows only about
                // RGB mode. For the other modes, we just do it ourself, rounding all values
                // to 8bits *before* color conversion. This is (very slightly) bad, but
                // enables us to re-use libtiff conversion functions. Which is *very* good.
                //
                // Possible modes for multi-sample-per-pixel images are :
                //  + PHOTOMETRIC_RGB    : done already.
                //  + PHOTOMETRIC_CIELAB : tif_color.c providdes conversion.
                //  + PHOTOMETRIC_YCBCR  : also. But there is complicated subsampling here. -> NYI.
                //  + PHOTOMETRIC_SEPARATED : Inks, like CMJK formats... --> NYI.
                //  + PHOTOMETRIC_LOGL and PHOTOMETRIC_LOGLUV :
                //          Use SGILOGDATAFMT_8BIT to ask tif_luv.c to convert to RGB 8 bits ???

                if( mode == PHOTOMETRIC_CIELAB )
                    initCIELabConversion(tif);

                char* scanlineBuffer = new char[TIFFScanlineSize(tif)];
                for(uint y=0; success && y<height; ++y)
                {
                    QRgb*   q = (QRgb*)img.scanLine(y);

                    success &= ( TIFFReadScanline(tif, scanlineBuffer, y) > 0);

                    switch(mode)
                    {
                    case PHOTOMETRIC_CIELAB:
                        {
                            short* p = (short*)scanlineBuffer; // a and b are signed, L is not...
                            for(uint x=0; x<width; ++x, p+=spp)
                            {
                                float  X=0, Y=0, Z=0; TIFFCIELabToXYZ(cielab, ushort(p[0])>>8, 
                                                                              p[1]>>8, p[2]>>8, &X, &Y, &Z);
                                uint32 r=0, g=0, b=0; TIFFXYZToRGB(cielab, X, Y, Z, &r, &g, &b);
                                q[x] = qRgb(r,g,b);
                            }
                        }
                        break;

                    default:
                        goto Failed;
                    }
                }

                delete[] scanlineBuffer;
                if(cielab) { _TIFFfree(cielab); cielab=0; }
            }
            else if( bps <= 32 && mode==PHOTOMETRIC_RGB ) // Also PHOTOMETRIC_YCBCR or PHOTOMETRIC_CIELAB ?
            {
                char* scanlineBuffer = new char[TIFFScanlineSize(tif)];
                for(uint y=0; success && y<height; ++y)
                {
                    uint32* p = (uint32*)scanlineBuffer; // RGB are not signed values.
                    QRgb*   q = (QRgb*)img.scanLine(y);
                    success &= ( TIFFReadScanline(tif, scanlineBuffer, y) > 0);

                    for(uint x=0; x<width; ++x, p+=spp)
                        q[x] = qRgb(p[0]>>24, p[1]>>24, p[2]>>24);
                }

                delete[] scanlineBuffer;
            }
            else
                goto Failed;
        }
        else if( spp==1 )
        {
            // Qt does not support more than 256 colors.
            int colors = (bps <8) ? 1<<bps : 256;

            // Beware of special parameters for bit bitmaps:
#if QT_VERSION < 0x040000
            img.create(width, height, 
                       (bps>1) ? 8 : 1, 
                       colors, 
                       (bps==1) ? (fillOrder==FILLORDER_MSB2LSB ? QImage::BigEndian : QImage::LittleEndian)
                                : QImage::IgnoreEndian);
#else
            img = QImage(width, height,
                (bps>1) ? QImage::Format_Indexed8 
                        : (fillOrder==FILLORDER_MSB2LSB ? QImage::Format_Mono : QImage::Format_MonoLSB));
#endif
            if( img.isNull() )
            {
                qWarning("Cannot allocate %d-bit image %dx%d", bps, (int)width, (int)height);
                return;
            }

            // Reading and translating color map, if any ?
            if( mode==PHOTOMETRIC_PALETTE )
            {
                uint16 *red=0, *green=0, *blue=0;
                if( ! TIFFGetField(tif, TIFFTAG_COLORMAP, &red, &green, &blue) )
                    qWarning("Missing palette data in color-palette TIFF image");
                else
                {
                    Q_CHECK_PTR( red ); Q_CHECK_PTR( green ); Q_CHECK_PTR( blue);
                    if( badIndians )
                    {
                        TIFFSwabArrayOfShort(red,   colors);
                        TIFFSwabArrayOfShort(green, colors);
                        TIFFSwabArrayOfShort(blue,  colors);
                    }
                    img.setNumColors(colors);
                    for(int c=0; c<colors; ++c)
                        img.setColor(c, qRgb(red[c]>>8,green[c]>>8,blue[c]>>8));
                }
            }
            else if( (mode==PHOTOMETRIC_MINISWHITE) || (mode==PHOTOMETRIC_MINISBLACK) )
            {
                float s = 255.0f/(colors-1);
                img.setNumColors(colors);
                for(int c=0; c<colors; ++c)
                    img.setColor(
                        (mode==PHOTOMETRIC_MINISBLACK) ? c : (colors-1-c),
                        qRgb((int)(s*c + 0.5f), (int)(s*c + 0.5f), (int)(s*c + 0.5f)));
            }
            else
                qWarning("Invalid palette in single channel TIFF image (mode=%d)", mode);

            // Reading indexes line by line. Always packed on a byte basis.
            if( bps == 2 )
            {
                char* scanlineBuffer = new char[TIFFScanlineSize(tif)];
                for(uint y=0; success && (y<height); ++y)
                {
                    success &= ( TIFFReadScanline(tif, scanlineBuffer, y) > 0);

                    // XXX should we have to use TIFFReverseBits(bits, size) here ?
                    // And/or to unpack the other way around ?

                    for(uint x=0; success && (x<width); ++x)
                        switch( x&3 ) {
                            case 3: img.setPixel(x, y, (scanlineBuffer[x>>2] >> 0 )& 3 ); break;
                            case 2: img.setPixel(x, y, (scanlineBuffer[x>>2] >> 2 )& 3 ); break;
                            case 1: img.setPixel(x, y, (scanlineBuffer[x>>2] >> 4 )& 3 ); break;
                            case 0: img.setPixel(x, y, (scanlineBuffer[x>>2] >> 6 )& 3 ); break;
                        }
                }
                delete scanlineBuffer;
            }
            else if( bps == 4 )
            {
                char* scanlineBuffer = new char[TIFFScanlineSize(tif)];
                for(uint y=0; success && (y<height); ++y)
                {
                    success &= ( TIFFReadScanline(tif, scanlineBuffer, y) > 0);

                    // XXX should we have to use TIFFReverseBits(bits, size) here ?
                    // And/or to unpack the other way around ?

                    for(uint x=0; success && (x<width); ++x)
                        img.setPixel( x, y, (x & 1) ? scanlineBuffer[x>>1] & 0xF
                                                    : (scanlineBuffer[x>>1] >> 4) & 0xF );
                }
                delete scanlineBuffer;
            }
            else if( bps == 8 || bps == 1)                          // Both 1-bit and 8-bits don't need extra works.
            {
                for(uint y=0; success && (y<height); ++y)
                    success &= ( TIFFReadScanline(tif, img.scanLine(y), y) > 0);
            }
            else if( bps<=16 )                                      // 16 bits gray/paletted image ?
            {
                uint16 *wine = (uint16*)new char[TIFFScanlineSize(tif)];
                Q_CHECK_PTR(wine);
                for(uint y=0; success && (y<height); ++y)
                {
                    success &= ( TIFFReadScanline(tif, wine, y) > 0);  // Already swabed, if needed.
                    for(uint x=0; x<width; ++x)
                        img.setPixel(x, y, wine[x]>>8);
                }
                delete[] wine;
            }
            else
                goto Failed;
        }
        else
        {
Failed:
            static const char *photoNames[] = {
                "min-is-white",				/* PHOTOMETRIC_MINISWHITE */
                "min-is-black",				/* PHOTOMETRIC_MINISBLACK */
                "RGB color",				/* PHOTOMETRIC_RGB */
                "palette color (RGB from colormap)",	/* PHOTOMETRIC_PALETTE */
                "transparency mask",			/* PHOTOMETRIC_MASK */
                "separated",				/* PHOTOMETRIC_SEPARATED */
                "YCbCr",					/* PHOTOMETRIC_YCBCR */
                "?",
                "CIE L*a*b*",				/* PHOTOMETRIC_CIELAB */
            };
#define	NPHOTONAMES	(sizeof (photoNames) / sizeof (photoNames[0]))
            const char *modeString = (mode<NPHOTONAMES) ? photoNames[mode] : "?";
            
            qWarning("Unsupported pixel format while reading TIFF (spp=%d, bps=%d, mode=%d (%s))",
                spp, bps, mode, modeString);

            success = 0;
            img = QImage();
        }

        TIFFClose(tif);

        status = success ? 0 : -2;      // Everything went OK ?
    }
    else
    {
        img = QImage();
        status = -1;                    // Error reading image...
    }
}

void commonWriteTIFF(HANDLER_CLASS* handler, const QImage& img, int& status)
{
    TIFFSetErrorHandler(QtTIFFErrorHandler);
    TIFFSetWarningHandler(QtTIFFWarningHandler);

    TIFF *tif = QtTIFFOpen(handler, "wBm");   // LZW bogus in H or L mode...

    if( tif )
    {
        int success = TIFFSetField(tif, TIFFTAG_IMAGEWIDTH,      img.width())
                   && TIFFSetField(tif, TIFFTAG_IMAGELENGTH,     img.height())
                   && TIFFSetField(tif, TIFFTAG_PLANARCONFIG,    PLANARCONFIG_CONTIG)
                   && TIFFSetField(tif, TIFFTAG_ORIENTATION,     ORIENTATION_TOPLEFT)
                   && TIFFSetField(tif, TIFFTAG_COMPRESSION,     COMPRESSION_LZW)
                   ;

        if( img.depth() > 8 )
        {
#if QT_VERSION < 0x040000
            bool alpha   = img.hasAlphaBuffer();
            QImage img32 = img.convertDepth(32);
#else
            bool alpha   = img.hasAlphaChannel();
            QImage img32 = img.convertToFormat( alpha ? QImage::Format_ARGB32 
                                                      : QImage::Format_RGB32 );
#endif
            success &= TIFFSetField(tif, TIFFTAG_BITSPERSAMPLE,   8)
                    && TIFFSetField(tif, TIFFTAG_SAMPLESPERPIXEL, (alpha ? 4 : 3) )
                    && TIFFSetField(tif, TIFFTAG_PHOTOMETRIC,     PHOTOMETRIC_RGB)
                    ;

            //  Note: TIFFWriteScanline does destroy the buffer passed when a swab
            //        is needed... So always make a copy !

            unsigned char* buffer = new unsigned char[4*img32.width()];
            Q_CHECK_PTR(buffer);

            for(int y=0; success && (y<img32.height()); ++y)
            {
                Q_UINT32 *p = (Q_UINT32 *)img32.scanLine(y);
                Q_CHECK_PTR(p);

                if( alpha )
                {
                    // Make sure to force memory order RGBA, whatever hardware we are
                    // running on:
                    for(int x=0; x<img32.width(); ++x)
                        buffer[4*x  ] = qRed  (p[x]),
                        buffer[4*x+1] = qGreen(p[x]),
                        buffer[4*x+2] = qBlue (p[x]),
                        buffer[4*x+3] = qAlpha(p[x]);
                }
                else
                {
                    // Make sure to force memory order RGB, whatever hardware we are
                    // running on:
                    for(int x=0; x<img32.width(); ++x)
                        buffer[3*x  ] = qRed  (p[x]),
                        buffer[3*x+1] = qGreen(p[x]),
                        buffer[3*x+2] = qBlue (p[x]);
                }

                success &= ( TIFFWriteScanline(tif, buffer, y) > 0);
            }
            delete[] buffer;

            status = success ? 0 : -2 ;
        }
        else if( img.depth()==8 )
        {
            success &= TIFFSetField(tif, TIFFTAG_BITSPERSAMPLE,   8)
                    && TIFFSetField(tif, TIFFTAG_SAMPLESPERPIXEL, 1)
                    ;

            if( img.isGrayscale() )
            {
                success &= TIFFSetField(tif, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISBLACK);
            }
            else
            {
                int colors = img.numColors();
                uint16 *red   = new uint16[colors]; Q_CHECK_PTR(red);
                uint16 *green = new uint16[colors]; Q_CHECK_PTR(green);
                uint16 *blue  = new uint16[colors]; Q_CHECK_PTR(blue);
                float scale=65535.0f/255.0f;
                for(int c=0; c<colors; ++c)
                {
                    QRgb rgb = img.color(c);
                    red[c]   = (int)(qRed(rgb)  *scale + 0.5f);
                    green[c] = (int)(qGreen(rgb)*scale + 0.5f);
                    blue[c]  = (int)(qBlue(rgb) *scale + 0.5f);
                }
                success &= TIFFSetField(tif, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_PALETTE);
                success &= TIFFSetField(tif, TIFFTAG_COLORMAP,    red, green, blue);
                delete[] blue;
                delete [] green;
                delete[] red;
            }

            // Write all lines, in order (planar does not support anything else ?)
            for(int y=0; success && (y<img.height()); ++y)
                success &= ( TIFFWriteScanline(tif, (tdata_t)img.scanLine(y), y) > 0);

            status = success ? 0 : -2;
        }
        else    // 1-bit bitmap.
        {
            bool msbFirst =
#if QT_VERSION < 0x040000
                (img.bitOrder() == QImage::BigEndian);
#else
                (img.format() == QImage::Format_Mono);
#endif

            success &= TIFFSetField(tif, TIFFTAG_BITSPERSAMPLE,   1)
                    && TIFFSetField(tif, TIFFTAG_SAMPLESPERPIXEL, 1)
                    && TIFFSetField(tif, TIFFTAG_FILLORDER,       msbFirst ? FILLORDER_MSB2LSB : FILLORDER_LSB2MSB)
                    ;

            if( img.color(0) == qRgb(255,255,255) )
                success &= TIFFSetField(tif, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISWHITE);
            else
                success &= TIFFSetField(tif, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISBLACK);

            // Write all lines, in order (planar does not support anything else ?)
            for(int y=0; success && (y<img.height()); ++y)
                success &= ( TIFFWriteScanline(tif, (tdata_t)img.scanLine(y), y) > 0);

            status = success ? 0 : -2;
        }

        TIFFFlush(tif);
        TIFFClose(tif);

    }
    else
        status = -1;
}
