// EXchess source code, (c) Daniel C. Homan  1997-2000
// Released under the GNU public license, see file license.txt

/* Search Functions */
#define TIME_FLAG 123456

#include <iostream.h>
#include <stdio.h>

#if DEBUG
 #include <string.h>
 #include <stdlib.h>
#endif

#include <fstream.h>
#include "define.h"
#include "chess.h"
#include "const.h"
#include "funct.h"
#include "hash.h"
#include "search.h"


/*----------------------- Search function ---------------------*/
// Driver for the search process.  1st initialize important data
// structures, the do iterative deeping until time runs out.
move search(position p, int time_limit, int T)
{
  char outstring[60], mstring[10];
  int g, done, pvi;
  long elapsed;
  position pv_pos;
  turn = T;
  root_wtm = p.wtm;
  nomove.t = 0;
  
#if DEBUG
 outfile.open("search.log");
#endif

  // logging the current position in FEN format
  if(logging) {
   logfile << "Starting search on position:\n";
   int lcount; square lsqr;
   for(int rank = 7; rank >= 0; rank--) {
     lcount = 0;
     for(int file = 0; file < 8; file++) {
       lsqr = p.sq[SQR(file,rank)];
       if(lsqr.type) {
	if(lcount) { logfile << lcount; lcount = 0; }
        if(lsqr.side) 
         logfile << name[lsqr.type];
        else logfile << bname[lsqr.type];
       } else lcount ++;          
     }
     if(lcount) logfile << lcount;
     if(rank) logfile << "/";
   }
   if(p.wtm) logfile << " w ";
   else logfile << " b ";
   if(p.castle&1) logfile << "K";
   if(p.castle&2) logfile << "Q";
   if(p.castle&4) logfile << "k";
   if(p.castle&8) logfile << "q";
   if(!p.castle)  logfile << "-";   
   if(p.ep)
    logfile << " " << char(FILE(p.ep) + 97) << (RANK(p.ep) + 1);
   else logfile << " -";
   logfile << "\n";
  }

  // Set start of search depth
  if(pc[0][0].t == p.last.t && pc[0][0].t) { 
   start_depth = MAX(last_depth-1, 1);
  } else if(pc[0][1].t == p.last.t && pc[0][1].t && !last_ponder) {
   start_depth = MAX(last_depth-2, 1);
  } else {
   start_depth = 1;
  }

  // setting counts to zero
  node_count = 0; eval_count = 0; time_count = 0;
  phash_count = 0; hash_count = 0; hmove_count = 0; q_count = 0;
  null_cutoff = 0; extensions = 0; internal_iter = 0;
  egtb_probes = 0; egtb_hits = 0;

  time_check_interval = 500;

  // setup some search parameters - for old hash entries
  if(h_id > 100 || h_id < 0) h_id = 1;
  h_id++;

  // is this a pondering session, if so - make the move from the pv
  if(ponder) {
   if(book && !pc[0][1].t) ponder_move = opening_book(p.hcode, p);
   else ponder_move = pc[0][1];
   if(ics) { ics = 0; print_move(p, ponder_move, mstring); ics = 1; }
   else print_move(p, ponder_move, mstring);
   if(ponder_move.t) {
     if(!exec_move(&p, ponder_move, 0)) { ponder_time = 0; return nomove; }
   } else { ponder_time = 0; return nomove; }
   if(!ALLEG && post) {
    cout << "Hint: " << mstring << "\n";
    cout.flush();
   }
   if(logging) logfile << "***** Ponder Move: " << mstring << " *****\n";
   // add ponder move to position list
   p_list[T-1] = p.hcode;
   // set ponder time doubling
   ponder_time_double = 0;
  }

  // initializing eval
  init_score(&p, T);

  // checking tablebases at root
  if(p.pieces[0] + p.pieces[1]
      + p.plist[0][PAWN][0] + p.plist[1][PAWN][0] <= EGTB) {
   root_tb_score = probe_tb(&p, 0);
  } else root_tb_score = -1;

  // checking book
  if(book && !ponder) {
    book_search = 1;
    bookm = opening_book(p.hcode, p);
    book_search = 0;
    if(bookm.t) {
     pc[0][0] = bookm; pc[0][1].t = 0;
     if(logging) {
      if(ics) { ics = 0; print_move(p, bookm, mstring); ics = 1; }
      else print_move(p, bookm, mstring);
      logfile << "Book Move Found: " << mstring << "\n";
     }
     return pc[0][0];
    } else {
     no_book++;
     if(no_book >= 3) book = 0;
    }
  }

  // if last search was a ponder and if user made expected move and
  // if ponder time is greater than limit, return best move
  if(last_ponder && ponder_time >= (time_limit<<ponder_time_double) &&
     p.last.t == ponder_move.t && pc[0][0].t) {
   if(logging) {
    if(ics) cout << "tellics whisper Guessed last move! Score: "
		 << g_last << " Ply: " << last_depth << "\n";
    if(ics) { ics = 0; print_move(p, pc[0][0], mstring); ics = 1; }
    else print_move(p, pc[0][0], mstring);
    logfile << "Guessed Last Move! Returning: " << mstring << "\n";
   }
   cout.flush();
   return pc[0][0];
  }
  if(last_ponder && p.last.t == ponder_move.t && pc[0][0].t && !book_search) {
   start_depth = MAX(last_depth-1,1);
   h_id--; // reset hash id to mark entries as current
   if(logging) {
    logfile << "Guessed last move, but searching deeper...\n";
    if(ics) cout << "tellics whisper Guessed last move, searching deeper...\n";
   }
   cout.flush();
   // set target time
   limit = (time_limit<<ponder_time_double) - ponder_time;
  } else {
   pc[0][0].t = NOMOVE;   // clear first move in the pv
   limit = time_limit;    // set target time
  }

  last_ponder = 0;
  ponder_time = 0;
  max_limit = int(MIN(4*time_limit, timeleft/6.0));
  if(!tsuite && !analysis_mode) {
   limit = MIN(max_limit, limit);
   if(timeleft < 500) {
    start_depth = 1;
    time_check_interval = 50;
   }
  }
  start_time = GetTime();
  last_depth = 1;

  // adjusting hash code for game stage
  or(p.hcode, stage_code[stage]);

  // initialize history table
  for(int i = 0; i < 64; i++)
   for(int j = 0; j < 64; j++) history[i][j] = 0;
  
  if(logging) {
   logfile << "Target search time: " << time_limit << "\n";
   logfile << "Starting depth: " << start_depth << "\n";
   logfile << "Game stage: " << stage << "\n";
  }
        
  // Main search loop for iterative deeping
  for(max_ply = start_depth; max_ply < MAXD-1; max_ply++)
  {
    sp[0] = p;
    node_count++;

    max_depth = max_ply;

    if(max_ply != start_depth) 
     { root_alpha = g_last-25; root_beta = g_last+25; }
    else 
     { root_alpha = -MATE; root_beta = +MATE; }
    
    done = 2;

    while(1) {
     g = pvs(root_alpha, root_beta, (max_ply-1)*UNIT_DEPTH+INIT_EXT, 0);
     if(g == -TIME_FLAG) break;
     if(g <= root_alpha) {
      if(done == 2) {
       root_beta = root_alpha+1; root_alpha = -MATE; 
      } else if(done == 1) {
       root_beta = +MATE; root_alpha = -MATE;
      } else break;
      done--;
     } else if(g >= root_beta) {
      if(done == 2) {
       root_alpha = root_beta-1; root_beta = +MATE; 
      } else if(done == 1) {
       root_beta = +MATE; root_alpha = -MATE;
      } else break;
      done--;
     } else break;
    }

    // writing pv into hash table to be searched first

    pv_pos = p; pvi = 0;

    while(pc[0][pvi].t) {
     put_move(pv_pos.hcode, pc[0][pvi].t);
     exec_move(&pv_pos, pc[0][pvi], 0);
     pvi++;
    } 
         
    if(post && !ics && g != -TIME_FLAG) {
      search_display(g, start_time, node_count, max_ply);
      best_depth = max_ply; 
    }

    // if time is up, or we found a mate... break
    if(inter() || g == -TIME_FLAG) break; 

    g_last = g;
    last_depth = max_ply;

    if((g == 0 && max_ply > MAXD-3) ||
        ((g > (MATE>>1) || g < -(MATE>>1)) && max_ply > 2)) break;

  }

  elapsed = GetTime() - start_time;  if(!elapsed) elapsed = 1;

  if(ponder) {
   last_ponder = 1; ponder_time = elapsed;
  }

  // adjusting hash code for game stage
  or(p.hcode, stage_code[stage]);

  if(!xboard && !ALLEG && post) {
   cout << "\nnode_count = " << node_count
        << " quiescent nodes = " << q_count
        << " eval_count = " << eval_count << "\n";
   cout << "hash hits = " << hash_count
        << " hash moves = " << hmove_count
        << " pawn hash hits = " << phash_count << "\n";
   cout << "node_rate = " << int(float(node_count*100)/float(elapsed))
        << " null cuts = " << null_cutoff
        << " exten = " << extensions
        << " int_iter = " << internal_iter << "\n";
   cout << "egtb_probes = " << egtb_probes 
        << " egtb_hits = " << egtb_hits << "\n";
  }

  if(logging) {
   logfile << "node_count = " << node_count
        << " quiescent nodes = " << q_count
        << " eval_count = " << eval_count << "\n";
   logfile << "hash hits = " << hash_count
        << " hash moves = " << hmove_count
        << " pawn hash hits = " << phash_count << "\n";
   logfile << "node_rate = " << int(float(node_count*100)/float(elapsed))
        << " null cuts = " << null_cutoff
        << " exten = " << extensions
        << " int_iter = " << internal_iter << "\n";
   logfile << "egtb_probes = " << egtb_probes 
        << " egtb_hits = " << egtb_hits << "\n";
  }

  // Kibitz the search results to ics server
  if(ics && !ALLEG && !ponder && logging) {
   if(xboard) sprintf(outstring, "tellics whisper ");
   else sprintf(outstring, "\n Search summary: ");
   cout << outstring;
   if(wbest > (MATE>>1)) {
     sprintf(outstring, "MATE+(in %i moves) ", (MATE-wbest)>>1);
   } else if(wbest < -(MATE>>1)) {
     sprintf(outstring, "MATE-(in %i moves) ", (MATE+wbest)>>1);
   } else if(wbest >= 0) {
     sprintf(outstring, "+%.2f, ", float(wbest)/value[PAWN]);
   } else {
     sprintf(outstring, "%.2f, ", float(wbest)/value[PAWN]);
   }
   cout << outstring;
   if (elapsed >= 5)
    sprintf(outstring, "%i ply, nps: %.0f ",
             wply, float(node_count*100)/elapsed);
   else sprintf(outstring, "%i ply ", wply);
   cout << outstring;
   cout << "\n";
  }

#if DEBUG
 outfile.close();
#endif

  // learning if applicable
  if(!ponder && learn_bk && BOOK_LEARNING) {
    if(!book && learn_count > 0 && wbest > +LEARN_SCORE)
      { book_learn(1); learned = learn_count; learn_count = -1; }
    else if(!book && learn_count > 0 && wbest < -LEARN_SCORE)
      { book_learn(0); learn_count = -1; }
    else if(learned && wbest < -LEARN_SCORE)
      { learn_count = learned; book_learn(-1);
        learned = 0; learn_count = -1; }
   }

  if(logging) {
   if(ics) { ics = 0; print_move(p, pc[0][0], mstring); ics = 1; }
   else print_move(p, pc[0][0], mstring);
   if(!ponder)  
    logfile << "Return move from search: " << mstring << "\n";
   if(ponder) {
     logfile << "***** Pondering ended ***** Time: " << ponder_time 
             << ", Time required doublings: " << ponder_time_double << "\n";
   }
  }

  return pc[0][0];

}

/* Function to update principle continuation */
// The principle continuation is stored as a "triangular" array.
// It is updated by doing a mem-copy of the principle continuation
// found at deeper depths to this depth + the move at this depth
// is stuffed in first.
inline void pc_update(move pcmove, int ply)
{
 int pci;
 pc[ply][ply].t = pcmove.t;
 for (pci = ply+1; ; pci++)
 {
  if(ply+1 < max_ply && pci >= max_ply)
   { if(!(pc[ply+1][pci].t)) break; }
  else if (pci >= max_ply) break;
  pc[ply][pci].t = pc[ply+1][pci].t;
 }
 pc[ply][pci].t = NOMOVE;
}

// Special sort to shuffle best move to the top of the list
inline void MSort(move_rec *Lb, move_rec *Ub)
{
  move_rec V, *I, *J;

   V = *Lb; J = Lb;
   for(I = Lb+1; I <= Ub; I++) {
       if (I->score > J->score) { J = I; }
   }
   *Lb = *J;
   *J = V;
}

/*-------------------- Principle variation function --------------------*/
// This function uses the improvement on the
// alpha-beta scheme called "principle variation search".
// After the first node, which is searched with the full alpha-beta window,
// the rest of the nodes are searched with a "null" window: alpha,alpha+1.
int tb_hit = 0;

int pvs(int alpha, int beta, int depth, int ply)
{

 alpha = MAX(alpha, -MATE+ply);

 int in_pv = 0, RR = R;
 int best = -MATE+ply, talpha = alpha, null_hash = 1;
 int score, old_depth = depth, mate_ext = 0;
 int mcount = 0, first = 1, razor = 0;
 int razor_margin, total_pieces;
 move smove, gmove, bmove = nomove;

#if DEBUG
// Debug archiving of search to a file
char space_string[50] = "\0";
char last_move_string[10];
for(int si = 0; si < ply; si++) strcat(space_string," ");
print_move(sp[ply], sp[ply].last, last_move_string);

outfile << space_string << "->Ply: " << ply << ", max_ply: "
        << max_ply << ", depth: " << depth << ", alpha: " << alpha
        << ", beta: " << beta << ", last: " << last_move_string << "\n";

#endif

 // check time and keyboard interrupt
 if(ply && time_count > time_check_interval
    && (max_ply != start_depth || ponder)) {
  if ((GetTime() - start_time >= limit && !ponder) || (inter())) {
   return -TIME_FLAG;
  }
  time_count = 0;
 }


 if(ply) {
  // add hash code for this position to position list
  p_list[turn+ply-1] = sp[ply].hcode;
  or(p_list[turn+ply-1], stage_code[stage]);

  // fifty move rule
  if(sp[ply].fifty >= 100) return(0);

  // avoid repeating a position if possible
  for(int ri = turn+ply-3; ri >= turn+ply-sp[ply].fifty-1; ri -= 2) {
   if(p_list[ri].key == p_list[turn+ply-1].key
      && p_list[ri].address == p_list[turn+ply-1].address) {
     return(0);
   }
  }

  // check hash table
  score = get_hash(sp[ply].hcode, alpha, beta,(depth>>5)+1,&mate_ext,&null_hash,&gmove);
  if(score != HASH_MISS) {
   hash_count++;
   pc[ply][ply].t = NOMOVE;
   return(score);
  } 

  // check egtb
  if(stage > 1) {
   total_pieces = sp[ply].pieces[0]+sp[ply].pieces[1]
                 +sp[ply].plist[0][PAWN][0]
                 +sp[ply].plist[1][PAWN][0];
   if(total_pieces == 2) return(0);
   if(total_pieces <= EGTB) {
    score = probe_tb(&sp[ply], ply);
    egtb_probes++;
    if(score != -1) {
     if(ply == 1) tb_hit = 1;
     egtb_hits++;
     return(score);
    }
   }
  }
 } else gmove.t = get_move(sp[ply].hcode);

 if(depth >= 384) RR += UNIT_DEPTH;

 if (R && ply && !sp[ply].check && null_hash && ply < MAXD-3
      && sp[ply].pieces[sp[ply].wtm] > 1 && sp[ply].last.t
      && sp[ply].material > beta-NULL_THRESH)
   {
     node_count++;
     sp[ply+1] = sp[ply]; sp[ply+1].wtm ^= 1;
     or(sp[ply+1].hcode,wtm);
     or(sp[ply+1].hcode,btm);
     sp[ply+1].last.t = NOMOVE; sp[ply+1].ep = 0;
     sp[ply+1].material = -sp[ply+1].material;
     sp[ply+1].fifty = 0;
     if(depth >= RR+UNIT_DEPTH) {
       score = -pvs(-beta, -beta+1, depth-UNIT_DEPTH-RR, ply+1);
     } else score = -qsearch(ply+1, -beta, -beta+1);
     if(score == TIME_FLAG) return -TIME_FLAG;
     /* verification search */
     if (score >= beta && depth >= RR+UNIT_DEPTH) {
      node_count++;
      sp[ply+1] = sp[ply];
      sp[ply+1].last.t = NOMOVE;
      score = pvs(beta-1, beta, depth-UNIT_DEPTH-RR, ply+1);
      if(score == TIME_FLAG) return -TIME_FLAG;
     }
     if (score >= beta) {
      null_cutoff++;
      put_hash(sp[ply].hcode, score, beta-1, beta, (depth>>5)+1, nomove, mate_ext);
      pc[ply][ply].t = NOMOVE;
      return beta;
     }
     // bruce moreland's mate extension idea
     if(score <= -(MATE>>1) && depth >= RR+UNIT_DEPTH) mate_ext = 1;
   }


 // Are we in the PV?
 if(!(ply&1)) {
    if(alpha <= MAX(root_alpha,-(MATE>>1)) 
        && beta >= MIN(root_beta,+(MATE>>1))) in_pv = 1;
 } else {
    if(alpha <= MAX(-root_beta,-(MATE>>1)) 
        && beta >= MIN(-root_alpha,+(MATE>>1))) in_pv = 1;
 }

 // set hash move
 hmove = gmove;  if(hmove.t) hmove_count++; 

 // internal iterative deepening - idea taken from crafty
 if(in_pv && !hmove.t && depth > (UNIT_DEPTH<<1)) {
    internal_iter++; null_hash = 0;
    sp[ply+1] = sp[ply];
    score = pvs(alpha, beta, depth-(UNIT_DEPTH<<1), ply+1);
    if(score == TIME_FLAG) { return -TIME_FLAG; }
    if(score <= alpha) {
     score = pvs(-MATE, beta, depth-(UNIT_DEPTH<<1), ply+1);
     if(score == TIME_FLAG) { return -TIME_FLAG; }
    }
    hmove.t = pc[ply+1][ply+1].t;
 } 
 
 // find the semi-legal moves for this position
 legalmoves(&sp[ply], &slist[ply]);

 if(in_pv) {
   for(mcount = 0; mcount < slist[ply].count; mcount++) {
    if(slist[ply].mv[mcount].m.t == hmove.t) continue;
    // making move
    sp[ply+1] = sp[ply];
    if(!exec_move(&sp[ply+1], slist[ply].mv[mcount].m, ply)) continue;        
    slist[ply].mv[mcount].score = -qsearch(ply+1,+MATE,-MATE);
   }
   mcount = 0;
 } else {
  // set razoring of uninteresting moves if material is low enough
  if(depth < RAZOR_DEPTH && !sp[ply].check 
     && !sp[ply].extend &&  max_ply > 2 && depth >= UNIT_DEPTH) {
   razor = 1;
   if(depth < (UNIT_DEPTH<<1)) razor_margin = 50;
   else razor_margin = 50+value[QUEEN];
  }
 }

 while (mcount < slist[ply].count && best < beta)
 {
   // increase node counts
   node_count++;  time_count++;

   // shuffle highest scored move to top of list
   MSort(&slist[ply].mv[mcount], &slist[ply].mv[slist[ply].count-1]);
    
   smove = slist[ply].mv[mcount].m;

   // making move
   sp[ply+1] = sp[ply];
   if(!exec_move(&sp[ply+1], smove, ply)) {
    mcount++;
    continue;
   }

#if DEBUG
 print_move(sp[ply+1], smove, last_move_string);
 outfile << space_string << "Move: " << last_move_string << "\n";
#endif

   // Set the extensions
   while(DEPTH(depth)+ply < (MAXD-3)) {
     // if we caused a check, extend
     if(sp[ply+1].check)
      { depth += CHECK_EXT; if(DEPTH(depth) > DEPTH(old_depth)) break; }
     // if last move was the only available move
     if(slist[ply].count == 1 && sp[ply].check)
      { depth += ONE_REPLY_TO_CHECK; if(DEPTH(depth) > DEPTH(old_depth)) break; }
     // if last move was a capture
     if(sp[ply+1].last.b.type&CAPTURE) {
      // if last move was a re-capture, extend
      if(sp[ply].last.b.type&CAPTURE && ply
         && sp[ply-1].material == sp[ply+1].material      // even exchange
         && sp[ply].sq[sp[ply+1].last.b.to].type > PAWN)
        { depth += RE_CAPT_EXT; if(DEPTH(depth) > DEPTH(old_depth)) break; }
     }
     // if mate_extension
     if(mate_ext) { depth += MATE_EXT; if(DEPTH(depth) > DEPTH(old_depth)) break; }
     // if last move was a pawn push and endgame, extend
     if(sp[ply+1].last.b.type&PAWN_PUSH &&
        (sp[ply+1].last.b.to > 48 || sp[ply+1].last.b.to < 17))
       { depth += PAWN_EXT; if(DEPTH(depth) > DEPTH(old_depth)) break; }
     // Don't go to qsearch if we are in check
     if (depth < UNIT_DEPTH && sp[ply+1].check) { depth = UNIT_DEPTH; }           
     break;
   }

   if(DEPTH(depth) > DEPTH(old_depth)) { extensions++; sp[ply+1].extend = 1; }
   else { sp[ply+1].extend = 0; }

   // razoring if possible, e.g. the current move is uninteresting
   // This is basically just a negative extension.
   if(razor && depth == old_depth
       && -sp[ply+1].material+razor_margin <= alpha) {
      depth -= UNIT_DEPTH;
   }

   if(!ply) tb_hit = 0;

   if (depth < UNIT_DEPTH) {
    score = -get_hash(sp[ply+1].hcode,-beta,-alpha, 0, &mate_ext, &null_hash, &gmove);
    if(score == -HASH_MISS) {
     score = -qsearch(ply+1, -beta, -alpha);
     put_hash(sp[ply+1].hcode, -score, -beta, -alpha, 0, nomove, 0);
    } else hash_count++;
   } else {
    if (first) {
     score = -pvs(-beta, -alpha, depth-UNIT_DEPTH, ply+1);
     if(!ply && tb_hit && score > (MATE>>1) 
         && score <= root_tb_score) score = alpha;
     if (score == TIME_FLAG) return -TIME_FLAG;    
    } else {
     score = -pvs(-alpha-1, -alpha, depth-UNIT_DEPTH, ply+1);
     if(!ply && tb_hit && score > (MATE>>1) 
         && score <= root_tb_score) score = alpha;
     if (score == TIME_FLAG) return -TIME_FLAG;
     if (score > alpha && score < beta) {
       score = -pvs(-beta, -alpha, depth-UNIT_DEPTH, ply+1);
       if (score == TIME_FLAG) return -TIME_FLAG;
     }
    }
   }

#if DEBUG
 outfile << space_string << "Returned value: " << score << "\n";
#endif
 
   // Panic mode for additional time 
   //  also display if we are failing low
   if(!ply && first && max_ply > 3 && !book_search) { 
     if(limit < max_limit && score <= g_last-30 
        && !tsuite && GetTime() - start_time >= (limit>>2)) {
       if(!ponder) { 
        limit *= 2;
        if(logging) logfile << "Extending search to: " << limit << "\n";
       } else if(ponder_time_double < 2) {
        ponder_time_double++;
        if(logging) logfile << "Doubling required ponder time\n";
       }
     }
     if(score <= alpha) {
      fail = -1;
      if(post && !ics) search_display(score, start_time, node_count, max_ply);
      if(logging) log_search(score, start_time, node_count, max_ply);
      fail = 0;
      return(score);
     }
   }

   first = 0;

   if(score > best) best = score;

   if (score > alpha) {
     alpha = score;
     bmove = smove;
     pc_update(smove, ply);
     if(!ply) {
      wbest = score; wply = max_ply;             // whisper variables
      if(score >= beta) fail = 1; 
      if(post && !ics)
        search_display(score, start_time, node_count, max_ply);
      if(logging) log_search(score, start_time, node_count, max_ply);      
      if(fail) fail = 0;
     }
     // update history list and killer moves
     if(score >= beta) {
      history[smove.b.from][smove.b.to] += depth+1;
      killer3[sp[ply].wtm] = killer2[sp[ply].wtm];
      killer2[sp[ply].wtm] = killer1[sp[ply].wtm];
      killer1[sp[ply].wtm] = smove.t;
     }
   }

   depth = old_depth;

   mcount++;
 }       


 // if it is stalemate; the score is even
 if (first && ply && !sp[ply].check) best = 0;

 // storing position in the hash table
 put_hash(sp[ply].hcode, best, talpha, beta, (depth>>5)+1, bmove, mate_ext);

 if(!bmove.t) pc[ply][ply].t = NOMOVE;

 return best;

}

/*--------------------- Quiescent search ---------------------*/
// This searches only non-losing captures.  Futility cut-offs
// are made if the capture is not likely to bring us up back
// above alpha.   A straight alpha-beta algorithm is used here.

int qsearch(int ply, int alpha, int beta)
{
  register int qi, best, score;  // best score, score for current move
  move_rec qmove;                // Move record for current move to search

  best = score_pos(&sp[ply],alpha,beta); eval_count++;

  if(ply >= MAXD-2  || best >= beta) return best;   // break if best > beta
                                                  //  or we are too deep
  // set futility cutoff
  if(best > alpha) alpha = best;
  delta_score = alpha - MAX(best,sp[ply].material) - 50;     
  delta_score = MAX(0,delta_score);
  if(alpha > best) best = alpha;

  // generate captures
  captures(&sp[ply], &slist[ply]);

  // loop trough possible captures, trying them in turn
  for (qi = 0; qi < slist[ply].count; qi++)
  {
    // shuffle highest scored move to top of list
    MSort(&slist[ply].mv[qi], &slist[ply].mv[slist[ply].count-1]);
    qmove = slist[ply].mv[qi];

    // increase node counts
    node_count++; q_count++; time_count++;

    if(sp[ply].sq[qmove.m.b.to].type == KING) return(MATE>>1);

    // execute move
    sp[ply+1] = sp[ply];
    qexec_move(&sp[ply+1], qmove.m, ply);

    // call next iteration
    score = -qsearch(ply+1, -beta, -alpha);

    if (score > alpha) {
     pc_update(qmove.m, ply);     
     if (score >= beta) return(score);  
     alpha = score;
     best = score;
    }

  }

  return best;
}






