#include <ChordSpace.hpp>
#include <System.hpp>
#include <iostream>
#include <cstdlib>
#include <cstdio>
#include <string>

static bool printPass = false;
static bool failureExits = true;
static int failedCount = 0;
static int passedCount = 0;
static int testCount = 0;
static int exitAfterFailureCount = 1;
static int maximumVoiceCountToTest = 4;

static void pass(std::string message) {
	if (printPass) {
        passedCount = passedCount + 1;
        testCount = passedCount + failedCount;
		csound::print("\nPASSED %6d of %d: %s\n", message.c_str(), passedCount, testCount);
	}
}

static void fail(std::string message) {
    failedCount = failedCount + 1;
    testCount = passedCount + failedCount;
	csound::print("========================================================================\n");
	csound::print("FAILED %6d of %6d:  %s\n", message.c_str(), failedCount, testCount);
	csound::print("========================================================================\n");
	if (failureExits && (failedCount >= exitAfterFailureCount)) {
		std::exit(-1);
	}
}

static bool test(bool passes, std::string message) {
	if (passes) {
		pass(message);
	} else {
		fail(message);
	}
    return passes;
}

static void testChordSpaceGroup(const csound::ChordSpaceGroup &chordSpaceGroup, std::string chordName) 
{
    csound::print("BEGAN test ChordSpaceGroup for %s...\n", chordName.c_str());
    csound::Chord originalChord = csound::chordForName(chordName);
    csound::Chord optti = originalChord.eOPTTI();
    csound::print("Original chord:\n%s\n", originalChord.information().c_str());
    Eigen::VectorXi pitv = chordSpaceGroup.fromChord(originalChord, true);
    csound::Chord reconstitutedChord = chordSpaceGroup.toChord(pitv[0], pitv[1], pitv[2], pitv[3], true)[0];
    csound::print("Reconstituted chord:\n%s\n", reconstitutedChord.information().c_str());
    test(originalChord == reconstitutedChord, "Reconstituted chord must be the same as the original chord.\n");
    csound::Chord revoicedOriginalChord = originalChord;
    revoicedOriginalChord.setPitch(1,  revoicedOriginalChord.getPitch(1) + 12.);
    revoicedOriginalChord.setPitch(2,  revoicedOriginalChord.getPitch(2) + 24.);
    csound::print("Revoiced original chord:\n%s\n", revoicedOriginalChord.information().c_str());
    pitv = chordSpaceGroup.fromChord(revoicedOriginalChord, true);
    csound::Chord reconstitutedRevoicedChord = chordSpaceGroup.toChord(pitv, true)[0];
    csound::print("Reconstituted revoiced chord:\n%s\n", reconstitutedRevoicedChord.information().c_str());
    test(revoicedOriginalChord == reconstitutedRevoicedChord, "Reconstituted revoiced chord must be the same as the original revoiced chord.\n");
    csound::Chord invertedChord = originalChord.I().eOP(); 
    csound::print("Inverted original chord:\n%s\n", invertedChord.information().c_str());
    pitv = chordSpaceGroup.fromChord(invertedChord);
    csound::Chord reconstitutedInvertedChord = chordSpaceGroup.toChord(pitv, true)[0];
    csound::print("Reconstituted inverted chord:\n%s\n", reconstitutedInvertedChord.information().c_str());
    test(invertedChord == reconstitutedInvertedChord,"Reconstituted inverted chord must be the same as the original inverted chord.\n");
    csound::print("ENDED test ChordSpaceGroup for %s.\n", chordName.c_str());
    csound::print("");
}

static void testAllOfChordSpaceGroup(int initialVoiceCount, int finalVoiceCount)
{
    for (int voiceCount = initialVoiceCount; voiceCount <= finalVoiceCount; ++voiceCount) {
        bool passes = true;
        csound::ChordSpaceGroup chordSpaceGroup;
        chordSpaceGroup.initialize(voiceCount, 48);
        chordSpaceGroup.list(true, true, true);
        for (int V = 0; V < chordSpaceGroup.countV; ++V) {
            for (int P = 0; P < chordSpaceGroup.countP; ++P) {
                for (int I = 0; I < chordSpaceGroup.countI; ++I) {
                    for (int T = 0; T < chordSpaceGroup.countT; ++T) {
                        csound::Chord fromPITV = chordSpaceGroup.toChord(P, I, T, V)[0];
                        csound::print("%8d     %8d     %8d     %8d => %s\n", P, I, T, V, fromPITV.toString().c_str());
                        Eigen::VectorXi pitv = chordSpaceGroup.fromChord(fromPITV);
                        csound::Chord frompitv = chordSpaceGroup.toChord(pitv(0), pitv(1), pitv(2), pitv(3))[0];     
                        csound::print("%8d     %8d     %8d     %8d <= %s\n", frompitv.toString().c_str(),pitv(0), pitv(1), pitv(2), pitv(3));
                        if(!test(fromPITV == frompitv, "fromPITV must match frompitv.\n")) {
                            csound::print("fromPITV (toChord):\n%s\n", fromPITV.information().c_str());
                            csound::print("frompitv (fromChord):\n%s\n", frompitv.information().c_str());
                        }
                        csound::print("\n");
                    }
                }
            }
        }
    }
}

/**
 * Consistency is tested as follows for a set of chords in R for 2 through 12
 * voices, for each equivalence class whether simple (O) or compound (OP). Note
 * that some tests are omitted because more than one chord within the same
 * fundamental domain may be equivalent, though these must be on the boundary
 * of the domain.
 * 
 * Additionally, for the compound equivalence classes, chord:isE() on the l.h.s.
 * will imply AND of each chord:iseE() for the constituent simple equivalent
 * classes on the r.h.s, and the same with the sides switched:
 * 
 * iseOP <=> iseO and iseP
 * iseOPT <=> iseOP and iseO and iseP and iseT and iseV
 * iseOPI <=> iseOP and iseO and iseP and iseI
 * iseOPTI <=> iseOPT and iseOPI and iseOP and iseO and iseP and iseT and iseI and iseV
 */
bool testEquivalence(std::string equivalence, const csound::Chord &chord, 
                      bool (csound::Chord::*iseE)() const, 
                      csound::Chord (csound::Chord::*eE)() const)
{
    // chord:eE().iseE() == true
    char description[0x5000];
    std::sprintf(description, "chord.e%s().ise%s() == true", equivalence.c_str(), equivalence.c_str());
    csound::Chord equivalent = (chord.*eE)();
    if (!((equivalent.*iseE)() == true)) {
        csound::print("Chord:\n%s\n", chord.information().c_str());
        csound::print("Equivalent:\n%s\n", equivalent.information().c_str());
        fail(description);
    } else {
        pass(description);
    }
    // (chord.iseE() == false) => (chord.eE() != chord)
    std::sprintf(description, "(chord:ise%s() == false) => (chord:e%s() != chord)", equivalence.c_str(), equivalence.c_str());
    if ((chord.*iseE)() == false ) {
        if (equivalent == chord) {
            csound::print("Chord:\n%s\n", chord.information().c_str());
            csound::print("Equivalent:\n%s\n", equivalent.information().c_str());
            fail(description);
        }
    } else {
        pass(description);
    }
    // (chord:eE() == chord) => (chord:iseE() == true)
    std::sprintf(description, "(chord:e%s() == chord) => (chord:ise%s() == true)", equivalence.c_str(), equivalence.c_str());
    if (equivalent == chord) {
        if (!((chord.*iseE)() == true)) {
            csound::print("Chord:\n%s\n", chord.information().c_str());
            csound::print("Equivalent:\n%s\n", equivalent.information().c_str());
            fail(description);
        }
    } else {
        pass(description);
    }
}

static void testCompoundEquivalence(std::string equivalence, 
    const csound::Chord &chord, 
    bool (csound::Chord::*iseE)() const, 
    std::vector<bool (csound::Chord::*)() const> &otherIseEs)
{
    char buffer[0x5000];
    int otherIsEsN = 0;
    for (int i = 0, n = otherIseEs.size(); i < n; ++i) {
        bool (csound::Chord::*iseE)() const = otherIseEs[i];
        if ((chord.*iseE)()) {
            otherIsEsN++;
        }
    }
    if ((chord.*iseE)()) {
        if (otherIsEsN == otherIseEs.size()) {
            std::sprintf(buffer, "%s: compound equivalence for:\n%s\n", equivalence.c_str(), chord.information().c_str());
            pass(buffer);
        } else {
            std::sprintf(buffer, "%s: missing compound equivalence for:\n%s\n", equivalence.c_str(), chord.information().c_str());
            fail(buffer);
        }
    } else {
        if (otherIsEsN == otherIseEs.size()) {
            std::sprintf(buffer, "%s: false compound equivalence for:\n%s\n", equivalence.c_str(), chord.information().c_str());
            fail(buffer);
        } else {
            pass(buffer);
            std::sprintf(buffer, "%s: compound equivalence for:\n%s\n", equivalence.c_str(), chord.information().c_str());
        }
    }
}

static void testEquivalences(int voiceCount, double range = 12.0)
{
    csound::print("\nTesting equivalences for %d voices over range %f...\n\n", voiceCount, range);
    csound::Chord chord = csound::iterator(voiceCount, -range);
    while (csound::next(chord, -range, range, 1.0)) {
        testEquivalence("O",    chord, &csound::Chord::iseO,    &csound::Chord::eO);
        testEquivalence("P",    chord, &csound::Chord::iseP,    &csound::Chord::eP);
        testEquivalence("T",    chord, &csound::Chord::iseT,    &csound::Chord::eT);
        testEquivalence("I",    chord, &csound::Chord::iseI,    &csound::Chord::eI);
        testEquivalence("V",    chord, &csound::Chord::iseV,    &csound::Chord::eV);
        testEquivalence("OP",   chord, &csound::Chord::iseOP,   &csound::Chord::eOP);
        testEquivalence("OPT",  chord, &csound::Chord::iseOPT,  &csound::Chord::eOPT);
        testEquivalence("OPI",  chord, &csound::Chord::iseOPI,  &csound::Chord::eOPI);
        testEquivalence("OPTI", chord, &csound::Chord::iseOPTI, &csound::Chord::eOPTI);
        std::vector<bool (csound::Chord::*)() const> otherIseEs;
        otherIseEs.push_back(&csound::Chord::iseO);
        otherIseEs.push_back(&csound::Chord::iseP);
        testCompoundEquivalence("OP", chord, &csound::Chord::iseOP, otherIseEs);
    }
}

int main(int argc, const char **argv) {
	std::cout << std::endl << "C H O R D S P A C E   U N I T   T E S T S" << std::endl << std::endl;
    csound::print("Behavior of std::fmod and std::remainder:\n");
    for (double pitch = -24.0; pitch < 24.0; pitch += 1.0) {
        double modulusFmod = std::fmod(pitch, csound::OCTAVE());
        double modulusRemainder = std::remainder(pitch, csound::OCTAVE());
        double pc = csound::epc(pitch);
        double modulus = csound::modulo(pitch, csound::OCTAVE());
        csound::print("Pitch: %9.4f  modulo: %9.4f  std::fmod: %9.4f  std::remainder: %9.4f  epc: %9.4f\n", pitch, modulus, modulusFmod, modulusRemainder, pc);
    }
    csound::Chord pcs = csound::chordForName("C major").epcs();
    csound::print("Should be C major scale:\n%s\n", pcs.information().c_str());
    for (double pitch = 36.0; pitch < 96.0; pitch += 1.0) {
        double conformed = csound::conformToPitchClassSet(pitch, pcs);
        csound::print("pitch: %9.4f  conformed: %9.4f\n", pitch, conformed);
    }
	std::cout << "EPSILON: " << csound::EPSILON() << std::endl;
	std::cout << "epsilonFactor: " << csound::epsilonFactor() << std::endl;
	std::cout << "EPSILON * epsilonFactor: " << csound::EPSILON() * csound::epsilonFactor() << std::endl;
	std::cout << "csound::eq_epsilon(0.15, 0.15): " << csound::eq_epsilon(0.15, 0.15) << std::endl;
	std::cout << "csound::eq_epsilon(0.1500001, 0.15): " << csound::eq_epsilon(0.1500001, 0.15) << std::endl;
	std::cout << "csound::gt_epsilon(14.0, 12.0): " << csound::gt_epsilon(14.0, 12.0) << std::endl;
	std::cout << "csound::ge_epsilon(14.0, 12.0): " << csound::ge_epsilon(14.0, 12.0) << std::endl;
	csound::Chord chord;
	std::cout << "Default chord: " << chord.toString() << std::endl;
	std::cout << "chord.count(0.0): " << chord.count(0.0) << std::endl;
	csound::Chord other;
	other.setPitch(1, 2);
	std::cout << "Other chord: " << other.toString() << std::endl;
	std::cout << "other.count(0.0): " << other.count(0.0) << std::endl;
	std::cout << "(chord == other): " << (chord == other) << std::endl;
	std::cout << "other.contains(2.0): " << other.contains(2.0) << std::endl;
	std::cout << "other.contains(2.00000001): " << other.contains(2.00000001) << std::endl;
	std::vector<double> result = other.min();
	std::cout << "other.min(): " << result[0] << ", " << result[1] << ", " << result.size() << std::endl;
	std::cout << "other.minimumInterval(): " << other.minimumInterval() << std::endl;
	std::cout << "other.maximumInterval(): " << other.maximumInterval() << std::endl;
	csound::Chord clone = other;
	std::cout << "clone = other: " << clone.toString() << std::endl;
	clone.setPitch(1, .5);
	std::cout << "clone: " << clone.toString() << std::endl;
	csound::Chord floor = clone.floor();
	std::cout << "floor: " << floor.toString() << std::endl;
	csound::Chord ceiling = clone.ceiling();
	std::cout << "ceiling: " << ceiling.toString() << std::endl;
	chord.setPitch(0, 1);
	chord.setPitch(1, 1);
	std::cout << "chord: " << chord.toString() << std::endl;
	std::cout << "chord.distanceToOrigin(): " << chord.distanceToOrigin() << std::endl;
	std::cout << "chord.distanceToUnisonDiagonal(): " << chord.distanceToUnisonDiagonal() << std::endl;
	std::cout << "chord.maximallyEven(): " << chord.maximallyEven().toString() << std::endl;
	std::cout << "chord.T(3): " << chord.T(3).toString() << std::endl;
	std::cout << "chord.I(): " << chord.I().toString() << std::endl;
	std::cout << "csound::epc(13.2): " << csound::epc(13.2) << std::endl;
	std::cout << "chord.isepcs(): " << chord.isepcs() << std::endl;
	chord.setPitch(2, 14);
	std::cout << "chord: " << chord.toString() << std::endl;
	std::cout << "chord.isepcs(): " << chord.isepcs() << std::endl;
	csound::Chord epcs = chord.epcs();
	std::cout << "chord.epcs(): " << epcs.toString() << std::endl;
	std::cout << "csound::epc(14.0): " << csound::epc(14.0) << std::endl;
	csound::Chord transposed = chord.T(5.0);
	std::cout << "chord::T(5.0): " << transposed.toString() << std::endl;
	std::cout << "transposed.iset(): " << transposed.iset() << std::endl;
	csound::Chord et = transposed.et();
	std::cout << "et = transposed.et(): " << et.toString() << std::endl;
	std::cout << "transposed.iseO(): " << transposed.iseO() << std::endl;
	csound::Chord eO = transposed.eO();
	std::cout << "transposed.eO(): " << eO.toString() << std::endl;
	std::cout << "eO.iseO(): " << eO.iseO() << std::endl;
	std::cout << "eO.iseP(): " << eO.iseP() << std::endl;
	csound::Chord eP = eO.eP();
	std::cout << "eP = eO.eP(): " << eP.toString() << std::endl;
	std::cout << "eP.iseT(): " << eP.iseT() << std::endl;
	csound::Chord eT= eP.eT();
	std::cout << "eT = eP.eT(): " << eT.toString() << std::endl;
	std::cout << "eT.iseT(): " << eT.iseT() << std::endl;
	csound::Chord eTT= eP.eTT();
	std::cout << "eTT = eP.eTT(): " << eTT.toString() << std::endl;
	std::cout << "eTT.iseT(): " << eTT.iseTT() << std::endl;
	std::cout << "eT.iseTT(): " << eT.iseTT() << std::endl;
	std::cout << "eTT: " << eTT.toString() << std::endl;
	csound::Chord inverse = eTT.I();
	std::cout << "csound::Chord inverse = eTT.I(): " << inverse.toString() << std::endl;
	csound::Chord inverseOfInverse = inverse.I();
	std::cout << "csound::Chord inverseOfInverse = inverse.I(): " << inverseOfInverse.toString() << std::endl;
	std::cout << "inverse.iseI(): " << inverse.iseI() << std::endl;
	csound::Chord eI = eTT.eI();
	std::cout << "csound::Chord eI = eTT.eI(): " << eI.toString() << std::endl;
	std::cout << "eI.iseI(): " << eI.iseI() << std::endl;
	std::cout << "eTT.iseI(): " << eTT.iseI() << std::endl;
	std::cout << "(inverse < eTT): " << (inverse < eTT) << std::endl;
	std::cout << "(eTT < inverse): " << (eTT < inverse) << std::endl;
	std::cout << "chord: " << chord.toString() << std::endl;
	std::cout << "chord.cycle(): " << chord.cycle().toString() << std::endl;
	std::cout << "chord.cycle(2): " << chord.cycle(2).toString() << std::endl;
	std::cout << "chord.cycle().cycle(): " << chord.cycle().cycle().toString() << std::endl;
	std::cout << "eI: " << eI.toString() << std::endl;
	std::cout << "eI.cycle(-1): " << eI.cycle(-1).toString() << std::endl;
	std::vector<csound::Chord> permutations = chord.permutations();
	std::cout << "Permutations, iseV, eV:" << std::endl;
	for (size_t i = 0; i < permutations.size(); i++) {
		const csound::Chord permutation = permutations[i];
		const csound::Chord &eV = permutation.eV();
		std::cout << i << " " << permutation.toString() << "  iseV: " << permutation.iseV() << "  eV: " << eV.toString() << "  iseP: " <<  permutation.iseP() << std::endl;
	}
	std::vector<csound::Chord> voicings = chord.voicings();
	std::cout << "voicing, iseV, eV:" << std::endl;
	for (size_t i = 0; i < voicings.size(); i++) {
		const csound::Chord voicing = voicings[i];
		const csound::Chord &eV = voicing.eV();
		std::cout << i << " " << voicing.toString() << "  iseV: " << voicing.iseV() << "  eV: " << eV.toString() << std::endl;
	}
    std::string tosplit = "C     D     E        G           B";
    csound::fill("C", 0., "M9", tosplit, true);
	csound::Chord C7 = csound::chordForName("C7");
	std::cout << "Should be C7:" << std::endl << C7.information().c_str() << std::endl;
	csound::Chord G7 = csound::chordForName("G7");
	std::cout << "Should be G7:" << std::endl << G7.information().c_str() << std::endl;
	csound::Chord CM9 = csound::chordForName("CM9");
	std::cout << "Should be CM9:" << std::endl << CM9.information().c_str() << std::endl;
    CM9.setPitch(0, 0.);
    CM9.setPitch(1, 4.);
    CM9.setPitch(2, 7.);
    CM9.setPitch(3,-1.);
    CM9.setPitch(4, 2.);
	std::cout << "Should be CM9:" << std::endl << CM9.information().c_str() << std::endl;
	csound::Chord Dm9 = csound::chordForName("Dm9");
	std::cout << "Should be Dm9:" << std::endl << Dm9.information().c_str() << std::endl;
    Dm9.setPitch(0, 2.);
    Dm9.setPitch(1, 5.);
    Dm9.setPitch(2, 9.);
    Dm9.setPitch(3, 0.);
    Dm9.setPitch(4, 4.);
	std::cout << "Should be Dm9:" << std::endl << Dm9.information().c_str() << std::endl;
    csound::Chord chordForName_ = csound::chordForName("CM9");
    csound::print("chordForName(%s): %s\n", "CM9", chordForName_.information().c_str());
    for (int voiceCount = 3; voiceCount <= maximumVoiceCountToTest; ++voiceCount) {
        testEquivalences(voiceCount);
    }
    csound::ChordSpaceGroup chordSpaceGroup;
    chordSpaceGroup.createChordSpaceGroup(4, csound::OCTAVE() * 5.0, 1.0);
    chordSpaceGroup.list(true, true, true);
    testChordSpaceGroup(chordSpaceGroup, "Gb7");
    csound::print("\nTesting all of chord space groups...\n\n");
    testAllOfChordSpaceGroup(3, 5);
	csound::print("\nFINISHED.\n\n");
	return 0;
}
