/*
   Copyright (C) 2006-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "mainwindow.h"

#include "case.h"
#include "config.h"
#include "log.h"
#include "pref.h"
#include "preview.h"
#include "settings.h"
#include "tabdisc.h"
#include "taboptions.h"
#include "tabproject.h"
#include "utils.h"

#include "cdinfo.h"
#ifdef ENABLE_AUDIOCD
# include "cddbquery.h"
#endif

#include <kapplication.h>
#include <kcolorbutton.h>
#include <kconfig.h>
#include <kdeversion.h>
#include <keditlistbox.h>
#include <kfiledialog.h>
#include <kfontdialog.h>
#include <kglobal.h>
#include <khelpmenu.h>
#include <kiconloader.h>
#include <kio/netaccess.h>
#include <klineedit.h>
#include <klocale.h>
#include <kmenu.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <kmimetype.h>
#include <kpushbutton.h>
#include <kstatusbar.h>
#include <kstdaccel.h>
#include <kstdaction.h>
#include <ktemporaryfile.h>
#include <kurl.h>
#include <kurlrequesterdlg.h>
#include <kurlrequester.h>

#include <QAction>
#include <QApplication>
#include <QFileDialog>
#include <QHBoxLayout>
#include <QImageReader>
#include <QKeySequence>
#include <QLabel>
#include <QMenuBar>
#include <QPrinter>
#include <QPrintDialog>
#include <QStackedWidget>
#include <QTabBar>
#include <QTimer>
#include <QToolBar>
#include <QVBoxLayout>

#include <iostream>
#include <assert.h>

#define KOVERARTIST_GROUP_SPACING 8
#define KOVERARTIST_BUTTONBLOCK_SPACING 6

#define UPDATE_TIMER_MSEC 200


MainWindow::MainWindow()
:Inherited()
,mTitle(QString("KoverArtist %1").arg(VERSION))
,mProject()
,mRender(new KoverArtist::Renderer(&mProject))
,mPrinter(0)
#ifdef ENABLE_AUDIOCD
,mCddbQuery(new CddbQuery(this))
#else
,mCddbQuery(0)
#endif
,mCaptionLblFont()
,mIdActiveDisc(-1)
,mNumTabsNoDisc(0)
,mCddbDiscId(-1)
,mLocalFileName()
,mPixPreview()
,mUpdPreviewFront(true)
,mUpdPreviewBack(true)
,mRendering(false)
,mPreview(0)
,mTmrUpdate(new QTimer(this))
,mTabBar(0)
,mStkSide(0)
,mBaseProject(0)
,mBaseOptions(0)
,mBaseDisc(0)
,mTabProject(0)
,mTabOptions(0)
{
#ifdef ENABLE_AUDIOCD
   connect(mCddbQuery, SIGNAL(status(const QString&)), this, SLOT(changeStatusbar(const QString&)));
   connect(mCddbQuery, SIGNAL(finished()), this, SLOT(cddbQueryAudioCdDone()));
#endif /*ENABLE_AUDIOCD*/

   KoverArtist::Case *c = KoverArtist::Case::findCase("cd-standard");
   if (c) mProject.setBox(*c);
   else mProject.setBox(KoverArtist::Case::standardCase());

   mTmrUpdate->setSingleShot(true);
   connect(mTmrUpdate, SIGNAL(timeout()), this, SLOT(updatePreviewTimeout()));

   mCaptionLblFont.setBold(true);

   QHBoxLayout* hbox;

   setWindowTitle(mTitle);
   resize(1024, 768);

   QWidget* base = new QWidget(this);
   setCentralWidget(base);

   QVBoxLayout* vbox = new QVBoxLayout(base);
   vbox->setMargin(4);
   vbox->setSpacing(0);

   hbox = new QHBoxLayout;
   vbox->addLayout(hbox, 0);

   mTabBar = new QTabBar(base);
   mTabBar->addTab(i18n("Project"));
   mTabBar->addTab(i18n("Options"));
   connect(mTabBar, SIGNAL(currentChanged(int)), SLOT(switchToTab(int)));
   hbox->addWidget(mTabBar, 0);

   hbox->addStretch(10);

   hbox = new QHBoxLayout;
   vbox->addLayout(hbox, 1000);

   mStkSide = new QStackedWidget(base);

   mTabProject = new TabProject(&mProject, base);
   mTabIdProject = mStkSide->addWidget(mTabProject);
   connect(mTabProject, SIGNAL(changed()), SLOT(updatePreview()));

   mTabOptions = new TabOptions(&mProject, base);
   mTabIdOptions = mStkSide->addWidget(mTabOptions);
   connect(mTabOptions, SIGNAL(changed()), SLOT(updatePreview()));

   mTabDisc = new TabDisc(&mProject, base);
   mTabIdDisc = mStkSide->addWidget(mTabDisc);
   connect(mTabDisc, SIGNAL(changed()), SLOT(updatePreview()));

   mStkSide->setMinimumWidth(mStkSide->sizeHint().width());
   hbox->addWidget(mStkSide);

   connect(mTabBar, SIGNAL(currentChanged(int)),
           mStkSide, SLOT(setCurrentIndex(int)));

   mPreview = new Preview(base);
   mPreview->setFrameStyle(QFrame::StyledPanel|QFrame::Sunken);
   mPreview->setScaledContents(false);
   mPreview->setAlignment(Qt::AlignLeft|Qt::AlignTop);
   mPreview->setMinimumSize(100, 100);
   connect(mPreview, SIGNAL(updateRequest()), SLOT(updatePreview()), Qt::QueuedConnection);
   hbox->addWidget(mPreview, 100);

   setupActions();
   setupMenuBar();
   setupToolBar();

   connect(&mProject, SIGNAL(discsChanged()), SLOT(updateDiscsTabBar()));

   mTmrUpdate->setSingleShot(true);
   mTmrUpdate->setInterval(UPDATE_TIMER_MSEC);
   connect(&mProject, SIGNAL(changed()), mTmrUpdate, SLOT(start()));

   mStkSide->setCurrentIndex(mTabBar->currentIndex());
   updateContents();
   switchToTab(mTabIdProject);
}


MainWindow::~MainWindow()
{
//    KSharedConfigPtr cfg = KGlobal::config();
//    if (mRecentFiles && cfg) mRecentFiles->saveEntries(cfg);

   delete mRender;

#ifdef ENABLE_AUDIOCD
   delete mCddbQuery;
#endif /*ENABLE_AUDIOCD*/
}


void MainWindow::setupActions()
{
   // File actions
   createAction("fileNew", i18n("&New"), "Ctrl+N", 0, SLOT(fileNew()), "document-new");
   createAction("fileOpen", i18n("&Open..."), "Ctrl+O", 0, SLOT(fileOpen()), "document-open");
   createAction("fileSave", i18n("&Save"), "Ctrl+S", 0, SLOT(fileSave()), "document-save");
   createAction("fileSaveAs", i18n("Save &As..."), 0, 0, SLOT(fileSaveAs()), "document-save-as");
   createAction("fileRevert", i18n("Re&vert"), 0, 0, SLOT(fileRevert()), "document-revert");
   createAction("filePrint", i18n("&Print..."), "Ctrl+P", 0, SLOT(filePrint()), "document-print");
   createAction("exit", i18n("&Quit"), "Ctrl+Q", 0, SLOT(close()), "application-exit");

   // Disc actions
   createAction("discNew", i18n("&New Disc"), 0, 0, SLOT(discNew()), "document-new");
   createAction("discDelete", i18n("&Delete Disc"), 0, 0, SLOT(discDelete()), "document-close");
   createAction("discClear", i18n("&Clear Disc"), 0, 0, SLOT(discClear()), "edit-clear");
   createAction("discFront", i18n("Move Disc to &Front"), 0, 0, SLOT(discFront()), "go-previous");
   createAction("discBack", i18n("Move Disc to &Back"), 0, 0, SLOT(discBack()), "go-next");
#ifdef ENABLE_AUDIOCD
   createAction("discImportCddb", i18n("&Import &Audio-CD"), 0, 0, SLOT(discImportCddb()), "mixer-cd");
#endif /*ENABLE_AUDIOCD*/
   createAction("discImport", i18n("&Import Disc"), 0, 0, SLOT(discImport()), "document-import");
   createAction("discExport", i18n("&Export Disc"), 0, 0, SLOT(discExport()), "document-export");

   // Extras
   createAction("searchCoverImages", i18n("&Search for cover images..."), 0, 0, SLOT(searchCoverImages()), "edit-find");

   // Settings
   createAction("settings", i18n("&Settings..."), 0, 0, SLOT(settings()), "configure");

   // Others
   createAction("redisplay", i18n("&Redisplay"), 0, 0, SLOT(redisplay()), "view-refresh");
   createAction("windowNew", i18n("New Window"), 0, 0, SLOT(windowNew()), "window-new");

   // Help
   createAction("about", i18n("About"), 0, 0, SLOT(about()));
}


void MainWindow::setupMenuBar()
{
   mMnuFile = menuBar()->addMenu(i18n("&File"));
   mMnuFile->addAction(mActions["windowNew"]);
   mMnuFile->addSeparator();
   mMnuFile->addAction(mActions["fileNew"]);
   mMnuFile->addAction(mActions["fileOpen"]);
   mMnuFile->addSeparator();
   mMnuFile->addAction(mActions["fileSave"]);
   mMnuFile->addAction(mActions["fileSaveAs"]);
   mMnuFile->addAction(mActions["fileRevert"]);
   mMnuFile->addSeparator();
   mMnuFile->addAction(mActions["filePrint"]);
   mMnuFile->addSeparator();
   mMnuFile->addAction(mActions["exit"]);

   mMnuDisc = menuBar()->addMenu(i18n("&Disc"));
   mMnuDisc->addAction(mActions["discNew"]);
   mMnuDisc->addAction(mActions["discDelete"]);
   mMnuDisc->addAction(mActions["discClear"]);
   mMnuDisc->addSeparator();
#ifdef ENABLE_AUDIOCD
   mMnuDisc->addAction(mActions["discImportCddb"]);
#endif /*ENABLE_AUDIOCD*/
   mMnuDisc->addAction(mActions["discImport"]);
   mMnuDisc->addAction(mActions["discExport"]);
   mMnuDisc->addSeparator();
   mMnuDisc->addAction(mActions["discFront"]);
   mMnuDisc->addAction(mActions["discBack"]);

   QMenu* mnuExtras = menuBar()->addMenu(i18n("&Extras"));
   mnuExtras->addAction(mActions["searchCoverImages"]);

   QMenu* mnuSettings = menuBar()->addMenu(i18n("&Settings"));
   mnuSettings->addAction(mActions["settings"]);

   menuBar()->addMenu(helpMenu());
}


void MainWindow::setupToolBar()
{
   QToolBar* toolBar = new QToolBar(i18n("Main Toolbar"), this);

   toolBar->addAction(mActions["fileNew"]);
   toolBar->addAction(mActions["fileOpen"]);
   toolBar->addAction(mActions["fileSave"]);
   toolBar->addAction(mActions["filePrint"]);
   toolBar->addSeparator();
   toolBar->addAction(mActions["discFront"]);
   toolBar->addAction(mActions["discBack"]);
   toolBar->addSeparator();
   toolBar->addAction(mActions["discNew"]);
   toolBar->addAction(mActions["discDelete"]);
   toolBar->addAction(mActions["discClear"]);
   toolBar->addAction(mActions["discImport"]);
   toolBar->addAction(mActions["discImportCddb"]);
   toolBar->addSeparator();
   toolBar->addAction(mActions["redisplay"]);

   addToolBar(toolBar);
}


QAction* MainWindow::createAction(const QString& aActionName, const QString& aText, const char* aShortcut,
                                  QObject* aObject, const char* aSlotName, const char* aIconName)
{
   assert(mActions.find(aActionName)==mActions.end());

   QString iconName = aIconName ? aIconName : aActionName.toLower();
   QPixmap iconPix = KIconLoader::global()->loadIcon(iconName, KIconLoader::MainToolbar);
   QIcon icon(iconPix);

   QAction* act = new QAction(icon, aText, this);
   if (aShortcut && *aShortcut) act->setShortcut(QKeySequence(aShortcut));
   connect(act, SIGNAL(triggered(bool)), aObject ? aObject : this, aSlotName);

   mActions[aActionName] = act;
   return act;
}


QPushButton* MainWindow::createButton(const char* aIcon, QWidget* aParent,
                                      const char* aSlot, QBoxLayout* aBox) const
{
   QPushButton *btn;
   btn = new QPushButton(BarIconSet(aIcon), 0, aParent);
   btn->setFixedSize(btn->sizeHint());
   connect(btn, SIGNAL(clicked()), aSlot);

   if (aBox) aBox->addWidget(btn);

   return btn;
}


QLabel* MainWindow::createCaptionLabel(const QString& aText, QWidget* aParent) const
{
   QLabel* lbl = new QLabel(aText, aParent);
   lbl->setFont(mCaptionLblFont);
   lbl->setFixedSize(lbl->sizeHint());
   return lbl;
}


void MainWindow::updateContents()
{
   updateTitle();
   updateDiscsTabBar();

   mTabProject->updateContents();
   mTabOptions->updateContents();
   mTabDisc->updateContents();

   scheduleUpdatePreview(50);
}


void MainWindow::updateTitle()
{
   QString url = mProject.url().path();
   if (url.isEmpty()) url = i18n("unnamed")+".koap";
   else
   {
      int idx = url.lastIndexOf('/');
      if (idx>=0) url = url.mid(idx+1);
   }
   QString titleStr = url;
   if (mProject.isModified()) titleStr += " ["+i18n("modified")+"]";
   titleStr += " - " + mTitle + " ";
   setWindowTitle(titleStr);
}


void MainWindow::updateDiscsTabBar()
{
   const int numDiscs = qMax(0, mProject.count());
   while (mTabBar->count()-2 > numDiscs)
      mTabBar->removeTab(mTabBar->count()-1);

   for (int i=mTabBar->count()-2; i<numDiscs; ++i)
      mTabBar->addTab(i18n("Disc %1").arg(i+1));
}


#if 0
bool MainWindow::load(const QString& aUrl)
{
   bool upd = updatesEnabled();
   setUpdatesEnabled(false);
   qApp->setOverrideCursor(Qt::WaitCursor);

   bool ok = mProject.load(aUrl, this);
   bool mod = mProject.isModified();
   updateFromProject();
   updateDiscFromProject();
   if (!mod) mProject.clearModified();
   changeCaption();

//    if (ok && mRecentFiles)
//       mRecentFiles->addURL(aUrl);

   qApp->restoreOverrideCursor();
   setUpdatesEnabled(upd);
   updatePreview(100);

   return ok;
}
#endif


bool MainWindow::save(const KUrl& aUrl)
{
   if (!mProject.save(aUrl, this)) return false;

//    if (mRecentFiles)
//       mRecentFiles->addURL(aUrl);

   mTmrUpdate->stop();
   changeCaption();

   return true;
}


void MainWindow::modified()
{
   if (mProject.isModified()) return;
   mProject.modified();
   changeCaption();
}


bool MainWindow::queryClose()
{
   if (mProject.isModified())
   {
      QString nm = mProject.url().path();
      if (nm.isEmpty()) nm = "unnamed.koap";

      QMessageBox::StandardButton ret =
         QMessageBox::question(this, i18n("Confirm Close") + " - KoverArtist",
               i18n("The project \"%1\" has been modified.\n"
                    "Do you want to save your changes or discard them?").arg(nm),
               QMessageBox::Save|QMessageBox::Discard|QMessageBox::Cancel);

      if (ret==QMessageBox::Cancel)
         return false;
      if (ret==QMessageBox::Save)
         if (!fileSave()) return false;
   }

   return true;
}


bool MainWindow::isImageFile(const QString& aFileName) const
{
   QString ext = KoverArtist::Project::fileExt(aFileName).toUpper();
   if (ext=="JPG") ext = "JPEG";

   QList<QByteArray> fmts = QImageReader::supportedImageFormats();
   QList<QByteArray>::const_iterator it;

   for (it = fmts.begin(); it != fmts.end(); ++it)
      if (*it == ext) return true;

   return false;
}


bool MainWindow::canDecode(const QString& aUrl) const
{
   QString fname = aUrl;
   if (KoverArtist::Project::canLoad(fname)) return true;
   if (KoverArtist::Disc::canLoad(fname)) return true;

#ifdef ENABLE_AUDIOCD
   if (CddbQuery::canDecode(aUrl)) return true;
#endif /*ENABLE_AUDIOCD*/

   return isImageFile(fname);
}


void MainWindow::dragEnterEvent(QDragEnterEvent *e)
{
#if 0
   bool ok = KURLDrag::canDecode(e);
   if (ok)
   {
      KURL::List urls;
      if (KURLDrag::decode(e, urls) && !urls.isEmpty())
         ok = canDecode(urls.first());
   }

   e->accept(ok);
#endif
}


bool MainWindow::imageDropped(const KUrl &aUrl, const QPoint& aPos)
{
#if 0
   QPopupMenu mnu(this);
   mnu.insertItem(i18n("Use for front side"), 1);
   mnu.insertItem(i18n("Use for back side"), 2);
   mnu.insertSeparator();
   mnu.insertItem(BarIconSet("cancel"), i18n("Cancel"), 0, 0,
                  QKeySequence(Qt::Key_Escape), 0);
   int ret = mnu.exec(aPos);
   if (ret<=0) return false;

   if (ret==1) mProject.imgFront().load(aUrl);
   else mProject.imgBack().load(aUrl);

   updateFromProject();
   modified();
#endif

   return true;
}


void MainWindow::dropEvent(QDropEvent *e)
{
#if 0
   KURL::List urls;
   bool ok = false;

   e->accept(false);

   if (!KURLDrag::decode(e, urls) || urls.isEmpty())
      return;

   const KURL &url = urls.first();
   KMimeType::Ptr mt = KMimeType::findByURL(url);

//    std::cout<<"Dropped "<<url.prettyURL()<<" mime type "<<mt->name()<<std::endl;

#ifdef ENABLE_AUDIOCD
   if (CddbQuery::canDecode(url.url()))
   {
      mCddbDiscId = mIdActiveDisc;
      if (mCddbDiscId<0) mCddbDiscId = 0;

      if (!mProject.disc(mCddbDiscId).isEmpty() && !confirmDiscNotEmpty(mCddbDiscId))
         return;

      e->accept(mCddbQuery->fromUrl(url));
      return;
   }
#endif /*ENABLE_AUDIOCD*/

   // Remote protocols do not get detected properly (at least with images in
   // Kde 3.5.1) so we need to download the file to determine the type in that
   // case
   if (!url.isLocalFile() && mt->name()=="application/octet-stream")
   {
      QString fname;
      if (!KIO::NetAccess::download(url, fname, this))
      {
         KMessageBox::error(this, KIO::NetAccess::lastErrorString());
         return;
      }
      mt = KMimeType::findByPath(fname);
      KIO::NetAccess::removeTempFile(fname);
   }

   QString mtName = mt->name();
   int idx = mtName.find('/');
   if (idx>0) mtName = mtName.left(idx);

   if (mtName=="image" || mt->is("image/png") || mt->is("image/jpeg"))
      ok = imageDropped(url, mapToGlobal(e->pos()));
   else if (KoverArtist::Project::canLoad(url.fileName()))
      ok = load(url);
   else if (KoverArtist::Disc::canLoad(url.fileName()))
   {
      mCddbDiscId = mIdActiveDisc;
      if (mCddbDiscId<0) mCddbDiscId = 0;
      KoverArtist::Disc& d = mProject.disc(mCddbDiscId);
      if (!d.isEmpty() && !confirmDiscNotEmpty(mCddbDiscId)) return;
      if (d.load(url, this))
      {
         modified();
         updatePreviewBack();
         updateDiscFromProject();
      }
   }

   e->accept(ok);
#endif
}


bool MainWindow::load(const KUrl& aUrl)
{
   if (!mProject.load(aUrl))
      return false;

   mTabDisc->setDisc(&mProject.disc(0));
   updateContents();

   return true;
}


void MainWindow::windowNew()
{
   MainWindow *mw = new MainWindow;
   mw->show();
}


void MainWindow::fileNew()
{
   if (!queryClose()) return;

   mProject.clear();

   KoverArtist::Case *c = KoverArtist::Case::findCase("cd-standard");
   if (c) mProject.setBox(*c);
   else mProject.setBox(KoverArtist::Case::standardCase());

   updateContents();
}


void MainWindow::fileRevert()
{
   if (mProject.url().isEmpty()) return;
   if (!queryClose()) return;
   load(mProject.url());
}


void MainWindow::fileOpen()
{
   if (!queryClose()) return;

   QString dirPath = mProject.dirPath();
   if (dirPath.isEmpty()) dirPath = QDir::currentPath();

   QString filter = QString("*.koap *.kiap *.kmf|%1\n*.koap|%2\n*.kiap|%3\n*.kmf|%4\n*|%5")
      .arg(i18n("All Supported Files"))
      .arg(i18n("KoverArtist Projects"))
      .arg(i18n("KinoArtist Projects"))
      .arg(i18n("KMediaFactory Projects"))
      .arg(i18n("All Files"));

   KUrl url = KFileDialog::getOpenUrl(KUrl(dirPath), filter, this,
                                      i18n("Open Project") + " - KoverArtist");
   if (!url.isEmpty()) load(url);
}


bool MainWindow::fileSave()
{
   if (mProject.url().isEmpty())
      return fileSaveAs();
   return save(mProject.url());
}


bool MainWindow::fileSaveAs()
{
   QString dirPath = mProject.url().path();
   if (dirPath.isEmpty()) dirPath = QDir::currentPath();

   QString filter = QString("*.koap|%1\n*|%2\n")
      .arg(i18n("KoverArtist Projects"))
      .arg(i18n("All Files"));
   
   KUrl url = KFileDialog::getSaveUrl(KUrl(dirPath), filter, this,
      i18n("Save Project") + " - KoverArtist");
      
   if (url.isEmpty()) return false;

   bool ok = mProject.save(url);
   updateContents();

   return ok;

#if 0
   QString filter = QString("*.koap|%1")
      .arg(i18n("KoverArtist Projects"));

   KFileDialog dlg(":KoverArtist::Project", filter, this, "dlg-save", true);
   dlg.setCaption(i18n("Save Project"));
   dlg.setOperationMode(KFileDialog::Saving);
   dlg.setMode(KFile::File);
   dlg.setURL(mProject.url());
   dlg.setSelection(mProject.url().fileName());

   int ret = dlg.exec();
   if (ret!=QDialog::Accepted) return false;

   KURL url = dlg.selectedURL();
   if (url.isEmpty() || !url.isValid()) return false;

   if (url.isLocalFile())
   {
      if (QFile::exists(url.path()))
      {
         int ret = KMessageBox::warningYesNo(this,
            i18n("A file '%1' already exists.\nDo you want to overwrite it?").arg(url.fileName()),
            i18n("Save file")+" - KoverArtist", KStdGuiItem::cont(),
            KStdGuiItem::cancel());
	 if (ret==KMessageBox::Cancel) return false;
      }
   }

   return save(url);
#endif
}


void MainWindow::filePrint()
{
   if (!mPrinter)
   {
      mPrinter = new QPrinter;
      mPrinter->setOrientation(QPrinter::Portrait);
//       mPrinter->setResolution(200);
   }

   QPrintDialog dlg(mPrinter, this);
   if (dlg.exec() == QDialog::Accepted)
   {
      logInfo("Printer resolution: %1 dpi").arg(mPrinter->resolution());
      KoverArtist::Renderer render(&mProject);
      render.render(mPrinter, false);
   }
}


void MainWindow::discClear()
{
   if (mIdActiveDisc<0) return;
   KoverArtist::DiscVector& discs = mProject.discs();
   KoverArtist::Disc& disc = discs[mIdActiveDisc];

   if (!disc.isEmpty())
   {
      int ret = QMessageBox::question(this, i18n("Clear disc") + " - KoverArtist",
                                      i18n("The current disc is not empty.\nContinue and clear it?"),
                                      QMessageBox::Ok|QMessageBox::Cancel, QMessageBox::Ok
                                     );
      if (ret != QMessageBox::Ok)
         return;
   }

   disc = KoverArtist::Disc();
   mTabDisc->updateContents();
   modified();
   updatePreviewBack();
}


void MainWindow::discNew()
{
   KoverArtist::DiscVector& discs = mProject.discs();
   if (mIdActiveDisc<0)
   {
      discs.append(KoverArtist::Disc());
   }
   else
   {
      KoverArtist::DiscVector::iterator it = mProject.discIt(mIdActiveDisc);
      discs.insert(it, KoverArtist::Disc());
   }

   updateDiscsTabBar();
   if (mIdActiveDisc>0) mTabDisc->updateContents();
   else mTabBar->setCurrentIndex(mTabIdDisc + discs.count() - 1);

   modified();
   updatePreviewBack();
}


void MainWindow::discDelete()
{
   if (mIdActiveDisc<0) return;
   KoverArtist::DiscVector& discs = mProject.discs();
   KoverArtist::DiscVector::iterator it = mProject.discIt(mIdActiveDisc);

   if (!it->isEmpty())
   {
      int ret = QMessageBox::question(this, i18n("Delete disc") + " - KoverArtist",
                                 i18n("The current disc is not empty.\nContinue and delete it?"),
                                 QMessageBox::Ok|QMessageBox::Cancel, QMessageBox::Ok
                                 );

      if (ret != QMessageBox::Ok) return;
   }

   discs.erase(it);

   if (mIdActiveDisc>=int(discs.count()))
      mIdActiveDisc = discs.count()-1;

   mTabBar->setCurrentIndex(mTabIdDisc + mIdActiveDisc);
   updateDiscsTabBar();
   mTabDisc->updateContents();
   modified();
   updatePreviewBack();
}


void MainWindow::discFront()
{
   if (mIdActiveDisc<1) return;
   KoverArtist::DiscVector& discs = mProject.discs();
   KoverArtist::Disc d = discs[mIdActiveDisc];
   discs[mIdActiveDisc] = discs[mIdActiveDisc-1];
   discs[mIdActiveDisc-1] = d;
   mTabBar->setCurrentIndex(mTabIdDisc + mIdActiveDisc - 1);
   modified();
   updatePreviewBack();
}


void MainWindow::discBack()
{
   if (mIdActiveDisc<0 || mIdActiveDisc>=mProject.count()-1) return;
   KoverArtist::DiscVector& discs = mProject.discs();
   KoverArtist::Disc d = discs[mIdActiveDisc];
   discs[mIdActiveDisc] = discs[mIdActiveDisc+1];
   discs[mIdActiveDisc+1] = d;
   mTabBar->setCurrentIndex(mTabIdDisc + mIdActiveDisc + 1);
   modified();
   updatePreviewBack();
}


bool MainWindow::confirmDiscNotEmpty(int aDiscIdx)
{
   int ret = QMessageBox::question(this, i18n("Import disc") + " - KoverArtist",
                           i18n("Disc #%1 is not empty.\nContinue and replace it?").arg(aDiscIdx+1),
                           QMessageBox::Ok|QMessageBox::Cancel, QMessageBox::Ok
                           );

   return ret==QMessageBox::Ok;
}


void MainWindow::cddbQueryAudioCdDone()
{
#ifdef ENABLE_AUDIOCD
   int num = mProject.count();
   if (mCddbDiscId<0)
   {
      mCddbDiscId = mIdActiveDisc;
      if (mCddbDiscId<0) mCddbDiscId = 0;
   }
   if (mCddbDiscId>=num)
   {
      mProject.resize(num+1);
      mCddbDiscId = num;
   }

   mProject.disc(mCddbDiscId) = mCddbQuery->disc();

   if (mProject.title().isEmpty() &&
       KoverArtist::Settings::instance()->cddbAutoSetTitle)
   {
      QString t = mCddbQuery->disc().title();
      int idx = t.indexOf('/');
      if (idx>0)
      {
         mProject.setTitle(t.mid(idx+1).trimmed());
         mProject.setSubTitle(t.left(idx).trimmed());
      }
      else mProject.setTitle(t);
   }

   updateContents();
   modified();

   mCddbDiscId = -1;
#endif /*ENABLE_AUDIOCD*/
}


void MainWindow::discImportCddb()
{
#ifdef ENABLE_AUDIOCD
   mCddbDiscId = mIdActiveDisc;
   if (mCddbDiscId<0) mCddbDiscId = 0;

   if (!mProject.disc(mCddbDiscId).isEmpty() && !confirmDiscNotEmpty(mCddbDiscId))
      return;

   mCddbQuery->cddbQueryAudioCd();
#else
   logWarning("Audio-cd support was disabled at compile time");
#endif /*ENABLE_AUDIOCD*/
}


void MainWindow::discImport()
{
   int id = mIdActiveDisc;
   if (id<0) id = 0;
   KoverArtist::Disc& disc = mProject.disc(id);

   if (!disc.isEmpty() && !confirmDiscNotEmpty(id))
      return;

   QString filter = QString("*.koad *.kiap *.kmf *.txt|%1\n*.koad|%2\n*.kiap|%3\n"
      "*.kmf|%4\n*.txt|%5\n*|%6")
      .arg(i18n("All Supported Files"))
      .arg(i18n("KoverArtist Discs"))
      .arg(i18n("KinoArtist Projects"))
      .arg(i18n("KMediaFactory Projects"))
      .arg(i18n("Text Files"))
      .arg(i18n("All Files"));

   KUrl url = KFileDialog::getOpenUrl(KUrl(), filter, this,
                                      i18n("Import Disc")+" - KoverArtist");
   if (url.isEmpty()) return;

   if (!discImport(disc, url)) return;

   if (mProject.title().isEmpty())
   {
      mProject.setTitle(disc.title());
      mTabProject->updateContents();
   }

   int num = disc.snapShots().count();
   if (mProject.imgFront().isNull() && num>0)
   {
      mProject.imgFront().load(disc.snapShots()[0]);
      updatePreviewFront();
   }
   if (mProject.imgBack().isNull() && num>1)
      mProject.imgBack().load(disc.snapShots()[1]);

   mTabDisc->updateContents();
   modified();
   updatePreviewBack();
}


bool MainWindow::discImport(KoverArtist::Disc& aDisc, const KUrl& aUrl)
{
   QString fileName;
   bool ok = false;

   if (aUrl.isLocalFile()) fileName = aUrl.path();
   else
   {
      if (!KIO::NetAccess::download(aUrl, fileName, this))
      {
         KMessageBox::error(this, KIO::NetAccess::lastErrorString());
         return false;
      }
   }

   QFileInfo fi(fileName);
   if (fi.exists() && fi.isReadable() && !fi.isDir())
   {
      ok = aDisc.loadFile(fileName);
   }
   else
   {
      KMessageBox::error(this, KIO::NetAccess::lastErrorString());
   }

   if (!aUrl.isLocalFile())
      KIO::NetAccess::removeTempFile(fileName);

   return ok;
}


void MainWindow::discExport()
{
   QString filter = QString("*.koad|%1\n*.txt|%2")
      .arg(i18n("KoverArtist Discs"))
      .arg(i18n("Text Files"));

   KUrl url = KFileDialog::getSaveUrl(KUrl(), filter, this,
                                      i18n("Export disc")+" - KoverArtist");
   if (url.isEmpty()) return;

   int id = mIdActiveDisc;
   if (id<0) id = 0;
   KoverArtist::Disc& disc = mProject.disc(id);

   discExport(disc, url);
}


bool MainWindow::discExport(const KoverArtist::Disc& aDisc, const KUrl& aUrl)
{
   QString filename;
   KTemporaryFile tempFile;
   tempFile.setAutoRemove(true);

   if (aUrl.isLocalFile()) filename = aUrl.path();
   else filename = tempFile.fileName();

   if (!aDisc.saveFile(filename)) return false;

   if (!aUrl.isLocalFile())
   {
      if (!KIO::NetAccess::upload(filename, aUrl, this))
         KMessageBox::error(this, KIO::NetAccess::lastErrorString());
   }

   return true;
}


void MainWindow::searchCoverImages()
{
   QString search = mProject.title() + ' ' + mProject.subTitle();
   search = search.trimmed();
   search.replace(' ', '+');

   if (search.isEmpty())
   {
      KMessageBox::error(this, i18n("Please set a title and try again."));
      mTabProject->setFocusTitle();
      mTabBar->setCurrentIndex(mTabIdProject);
      return;
   }

   KMessageBox::information(this,
         i18n("The cover search is performed by querying an "
         "internet search engine for images, using the "
         "title and subtitle as keywords. You can then "
         "drag&drop the selected image(s) from the browser "
         "into the KoverArtist window."),
         i18n("Search For Cover Images")+" - KoverArtist",
         "search_cover_images", 0);

   system("kde-open '" + KoverArtist::Settings::instance()->coverImageSearchUrl.arg(search).toLocal8Bit() + "'");

   changeStatusbar(i18n("Starting external browser..."));
}


#if 0
void MainWindow::numDiscsChanged()
{
   int i, idx, tabs = mTabBar->count();

   int numDiscs = mProject.count();
   mEdtNumDiscs->setText(QString("%1").arg(numDiscs));

   while (tabs-mNumTabsNoDisc > numDiscs)
      mTabBar->removeTab(--tabs);

   while (tabs-mNumTabsNoDisc < numDiscs)
   {
      idx = ++tabs - mNumTabsNoDisc;
//       mTabBar->addTab(new QTab(BarIconSet("cd"), QString("%1").arg(idx)));
      mTabBar->addTab(QString("%1").arg(idx));
   }

   // Set identifers of the other tabs
   mTabBar->tabAt(0)->setIdentifier(mTabStack->id(mBaseProject));
   mTabBar->tabAt(1)->setIdentifier(mTabStack->id(mBaseOptions));

   // Set identifers of the disc tabs
   tabs = mTabBar->count();
   for (i=mNumTabsNoDisc,idx=KOA_TABID_DISC; i<tabs; ++i,++idx)
      mTabBar->tabAt(i)->setIdentifier(idx);

   modified();
   updateDiscFromProject();
}


void MainWindow::slotDiscsInc()
{
   int num = mProject.count()+1;
   if (num<=999)
   {
      mProject.resize(num);
      updateDiscsTabBar();
      modified();
   }
}


void MainWindow::slotDiscsDec()
{
   int num = mProject.count()-1;
   if (num>=1)
   {
      mProject.resize(num);
      updateDiscsTabBar();
      modified();
   }
}


void MainWindow::slotDiscsEdited()
{
   int num = mEdtNumDiscs->text().toInt();
   if (num<1) num = 1;
   else if (num>999) num = 999;
   if (num!=mProject.count())
   {
      mProject.resize(num);
      updateDiscsTabBar();
      modified();
   }
}
#endif


void MainWindow::switchToTab(int aId)
{
   bool discActive = false;

   if (aId >= mTabIdDisc)
   {
      discActive = true;
      mIdActiveDisc = aId - mTabIdDisc;

      if (mIdActiveDisc >= mProject.count()) mTabDisc->setDisc(NULL);
      else mTabDisc->setDisc(&mProject.disc(mIdActiveDisc));

      mStkSide->setCurrentIndex(mTabIdDisc);
      changeCaption();
   }
   else
   {
      mIdActiveDisc = -1;
      mStkSide->setCurrentIndex(aId);
   }

   mActions["discClear"]->setEnabled(discActive);
   mActions["discDelete"]->setEnabled(discActive && mProject.count()>1);
   mActions["discFront"]->setEnabled(discActive && mIdActiveDisc>0);
   mActions["discBack"]->setEnabled(discActive && mIdActiveDisc<mProject.count()-1);
   mActions["discImport"]->setEnabled(discActive);
#ifdef ENABLE_AUDIOCD
   mActions["discImportCddb"]->setEnabled(discActive);
#endif /*ENABLE_AUDIOCD*/
   mActions["discExport"]->setEnabled(discActive);
}


void MainWindow::settings()
{
   KoverArtistPreferences dlg;
   if (dlg.exec())
   {
      // redo your settings
   }
}


void MainWindow::changeStatusbar(const QString& text)
{
   // display the text on the statusbar
   logInfo(text);
   statusBar()->showMessage(text, 5000);
}


void MainWindow::changeCaption(const QString& text)
{
   bool mod = mProject.isModified();
   QString str = text.isNull() ? mProject.url().fileName() : text;
   if (str.isEmpty()) str = i18n("unnamed.koap");

   // TODO set caption
//    setCaption(kapp->makeStdCaption(str, true, mod));

   mActions["fileSave"]->setEnabled(mod);
   mActions["fileRevert"]->setEnabled(mProject.url().isValid());
}


void MainWindow::scheduleUpdatePreview(int aTmout)
{
   if (aTmout<=0) aTmout = 250;
//    logDebug(QString("scheduleUpdatePreview %1 msec").arg(aTmout));

   if (mRendering) mRender->abortRendering();
   mTmrUpdate->stop();
   mTmrUpdate->start(aTmout);
}


void MainWindow::updatePreview(int aTmout)
{
   mUpdPreviewFront = true;
   mUpdPreviewBack = true;
   scheduleUpdatePreview(aTmout);
}


void MainWindow::updatePreviewFront(int aTmout)
{
   mUpdPreviewFront = true;
   scheduleUpdatePreview(aTmout);
}


void MainWindow::updatePreviewBack(int aTmout)
{
   mUpdPreviewBack = true;
   scheduleUpdatePreview(aTmout);
}


void MainWindow::updatePreviewNow()
{
   mUpdPreviewFront = true;
   mUpdPreviewBack = true;
   logDebug("updatePreviewNow");
   updatePreviewTimeout();
}


void MainWindow::updatePreviewTimeout()
{
   if (!updatesEnabled() || (!mUpdPreviewFront && !mUpdPreviewBack))
      return;

   if (mRendering)
   {
      scheduleUpdatePreview();
      return;
   }

   mRendering = true;
   qApp->setOverrideCursor(Qt::BusyCursor);

   logDebug("Updating preview");

   int wp = mPreview->contentsRect().width()-4;
   int hp = mPreview->contentsRect().height()-4;

   if (mPixPreview.width()!=wp || mPixPreview.height()!=hp)
   {
      mPixPreview = QPixmap(wp, hp);
      mUpdPreviewFront = true;
      mUpdPreviewBack = true;
   }

   if (mUpdPreviewFront && mUpdPreviewBack)
      mPixPreview.fill(mPreview->palette().color(QPalette::Background));

   mRender->setEnabledFront(mUpdPreviewFront);
   mRender->setEnabledBack(mUpdPreviewBack);

   mRender->render(&mPixPreview, true);
   mPreview->setPixmap(mPixPreview);

   qApp->restoreOverrideCursor();
   mUpdPreviewFront = false;
   mUpdPreviewBack = false;
   mRendering = false;

   if (mTmrUpdate->isActive())
   {
      mTmrUpdate->stop();
      mTmrUpdate->start(500);
   }
}


void MainWindow::redisplay()
{
   updateContents();
   updatePreviewNow();
}
