/*
   Copyright (C) 2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>.
*/
#ifndef log_h
#define log_h

#include <QString>

enum LogLevel
{
   LOG_TRACE, LOG_DEBUG, LOG_INFO, LOG_WARNING, LOG_ERROR
};


/**
 * Output log messages.
 *
 * To write a log message, you most probably want to use the simplified interface:
 * <pre>
 * logDebug("the log #%1 message").arg(num);
 * </pre>
 *
 * The available log functions are: logTrace, logDebug, logInfo, logWarning, logError.
 */
class Logger
{
public:
   Logger(const char* file, const int line, LogLevel level = (LogLevel)-1);
   ~Logger();
   QString& operator()(LogLevel level, QString message);
   void operator<<(const QString& message);

   /**
    * Output a log line
    *
    * @param level - the log level
    * @param message - the log message
    * @param file - the file of the occurence
    * @param line - the line of the occurence
    */
   void log(LogLevel level, const QString& message, const char* file = 0, int line = -1);

   /**
    * Returns the minimum log level.
    */
   static LogLevel minLogLevel() { return sMinLogLevel; }
   
   /**
    * Set the minimum log level.
    */
   static void setMinLogLevel(LogLevel minLevel) { sMinLogLevel = minLevel; }
   
private:
   const char* mFile;
   const int mLine;
   LogLevel mLevel;
   QString mMessage;
   static LogLevel sMinLogLevel;
};


#define logTrace(message)   Logger(__FILE__,__LINE__,LOG_TRACE)<<QString(message)
#define logDebug(message)   Logger(__FILE__,__LINE__,LOG_DEBUG)<<QString(message)
#define logInfo(message)    Logger(__FILE__,__LINE__,LOG_INFO)<<QString(message)
#define logWarning(message) Logger(__FILE__,__LINE__,LOG_WARNING)<<QString(message)
#define logError(message)   Logger(__FILE__,__LINE__,LOG_ERROR)<<QString(message)

#endif /*log_h*/
