/*
   Copyright (C) 2008-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>.
*/
#ifndef koverartistimage_h
#define koverartistimage_h

#include <QtCore/QString>
#include <QtGui/QPixmap>
#include <QtGui/QColor>

class QDomElement;
class QDomDocument;


namespace KoverArtist
{

/** The position of an image.
 */
enum ImagePosition
{
   Centered=0, Tiled, CenteredTiled, CenteredMaxpect, Scaled, ScaleCrop
};



/** An image with layout information.
 */
class Image
{
public:
   Image();
   Image(const Image& o);
   Image& operator=(const Image& o);

   /** Returns the image url. */
   const QString& url() const { return mUrl; }

   /** Load the image from a file. Returns true on success. */
   bool load(QString fileName);

   /** Set the image url without loading the image. */
   void setUrl(const QString& fileName) { mUrl = fileName; }

   /** Returns the image. */
   const QPixmap& pixmap() const { return mPixmap; }
   QPixmap& pixmap() { return mPixmap; }

   /** Test if the image is valid. */
   bool isNull() const { return mPixmap.isNull(); }

   /** Get/set the image position. */
   ImagePosition position() const { return mPosition; }
   void setPosition(ImagePosition ip) { mPosition = ip; }

   /** Get/set indicator if front image is drawn on sides too. */
   bool onSides() const { return mOnSides; }
   void setOnSides(bool f) { mOnSides = f; }

   /** Get/set image transparency: 0=fully transparent, 255=opaque. */
   int transparency() const { return mTransparency; }
   void setTransparency(int t) { mTransparency = t; }

   /** Fill the DOM element de with the object's contents. */
   void toDom(QDomDocument& doc, QDomElement& de, const QString& projectDir) const;

   /** Set the object from the DOM element de. */
   bool fromDom(const QDomElement& de, const QString& projectDir);

   /** Clear the object. */
   void clear();

   /** Returns the image-position ip as (untranslated) string. */
   static const char* imagePositionAsString(ImagePosition ip);

   /** Translates the (untranslated) string str to a image-position. */
   static ImagePosition imagePositionFromString(const QString& str);

protected:
   /** Remove black lines from left/right side of the image by cutting the parts off. */
   QPixmap imageToPixmapBordersRemoved(QImage& img);

protected:
   QString mUrl;
   ImagePosition mPosition;
   QPixmap mPixmap;
   bool mOnSides;
   int mTransparency;
};


} //namespace

#endif /*koverartistimage_h*/
