/* Copyright (C) 1998 Ulrich Drepper, <drepper@cygnus.com>.
   The GPL applies to this file.
   As a special restriction the file must not be used in this or a modified
   form on Microsoft and Be systems.  */

#include <errno.h>
#include <error.h>
#include <float.h>
#include <gmp.h>
#include <math.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/param.h>

#include "zelibm.h"

#ifndef MAX
# define MAX(a,b) ((a) > (b) ? (a) : (b))
#endif
#ifndef MIN
# define MIN(a,b) ((a) < (b) ? (a) : (b))
#endif


/* Number of bits in the mantissa.  */
#define MANT_DIG	DBL_MANT_DIG

/* Limit where implementations of monotic series evaluations can stop.  */
mpq_t limit;

/* Number of errors > 0.5ulp. */
int bigerror;


/* Nonzero if X should be used.  */
int with_x;

/* Window size the user asked for.  */
unsigned int user_width = 640;
unsigned int user_height = 480;

/* How many samples.  */
unsigned int N = 30000;

/* List of all supported tests.  */
struct
{
  const char *name;
  double from;
  double to;
  double (*lmf) (double);
  double (*zef) (double);
  double (*diff) (double, double);
} fcts[] =
{
  { "exp", 0.0, 0.5, exp, zeexp, zeexp_diff },
  { "sin", -2.0, 2.0, sin, zesin, zesin_diff },
  { "cos", -2.0, 2.0, cos, zecos, zecos_diff },
  { "sinh", -2.0, 2.0, sinh, zesinh, zesinh_diff },
  { "cosh", -2.0, 2.0, cosh, zecosh, zecosh_diff },
  { "sqrt", 0.0, 2.0, sqrt, zesqrt, zesqrt_diff },
  { "tan", -2.0, 2.0, tan, zetan, zetan_diff },
  { "tanh", -2.0, 2.0, tanh, zetanh, zetanh_diff },
};


static double *runtest_uniform (int *pargc, char *argc[], const char *name,
				double low, double high,
				double (*lm) (double), double (*ze) (double),
				double (*diff) (double, double),
				unsigned long int n);


int
main (int argc, char *argv[])
{
  int cnt;

  /* Check whether X output is wanted.  */
  while (argc > 1)
    {
      if (strcmp (argv[1], "-x") == 0)
	{
	  with_x = 1;
	  ++argv;
	  --argc;
	  continue;
	}
      if (strcmp (argv[1], "-N") == 0 && argc > 2)
	{
	  N = atol (argv[2]);
	  argv += 2;
	  argc -= 2;
	  continue;
	}
      break;
    }

  if (argc < 2)
    {
      printf ("missing function name\n");
      exit (1);
    }

  /* Set the limit to 2^-L where L is 100% larger than the number of bits
     in the mantissa.  This should guarantee that no term which has any
     effect is left out.  */
  mpq_init (limit);
  mpq_set_ui (limit, 1, 1);
  mpz_mul_2exp (mpq_denref (limit), mpq_denref (limit), 2 * MANT_DIG);

  /* We occasionally need the floating-point representation.  */
  mpf_set_default_prec (4 * MANT_DIG);

  /* Run test desired test.  */
  for (cnt = 0; cnt < sizeof (fcts) /  sizeof (fcts[0]); ++cnt)
    if (strcmp (fcts[cnt].name, argv[1]) == 0)
      {
	runtest_uniform (&argc, argv, argv[1], fcts[cnt].from, fcts[cnt].to,
			 fcts[cnt].lmf, fcts[cnt].zef, fcts[cnt].diff,
			 N);
	exit (0);
      }

  printf ("function `%s' not known\n", argv[1]);
  return 1;
}


struct thread_params
{
  int nprocs;
  int thd_no;
  double low;
  double high;
  double (*lm) (double);
  double (*ze) (double);
  double (*diff) (double, double);
  volatile unsigned long int *counter;
  unsigned long int max;
  double *result;
  pthread_mutex_t *result_lock;
};


void *
compute (void *parm)
{
  struct thread_params *thd_parm = (struct thread_params *) parm;
  unsigned long int max = thd_parm->max;
  double low = thd_parm->low;
  double high = thd_parm->high;
  int nprocs = thd_parm->nprocs;

  while (*thd_parm->counter < max)
    {
      double x = zerandom (low, high);
      double lm_res = thd_parm->lm (x);
      double ze_res = thd_parm->ze (x);
      double diff_res = thd_parm->diff (x, lm_res);
      double e = diff_res / ldexp (1.0, ilogb (ze_res) - 52);
      unsigned long int c;

      /* XXX Improve this.  If possible we should use atomic operations.  */
      if (nprocs > 1)
	pthread_mutex_lock (thd_parm->result_lock);

      c = *thd_parm->counter;
      if (c < max)
	{
	  thd_parm->result[2 * c] = x;
	  thd_parm->result[2 * c + 1] = e;
	  *thd_parm->counter = c + 1;
	}

      if (nprocs > 1)
	pthread_mutex_unlock (thd_parm->result_lock);

      if (e > 0.5)
	{
	  printf ("x = %.20g, ln = %.20g, ze = %.20g, e = %.20e\n",
		  x, lm_res, ze_res, e);
	  ++bigerror;
	}
    }

  return NULL;
}

static double *
runtest_uniform (int *pargc, char *argv[], const char *name,
		 double low, double high,
		 double (*lm) (double), double (*ze) (double),
		 double (*diff) (double, double), unsigned long int n)
{
#ifdef _SC_NPROCESSORS_ONLN
  int nprocs = MAX (MIN (sysconf (_SC_NPROCESSORS_ONLN), 1), 1);
#else
  int nprocs = 1;
#endif
  pthread_t thds[nprocs];
  struct thread_params thd_parm[nprocs];
  int cnt;
  void *handle = NULL;
  unsigned long int counter = 0;
  double *result;
  pthread_mutex_t result_lock;
  void *ignore;
  double maxerror = 1.0;

  /* Allocate the memory for the result.  */
  result = (double *) calloc (n,  2 * sizeof (double));
  if (result == NULL)
    error (EXIT_FAILURE, errno, "cannot create array for result");

  for (cnt = 0; cnt < nprocs; ++cnt)
    {
      thd_parm[cnt].nprocs = nprocs + with_x;
      thd_parm[cnt].thd_no = cnt;
      thd_parm[cnt].low = low;
      thd_parm[cnt].high = high;
      thd_parm[cnt].lm = lm;
      thd_parm[cnt].ze = ze;
      thd_parm[cnt].diff = diff;
      thd_parm[cnt].counter = &counter;
      thd_parm[cnt].max = n;
      thd_parm[cnt].result = result;
      thd_parm[cnt].result_lock = &result_lock;
    }

  if (with_x)
    handle = create_window (pargc, argv, name, low, high, & maxerror,
			    &counter, n);

  if (with_x || nprocs > 1)
    pthread_mutex_init (&result_lock, NULL);

  for (cnt = with_x ? 0 : 1; cnt < nprocs; ++cnt)
    pthread_create (&thds[cnt], NULL, compute, &thd_parm[cnt]);

  if (with_x)
    display_loop (handle, &counter, result);
  else
    compute (&thd_parm[0]);

  /* Join all the other threads.  */
  for (cnt = with_x ? 0 : 1; cnt < nprocs; ++cnt)
    pthread_join (thds[cnt], &ignore);

  return result;
}
