/* MAIN ASSEMBLING FUNCTIONS */
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>

#include "regs.h"
#include "compile.h"
#include "hash.h"
#include "z80-asm.h"
#include "instr.h"
#include "asm.h"



int WARNINGS=0;

unsigned char *memory_ptr;
int disable_defx;
unsigned short end;
int line;

struct argument_type
{
 unsigned char type;
 int value;
 char *text;            /* this item contains defs string */
 unsigned char label;   /* if type==A_NUM||type==A_PODLE_NUM, indicates if number was from label */
 struct argument_type *next,*previous;
};

struct lex_type
{
 int instruction;
 struct argument_type *arg;
}lex;

int pruchod;
unsigned short address;


#ifdef UNIX

char * 
strupr (char *txt)
{
 char *p;
 for (p=txt;*p;p++)
  *p=toupper(*p);

 return txt;
}

#endif /* UNIX */


void
asm_close(void)
{
 free_hash_table();
}


void
asm_init(void)
{
 hash_table_init();
 address=0;
 end=0;   /* funckin' DJGPP doesn't initialize global variables in modules */
          /* with zero so variable end contains garbage in programs using */
	  /* asm.a library */
}


/* binary finds txt in seznam */

int
convert(const struct seznam_type *seznam,int seznam_size,char *txt)
{
int a,b,begin,end;
begin=0;end=seznam_size-1;

if (!(*txt))return 0;
while (1)
 {
 a=((end-begin)>>1)+begin;
 if (!(b=strcmp(seznam[a].name,txt)))return seznam[a].code;
 if (b>0)end=a-1;
 if (b<0)begin=a+1;
 if (end<begin)return -1;
 }
}


/* return value: 1-if txt is a number (value i set)    */
/*               0-NAN                                 */

int
test_number(char *txt,int *value)
{
int base=10;
char *p,*q;
int sign=1;
p=txt;

/* character constant */
if (txt[0]==39&&txt[2]==39&&strlen(txt)==3)  /* 39=' */
{
 *value=(unsigned char)(txt[1]);
 return 1;
}
if ((*p)=='-'){sign=-1;p++;}
if ((*p)=='%'){base=2;p++;}
else if ((*p)=='0'&&tolower(p[1])=='x'){base=16;p+=2;}
q=p;

while (*p)
{
 if ((tolower(*p)>('0'+((base-1)>10?9:base))&&tolower(*p)<'a')||tolower(*p)>('a'+base-11)||(*p)<'0')return 0;
 p++;
}
*value=sign*strtol(q,NULL,base);
return 1;
}


int
convert_arg(struct argument_type *arg,char *txt)
{
int a,mode=0,l=strlen(txt);

/* nothing */
if (!*txt){arg->type=A_EMPTY;arg->value=0;return 0;}

/* number */
if (test_number(txt,&a))
 {
 arg->type=A_NUM;
 arg->value=a;
 return 0;
 }

/* string */
if ((*txt)=='"'&&txt[l-1]=='"')
{
 arg->type=A_STRING;
 arg->text=malloc(l-1);
 if(!arg->text){fprintf(stderr,"Memory allocation error.\n");exit(1);}
 memcpy(arg->text,txt+1,l-2);
 arg->text[l-2]=0;
 return 0;
}

strupr(txt);

/* register */
a=convert(reg,N_REGISTERS,txt);
if (a!=-1){arg->type=A_REG;arg->value=a;return 0;}

/* flag */
a=convert(flag,N_FLAGS,txt);
if (a!=-1){arg->type=A_FLAG;arg->value=a;return 0;}

if (txt[0]!='('||txt[strlen(txt)-1]!=')')
{
 /* label */
 arg->value=is_in_table(txt);
 if (arg->value==-1)return 1;
 arg->type=A_NUM;
 arg->label=1;
 return 0;
}

txt=memmove(txt,txt+1,strlen(txt));
txt[strlen(txt)-1]=0;

/* (register) */
arg->value=convert(reg,N_REGISTERS,txt);
if (arg->value!=-1){arg->type=A_PODLE_REG;return 0;}

/* (number) */
if (test_number(txt,&a))
 {
 arg->type=A_PODLE_NUM;
 arg->value=a;
 return 0;
 }
if (txt[0]=='I'&&txt[2]=='+')
 {
 if (txt[1]=='Y')mode=1;
 if (txt[1]=='X')mode=0;
 }
else 
 {
 arg->value=is_in_table(txt);
 if (arg->value==-1)return 1;
 arg->type=A_PODLE_NUM;
 arg->label=1;
 return 0;
 }

/* (IX+num) or (IY+num) */
txt=memmove(txt,txt+3,strlen(txt)-2);
if (test_number(txt,&a))
 {
 if (mode)arg->type=A_PODLE_IY_PLUS;
 else arg->type=A_PODLE_IX_PLUS;
 arg->value=a;
 return 0;
 }
 
return 1;
}


/* converts line into structure lex */
/* return value:
   0=O.K.
   1=memory allocation error
   2=unknown label
   3=label previously defined
*/

int
lexical_analysis(char *line)
{
char txt1[MAX_TEXT];
char *p,*b=NULL;
struct argument_type *a,*c;

take_line(line);
lex.instruction=I_EMPTY;
lex.arg=NULL;
a=NULL;c=NULL;

if ((*line)==';')return 0;  /* ignore comment */

/* LABEL */

if (pruchod==2)
 {
 for (p=line;(*p)!=32&&(*p)!=9&&(*p);p++);  /* skip label */
 if (!(*p))return 0;
 }
else
 {
 for (p=line,b=txt1;(*p)!=32&&(*p)!=9&&(*p);b++,p++)
  *b=toupper(*p);
 *b=0;
 if (p!=line)
  {
  if (is_in_table(txt1)==-1)return 3;
  b=malloc(MAX_LABEL);
  if (!b){fprintf(stderr,"Error: Not enough memory.\n");return 1;}
  b[MAX_LABEL-1]=0;
  memcpy(b,txt1,MAX_LABEL-1);
  if (add_to_table(b,address))return 1;
  }
 }

/* SPACE */

while((*p)==9||(*p)==32)  /* skip spaces */
 p++;
if ((*p)==';')return 0;
if (!(*p))return 0;

/* INSTRUCTION */

for (b=txt1;((*p)!=32&&(*p)!=9&&(*p));p++,b++)
 *b=toupper(*p);
*b=0;
lex.instruction=convert(instruction,N_INSTRUCTIONS,txt1);

/* SPACE */

while((*p)==9||(*p)==32)
 p++;
if (!(*p))return 0;
if ((*p)==';')return 0;

a=(struct argument_type *)malloc(sizeof(struct argument_type));
if (!a) {fprintf(stderr,"Error: Not enough memory.\n");return 1;}
a->next=NULL;
c=a;

/* ARGUMENTS */

while(1)
 {
 int quotes=0;
 /* arguments are separated by ' ' or ','; separators in quotes are ignored */
 /* we stop reading on terminating null character */
 for (b=txt1;(((*p)!=32&&(*p)!=9&&(*p!=','))||(quotes&1))&&(*p);p++,b++)
 {
  if (*p=='"')quotes++;
  *b=*p;
 }
 *b=0;
 
 a->next=(struct argument_type *)malloc(sizeof(struct argument_type));
 if (!a->next){fprintf(stderr,"Error: Not enough memory.\n");free(a);return 1;}
 a->next->label=0;
 a->next->value=0;
 a->next->type=A_EMPTY;
 a->next->next=NULL;
 a->next->previous=(a==c)?NULL:a;
 a=a->next;
 
 if (convert_arg(a,txt1)){free (c);return 2;}

 while((*p)==9||(*p)==32||(*p)==',')
  p++;
 if (!(*p)||(*p)==';'){lex.arg=c->next;free(c);return 0;}
 }
}


void
out(char a)
{
if (pruchod==1){address++;return;}

 if (WARNINGS)
 {
  if (memory_ptr[address]) 
   fprintf(stderr,"Warning: overwriting code at address 0x%04x (%05d)\n",address,address);
 }

memory_ptr[address]=a;
if (address>end)end=address;
address++;
}


int
compile(void)
/* return value: 0=OK; 1=error */
{
int a;
char txt[MAX_TEXT];
struct argument_type *t;
int ret=0;

a=lexical_analysis(txt);
switch (a)
 {
 case 1:
 return 1;

 case 2:
 error(line,txt,"Label not defined.");
 ret=1;
 break;

 case 3:
 error(line,txt,"Previous declaration of label.");
 ret=1;
 break;
 }

if (!ret)
switch (lex.instruction)
 {
 case I_ORG:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->type!=A_NUM){error(line,txt,"Invalid argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 address=lex.arg->value;
 break;

 case I_ILLEGAL:
 error(line,txt,"Illegal instruction.");
 ret=1;
 break;

 case I_EMPTY:
 break;

 case I_HALT:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else out(0x76);
 break;

 case I_CCF:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else out(0x3f);
 break;
 
 case I_CPD:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xa9);}
 break;
 
 case I_CPDR:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xb9);}
 break;
 
 case I_CPI:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xa1);}
 break;
 
 case I_CPIR:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xb1);}
 break;
 
 case I_CPL:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else out(0x2f);
 break;
 
 case I_DAA:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else out(0x27);
 break;
 
 case I_DI:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else out(0xf3);
 break;
 
 case I_EI:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else out(0xfb);
 break;
 
 case I_EXX:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else out(0xd9);
 break;
 
 case I_IND:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xaa);}
 break;
 
 case I_INI:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xa2);}
 break;
 
 case I_INDR:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xba);}
 break;
 
 case I_INIR:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xb2);}
 break;
 
 case I_LDD:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xa8);}
 break;
 
 case I_LDI:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xa0);}
 break;
 
 case I_LDDR:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xb8);}
 break;
 
 case I_LDIR:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xb0);}
 break;
 
 case I_NEG:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0x44);}
 break;
 
 case I_NOP:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else out(0x00);
 break;
 
 case I_OTDR:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xbb);}
 break;
 
 case I_OTIR:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xb3);}
 break;
 
 case I_OUTD:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xab);}
 break;
 
 case I_OUTI:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0xa3);}
 break;
 
 case I_RETI:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0x4d);}
 break;
 
 case I_RETN:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0x45);}
 break;
 
 case I_RLA:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else out(0x17);
 break;
 
 case I_RLCA:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else out(0x07);
 break;
 
 case I_RLD:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0x6f);}
 break;
 
 case I_RRA:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else out(0x1f);
 break;
 
 case I_RRCA:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else out(0x0f);
 break;
 
 case I_RRD:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else {out(0xed);out(0x67);}
 break;
 
 case I_SCF:
 if (lex.arg){error(line,txt,"Extra argument.");ret=1;}
 else out(0x37);
 break;

 case I_ADC:
 if (!lex.arg){error(line,txt,"Missing arguments.");ret=1;break;}
 if ((lex.arg->type!=A_REG)||(lex.arg->value!=R_A&&lex.arg->value!=R_HL))
  {error(line,txt,"Invalid first argument.");ret=1;break;}
 if (!lex.arg->next)
  {error(line,txt,"Missing second argument.");ret=1;break;}
 if (lex.arg->next->next)
  {error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_adc_sbc(lex.arg->value,lex.arg->next->type,lex.arg->next->value,0x88,0x0a);
 if (a==1) {error(line,txt,"Invalid second argument.");ret=1;break;}
 if (a==2) {error(line,txt,"Value out of range.");ret=1;break;}
 if (a==3) {error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_ADD:
 if (!lex.arg){error(line,txt,"Missing arguments.");ret=1;break;}
 if ((lex.arg->type!=A_REG)||(lex.arg->value!=R_A&&lex.arg->value!=R_HL&&lex.arg->value!=R_IX&&lex.arg->value!=R_IY))
  {error(line,txt,"Invalid first argument.");ret=1;break;}
 if (!lex.arg->next)
  {error(line,txt,"Missing second argument.");ret=1;break;}
 if (lex.arg->next->next)
  {error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_add(lex.arg->value,lex.arg->next->type,lex.arg->next->value);
 if (a==1) {error(line,txt,"Invalid second argument.");ret=1;break;}
 if (a==2) {error(line,txt,"Value out of range.");ret=1;break;}
 if (a==3) {error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_AND:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_logical(lex.arg->type,lex.arg->value,0xa0);
 if (a==1) {error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2) {error(line,txt,"Value out of range.");ret=1;break;}
 if (a==3) {error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_BIT:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->type!=A_NUM||(lex.arg->type==A_NUM&&lex.arg->value>7))
  {error(line,txt,"Invalid first argument.");ret=1;break;}
 if (!lex.arg->next){error(line,txt,"Missing second argument.");ret=1;break;}
 if (lex.arg->next->next)
  {error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_bit(lex.arg->value,lex.arg->next->type,lex.arg->next->value,0x40);
 if (a==1){error(line,txt,"Invalid second argument.");ret=1;break;}
 if (a==2){error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_CP:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_logical(lex.arg->type,lex.arg->value,0xb8);
 if (a==1) {error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2) {error(line,txt,"Value out of range.");ret=1;break;}
 if (a==3) {error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_DEC:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_inc_dec(lex.arg->type,lex.arg->value,0x05,0x0b);
 if (a==1){error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2){error(line,txt,"Offset out of range.");ret=1;break;}
 break;
 
 case I_EX:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (!lex.arg->next){error(line,txt,"Missing second argument.");ret=1;break;}
 if (lex.arg->next->next){error(line,txt,"Too many arguments.");ret=1;break;}
 if (lex.arg->next->type!=A_REG){error(line,txt,"Invalid second argument.");ret=1;break;}
 if (c_ex(lex.arg->type,lex.arg->value,lex.arg->next->value))
  {error(line,txt,"Invalid argument.");ret=1;}
 break;

 case I_IM:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 if (lex.arg->type!=A_NUM||(lex.arg->type==A_NUM&&lex.arg->value>2))
  {error(line,txt,"Invalid argument.");ret=1;break;}
 c_im(lex.arg->value);
 break;
 
 case I_IN:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (!lex.arg->next){error(line,txt,"Missing second argument.");ret=1;break;}
 if (lex.arg->next->next){error(line,txt,"Too many arguments.");ret=1;break;}
 if (lex.arg->type!=A_REG){error(line,txt,"Invalid first argument.");ret=1;break;}
 a=c_in(lex.arg->value,lex.arg->next->type,lex.arg->next->value);
 if (a==1) {error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2) {error(line,txt,"Value out of range.");ret=1;break;}
 break;

 case I_INC:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_inc_dec(lex.arg->type,lex.arg->value,0x04,0x03);
 if (a==1){error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2){error(line,txt,"Offset out of range.");ret=1;break;}
 break;
 
 case I_OR:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_logical(lex.arg->type,lex.arg->value,0xb0);
 if (a==1) {error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2) {error(line,txt,"Value out of range.");ret=1;break;}
 if (a==3) {error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_OUT:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (!lex.arg->next){error(line,txt,"Missing second argument.");ret=1;break;}
 if (lex.arg->next->next){error(line,txt,"Too many arguments.");ret=1;break;}
 if (lex.arg->next->type!=A_REG){error(line,txt,"Invalid second argument.");ret=1;break;}
 a=c_out(lex.arg->type,lex.arg->value,lex.arg->next->value);
 if (a==1) {error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2) {error(line,txt,"Value out of range.");ret=1;break;}
 break;

 case I_POP:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 if (lex.arg->type!=A_REG){error(line,txt,"Invalid argument.");ret=1;break;}
 if (c_push_pop(lex.arg->value,0x01))
  {error(line,txt,"Invalid argument.");ret=1;}
 break;
 
 case I_PUSH:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 if (lex.arg->type!=A_REG){error(line,txt,"Invalid argument.");ret=1;break;}
 if (c_push_pop(lex.arg->value,0x05))
  {error(line,txt,"Invalid argument.");ret=1;}
 break;
 
 case I_RES:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->type!=A_NUM||(lex.arg->type==A_NUM&&lex.arg->value>7))
  {error(line,txt,"Invalid first argument.");ret=1;break;}
 if (!lex.arg->next){error(line,txt,"Missing second argument.");ret=1;break;}
 if (lex.arg->next->next)
  {error(line,txt,"Too many arguments.");ret=1;break;}
 if (c_bit(lex.arg->value,lex.arg->next->type,lex.arg->next->value,0x80))
  {error(line,txt,"Invalid second argument.");ret=1;}
 break;

 case I_RET:
 if (!lex.arg){out(0xc9);break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 if (lex.arg->type==A_REG)
 {
  if (lex.arg->value!=R_C){error(line,txt,"Invalid argument.");ret=1;break;}
 }
 else {if (lex.arg->type!=A_FLAG){error(line,txt,"Invalid argument.");ret=1;break;}}
 if (c_ret(lex.arg->value))
  {error(line,txt,"Invalid argument.");ret=1;}
 break;
 
 case I_RL:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_shift_rot(lex.arg->type,lex.arg->value,0x10);
 if (a==1){error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2){error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_RLC:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_shift_rot(lex.arg->type,lex.arg->value,0x00);
 if (a==1){error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2){error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_RR:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_shift_rot(lex.arg->type,lex.arg->value,0x18);
 if (a==1){error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2){error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_RRC:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_shift_rot(lex.arg->type,lex.arg->value,0x08);
 if (a==1){error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2){error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_RST:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 if (lex.arg->type!=A_NUM){error(line,txt,"Invalid argument type.");ret=1;break;}
 if (c_rst(lex.arg->value))
  {error(line,txt,"Invalid argument value.");ret=1;}
 break;

 case I_SBC:
 if (!lex.arg){error(line,txt,"Missing arguments.");ret=1;break;}
 if ((lex.arg->type!=A_REG)||(lex.arg->value!=R_A&&lex.arg->value!=R_HL))
  {error(line,txt,"Invalid first argument.");ret=1;break;}
 if (!lex.arg->next)
  {error(line,txt,"Missing second argument.");ret=1;break;}
 if (lex.arg->next->next)
  {error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_adc_sbc(lex.arg->value,lex.arg->next->type,lex.arg->next->value,0x98,0x02);
 if (a==1) {error(line,txt,"Invalid second argument.");ret=1;break;}
 if (a==2) {error(line,txt,"Value out of range.");ret=1;break;}
 break;

 case I_SET:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->type!=A_NUM||(lex.arg->type==A_NUM&&lex.arg->value>7))
  {error(line,txt,"Invalid first argument.");ret=1;break;}
 if (!lex.arg->next){error(line,txt,"Missing second argument.");ret=1;break;}
 if (lex.arg->next->next)
  {error(line,txt,"Too many arguments.");ret=1;break;}
 if (c_bit(lex.arg->value,lex.arg->next->type,lex.arg->next->value,0xc0))
  {error(line,txt,"Invalid second argument.");ret=1;}
 break;

 case I_SLA:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_shift_rot(lex.arg->type,lex.arg->value,0x20);
 if (a==1){error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2){error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_SRA:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_shift_rot(lex.arg->type,lex.arg->value,0x28);
 if (a==1){error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2){error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_SRL:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_shift_rot(lex.arg->type,lex.arg->value,0x38);
 if (a==1){error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2){error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_SUB:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_logical(lex.arg->type,lex.arg->value,0x90);
 if (a==1) {error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2) {error(line,txt,"Value out of range.");ret=1;break;}
 if (a==3) {error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_XOR:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_logical(lex.arg->type,lex.arg->value,0xa8);
 if (a==1) {error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2) {error(line,txt,"Value out of range.");ret=1;break;}
 if (a==3) {error(line,txt,"Offset out of range.");ret=1;break;}
 break;

 case I_JP:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (!lex.arg->next)
  {
  if (c_jp(lex.arg->type,lex.arg->value,A_EMPTY,0))
   {error(line,txt,"Invalid argument.");ret=1;break;}
  }
 else 
  {
  if (lex.arg->next->next){error(line,txt,"Too many arguments.");ret=1;break;}
  if (c_jp(lex.arg->type,lex.arg->value,lex.arg->next->type,lex.arg->next->value))
   {error(line,txt,"Invalid argument.");ret=1;break;}
  }
 break;
 
 case I_CALL:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (!lex.arg->next)
  {
  if (c_call(lex.arg->type,lex.arg->value,A_EMPTY,0))
   {error(line,txt,"Invalid argument.");ret=1;break;}
  }
 else 
  {
  if (lex.arg->next->next){error(line,txt,"Too many arguments.");ret=1;break;}
  if (c_call(lex.arg->type,lex.arg->value,lex.arg->next->type,lex.arg->next->value))
   {error(line,txt,"Invalid argument.");ret=1;break;}
  }
 break;
 
 case I_JR:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (!lex.arg->next)
  a=c_jr(lex.arg->type,lex.arg->value,A_EMPTY,0,lex.arg->label);
 else 
  {
  if (lex.arg->next->next){error(line,txt,"Too many arguments.");ret=1;break;}
  a=c_jr(lex.arg->type,lex.arg->value,lex.arg->next->type,lex.arg->next->value,lex.arg->next->label);
  }
 if(a==1){error(line,txt,"Invalid argument.");ret=1;break;}
 if(a==2){error(line,txt,"Value out of range.");ret=1;break;}
 break;
 
 case I_DJNZ:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_djnz(lex.arg->type,lex.arg->value,lex.arg->label);
 if(a==1){error(line,txt,"Invalid argument.");ret=1;break;}
 if(a==2){error(line,txt,"Value out of range.");ret=1;break;}
 break;
 
 case I_LD:
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 if (!lex.arg->next){error(line,txt,"Missing argument.");ret=1;break;}
 if (lex.arg->next->next){error(line,txt,"Too many arguments.");ret=1;break;}
 a=c_ld(lex.arg->type,lex.arg->value,lex.arg->next->type,lex.arg->next->value);
 if (a==1){error(line,txt,"Invalid argument.");ret=1;break;}
 if (a==2){error(line,txt,"Value out of range.");ret=1;break;}
 break;

 case I_DEFS:
 if (disable_defx){error(line,txt,"Forbidden instruction.");ret=1;break;}
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 t=lex.arg;
 do
 {
  unsigned a;
  if (t->type!=A_STRING){error(line,txt,"Invalid argument. String expected.");ret=1;break;}
  for (a=0;a<strlen(t->text);a++)
   out(t->text[a]);
  free(t->text);
  t=t->next;
 }
 while (t);
 break;

 case I_DEFB:
 if (disable_defx){error(line,txt,"Forbidden instruction.");ret=1;break;}
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 t=lex.arg;
 do
 {
 if (t->type!=A_NUM){error(line,txt,"Invalid argument. Number expected.");ret=1;break;}
 if (t->value>255||t->value<-128){error(line,txt,"Value out of range.");ret=1;break;}
 out(t->value);
 t=t->next;
 }
 while (t);
 break;
 
 case I_DEFW:
 if (disable_defx){error(line,txt,"Forbidden instruction.");ret=1;break;}
 if (!lex.arg){error(line,txt,"Missing argument.");ret=1;break;}
 t=lex.arg;
 do
 {
 if (t->type!=A_NUM){error(line,txt,"Invalid argument. Number expected.");ret=1;break;}
 if (t->value>65535||t->value<-32768){error(line,txt,"Value out of range.");ret=1;break;}
 out(t->value&255);out((t->value>>8)&255);
 t=t->next;
 }
 while (t);
 break;
 
 default:
 error(line,txt,"Instruction not implemented.");
 ret=1;
 break;
 }
line++;

if (lex.arg)
 {
 t=lex.arg;
 while((t=t->next))
  free(t->previous);
 free(t);
 }

return ret;
}
